/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <string.h>
#include <IV-look/kit.h>
#include <IV-look/button.h>
#include <IV-look/choice.h>
#include <IV-look/telltale.h>
#include <IV-look/menu.h>
#include <InterViews/layout.h>
#include <InterViews/session.h>
#include <InterViews/rule.h>
#include <InterViews/hit.h>
#include <InterViews/observe.h>
#include <InterViews/border.h>
#include <InterViews/patch.h>
#include <InterViews/deck.h>
#include <InterViews/page.h>
#include <InterViews/glyph.h>
#include <InterViews/font.h>
#include <InterViews/color.h>
#include <OS/string.h>

#include "texte.h"
#include "global.h"
#include "regel_tab.h"

/*
 *------------------------------------------------------------------------------
 *-------------- Deklaration und Definition der Klasse : 'Tabelle' -------------
 *------------------------------------------------------------------------------
 */

declarePtrList(ZustandListe,TelltaleState)
implementPtrList(ZustandListe,TelltaleState)

enum FeldArt { XLeiste, YLeiste, TabFeld, TabKopf };

/*
 *------------------------------------------------------------------------------
 * Klasse : Tabelle
 * Zweck  : Erzeugt eine Tabelle mit gegebener Anzahl von Spalten und Zeilen.
 *          Ausserdem werden Funktionen definiert, die Texte in einzelne
 *          Tabellen-Felder schreiben, und es ist moeglich Callbacks anzugeben,
 *          die ausgefuehrt werden, wenn der Mauszeiger ein Tabellen-Feld
 *          betritt oder anklickt.
 *------------------------------------------------------------------------------
 */
class Tabelle : public InputHandler, public Observer
{
  public : Tabelle(int spalte, int zeile, Action* move, Action* press, Style*);
           ~Tabelle();
           virtual void move(const Event&);
           virtual void press(const Event&);
           void eintragen(FeldArt, int spalte, int zeile, String text);
           void alle_loeschen();
           void markiert(FeldArt&, int& spalte, int& zeile);
           void markiere(FeldArt, int spalte, int zeile);
           void lingvar_menue(Menu*);
           void fuzzyset_menue(Menu*);

  // lokale Funktionen
  private: Glyph* _tab_feld(FeldArt, NfcLabel*);   // erzeugt ein Tabellen-Feld

  // Grafikelemente
  private: NfcLabel **_x_leiste,                   // Text-Labels der Felder
                    **_y_leiste,
                    **_tabelle;
           Action *_move_callback,                 // Callbacks
                  *_press_callback;
           Glyph* _tab_glyph;                      // Tabellen-Grafik
           Page* _blatt;                           // Hintergrund
           WidgetKit* _wkit;
           LayoutKit* _lkit;
           Coord _feld_breite;
           const Color* _markierung;

  // Zustands-Variablen
  private: ZustandListe _liste;                    // Tabellen-Feld Zustaende
           int _anz_spalten,                       // Azahl der Spalten und
               _anz_zeilen;                        // Zeilen in der Tabelle
           GlyphIndex _markiert,                   // welches Feld ist markiert?
                      _ausgewaehlt;                // in welchem ist die Maus?
           boolean _mit_fuz_menue;
};

Tabelle::Tabelle(int spalten,
                 int zeilen,
                 Action* move_callback,
                 Action* press_callback,
                 Style* style)
  : InputHandler(nil, style)
{
  _wkit = WidgetKit::instance();
  _lkit = LayoutKit::instance();

  _anz_spalten    = spalten;
  _anz_zeilen     = zeilen;
  _markiert       = -1;
  _ausgewaehlt    = -1;
  _mit_fuz_menue  = false;
  _move_callback  = move_callback;
  _press_callback = press_callback;
  Resource::ref(_move_callback);
  Resource::ref(_press_callback);

  String name;
  Display* display = Session::instance()->default_display();
  _wkit->style()->find_attribute("Markierung", name);
  _markierung = Color::lookup(display, name);

  // Feldbreite berechnen (fuer 7 Zeichen)
  FontBoundingBox font_box;
  _wkit->font()->font_bbox(font_box);
  _feld_breite = 7 * (font_box.left_bearing() + font_box.right_bearing());

  // die einzelnen Felder fuer die Texte in der Tabelle erzeugen
  _x_leiste = new NfcLabel*[_anz_spalten];
  _y_leiste = new NfcLabel*[_anz_zeilen];
  _tabelle  = new NfcLabel*[_anz_spalten * _anz_zeilen];

  for(int spalte = 0; spalte < _anz_spalten; spalte++)
    _x_leiste[spalte] = new NfcLabel(LeerStr, _wkit);
  for(int zeile = 0; zeile < _anz_zeilen; zeile++)
    _y_leiste[zeile] = new NfcLabel(LeerStr, _wkit);
  for(int feld = 0; feld < _anz_spalten * _anz_zeilen; feld++)
    _tabelle[feld] = new NfcLabel(LeerStr, _wkit);

  // die einzelnen Felder zu einer Tabelle zusammensetzen
  _tab_glyph       = _lkit->vbox((_anz_spalten + 1) * (_anz_zeilen + 1) + 2);
  Glyph* tab_zeile = _lkit->hbox(_anz_spalten + 1);

  tab_zeile->append(_tab_feld(TabKopf, new NfcLabel(LeerStr, _wkit)));

  // X-Leiste der Tabelle erstellen
  for(spalte = 0; spalte < _anz_spalten; spalte++)
    tab_zeile->append(_tab_feld(XLeiste, _x_leiste[spalte]));
  _tab_glyph->append(_lkit->vcenter(tab_zeile));

  // Eigentliche Tabelle und deren Y-Leiste erstellen
  for(zeile = 0; zeile < _anz_zeilen; zeile++) {
    tab_zeile = _lkit->hbox(_anz_spalten + 1);
    tab_zeile->append(_tab_feld(YLeiste, _y_leiste[zeile]));
    for(spalte = 0; spalte < _anz_spalten; spalte++)
      tab_zeile->append(
        _tab_feld(TabFeld, _tabelle[zeile * _anz_spalten + spalte])
      );
    _tab_glyph->append(tab_zeile);
  }
  _blatt = new Page(_tab_glyph);
  body(_blatt);
}

Tabelle::~Tabelle()
{
  delete[] _x_leiste;
  delete[] _y_leiste;
  delete[] _tabelle;
  Resource::unref(_move_callback);
  Resource::unref(_press_callback);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : void Tabelle::move(Event&)
 * Parameter : Event& - Ereignis
 * Zweck : Registriert Mausbewegung und stellt fest, ob sich der Zeiger
 *         innerhalb eines Tabellen-Feldes befindet. Falls ja, und das Feld
 *         ist noch nicht ausgewaehlt, wird der Callback ausgefuehrt und das
 *         Feld gilt als ausgewaehlt.
 *------------------------------------------------------------------------------
 */
void Tabelle::move(const Event& e)
{
  GlyphIndex spalte, zeile, feld_nr;
  spalte = zeile = feld_nr = -1;
  Hit h(&e);

  repick(2, h);
  if (h.any()) {
    zeile   = h.index(2); 
    spalte  = h.index(3); 
    feld_nr = zeile * (_anz_spalten + 1) + spalte;
    if(feld_nr != _ausgewaehlt) {
      _ausgewaehlt = feld_nr;
      if(_move_callback != nil)
        _move_callback->execute();
    }
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : void Tabelle::press(Event&)
 * Parameter : Event& - Ereignis
 * Zweck : Registriert Tastendruck der Maus und fuehrt Callback aus.
 *------------------------------------------------------------------------------
 */
void Tabelle::press(const Event&)
{
  if(_press_callback != nil)
    _press_callback->execute();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : void Tabelle::eintragen(FeldArt, int, int, String)
 * Parameter : FeldArt    - Art des Tabellen-Feldes
 *             int spalte - Tabellenspalte
 *             int zeile  - Tabellenzeile
 *             String     - Text
 * Zweck : Traegt Text in Tabellen-Feld ein.
 *------------------------------------------------------------------------------
 */
void Tabelle::eintragen(FeldArt art, int spalte, int zeile, String eintrag)
{
  if(spalte >= 0 && spalte < _anz_spalten &&
     zeile  >= 0 && zeile  < _anz_zeilen)  {
    switch(art) {
      case XLeiste:
        _x_leiste[spalte]->text(eintrag);
        break;
      case YLeiste:
        _y_leiste[zeile]->text(eintrag);
        break;
      case TabFeld:
        _tabelle[zeile * _anz_spalten + spalte]->text(eintrag);
        break;
      case TabKopf:
        break;
    }
    redraw();
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : void Tabelle::alle_loeschen()
 * Zweck : Loescht alle Texte in den Tabellen-Feldern.
 *------------------------------------------------------------------------------
 */
void Tabelle::alle_loeschen()
{
  int anzahl = _anz_spalten * _anz_zeilen;
  for(int i = 0; i < anzahl; i++)
    _tabelle[i]->text(LeerStr);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : void Tabelle::markiert(FeldArt&, int&, int&)
 * Parameter : FeldArt     - Art des Tabellen-Feldes
 *             int& spalte - Tabellenspalte
 *             int& zeile  - Tabellenzeile
 * Zweck : belegt Parameter mit Werten des momentan markierten Feldes.
 *------------------------------------------------------------------------------
 */
void Tabelle::markiert(FeldArt& art, int& spalte, int& zeile)
{
  if(_ausgewaehlt == -1) {
    art    = TabKopf;
    spalte = zeile = 0;
  } else {
    spalte = (int) _ausgewaehlt % (_anz_spalten + 1);
    zeile  = (int) _ausgewaehlt / (_anz_spalten + 1);
    art    = TabKopf;
    if(spalte != 0 || zeile != 0) {
      if(spalte == 0)
        art = YLeiste;
      else
        spalte--;
      if(zeile == 0)
        art = XLeiste;
      else
        zeile--;
      if(art == TabKopf)
        art = TabFeld;
    }
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : void Tabelle::markiere(FeldArt, int, int)
 * Parameter : FeldArt    - Art des Tabellen-Feldes
 *             int spalte - Tabellenspalte
 *             int zeile  - Tabellenzeile
 * Zweck : markiert angegebenes Tabellen-Fald
 *------------------------------------------------------------------------------
 */
void Tabelle::markiere(FeldArt art, int spalte, int zeile)
{
  GlyphIndex nr;
  switch(art) {
    case XLeiste:
      nr = spalte + 1;
      break;
    case YLeiste:
      nr = (zeile + 1) * (_anz_spalten + 1);
      break;
    case TabFeld:
      nr = (zeile + 1) * (_anz_spalten + 1) + spalte + 1;
      break;
    case TabKopf:
      nr = 0;
      break;
  }

  TelltaleState* t;
  if(_markiert != -1) {
    t = _liste.item(_markiert);  // alte Markierung loeschen
    t->attach(this);
    t->set(TelltaleState::is_active, false);
    t->detach(this);
  }

  t = _liste.item(nr);           // neue Markierung setzen
  t->attach(this);
  t->set(TelltaleState::is_active, true);
  t->detach(this);

  // Menue positionieren
  if(_mit_fuz_menue) {
    GlyphIndex menue_nr = _blatt->count() - 1;   // Nr. des Fuzzyset-Menues
    if(art == TabFeld) {
      spalte++;
      zeile++;
      Allotment allox, alloy;
      _tab_glyph->component(zeile)->allotment(spalte, Dimension_X, allox);
      _tab_glyph->allotment(zeile, Dimension_Y, alloy);
      _blatt->move(
        menue_nr, allox.end() - _blatt->x(), alloy.end() - _blatt->y()
      );
      if(!_blatt->showing(menue_nr))
        _blatt->show(menue_nr, true);
    } else
      _blatt->show(menue_nr, false);
  }

  _ausgewaehlt = _markiert = nr;
  redraw();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : void Tabelle::lingvar_menue(Menu*)
 * Parameter : Menu* - Menue
 * Zweck : Installiert im Tabellen-Kopf ein Menue. Mit Hilfe dieser Funktion
 *         wird in der RegelTabelle ein Menue installiert, mit dem festgestellt
 *         wird, welche ling. Ausgabe-Variable ausgewaehlt wurde (falls es
 *         mehrere gibt).
 *------------------------------------------------------------------------------
 */
void Tabelle::lingvar_menue(Menu* menue)
{
  LayoutKit* lkit = LayoutKit::instance();

  _blatt->insert(0, lkit->center(menue, 0, 1));
  _blatt->move(0, 0, 0);
  _blatt->show(0, true);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : void Tabelle::fuzzyset_menue(Menu*)
 * Parameter : Menu* - Menue 
 * Zweck : Installiert im jeweils aktuellen Tabellen-Feld ein Menue.
 *         Mit Hilfe dieser Funktion wird in der RegelTabelle ein Menue
 *         installiert, mit dem sich Fuzzy-Sets auswaehlen lassen.
 *------------------------------------------------------------------------------
 */
void Tabelle::fuzzyset_menue(Menu* menue)
{
  LayoutKit* lkit = LayoutKit::instance();

  if(_mit_fuz_menue)
    _blatt->replace(_blatt->count() - 1, lkit->center(menue, 1, 1));
  else
    _blatt->append(lkit->center(menue, 1, 1));
  _mit_fuz_menue = true;
  _blatt->show(_blatt->count() - 1, false);
}

/*
 *-------------------------- lokale Funktionen ---------------------------------
 */

Glyph* Tabelle::_tab_feld(FeldArt art, NfcLabel* label)
{
  TelltaleState* status = new TelltaleState(TelltaleState::is_enabled);
  _liste.append(status);
  Glyph *normal,
        *markiert;

  Coord rand_dick  = 3;     // Breite der Umrandung im markierten Fall
  Coord rand_duenn = 1;     // Breite der Umrandung im normalen Fall
  Coord abstand    = 5;     // Abstand Text <-> Umrandung

  Coord rechts = (art == YLeiste || art == TabKopf) ? rand_dick : rand_duenn;
  Coord unten  = (art == XLeiste || art == TabKopf) ? rand_dick : rand_duenn;

  Glyph* basis = _lkit->hbox(
                   _lkit->vbox(
                     _lkit->vcenter(_lkit->margin(label, abstand, abstand)),
                     _lkit->hglue(_feld_breite),
                     new HRule(_wkit->foreground(), unten)
                   ), new VRule(_wkit->foreground(), rechts)
                 );
  markiert = new Border(basis, _markierung, rand_dick);

  switch(art) {
    case XLeiste:
      normal = _lkit->vbox(
                 _lkit->vcenter(new HRule(_wkit->foreground(), 1)),
                 basis
               );
      break;
    case YLeiste:
      normal = _lkit->hbox(
                 new VRule(_wkit->foreground(), 1),
                 basis
               );
      break;
    case TabKopf:
      normal = _lkit->hbox(
                 new VRule(_wkit->foreground(), 1),
                 _lkit->vbox(
                   _lkit->vcenter(new HRule(_wkit->foreground(), 1)),
                   basis
                 )
               );
      break;
    case TabFeld:
      normal = basis;
      break;
  }
  return new ChoiceItem(status, normal, markiert);
}


/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : RegelTabMaske --------------------
 *------------------------------------------------------------------------------
 */

declareActionCallback(RegelTabMaske);
implementActionCallback(RegelTabMaske);

RegelTabMaske::RegelTabMaske(const LingVarSatz* lingvarsatz)
{
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();

  _lingvarsatz     = lingvarsatz;
  _maske           = new Regel(
                       _lingvarsatz->anz_lingvars(EingabeVar),
                       _lingvarsatz->anz_lingvars(AusgabeVar)
                     );
  _menue_balken    = nil;
  _lingvar_auswahl = _fuzzyset_auswahl = -1;

  // ling. Variablen stehen in der Maske in der Reihenfolge ihres
  // Auftretens im 'lingvarsatz'; jeweils fuer Praemisse und Konklusion.
  // zu Beginn sind alle Bedingungen noch 'leer'
  int anzahl = _lingvarsatz->anz_lingvars(EingabeVar);
  for(int i = 0; i < anzahl; i++)
    _maske->bedingung(
      Praemisse, i, lingvarsatz->hol_lingvar(EingabeVar, i)->name(), LeerStr
    );

  anzahl = _lingvarsatz->anz_lingvars(AusgabeVar);
  for(i = 0; i < anzahl; i++)
    _maske->bedingung(
      Konklusion, i, lingvarsatz->hol_lingvar(AusgabeVar, i)->name(), LeerStr
    );

  if(lingvarsatz->anz_lingvars(EingabeVar) > 2) {
    _menue_balken = wkit->menubar();
    _mach_menue_balken(EingabeVar, 0);
  }
  if(lingvarsatz->anz_lingvars(AusgabeVar) > 1) {
    if(_menue_balken == nil)
      _menue_balken = wkit->menubar();
    _mach_menue_balken(AusgabeVar, lingvarsatz->anz_lingvars(EingabeVar));
  }

  if(_menue_balken != nil)
    _glyph = lkit->variable_span(_menue_balken, 0, 0);
  else
    _glyph = lkit->vglue(0, 0, 0);
}

RegelTabMaske::~RegelTabMaske() { delete _maske; }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabMaske::maske()
 * Rueckgabewert : Regel, die der Belegung der einzelnen linguistischen
 *                 Variablen entspricht.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
Regel* RegelTabMaske::maske() const { return _maske; }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabMaske::setz_maske(LingVar*, int)
 * Parameter : LingVar* - ling. Variable
 *             int      - Nummer eines Fuzzy-Sets
 * Zweck : setzt in der Maske die angegebene ling. Variable auf das Fuzzy-Set.
 *------------------------------------------------------------------------------
 */
void RegelTabMaske::setz_maske(LingVar* lingvar, int fuzzyset_nr)
{
  if(fuzzyset_nr >= 0 && fuzzyset_nr < lingvar->anz_fuzzysets()) {
    LingVarTyp typ = lingvar->typ();
    RegelAusdruck ausdruck = (typ == EingabeVar) ? Praemisse : Konklusion;
    String var_name = lingvar->name();
    String fuz_name = lingvar->hol_fuzzyset(fuzzyset_nr)->name();

    // Maske entsprechend aktualisieren
    boolean fertig;
    int i = _maske->groesse(ausdruck);
    do {
      fertig = _lingvarsatz->hol_lingvar(typ, --i)->name() == var_name;
      fertig = fertig || i < 0;
    } while(!fertig);
    if(i >= 0) {
      _maske->bedingung(ausdruck, i, var_name, fuz_name);

      // testen, ob Menue fuer Variable existiert und seine Nr feststellen
      int menue_nr = i;
      boolean mit_menue = true;
      if(_menue_balken == nil)
        mit_menue = false;
      else {
        int anz_pra = _lingvarsatz->anz_lingvars(EingabeVar);
        int anz_kon = _lingvarsatz->anz_lingvars(AusgabeVar);
        if(typ == EingabeVar && anz_pra <= 2)
          mit_menue = false;
        if(typ == AusgabeVar) {
          if(anz_pra <= 1)
            mit_menue = false;
          if(anz_pra > 2)
            menue_nr += anz_pra;
        }
      }
      
      // Menue entsprechend aktualisieren
      if(mit_menue)  {
        fuzzyset_nr += 2;      // ersten Menue-Eintraege sind keine Fuzzysets
        if(menue_nr < _menue_balken->item_count()) {
          Menu* menue = _menue_balken->item(menue_nr)->menu();
          if(menue->item_count() > fuzzyset_nr)
            menue->item(fuzzyset_nr)->state()->set(
              TelltaleState::is_chosen, true
            );
        }
      }
    }
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabMaske::masken_glyph()
 * Rueckgabewert : der RegelTabMaske entsprechendes Grafikelement
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
Glyph* RegelTabMaske::masken_glyph() { return _glyph; }

void RegelTabMaske::_callback()
{
  if(_lingvar_auswahl >= 0 && _fuzzyset_auswahl >= 0) {
    LingVarTyp typ = EingabeVar;
    int lingvar_nr = _lingvar_auswahl;

    if(_lingvar_auswahl >= _lingvarsatz->anz_lingvars(EingabeVar)) {
      typ = AusgabeVar;
      lingvar_nr = _lingvar_auswahl - _lingvarsatz->anz_lingvars(EingabeVar);
    }
    LingVar* lingvar = _lingvarsatz->hol_lingvar(typ, lingvar_nr);
    String lingvar_name, fuzzyset_name;

    if(_fuzzyset_auswahl >= lingvar->anz_fuzzysets())
      fuzzyset_name = LeerStr;
    else
      fuzzyset_name = lingvar->hol_fuzzyset(_fuzzyset_auswahl)->name();
    RegelAusdruck ausdruck = (typ == EingabeVar) ? Praemisse : Konklusion;
    _maske->bedingung(ausdruck, lingvar_nr, lingvar->name(), fuzzyset_name);
  }
  neue_maske();
}

/*
 *------------------------- lokale Funktionen ----------------------------------
 */

void RegelTabMaske::_mach_menue_balken(LingVarTyp typ, int lingvar_nr)
{
  Action* cb =
     new ActionCallback(RegelTabMaske)(this,&RegelTabMaske::_callback);
  MenuItem* eintrag;

  int anzahl = _lingvarsatz->anz_lingvars(typ);
  for(int i = 0; i < anzahl; i++) {
    LingVar* lingvar = _lingvarsatz->hol_lingvar(typ, i);
    if(lingvar->anz_fuzzysets() > 0) {
      eintrag = _mach_menue(cb, lingvar);
      eintrag->action(new NfcAction(&_lingvar_auswahl, lingvar_nr++, nil));
      _menue_balken->append_item(eintrag);
    }
  }
}

MenuItem* RegelTabMaske::_mach_menue(Action* cb, LingVar* lingvar)
{
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();

  Menu* menue = wkit->pulldown();
  TelltaleGroup* group = new TelltaleGroup;
  MenuItem* eintrag;
  int anzahl = lingvar->anz_fuzzysets();

  eintrag = wkit->radio_menu_item(group, wkit->label(Txt(45)));
  eintrag->action(new NfcAction(&_fuzzyset_auswahl, anzahl, cb));
  eintrag->state()->set(TelltaleState::is_chosen,true);
  menue->append_item(eintrag);
  menue->append_item(wkit->menu_item_separator());

  for(int i = 0; i < anzahl; i++) {
    eintrag = wkit->radio_menu_item(
                group, wkit->label(lingvar->hol_fuzzyset(i)->name())
              );
    eintrag->action(new NfcAction(&_fuzzyset_auswahl, i, cb));
    menue->append_item(eintrag);
  }
  eintrag = wkit->menubar_item(wkit->label(lingvar->name()));
  eintrag->menu(menue);
  return eintrag;
}


/*
 *------------------------------------------------------------------------------
 *-------------------- Definition der Klasse : RegelTabelle --------------------
 *------------------------------------------------------------------------------
 */

declareActionCallback(RegelTabelle);
implementActionCallback(RegelTabelle);

RegelTabelle::RegelTabelle(
                const RegelBasis* regeln, const LingVarSatz* lingvars
              ) : RegelTabMaske(lingvars)
{
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  NfcKit* nfc_kit = NfcKit::instance();
  Action *cb1, *cb2;

  _regelbasis    = regeln;
  _lingvarsatz   = lingvars;
  _menue_auswahl = 0;

  // zuerst maximale Anzahl darzustellender Fuzzy-Sets bestimmen ...
  _max_fuzzysets = 0;
  int anzahl = _lingvarsatz->anz_lingvars(EingabeVar);
  for(int i = 0; i < anzahl; i++)
    _max_fuzzysets = Max(
                       _max_fuzzysets,
                       _lingvarsatz->hol_lingvar(EingabeVar, i)->anz_fuzzysets()
                     );

  if(_max_fuzzysets > 0) {

    // ... damit dann entsprechende Tabelle erzeugen ...
    cb1 = new ActionCallback(RegelTabelle)(this,&RegelTabelle::_move_callback);
    cb2 = new ActionCallback(RegelTabelle)(this,&RegelTabelle::_press_callback);
    _tabelle = new Tabelle(
                 _max_fuzzysets, _max_fuzzysets, cb1, cb2, wkit->style()
               );
    if(_lingvarsatz->anz_lingvars(AusgabeVar) > 1)
      _tabelle->lingvar_menue(_mach_menue());

    cb1 = new ActionCallback(RegelTabelle)(this,&RegelTabelle::_x_leiste_cb);
    _x_leiste_auswahl = new AuswahlButton(wkit, cb1);
    cb1 = new ActionCallback(RegelTabelle)(this,&RegelTabelle::_y_leiste_cb);
    _y_leiste_auswahl = new AuswahlButton(wkit, cb1, Dimension_Y);

    anzahl = _lingvarsatz->anz_lingvars(EingabeVar);
    for(i = 0; i < anzahl; i++) {
      String eintrag = _lingvarsatz->hol_lingvar(EingabeVar, i)->name();
      _x_leiste_auswahl->zur_auswahl(eintrag);
      _y_leiste_auswahl->zur_auswahl(eintrag);
    }
    if(anzahl > 1)
      _y_leiste_auswahl->auswahl_nr(1);
    _x_leiste = _x_leiste_auswahl->auswahl();
    _y_leiste = _y_leiste_auswahl->auswahl();

    // Glyph definieren, der Regeltabelle darstellt
    Coord rand = 10;
    _glyph =  nfc_kit->text_drauf(
                Txt(59),
                lkit->vbox(
                  lkit->vcenter(lkit->hbox(masken_glyph(), lkit->hglue())),
                  lkit->vglue(rand, 0, 0),
                  lkit->hbox(
                    lkit->hglue(0, 0, 0),
                     _x_leiste_auswahl,
                     lkit->hglue(0, 0, 0)
                  ), lkit->vglue(rand, 0, 0),
                  lkit->hbox(
                    lkit->hglue(rand, 0, 0),
                    lkit->vbox(
                      lkit->vglue(0, fil, 0),
                      _y_leiste_auswahl,
                      lkit->vglue(0, fil, 0)
                    ),
                    lkit->hglue(rand, 0, 0),
                    _tabelle,
                    lkit->hglue(rand, 0, 0)
                  ), lkit->vglue(rand, 0, 0)
                ), true
              );
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabelle::zeig_regeln(RegelBasis*)
 * Parameter : RegelBasis* - Regel-Basis
 * Zweck : stellt angegebene Regel-Basis in der Tabelle dar und merkt sich
 *         den Zeiger auf die Regel-Basis.
 *------------------------------------------------------------------------------
 */
void RegelTabelle::zeig_regeln(const RegelBasis* regelbasis)
{
  _regelbasis = regelbasis;

  // zuerst Beschriftung der X/Y-Leiste der Tabelle durchfuehren ...
  int anz_x = _lingvarsatz->hol_lingvar(_x_leiste)->anz_fuzzysets();
  int anz_y = _lingvarsatz->hol_lingvar(_y_leiste)->anz_fuzzysets();
  for(int i = 0; i < _max_fuzzysets; i++) {
    if(i < anz_x)
      _tabelle->eintragen(
        XLeiste, i, 0,
        _lingvarsatz->hol_lingvar(_x_leiste)->hol_fuzzyset(i)->name()
      );
    else
      _tabelle->eintragen(XLeiste, i, 0, LeerStr);
    if(i < anz_y)
      _tabelle->eintragen(
        YLeiste, 0, i,
        _lingvarsatz->hol_lingvar(_y_leiste)->hol_fuzzyset(i)->name()
      );
    else
      _tabelle->eintragen(YLeiste, 0, i, LeerStr);
  }

  // ... dann die Regeln in die Tabellenfelder eintragen
  _tabelle->alle_loeschen();
  if(regelbasis != nil) {
    int anzahl = _regelbasis->anzahl_regeln();
    for(i = 0; i < anzahl; i++)
      zeig_regel(_regelbasis->hol_regel(i));
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabelle::zeig_regel(Regel*)
 * Parameter : Regel* - Regel
 * Zweck : stellt angegebene Regel in der Tabelle dar.
 *------------------------------------------------------------------------------
 */
void RegelTabelle::zeig_regel(Regel* regel)
{
  String kon_name;
  kon_name = _lingvarsatz->hol_lingvar(AusgabeVar, _menue_auswahl)->name();

  int spalte, zeile;
  if(_hol_tab_koo(regel, spalte, zeile))
    _tabelle->eintragen(
      TabFeld, spalte, zeile, regel->fuzzyset(Konklusion, kon_name)
    );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabelle::markiere(int)
 * Parameter : int - Nummer einer Regel
 * Zweck : markiert das Tabellen-Feld, in dem sich die Regel mit der angegebenen
 *         Nummer befindet, falls ein solches Feld gefunden wird.
 *------------------------------------------------------------------------------
 */
void RegelTabelle::markiere(int nr)
{
  if(_regelbasis == nil || nr == markiert())
    return;

  Regel* regel = _regelbasis->hol_regel(nr);
  if(regel != nil) {
    int spalte, zeile;
    if(_hol_tab_koo(regel, spalte, zeile))
      _tabelle->markiere(TabFeld, spalte, zeile);
    else
      _tabelle->markiere(TabKopf, 0, 0);
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabelle::markiert()
 * Rueckgabewert : Nummer der Regel, dessen Tabellen-Feld markiert ist. Falls
 *                 sich keine Regel im markierten Feld befinet : -1.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
int RegelTabelle::markiert()
{
  FeldArt art;
  int regel_nr = -1;
  int spalte, zeile, akt_spalte, akt_zeile;

  _tabelle->markiert(art, spalte, zeile);
  if(art != TabFeld)
    return regel_nr;

  // Ueberpruefen, ob in diesem Tabellenfeld eine Regel dargestellt wird
  if(_regelbasis != nil) {
    int anzahl_regeln = _regelbasis->anzahl_regeln();
    for(int i = 0; regel_nr == -1 && i < anzahl_regeln; i++)
      if(_hol_tab_koo(_regelbasis->hol_regel(i), akt_spalte, akt_zeile))
        if(akt_spalte == spalte && akt_zeile == zeile)
          regel_nr = i;
  }
  return regel_nr;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabelle::glyph()
 * Rueckgabewert : der RegelTabelle entsprechendes Grafikelement.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
Glyph* RegelTabelle::glyph() { return _glyph; }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabelle::neue_maske()
 * Zweck : Callback, das ausgefuehrt wird, wenn sich die Maske aendert. Traegt
 *         die Regeln, die durch die Maske kommen in die Tabelle ein.
 *------------------------------------------------------------------------------
 */
void RegelTabelle::neue_maske() { zeig_regeln(_regelbasis); }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabelle::konklusion()
 * Rueckgabewert : linguistische Variable, die als Konklusion in der Tabelle
 *                 dargestellt wird. Diese kann ueber Menue ausgewaehlt
 *                 werden, wenn mehrere Ausgabe-Variablen existieren.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
LingVar* RegelTabelle::konklusion()
{
  return _lingvarsatz->hol_lingvar(AusgabeVar, _menue_auswahl);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelTabelle::maskiere(Regel*, boolean)
 * Parameter : Regel*  - Regel
 *             boolean - gesamte Regel maskieren ?
 * Rueckgabewert : true  <=> alle Bedingungen, die in der Maske enthalten sind,
 *                           sind auch in der Regel enthalten.
 *                 false <=> sonst
 * Zweck : Ueberpruefen ob Regel durch die Maske kommt. Ueber den 2.ten
 *         Parameter kann angegeben werden, ob auch die Variablen ueberprueft
 *         werden sollen, die momentan in der Tabelle dargestellt werden.
 *------------------------------------------------------------------------------
 */
boolean RegelTabelle::maskiere(Regel* regel, boolean komplett)
{
  if(_x_leiste == _y_leiste)
    return false;

  boolean ok = true;
  RegelAusdruck ausdruck;

  for(ausdruck = Praemisse; ok && ausdruck <= Konklusion; ausdruck++) {
    int anzahl = maske()->groesse(ausdruck);
    for(int i = 0; ok && i < anzahl; i++) {
      Bedingung* bedingung = maske()->bedingung(ausdruck, i);

      boolean ueberpruefen = true;
      if(ausdruck == Konklusion)
        ueberpruefen = bedingung->lingvar() != konklusion()->name();
      else
        if(!komplett) {
          if(bedingung->lingvar() == _x_leiste)
            ueberpruefen = false;
          if(bedingung->lingvar() == _y_leiste)
            ueberpruefen = false;
        }

      if(ueberpruefen) {
        if(bedingung->fuzzyset() == LeerStr)
          ok = !regel->enthalten(ausdruck, bedingung->lingvar());
        else
          ok = regel->enthalten(ausdruck, bedingung);
      } else
        ok = regel->enthalten(ausdruck, bedingung->lingvar());
    }
  }
  return ok;
}

/*
 *------------------------- lokale Funktionen ----------------------------------
 */

void RegelTabelle::_move_callback()
{
  if(_x_leiste == _y_leiste)
    return;

  int spalte, zeile;
  FeldArt art;
  _tabelle->markiert(art, spalte, zeile);

  int anz_x = _lingvarsatz->hol_lingvar(_x_leiste)->anz_fuzzysets();
  int anz_y = _lingvarsatz->hol_lingvar(_y_leiste)->anz_fuzzysets();
  if(spalte < anz_x && zeile < anz_y) {
    if(art == TabFeld) {
      setz_maske(_lingvarsatz->hol_lingvar(_x_leiste), spalte);
      setz_maske(_lingvarsatz->hol_lingvar(_y_leiste), zeile);
    }
    _tabelle->markiere(art, spalte, zeile);
  }
}

void RegelTabelle::_press_callback()
{
  int regel_nr = markiert();
  if(regel_nr != -1 && angebunden() != nil)
    angebunden()->markiere(regel_nr);
}

void RegelTabelle::_neue_leiste(DimensionName dim)
{
  String auswahl;
  if(dim == Dimension_X) {
    auswahl = _x_leiste_auswahl->auswahl();
    if(auswahl != _x_leiste) {
      _x_leiste = auswahl;
      zeig_regeln(_regelbasis);
    }
  } else {
    auswahl = _y_leiste_auswahl->auswahl();
    if(auswahl != _y_leiste) {
      _y_leiste = auswahl;
      zeig_regeln(_regelbasis);
    }
  }
}

void RegelTabelle::_neue_konklusion() { zeig_regeln(_regelbasis); }

Menu* RegelTabelle::_mach_menue()
{
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();

  TelltaleGroup* group = new TelltaleGroup;
  Menu* menue = wkit->pulldown();
  MenuItem* eintrag;
  Action* cb;

  cb = new ActionCallback(RegelTabelle)(this,&RegelTabelle::_neue_konklusion);
  int anzahl = _lingvarsatz->anz_lingvars(AusgabeVar);
  for(int i = 0; i < anzahl; i++) {
    eintrag = wkit->radio_menu_item(
                group,
                lkit->hcenter(
                  wkit->label(_lingvarsatz->hol_lingvar(AusgabeVar, i)->name())
                )
              );
    if(i == _menue_auswahl)
      eintrag->state()->set(TelltaleState::is_chosen,true);
    eintrag->action(new NfcAction(&_menue_auswahl, i, cb));
    menue->append_item(eintrag);
  }

  eintrag = wkit->menubar_item(lkit->fixed_span(wkit->label("o"), 5, 5));
  eintrag->menu(menue);
  Menu* menue_balken = wkit->menubar();
  menue_balken->append_item(eintrag);

  return menue_balken;
}

boolean RegelTabelle::_hol_tab_koo(Regel* regel, int& spalte, int& zeile)
{
  if(maskiere(regel, false))
    if(regel->enthalten(Konklusion, konklusion()->name())) {
    
      String fuz_name1 = regel->fuzzyset(Praemisse, _x_leiste);
      String fuz_name2 = regel->fuzzyset(Praemisse, _y_leiste);
      if(fuz_name1 != LeerStr && fuz_name2 != LeerStr) {
        boolean gefunden = false;
        LingVar* var_x = _lingvarsatz->hol_lingvar(_x_leiste);
        for(int x = 0; !gefunden && x < var_x->anz_fuzzysets(); x++)
          if(var_x->hol_fuzzyset(x)->name() == fuz_name1) {
            gefunden = true;
            spalte = x;
          }
        gefunden = false;
        LingVar* var_y = _lingvarsatz->hol_lingvar(_y_leiste);
        for(int y = 0; !gefunden && y < var_y->anz_fuzzysets(); y++)
          if(var_y->hol_fuzzyset(y)->name() == fuz_name2) {
            gefunden = true;
            zeile = y;
          }
        return spalte != var_x->anz_fuzzysets() &&
                zeile != var_y->anz_fuzzysets();
      }
    }
  return false;
}

/*
 *------------------------------------------------------------------------------
 *-------------------- Definition der Klasse : GrafRegelEd ---------------------
 *------------------------------------------------------------------------------
 */

declareActionCallback(GrafRegelEd);
implementActionCallback(GrafRegelEd);

GrafRegelEd::GrafRegelEd(
               NfcAusgabe* ausgabe, RegelBasis* regeln, LingVarSatz* lingvars
             )
  : RegelTabelle(regeln, lingvars)
{
  ausgabe_feld = ausgabe;
  _regelbasis  = regeln;

  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();

  // Menue fuer Konklusion erzeugen
  _akt_regel     = -1;
  _menue_auswahl = -1;
  _cb_ptr = new ActionCallback(GrafRegelEd)(this,&GrafRegelEd::_callback);
  _menue_balken = wkit->menubar();
  _menue_balken->append_item(_mach_menue(konklusion()));
  tabelle()->fuzzyset_menue(_menue_balken);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafRegelEd::markiere(int)
 * Parameter : int - Nummer einer Regel
 * Zweck : Markiert Regel mit der angegebenen Nummer, falls sich die ling.
 *         Variable in der Konklusion geaendert hat, wird das Menue zur Auswahl
 *         der Fuzzy-Sets ausgetauscht.
 *------------------------------------------------------------------------------
 */
void GrafRegelEd::markiere(int regel_nr)
{
  if(_konklusion == nil || _konklusion != konklusion())
    _menue_balken->replace_item(0, _mach_menue(konklusion()));
  _akt_regel = regel_nr;
  if(regel_nr != -1)
    RegelTabelle::markiere(regel_nr);
}

/*
 *------------------------- lokale Funktionen ----------------------------------
 */

boolean GrafRegelEd::_parse_regel(Regel& regel, const LingVarSatz* lingvarsatz)
{
  // Anzahl der Variablen in Praemisse und Konklusion bestimmen
  int anz_pra = 0;
  int anz_kon = 0;
  for(RegelAusdruck ausdruck = Praemisse; ausdruck <= Konklusion; ausdruck++) {
    int anzahl = regel.groesse(ausdruck);
    for(int i = 0; i < anzahl; i++)
      if(regel.bedingung(ausdruck, i)->fuzzyset() != LeerStr) {
        if(ausdruck == Praemisse)
          anz_pra++;
        else
          anz_kon++;
      }
  }
  if(anz_pra > 0 && anz_kon > 0) {
    boolean ok = true;
    Regel neue_regel(anz_pra, anz_kon);
    for(ausdruck = Praemisse; ok && ausdruck <= Konklusion; ausdruck++) {
      int pos = 0;
      int anzahl = regel.groesse(ausdruck);
      for(int i = 0; ok && i < anzahl; i++) {
        ok = true;
        Bedingung* bedingung = regel.bedingung(ausdruck, i);
        if(bedingung->fuzzyset() != LeerStr) {
          LingVar* lingvar = lingvarsatz->hol_lingvar(bedingung->lingvar());
          if(lingvar != nil)
            if(lingvar->hol_fuzzyset(bedingung->fuzzyset()) != nil)
              neue_regel.bedingung(
                ausdruck, pos++, bedingung->lingvar(), bedingung->fuzzyset()
              );
            else
              ok = false;
        }
      }
    }
    regel = neue_regel;
    return ok;
  } else
    return false;
}

void GrafRegelEd::_callback()
{
  Regel regel;

  if(_menue_auswahl == konklusion()->anz_fuzzysets()) {

    // Loeschen einer Regel, also ->
    // zuerst Regel suchen
    boolean gefunden = false;
    int anzahl = _regelbasis->anzahl_regeln();
    for(int i = 0; !gefunden && i < anzahl; i++) {
      regel = *_regelbasis->hol_regel(i);
      if(maskiere(&regel, true))
        gefunden = true;
    }
    // falls gefunden : loeschen
    if(gefunden) {
      _regelbasis->del_regel(&regel);
      ausgabe_feld->ausgeben(Meldung, Txt(57));     // Regel geloescht
      zeig_regeln(_regelbasis);
      if(angebunden() != nil)
        angebunden()->zeig_regeln(_regelbasis);
    } else
      ausgabe_feld->ausgeben(Fehler, Txt(56));      // nicht gefunden
  } else {

    // Setzen einer Regel
    setz_maske(konklusion(), _menue_auswahl);
    regel = *maske();
    if(_parse_regel(regel, lingvarsatz())) {
      _regelbasis->add_regel(&regel);
      ausgabe_feld->ausgeben(Meldung, Txt(58));     // Regel gesetzt
      zeig_regel(&regel);
      if(angebunden() != nil)
        angebunden()->zeig_regeln(_regelbasis);
    }
  }
}

MenuItem* GrafRegelEd::_mach_menue(LingVar* lingvar)
{
  _konklusion = lingvar;

  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();

  Menu* menue = wkit->pulldown();
  MenuItem* eintrag;

  // erster Eintrag ist der Name der ling. Variablen
  eintrag = wkit->menu_item(lkit->hcenter(wkit->label(lingvar->name())));
  menue->append_item(eintrag);
  menue->append_item(wkit->menu_item_separator());

  // als Naechstes folgen die Namen der Fuzzy-Sets
  int anzahl = lingvar->anz_fuzzysets();
  for(int i = 0; i < anzahl; i++) {
    eintrag = wkit->menu_item(
                lkit->hcenter(wkit->label(lingvar->hol_fuzzyset(i)->name()))
              );
    eintrag->action(new NfcAction(&_menue_auswahl, i, _cb_ptr));
    menue->append_item(eintrag);
  }

  // letzter Eintrag zum Loeschen eines Fuzzy-Sets
  menue->append_item(wkit->menu_item_separator());
  eintrag = wkit->menu_item(lkit->hcenter(wkit->label(Txt(60))));
  eintrag->action(
    new NfcAction(&_menue_auswahl, lingvar->anz_fuzzysets(), _cb_ptr)
  );
  menue->append_item(eintrag);

  // Hauptmenue erstellen und in die 'Page' eintragen
  eintrag = wkit->menubar_item(lkit->fixed_span(wkit->label("o"), 5, 5));
  eintrag->menu(menue);
  return eintrag;
}
