/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <IV-look/telltale.h>
#include <IV-look/kit.h>
#include <IV-look/dialogs.h>
#include <IV-look/button.h>
#include <IV-look/fchooser.h>
#include <InterViews/background.h>
#include <InterViews/hit.h>
#include <InterViews/session.h>
#include <InterViews/style.h>
#include <InterViews/display.h>
#include <InterViews/action.h>

#include "texte.h"
#include "datei.h"
#include "nfclook.h"
#include "regel_tafel.h"
#include "nfc_ed.h"

declareActionCallback(NfcEditor);
implementActionCallback(NfcEditor);
#define AMF ActionMemberFunction(NfcEditor)

/*
 *------------------------------------------------------------------------------
 * Klasse : Copyright
 * Zweck  : Erzeugt Popup-Fenster mit Copyright-Informationen, das nach
 *          Anklicken verschwindet.
 *------------------------------------------------------------------------------
 */
class Copyright : public Dialog
{
  public : Copyright(Style*);
           virtual void press(const Event&);
};

Copyright::Copyright(Style* style)
  : Dialog(nil, style)
{
  Coord rand = 15;
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();

  wkit->begin_style("NfcInfoStyle","NfcInfoFont");
  body(
    new Background(
      wkit->push_button(
        lkit->center(
          lkit->margin(
            lkit->vbox(
              lkit->hbox(wkit->label(Txt(160))),
              lkit->hbox(wkit->label(" ")),
              lkit->hbox(wkit->label(Txt(161))),
              lkit->hbox(wkit->label(" ")),
              lkit->hbox(wkit->label(Txt(162))),
              lkit->hbox(wkit->label(" ")),
              lkit->hbox(wkit->label(Txt(163))),
              lkit->hbox(wkit->label(Txt(164)))
            ), rand, rand
          )
        ), nil
      ), wkit->background()
    )
  );
  wkit->end_style();

  // Callback fuer Button braucht man nicht wegen 'Copyright::press'.
  // Button braucht man trotzdem, weil 'Copyright::press' sonst nicht aufgerufen
  // wird, wenn "Leerraum" innerhalb des Popup-Fensters angeklickt wird.

  wkit->end_style();
}

void Copyright::press(const Event&) { dismiss(true); }


/*
 *------------------------------------------------------------------------------
 * Eine Variable vom Typt 'struct FensterStrukt' enthaelt alle Daten, 
 * die noetig sind, um mit Hilfe eines Buttons eines der Editor-Fenster
 * aufzurufen. Fur jeden Button existiert also eine FensterStrukt.
 * Die FensterStrukt wird durch die Funktion NfcEd::_init() initialisiert;
 * d.h, hier werden die die Callbacks (als Action*) und die Buttons erzeugt.
 * Anmerkung: die Variable 'modus' enth"alt eine Kennung; wenn mehrere Buttons
 *            die gleichen Callbacks haben, wird bei Betaetigung der Buttons die
 *            Variable NfcEd::_auswahl mit dem Wert von 'modus' initialisiert;
 *            die Callbacks koennen so feststellen, von welchem Button sie
 *            aufgerufen wurden.
 *------------------------------------------------------------------------------
 */
struct FensterStrukt
{
  int bt_text;                  // Text fuer Aufruf-Button
  int modus;                    // Kennung (s.o); -1 <=> keine Kennung noetig
  Button* button;               // Button-Glyph
  Action *cb, *ok_cb, *nook_cb; // 1 Callbacks fuer Button und 2 fuer Editor
  Coord x_pos, y_pos;           // falls != -1 : Fensterposition
};

FensterStrukt strukt[] = {
  { 122, 0, nil, nil, nil, nil, -1, -1 }, // Steuer-Var (strukt[0])
  { 123, 1, nil, nil, nil, nil, -1, -1 }, // Fehler-Var (strukt[1])
  { 124, 0, nil, nil, nil, nil, -1, -1 }, // Steuer-Regeln (strukt[2])
  { 125, 1, nil, nil, nil, nil, -1, -1 }, // Fehler-Regeln (strukt[3])
  { 126, 0, nil, nil, nil, nil, -1, -1 }, // Anwendung definieren (strukt[4])
  { 127, 1, nil, nil, nil, nil, -1, -1 }, // Anwendung starten (strukt[5])
  { 139, 0, nil, nil, nil, nil, -1, -1 }, // Protokoll definieren (strukt[6])
  { 135, 1, nil, nil, nil, nil, -1, -1 }, // Controller Oberflaeche (strukt[7])
  { -1 }                                  // Textnr. -1 markiert Ende des Arrays
};

/*
 *-------------------- Definition der Klasse : NfcEditor -----------------------
 */

NfcEditor::NfcEditor()
{
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  NfcKit* nfc_kit = NfcKit::instance();

  _wissensbank         = new Wissensbank;
  _nfc_fenster         = nil;
  _auswahl             = 0;
  _menue_auswahl       = 0;
  _unter_menue_auswahl = 0;
  copyright_angezeigt  = false;

  _init(strukt);
  Glyph* lingvar_bt = _button_paar(strukt[0].button, strukt[1].button,Txt(63));
  Glyph* regel_bt   = _button_paar(strukt[2].button, strukt[3].button,Txt(64));
  Glyph* anwend_bt  = _button_paar(strukt[4].button, strukt[5].button,Txt(103));
  Glyph* control_bt = _button_paar(strukt[6].button, strukt[7].button,Txt(134));

  _ausgabe_feld = new NfcAusgabe(new AusgabeFeld(35, 6));
  _datei_arbeit = nil;
  _lv_editor    = nil;
  _rg_editor    = nil;
  _an_editor    = nil;
  _controll_ed  = nil;
  _prot_ed      = nil;

  Coord rand = 10;
  body(
    lkit->vbox(
      lkit->vcenter(
        lkit->hbox(lkit->variable_span(_mach_menue(), 0, 0), lkit->hglue())
      ),
      lkit->margin(
        lkit->vbox(
          lkit->vcenter(
            lkit->hbox(
              lingvar_bt, lkit->hglue(rand),
              regel_bt,   lkit->hglue(rand),
              anwend_bt,  lkit->hglue(rand),
              control_bt
            )
          ), lkit->vglue(rand, 0, 0),
          nfc_kit->text_drauf(Txt(40), _ausgabe_feld, true)
        ), rand, rand
      )
    )
  );
}

NfcEditor::~NfcEditor()
{
  // offene Fenster schliessen
  _lv_fenster(false, nil);
  _rg_fenster(false, nil);
  _an_fenster(false);
  _ct_fenster(false);

  if(_prot_ed != nil)
    Resource::unref(_prot_ed);

  // Wissensbank loeschen
  delete _wissensbank;

  // Callbacks freigeben
  for(FensterStrukt* fs = strukt; fs->bt_text != nil; fs++) {
    if(fs->ok_cb != nil)
      Resource::unref(fs->ok_cb);
    if(fs->nook_cb != nil)
      Resource::unref(fs->nook_cb);
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcEditor::allocate(Canvas*, const Allocation&, Extension&)
 * Zweck : Zeiger auf Elternfenster merken
 *------------------------------------------------------------------------------
 */

void NfcEditor::allocate(Canvas* c, const Allocation& a, Extension& e)
{
  _nfc_fenster = c->window();
  MonoGlyph::allocate(c, a, e);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcEditor::pick(Canvas*, const Allocation&, int, Hit&)
 * Zweck : Bei der 1. Mausbewegung Coyright-Fenster einblenden
 *------------------------------------------------------------------------------
 */

void NfcEditor::pick(Canvas* c, const Allocation& a, int depth, Hit& h)
{
  MonoGlyph::pick(c, a, depth, h);

  if(copyright_angezeigt)
    return;

  Dialog* copyright = new Copyright(WidgetKit::instance()->style());
  if(_nfc_fenster != nil) {
    copyright->post_for(_nfc_fenster);
    copyright_angezeigt = true;
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcEditor::mach_lv_ed()
 *                   NfcEditor::lv_ed_abbr()
 *                   NfcEditor::lv_ed_ende()
 * Zweck : Callback-Funktionen zum Erzeugen eines Editor-Fensters und zur
 *         Auswertung des Ergebnisses
 *------------------------------------------------------------------------------
 */

void NfcEditor::mach_lv_ed()
{
  boolean steuer_var = _auswahl == 0;

   // solange Ctrl-Oberflaeche offen - Wissensbank nicht aendern
  //  weil : LingVarBrowser im Protokoll enthaelt Namen der LingVar's - diese
 //          duerfen daher nicht geloescht werden
  if(_controll_ed != nil) {
    _ausgabe_feld->ausgeben(Fehler, Txt(223));
    int strukt_nr = steuer_var ? 0 : 1;
    strukt[strukt_nr].button->state()->set( TelltaleState::is_chosen, false);
    return;
  }

  if(steuer_var)
    _lv_fenster(true, steuer_var);
  else
    if(_wissensbank->steuer_var_ok(_ausgabe_feld)) {
      _wissensbank->mach_fehler_var();
      _lv_fenster(true, steuer_var);
    } else
      strukt[1].button->state()->set(TelltaleState::is_chosen, false);
}

void NfcEditor::lv_ed_abbr() { _lv_fenster(false, _auswahl == 0); }

void NfcEditor::lv_ed_ende()
{
  // solange Ctrl-Oberflaeche offen - ...blabla (siehe oben)
  if(_controll_ed != nil) {
    _ausgabe_feld->ausgeben(Warnung, Txt(223));
    return;
  }

  boolean steuer_var = _auswahl == 0;
  if(steuer_var)
    _wissensbank->steuer_var(_lv_editor->lingvar());
  else
    _wissensbank->fehler_var(_lv_editor->lingvar());
  _lv_fenster(false, steuer_var);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcEditor::mach_rg_ed()
 *                   NfcEditor::rg_ed_abbr()
 *                   NfcEditor::rg_ed_abbr()
 * Zweck : Callback-Funktionen zum Erzeugen eines Editor-Fensters und zur
 *         Auswertung des Ergebnisses
 *------------------------------------------------------------------------------
 */

void NfcEditor::mach_rg_ed()
{
  boolean steuer_regeln = _auswahl == 0;

   // solange Ctrl-Oberflaeche offen - Wissensbank nicht aendern
  //  weil : RegelBrowser in Protokoll enthaelt Ptr. auf Steuer-Regeln
  if(_controll_ed != nil) {
    _ausgabe_feld->ausgeben(Fehler, Txt(223));
    int strukt_nr = steuer_regeln ? 2 : 3;
    strukt[strukt_nr].button->state()->set(TelltaleState::is_chosen, false);
    return;
  }

  if(steuer_regeln) {
    if(_wissensbank->steuer_var_ok(_ausgabe_feld))
      _rg_fenster(true, steuer_regeln);
    else
      strukt[2].button->state()->set(TelltaleState::is_chosen, false);
  } else
    if(_wissensbank->fehler_var_ok(_ausgabe_feld))
      _rg_fenster(true, steuer_regeln);
    else
      strukt[3].button->state()->set(TelltaleState::is_chosen, false);
}

void NfcEditor::rg_ed_abbr() { _rg_fenster(false, _auswahl == 0); }

void NfcEditor::rg_ed_ende()
{
  // solange Ctrl-Oberflaeche offen - ...blabla (siehe oben)
  if(_controll_ed != nil) {
    _ausgabe_feld->ausgeben(Warnung, Txt(223));
    return;
  }

  boolean steuer_regeln = _auswahl == 0;
  if(steuer_regeln)
    _wissensbank->steuer_regeln(_rg_editor->regelbasis());
  else
    _wissensbank->fehler_regeln(_rg_editor->regelbasis());
  _rg_fenster(false, steuer_regeln);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcEditor::mach_an_ed()
 *                   NfcEditor::an_ed_abbr()
 *                   NfcEditor::an_ed_ende()
 * Zweck : Callback-Funktionen zum Erzeugen eines Editor-Fensters und zur
 *         Auswertung des Ergebnisses
 *------------------------------------------------------------------------------
 */

void NfcEditor::mach_an_ed()
{
  if(_auswahl == 0) {
    if(_wissensbank->anwendung() != nil && _wissensbank->anwendung()->aktiv()) {

      // Anwendungsprogramm laeuft noch
      _ausgabe_feld->ausgeben(Fehler, Txt(231));
      strukt[4].button->state()->set(TelltaleState::is_chosen, false);
    } else {
      if(_wissensbank->steuer_var_ok(_ausgabe_feld))

        // alles ok -> Fenster oeffnen
        _an_fenster(true);
      else

        // noch keine Steuer-Var definiert
        strukt[4].button->state()->set(TelltaleState::is_chosen, false);
    }
  } else {
    if(_wissensbank->anwendung_ok(_ausgabe_feld))
      _wissensbank->anwendung()->starten(_ausgabe_feld);
    strukt[5].button->state()->set(TelltaleState::is_chosen, false);
  }
}

void NfcEditor::an_ed_abbr() { _an_fenster(false); }

void NfcEditor::an_ed_ende()
{
  // zuerst testen, ob evtl. Anw.prg gestartet wurde waehrend Fenster auf war
  // falls ja -> Aenderungen koennen nicht uebernommen werden
  if(_wissensbank->anwendung() == nil || !_wissensbank->anwendung()->aktiv())
    _wissensbank->anwendung(_an_editor->anwendung());
  else
    _ausgabe_feld->ausgeben(Warnung, Txt(232));
  _an_fenster(false);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcEditor::mach_ct_ed()
 *                   NfcEditor::ct_ed_ende()
 *                   NfcEditor::pt_ed_ende()
 *                   NfcEditor::pt_ed_abbr()
 * Zweck : Callback-Funktionen zum Erzeugen des Protokoll-Fensters und der
 *         Controller-Oberflaeche, sowie Funktionen zur Auswertung der Erg.
 *------------------------------------------------------------------------------
 */

void NfcEditor::mach_ct_ed()
{
  if(_auswahl == 0)
   _pt_fenster(true);
  else {
    if(_wissensbank->steuer_var_ok(_ausgabe_feld))
      _ct_fenster(true);
    else
      strukt[7].button->state()->set(TelltaleState::is_chosen, false);
  }
}

void NfcEditor::ct_ed_ende() { _ct_fenster(false); }

void NfcEditor::pt_ed_ende()
{
  _protokoll_strukt = *_prot_ed->prot_struktur();
  _pt_fenster(false);
}

void NfcEditor::pt_ed_abbr() { _pt_fenster(false); }

/*
 *---------------------------- lokale Funktionen -------------------------------
 */

void NfcEditor::_init(FensterStrukt* dat)
{
  WidgetKit* wkit = WidgetKit::instance();
  wkit->begin_style("NfcInfoStyle", "NfcInfoFont");

  // Callbacks installieren
  dat[0].cb      = new ActionCallback(NfcEditor)(this,&NfcEditor::mach_lv_ed);
  dat[0].ok_cb   = new ActionCallback(NfcEditor)(this,&NfcEditor::lv_ed_ende);
  dat[0].nook_cb = new ActionCallback(NfcEditor)(this,&NfcEditor::lv_ed_abbr);
  dat[1].cb      = new ActionCallback(NfcEditor)(this,&NfcEditor::mach_lv_ed);
  dat[1].ok_cb   = new ActionCallback(NfcEditor)(this,&NfcEditor::lv_ed_ende);
  dat[1].nook_cb = new ActionCallback(NfcEditor)(this,&NfcEditor::lv_ed_abbr);

  dat[2].cb      = new ActionCallback(NfcEditor)(this,&NfcEditor::mach_rg_ed);
  dat[2].ok_cb   = new ActionCallback(NfcEditor)(this,&NfcEditor::rg_ed_ende);
  dat[2].nook_cb = new ActionCallback(NfcEditor)(this,&NfcEditor::rg_ed_abbr);
  dat[3].cb      = new ActionCallback(NfcEditor)(this,&NfcEditor::mach_rg_ed);
  dat[3].ok_cb   = new ActionCallback(NfcEditor)(this,&NfcEditor::rg_ed_ende);
  dat[3].nook_cb = new ActionCallback(NfcEditor)(this,&NfcEditor::rg_ed_abbr);

  dat[4].cb      = new ActionCallback(NfcEditor)(this,&NfcEditor::mach_an_ed);
  dat[4].ok_cb   = new ActionCallback(NfcEditor)(this,&NfcEditor::an_ed_ende);
  dat[4].nook_cb = new ActionCallback(NfcEditor)(this,&NfcEditor::an_ed_abbr);
  dat[5].cb      = new ActionCallback(NfcEditor)(this,&NfcEditor::mach_an_ed);
  dat[5].ok_cb   = nil;
  dat[5].nook_cb = nil;

  dat[6].cb      = new ActionCallback(NfcEditor)(this,&NfcEditor::mach_ct_ed);
  dat[6].ok_cb   = new ActionCallback(NfcEditor)(this,&NfcEditor::pt_ed_ende);
  dat[6].nook_cb = new ActionCallback(NfcEditor)(this,&NfcEditor::pt_ed_abbr);
  dat[7].cb      = new ActionCallback(NfcEditor)(this,&NfcEditor::mach_ct_ed);
  dat[7].ok_cb   = new ActionCallback(NfcEditor)(this,&NfcEditor::ct_ed_ende);
  dat[7].nook_cb = nil;

  for(FensterStrukt* fs = dat; fs->bt_text != -1; fs++) {

    // Button initialisieren
    if(fs->modus == -1)
      fs->button = wkit->check_box(Txt(fs->bt_text), fs->cb);
    else
      fs->button = wkit->check_box(
                     Txt(fs->bt_text),
                     new NfcAction(&_auswahl, fs->modus, fs->cb)
                   );

    // ok-Callback (endguelitig) initialisieren
    if(fs->ok_cb != nil) {
      if(fs->modus != -1)
        fs->ok_cb = new NfcAction(&_auswahl, fs->modus, fs->ok_cb);
      Resource::ref(fs->ok_cb);
    }

    // nook-Callback (endguelitig) initialisieren
    if(fs->nook_cb != nil) {
      if(fs->modus != -1)
        fs->nook_cb = new NfcAction(&_auswahl, fs->modus, fs->nook_cb);
      Resource::ref(fs->nook_cb);
    }
  }

  wkit->end_style();
}

void NfcEditor::_lv_fenster(boolean auf, boolean steuer_var)
{
  FensterStrukt& fs1 = (steuer_var == true) ? strukt[0] : strukt[1];
  FensterStrukt& fs2 = (steuer_var == true) ? strukt[1] : strukt[0];

  if(auf) {
    if(_lv_editor == nil) {
      fs1.button->state()->set(TelltaleState::is_enabled, false);
      fs2.button->state()->set(TelltaleState::is_enabled, false);

      if(steuer_var) {
        _lv_editor = new LingVarEd(
                       _wissensbank->steuer_var(),
                       steuer_var,
                       fs1.ok_cb,
                       fs1.nook_cb
                     );
      } else {
        _lv_editor = new LingVarEd(
                       _wissensbank->fehler_var(),
                       steuer_var,
                       fs1.ok_cb,
                       fs1.nook_cb
                     );
      }
      _lv_editor->einblenden(
        _nfc_fenster, Txt(73), Txt(77), fs1.x_pos, fs1.y_pos
      );
    }
  } else {
    if(_lv_editor != nil) {
      _lv_editor->ausblenden();
      _lv_editor->position(fs1.x_pos, fs1.y_pos);
      delete _lv_editor;
      _lv_editor = nil;
      fs1.button->state()->set(TelltaleState::is_chosen, false);
      fs1.button->state()->set(TelltaleState::is_enabled, true);
      fs2.button->state()->set(TelltaleState::is_enabled, true);
    }
  }
}

void NfcEditor::_rg_fenster(boolean auf, boolean steuer_regeln)
{
  FensterStrukt& fs1 = (steuer_regeln ? strukt[2] : strukt[3]);
  FensterStrukt& fs2 = (steuer_regeln ? strukt[3] : strukt[2]);

  if(auf) {
    fs1.button->state()->set(TelltaleState::is_enabled, false);
    fs2.button->state()->set(TelltaleState::is_enabled, false);

    const LingVarSatz* lingvarsatz;
    const RegelBasis* regelbasis;
    if(_rg_editor == nil) {
      if(steuer_regeln) {
        lingvarsatz = _wissensbank->steuer_var();
        if(_wissensbank->steuer_regeln_ok(nil))
          regelbasis = _wissensbank->steuer_regeln();
        else
          regelbasis = nil;
      } else {
        lingvarsatz = _wissensbank->fehler_var();
        if(_wissensbank->fehler_regeln_ok(nil))
          regelbasis = _wissensbank->fehler_regeln();
        else
          regelbasis = nil;
      }
      _rg_editor = new RegelEd(
                     lingvarsatz, regelbasis, fs1.ok_cb, fs1.nook_cb
                   );
      _rg_editor->einblenden(
        _nfc_fenster, Txt(74), Txt(76), fs1.x_pos, fs1.y_pos
      );
    }
  } else {
    if(_rg_editor != nil) {
      _rg_editor->ausblenden();
      _rg_editor->position(fs1.x_pos, fs1.y_pos);
      delete _rg_editor;
      _rg_editor = nil;
      fs1.button->state()->set(TelltaleState::is_chosen, false);
      fs1.button->state()->set(TelltaleState::is_enabled, true);
      fs2.button->state()->set(TelltaleState::is_enabled, true);
    }
  }
}

void NfcEditor::_an_fenster(boolean auf)
{
  FensterStrukt& fs = strukt[4];

  if(auf) {
    if(_an_editor == nil) {
      fs.button->state()->set(TelltaleState::is_enabled, false);

      Anwendung* anwendung;
      if(_wissensbank->anwendung_ok(nil))
        anwendung = _wissensbank->anwendung();
      else
        anwendung = nil;
      _an_editor = new AnwendungsEd(
                     anwendung,
                     _wissensbank->steuer_var(),
                     fs.ok_cb, fs.nook_cb
                   );
      _an_editor->einblenden(
        _nfc_fenster, Txt(103), Txt(104), fs.x_pos, fs.y_pos
      );
    }
  } else {
    if(_an_editor != nil) {
      _an_editor->ausblenden();
      _an_editor->position(fs.x_pos, fs.y_pos);
      delete _an_editor;
      _an_editor = nil;
      fs.button->state()->set(TelltaleState::is_chosen, false);
      fs.button->state()->set(TelltaleState::is_enabled, true);
    }
  }
}

void NfcEditor::_pt_fenster(boolean auf)
{
  FensterStrukt& fs = strukt[6];

  if(auf) {
    if(_prot_ed == nil) {
      Style* style = WidgetKit::instance()->style();
      _prot_ed = new ProtokollEd(style, fs.ok_cb, fs.nook_cb);
      Resource::ref(_prot_ed);
    }
    _prot_ed->einblenden(_nfc_fenster, &_protokoll_strukt);
  } else {

    // _prot_ed ist Unterklasse von Dialog (benutzt Transient - Window) und
   //  macht sich selbst unsichtbar; wird nicht geloescht -> Resource

    fs.button->state()->set(TelltaleState::is_chosen, false);
  }
}

void NfcEditor::_ct_fenster(boolean auf)
{
  FensterStrukt& fs = strukt[7];

  if(auf) {
    if(_controll_ed == nil) {
      fs.button->state()->set(TelltaleState::is_enabled, false);

      _controll_ed = new ControllerEd(
                       _wissensbank, &_protokoll_strukt, fs.ok_cb
                     );
      _controll_ed->einblenden(
        _nfc_fenster, Txt(132), Txt(133), fs.x_pos, fs.y_pos
      );
    }
  } else {
    if(_controll_ed != nil) {
      _controll_ed->ausblenden();
      _controll_ed->position(fs.x_pos, fs.y_pos);
      delete _controll_ed;
      _controll_ed = nil;
      fs.button->state()->set(TelltaleState::is_chosen, false);
      fs.button->state()->set(TelltaleState::is_enabled, true);
    }
  }
}

MenueStruktur datei_menue[] = {
  { 145, nil },                       // "Steuer-Variable"
  { 146, nil },                       // "Fehler-Variable"
  { 147, nil },                       // "Steuer-Regeln"
  { 148, nil },                       // "Fehler-Regeln"
  { -1 }
};

MenueStruktur info_menue[] = {
  { 160, nil },                       // blabla
  { 161, nil },
  { 162, nil },
  { 163, nil },
  { 164, nil },
  { -1 }
};

MenueStruktur ende_menue[] = {
  {  87, nil },                       // "Ende"
  { -1 }
};

MenueStruktur menue_bar[] = {
  { 159, nil, info_menue,  3 },       // "Info"
  { 144, nil, datei_menue, 3 },       // "laden"
  { 143, nil, datei_menue, 3 },       // "speichern"
  { 001, nil, datei_menue, 3 },       // "loeschen"
  { 138, nil, ende_menue,  3 },       // "Ende"
  { -1 }
};

Menu* NfcEditor::_mach_menue()
{
  // Callbacks installieren
  Action* cb = new ActionCallback(NfcEditor)(this,&NfcEditor::_datei);
  for(int i = 0; i < 4; i++)
    datei_menue[i].cb = new NfcAction(&_unter_menue_auswahl, i, cb);
  for(i = 0; i < 3; i++)
    menue_bar[i + 1].cb = new NfcAction(&_menue_auswahl, i, nil);
  ende_menue[0].cb  = new ActionCallback(NfcEditor)(this,&NfcEditor::_prg_ende);

  return NfcKit::instance()->menue(
           menue_bar, WidgetKit::instance(), LayoutKit::instance()
         );
}

void NfcEditor::_datei()
{
  if(_controll_ed != nil) {
    _ausgabe_feld->ausgeben(Fehler, Txt(223));
    return;
  }

  if(_datei_arbeit == nil)
    _datei_arbeit = new DateiArbeit(_nfc_fenster, _ausgabe_feld);
  switch(_menue_auswahl) {
    case 0:      // laden
      switch(_unter_menue_auswahl) {
        case 0:
          _datei_arbeit->laden(true, _wissensbank->steuer_var_ref());     break;
        case 1:
          _datei_arbeit->laden(false, _wissensbank->fehler_var_ref());    break;
        case 2:
          _datei_arbeit->laden(true, _wissensbank->steuer_regeln_ref());  break;
        case 3:
          _datei_arbeit->laden(false, _wissensbank->fehler_regeln_ref()); break;
        default: break;
      }
      break;
    case 1:      // speichern
      switch(_unter_menue_auswahl) {
        case 0:
          _datei_arbeit->speichern(true, _wissensbank->steuer_var());     break;
        case 1:
          _datei_arbeit->speichern(false, _wissensbank->fehler_var());    break;
        case 2:
          _datei_arbeit->speichern(true, _wissensbank->steuer_regeln());  break;
        case 3:
          _datei_arbeit->speichern(false, _wissensbank->fehler_regeln()); break;
        default: break;
      }
      break;
    case 2:      // loeschen
      switch(_unter_menue_auswahl) {
        case 0:
          *_wissensbank->steuer_var_ref()    = LingVarSatz();
          _ausgabe_feld->ausgeben(Meldung, Txt(185));
          break;
        case 1:
          *_wissensbank->fehler_var_ref()    = LingVarSatz();
          _ausgabe_feld->ausgeben(Meldung, Txt(186));
          break;
        case 2:
          *_wissensbank->steuer_regeln_ref() = RegelBasis();
          _ausgabe_feld->ausgeben(Meldung, Txt(187));
          break;
        case 3:
          *_wissensbank->fehler_regeln_ref() = RegelBasis();
          _ausgabe_feld->ausgeben(Meldung, Txt(188));
          break;
        default: break;
      }
      break;
    default: break;
  }
}

void NfcEditor::_prg_ende() { Session::instance()->quit(); }

Glyph* NfcEditor::_button_paar(Glyph* bt1, Glyph* bt2, char* text)
{
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  NfcKit* nkit    = NfcKit::instance();
  Coord rand      = 5;

  return lkit->vcenter(
           nkit->text_drauf(
             text,
             lkit->vbox(
               lkit->vglue(rand, 0, 0), bt1,
               lkit->vglue(rand, 0, 0), bt2
            ), true
           )
         );
}
