/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include "controller.h"

/*
 *-----------------------------------------------------------------------------
 * Klasse : EingabeSchicht
 * Zweck  : Diese Klasse uebernimmt die Fuzzifizierung der Messwerte. Sie
 *          enthaelt ein Array, in dem die 'Trefferquoten' jedes Fuzzy-Sets
 *          aus den Eingabe-Variablen gespeichert wird. Dies hat den Vorteil,
 *          dass die 'Trefferquoten' nur einmal berechnet werden muessen und
 *          dass schnell auf diese zugegriffen werden kann.
 *------------------------------------------------------------------------------
 */
class EingabeSchicht
{
  public : EingabeSchicht(LingVarSatz*);
           ~EingabeSchicht();
           void messwerte(FuzzyTyp*);
           FuzzyTyp fuzzy_messwert(int var_nr, int fuz_nr);
  private: FuzzyTyp** _var_array;
           int _anzahl_var;
           LingVarSatz* _lingvarsatz;
};

EingabeSchicht::EingabeSchicht(LingVarSatz* lingvarsatz)
{
  _lingvarsatz = lingvarsatz;
  _anzahl_var  = lingvarsatz->anz_lingvars(EingabeVar);
  _var_array   = new FuzzyTyp*[_anzahl_var];

  for(int i = 0; i < _anzahl_var; i++) {
    int anz_fuz = lingvarsatz->hol_lingvar(EingabeVar, i)->anz_fuzzysets();
    _var_array[i] = new FuzzyTyp[anz_fuz];
  }
}

EingabeSchicht::~EingabeSchicht()
{
  for(int i = 0; i < _anzahl_var; i++)
    delete[] _var_array[i];
  delete[] _var_array;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : EingabeSchicht::messwerte(FuzzyTyp*)
 * Parameter : FuzzyTyp werte - Messwerte (einen fuer jede Eingabe-Variable)
 * Zweck : berechnet fuer alle FuzzySets der Eingabe-Variablen die Zugehoerig-
 *         keitswerte.
 *------------------------------------------------------------------------------
 */
void EingabeSchicht::messwerte(FuzzyTyp* werte)
{
  for(int var = 0; var < _anzahl_var; var++) {
    LingVar* lingvar = _lingvarsatz->hol_lingvar(EingabeVar, var);
    for(int fuz = 0; fuz < lingvar->anz_fuzzysets(); fuz++)
      _var_array[var][fuz] = lingvar->hol_fuzzyset(fuz)->wert(werte[var]);
  }
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : EingabeSchicht::fuzzy_messwert(int var, int fuz)
 * Parameter : int var - Nr. der Eingabe-Variablen
 *             int fuz - Nr. des Fuzzy-Sets
 * Rueckgabewert: Zugehoerigkeitswert
 * Zweck : liefert Zugehoerigkeitswert fuer spezifiziertes Fuzzy-Set
 *------------------------------------------------------------------------------
 */
inline FuzzyTyp EingabeSchicht::fuzzy_messwert(int var, int fuz)
{
  return _var_array[var][fuz];
}


/*
 *------------------------------------------------------------------------------
 *--------------Deklaration und Definition der Klasse : RegelKnoten ------------
 *------------------------------------------------------------------------------
 */

/*
 *------------------------------------------------------------------------------
 * Klasse : RegelKnoten
 * Zweck  : Darstellung der einzelnen Regeln. Die Variablen und Fuzzy-Sets
 *          werden als Indizes gespeichert, was die Zugriffsgeschwindigkeit
 *          gegenueber der 'RegelBasis' deutlich erhoeht.
 *          Anmerkung : Die Regeln in 'RegelBasis' bestehen aus Strings der
 *            jeweiligen Variablen und Fuzzy-Sets, wodurch sie allgemeiner und
 *            von dem 'LingVarSatz' unabhaengig definierbar werden.
 *            Die Klasse RegelKnoten erhaelt ihre Bedeutung nur im Zusammenhang
 *            mit einem bestimmten 'LingVarSatz'.
 *
 *          Ausserdem werden diverse Daten wie Treffergrade und Steuerwerte
 *          gespeichert. Zum Erlernen einer RegelBasis wird noch ein Zaehler
 *          und eine Variable zum Speichern von Fehlern mitgefuehrt.
 *------------------------------------------------------------------------------
 */
class RegelKnoten
{
  public : RegelKnoten(Regel*, LingVarSatz*);
           RegelKnoten(int nr, LingVarSatz*);
           ~RegelKnoten();

           int pra(int nr);
           int kon(int nr);

           void treffer_grad(FuzzyTyp grad);
           FuzzyTyp treffer_grad();

           void steuer_wert(int nr, FuzzyTyp wert);
           FuzzyTyp steuer_wert(int nr);
           int sgn_steuer_werte();

           void mach_regel(LingVarSatz*, RegelBasis*);

           int& zaehler1();
           int& zaehler2();
           FuzzyTyp& fehler();
  private: int* _praemisse;
           int* _konklusion;
           int _gr_kon;
           FuzzyTyp _treffer_grad;
           FuzzyTyp* _steuer_wert;

           FuzzyTyp _fehler;         // Fehler wird gebraucht beim Regeln-Lernen
           int _zaehler1, _zaehler2; // die Zaehler auch
};

/*
 *------------------------------------------------------------------------------
 * Konstruktor fuer RegelKnoten, bei dem eine bestimmte Regel vorgegeben ist.
 *------------------------------------------------------------------------------
 */
RegelKnoten::RegelKnoten(Regel* regel, LingVarSatz* lingvarsatz)
{
  int gr_pra = lingvarsatz->anz_lingvars(EingabeVar);
  _gr_kon    = lingvarsatz->anz_lingvars(AusgabeVar);

  _praemisse    = new int[gr_pra];
  _konklusion   = new int[_gr_kon];
  _steuer_wert  = new FuzzyTyp[_gr_kon];
  _treffer_grad = 0;
  _fehler       = 0;

  int pra_nr = 0;
  int kon_nr = 0;
  for(int i = 0; i < gr_pra + _gr_kon; i++) {
    LingVar* var = lingvarsatz->hol_lingvar(i);
    if(var->typ() == EingabeVar) {
      if(regel->enthalten(Praemisse, var->name())) {
         int nr = var->fuz_nr(regel->fuzzyset(Praemisse, var->name()));
        _praemisse[pra_nr++] = nr;
      } else
        _praemisse[pra_nr++] = -1;
    } else
      if(regel->enthalten(Konklusion, var->name())) {
         int nr = var->fuz_nr(regel->fuzzyset(Konklusion, var->name()));
        _konklusion[kon_nr++] = nr;
      } else
        _konklusion[kon_nr++] = -1;
  }
}


/*
 *------------------------------------------------------------------------------
 * Konstruktor fuer RegelKnoten, bei dem keine Regel, sondern eine Nummer nr
 * vorgegeben wird. Die Regel die hier zur Initialisierung genommen wird,
 * ist die so-und-so-vielte aller mgl. Kombinationen der Fuzzy-Sets aus dem
 * 'LingVarSatz', wobei gilt : so-und-so-vielte = nr.
 *------------------------------------------------------------------------------
 */
RegelKnoten::RegelKnoten(int nr, LingVarSatz* lingvarsatz)
{
  int gr_pra = lingvarsatz->anz_lingvars(EingabeVar);
  _gr_kon    = lingvarsatz->anz_lingvars(AusgabeVar);

  _praemisse    = new int[gr_pra];
  _konklusion   = new int[_gr_kon];
  _steuer_wert  = new FuzzyTyp[_gr_kon];
  _treffer_grad = 0;
  _fehler       = 0;

  int anz_fuz, faktor = 1;
  for(int i = 0; i < gr_pra; i++) {
    anz_fuz = lingvarsatz->hol_lingvar(EingabeVar, i)->anz_fuzzysets();
    _praemisse[i] = (nr / faktor) % anz_fuz;
    faktor *= anz_fuz;
  }
  for(i = 0; i < _gr_kon; i++) {
    anz_fuz = lingvarsatz->hol_lingvar(AusgabeVar, i)->anz_fuzzysets();
    _konklusion[i] = (nr / faktor) % anz_fuz;
    faktor *= anz_fuz;
  }
}


RegelKnoten::~RegelKnoten()
{
  if(_praemisse != nil)
    delete[] _praemisse;
  if(_konklusion != nil)
    delete[] _konklusion;
  if(_steuer_wert != nil)
    delete[] _steuer_wert;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelKnoten::sgn_steuer_werte()
 * Rueckgabewert : int - ' 1' <=> alle Steuerwerte sind groesser als 0.
 *                       ' 0' <=> Vorzeichen der Steuerwerte sind verschieden.
 *                       '-1' <=> alle Steuerwerte sin  kleiner als 0.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
int RegelKnoten::sgn_steuer_werte()
{
  if(_steuer_wert[0] > 0) {
    for(int i = 1; i < _gr_kon; i++)
      if(_steuer_wert[i] < 0)
        return 0;
    return 1;
  } else {
    for(int i = 1; i < _gr_kon; i++)
      if(_steuer_wert[i] > 0)
        return 0;
    return -1;
  }
}


/*
 *------------------------------------------------------------------------------
 * es folgen verschiedene Inline-Funktionen, die den Zugriff auf die Daten
 * des RegelKnotens ermoeglichen, bzw. diese Daten initialisieren.
 *------------------------------------------------------------------------------
 */

inline int RegelKnoten::pra(int nr) { return _praemisse[nr]; }

inline int RegelKnoten::kon(int nr) { return _konklusion[nr]; }

inline void RegelKnoten::treffer_grad(FuzzyTyp grad) { _treffer_grad = grad; }

inline FuzzyTyp RegelKnoten::treffer_grad() { return _treffer_grad; }

inline void RegelKnoten::steuer_wert(int nr, FuzzyTyp wert)
{
  _steuer_wert[nr] = wert;
}

inline FuzzyTyp RegelKnoten::steuer_wert(int nr) { return _steuer_wert[nr]; }

inline int& RegelKnoten::zaehler1() { return _zaehler1; }

inline int& RegelKnoten::zaehler2() { return _zaehler2; }

inline FuzzyTyp& RegelKnoten::fehler() { return _fehler; }


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelKnoten::mach_regel(LingVarSatz*, RegelBasis*)
 * Parameter : LingVarSatz* - zugrundeliegender LingVarSatz
 *             RegelBasis*  - wo soll Regel angehaengt werden
 * Zweck : Fuegt eine dem RegelKnoten entsprechende Regel an die uebergebene
 *         RegelBasis an.
 *------------------------------------------------------------------------------
 */
void RegelKnoten::mach_regel(LingVarSatz* lingvarsatz, RegelBasis* regelbasis)
{
  int gr_pra = lingvarsatz->anz_lingvars(EingabeVar);
  int pra    = gr_pra;
  int kon    = _gr_kon;

  // leere Bedingungen abziehen
  for(int i = 0; i < gr_pra; i++)
    if(_praemisse[i] == -1)
      pra--;
  for(i = 0; i < _gr_kon; i++)
    if(_konklusion[i] == -1)
      kon--;

  int zaehler = 0;
  Regel regel(pra, kon);
  for(i = 0; i < gr_pra; i++)
    if(_praemisse[i] != -1) {
      LingVar* var  = lingvarsatz->hol_lingvar(EingabeVar, i);
      FuzzySet* fuz = var->hol_fuzzyset(_praemisse[i]);
      regel.bedingung(Praemisse, zaehler, var->name(), fuz->name());
      zaehler++;
    }
  zaehler = 0;
  for(i = 0; i < _gr_kon; i++) {
    LingVar* var  = lingvarsatz->hol_lingvar(AusgabeVar, i);
    FuzzySet* fuz = var->hol_fuzzyset(_konklusion[i]);
    regel.bedingung(Konklusion, zaehler, var->name(), fuz->name());
    zaehler++;
  }

  regelbasis->add_regel(&regel);
}


/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : FuzController --------------------
 *------------------------------------------------------------------------------
 */

/*
 *------------------------------------------------------------------------------
 * Konstruktor fuer FuzController.
 * Anmerkung : Wenn die uebergebene RegelBasis == nil ist, wird eine RegelBasis
 *             erzeugt, die alle Regeln enthaelt, die aufgrund des entspr.
 *             LingVarSatz'es denkbar sind.
 *------------------------------------------------------------------------------
 */
FuzController::FuzController(
                 const LingVarSatz* lingvarsatz, const RegelBasis* regelbasis
               )
{
  _lingvarsatz     = new LingVarSatz(*lingvarsatz);
  _anz_eingabe_var = _lingvarsatz->anz_lingvars(EingabeVar);
  _anz_ausgabe_var = _lingvarsatz->anz_lingvars(AusgabeVar);

  _eingabe       = new EingabeSchicht(_lingvarsatz);
  _ausgabe       = new FuzzyTyp[_anz_ausgabe_var];
  _treffer_summe = new FuzzyTyp[_anz_ausgabe_var];
  _regeln        = nil;
  _init_regelbasis(regelbasis);
}


FuzController::~FuzController()
{
  for(int i = 0; i < _merk_anz_regeln; i++)
    delete _regeln[i];
  delete[] _regeln;
  delete[] _ausgabe;
  delete[] _treffer_summe;
  delete _eingabe;
  delete _lingvarsatz;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzController::berechne_steuerwerte(FuzzyTyp*)
 * Parameter : FuzzyTyp* eingaben - Messwerte (einen fuer jede Eingabe-Variable)
 * Rueckgabewert : FuzzyTyp       - Steuerwert (dito fuer jede Ausgabe-Variable)
 * Zweck : berechnet aus gegebenen Messwerte die entsprechenden Steuerwerte.
 *------------------------------------------------------------------------------
 */
FuzzyTyp* FuzController::berechne_steuerwerte(FuzzyTyp* eingaben)
{
  _eingabe->messwerte(eingaben);

  for(int i = 0; i < _anz_ausgabe_var; i++) {
    _treffer_summe[i] = 0;
    _ausgabe[i]       = 0;
  }

  int fuz_nr;
  FuzzyTyp wert, gewicht;
  for(int regel = 0; regel < _anz_regeln; regel++) {

    // Treffergrade fuer die Regeln berechnen
    FuzzyTyp treffer = 1;
    for(int var = 0; var < _anz_eingabe_var; var++)
      if((fuz_nr = _regeln[regel]->pra(var)) != -1)
        treffer = _t_norm(treffer, _eingabe->fuzzy_messwert(var, fuz_nr));
    _regeln[regel]->treffer_grad(treffer);

    // Steuerwerte fuer jede Ausgabevariable berechnen
    for(var = 0; var < _anz_ausgabe_var; var++)
      if((fuz_nr = _regeln[regel]->kon(var)) != -1) {
        if(treffer <= 0) {
          wert    = 0;
          gewicht = 0;
        } else
          wert = _defuz(
                   treffer,
                   _lingvarsatz->hol_lingvar(
                     AusgabeVar, var
                   )->hol_fuzzyset(fuz_nr),
                   gewicht
                 );
        _regeln[regel]->steuer_wert(var, wert);  // Wert merken
        _ausgabe[var] += wert * gewicht;         // Werte aufaddieren
        _treffer_summe[var] += gewicht;          // Gesamtgewicht bilden
      }
  }
  for(int var = 0; var < _anz_ausgabe_var; var++)
    if(_treffer_summe[var] != 0)
      _ausgabe[var] /= _treffer_summe[var];      // Mittelwert bilden
    else
      _ausgabe[var] = 0;                         // sollte nicht sein

  return _ausgabe;
}


LingVarSatz* FuzController::lingvarsatz() { return _lingvarsatz; }


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzController::_defuz(FuzzyTyp, FuzzySet*, FuzzyTyp&)
 * Parameter : FuzzyTyp wert       - unscharfer Wert
 *             FuzzySet* fuzzy_set - Zeiger auf ein FuzzySet
 *             FuzzyTyp& gewicht   - Gewicht des Ergebnisses
 * Rueckgabewert : FuzzyTyp        - berechneter scharfer Wert
 * Zweck : berechnet aus einem unscharfen Wert anhand eines gegebenen FuzzySets
 *         einen scharfen Wert; Dieser Wert wird mit einem ebenfalls zurueck-
 *         gegebenen Gewicht multipliziert, dieses kann zur Bildung eines
 *         gewichteten Mittelwertes herangezogen werden.
 *         Defuzzifizierungsverfahren : 'Mean Of Maxima'
 *         bzw. bei monotonen Funktionen : Inverses bilden
 *------------------------------------------------------------------------------
 */
FuzzyTyp FuzController::_defuz(
           FuzzyTyp wert, FuzzySet* fuzzyset, FuzzyTyp& gewicht
         )
{
  gewicht = wert;
  if(wert == 0)
    return 0;

  FuzzyTyp x1, y1, x2, y2;        // Endpunkte der Linie
  FuzzyTyp sx, x;                 // Hilfspunkte
  FuzzyTyp m  = 0;                // Summe der Mittelpunkte
  int zaehler = 0;                // Anzahl der einzelnen Mittelpunkte

  int anzahl = fuzzyset->anz_intervalle();
  if(anzahl == 1) {

    // Fuzzy-Set ist monoton => Inverses bilden
    if(fuzzyset->hol_intervall(0)->invertiere(wert, m))
      zaehler = 1;
  } else

    // Defuzifizieren nach MOM
    for(int i = 0; i < anzahl; i++) {
      FsIntervall* linie = fuzzyset->hol_intervall(i);
      linie->links(x1, y1);
      linie->rechts(x2, y2);
      if(y1 > wert && y2 > wert) {
        m += (x1 + x2) / 2;
        zaehler++;
      } else
        if(linie->invertiere(wert, sx)) {
          x  = (y1 > y2) ? x1 : x2;
          m += (sx + x) / 2;
          zaehler++;
        }
    }
  if(zaehler == 0) {

    // kein Schnittpunkt gefunden => maximalen y_Wert liefern
    FuzzyTyp max = 0;
    for(int i = 0; i < anzahl; i++) {
      FsIntervall* linie = fuzzyset->hol_intervall(i);
      linie->links(x1, y1);
      linie->rechts(x2, y2);
      if(y1 > max)
        max = y1;
      if(y2 > max)
        max = y2;
    }
    return max;
  } else
    return m / zaehler;
}


/*
 *------------------------------ lokale Funktionen -----------------------------
 */

void FuzController::_init_regelbasis(const RegelBasis* regelbasis)
{
  // evtl. vorhandene alte RegelKnoten loeschen
  if(_regeln != nil) {
    for(int i = 0; i < _merk_anz_regeln; i++)
      delete _regeln[i];
    delete[] _regeln;
  }

  if(regelbasis != nil)

    // Regelbasis vorgegeben
    _anz_regeln = regelbasis->anzahl_regeln();
  else {

    // Regeln aus allen mgl. Kombinationen von Fuzzy-Sets bilden
    _anz_regeln = 1;
    for(int i = 0; i < _anz_eingabe_var; i++)
      _anz_regeln *= _lingvarsatz->hol_lingvar(EingabeVar, i)->anz_fuzzysets();
    for(i = 0; i < _anz_ausgabe_var; i++)
      _anz_regeln *= _lingvarsatz->hol_lingvar(AusgabeVar, i)->anz_fuzzysets();
  }

  _merk_anz_regeln = _anz_regeln;
  _regeln          = new RegelKnoten*[_anz_regeln];

  for(int i = 0; i < _anz_regeln; i++)
    if(regelbasis != nil)
      _regeln[i] = new RegelKnoten(regelbasis->hol_regel(i), _lingvarsatz);
    else
      _regeln[i] = new RegelKnoten(i, _lingvarsatz);

}


/*
 *------------------------------------------------------------------------------
 *---------------- Definition der Klasse : NeuroFuzController ------------------
 *------------------------------------------------------------------------------
 */

NeuroFuzController::NeuroFuzController(
                      const LingVarSatz* steuer_vars,
                      const LingVarSatz* fehler_vars,
                      const RegelBasis* steuer_regeln,
                      const RegelBasis* fehler_regeln
                    ) : FuzController(steuer_vars, steuer_regeln)
{
  _protokoll = nil;

  _fehler_ctrl      = new FuzController(fehler_vars, fehler_regeln);
  _lern_faktor      = 0;    // Lernfaktor fuer Optimierung der Fuzzy-Sets
  _erster_durchlauf = true;
}


NeuroFuzController::~NeuroFuzController()
{ 
  // Protokoll informieren, dass LingVarSatz geloescht wurde
  if(_protokoll != nil)
    _protokoll->init((LingVarSatz*)nil);

  delete _fehler_ctrl;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NeuroFuzController::berechne_steuerwerte(FuzzyTyp*)
 * Parameter : FuzzyTyp*    - Messwerte (einen fuer jede Eingabe-Variable)
 * Rueckgabewert : FuzzyTyp - Steuerwert (einen fuer jede Ausgabe-Variable)
 * Zweck : berechnet aus gegebenen Messwerten die entsprechenden Steuerwerte.
 *         Danach wird der Systemfehler berechnet, mit dem entweder die
 *         FuzzySets optimiert oder Steuerregeln erlernt werden.
 *------------------------------------------------------------------------------
 */
FuzzyTyp* NeuroFuzController::berechne_steuerwerte(FuzzyTyp* eingaben)
{
  FuzzyTyp system_fehler = *_fehler_ctrl->berechne_steuerwerte(eingaben);

  // Fuzzy-Sets optimieren
  if(!_erster_durchlauf)
    _optimieren(system_fehler);
  else
    _erster_durchlauf = false;

  // Steuerwerte berechnen
  FuzzyTyp* ausgaben = FuzController::berechne_steuerwerte(eingaben);

  // Protokoll erstellen
  _protokollieren(eingaben, ausgaben, system_fehler);

  // Steuerwerte zurueckliefern
  return ausgaben;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NeuroFuzController::reset()
 * Zweck : Controll-Vorgang von neuem Starten (z.B., wenn Pendel umkippt).
 *------------------------------------------------------------------------------
 */
void NeuroFuzController::reset()
{
  _erster_durchlauf = true;
  if(_protokoll != nil)
    _protokoll->fehlschlag();
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NeuroFuzController::lern_faktor(float)
 * Parameter : float - zu benutzender Lernfaktor
 * Zweck : Setzt gewuenschten Lernfaktor fuer das Optimieren der FuzzySets;
 *         bei 'Lernfaktor == 0' wird nicht optimiert.
 *------------------------------------------------------------------------------
 */
void NeuroFuzController::lern_faktor(float faktor) { _lern_faktor = faktor; }


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NeuroFuzController::protokoll(Protokoll*)
 * Parameter : Protokoll* - zu benutzendes Protokoll
 * Zweck : Setzt das gewuenschte Protokoll; bei 'Protokoll == nil' wird nicht
 *         protokolliert.
 *------------------------------------------------------------------------------
 */
void NeuroFuzController::protokoll(Protokoll* protokoll)
{
  _protokoll = protokoll;
  if(_protokoll != nil)
    _protokoll->init(_lingvarsatz);
}

/*
 *---------------------------- lokale Funktionen -------------------------------
 */

void NeuroFuzController::_init_regelbasis(const RegelBasis* regelbasis)
{
  FuzController::_init_regelbasis(regelbasis);
  _erster_durchlauf = true;
}

void NeuroFuzController::_optimieren(FuzzyTyp system_fehler)
{
  // Anmerkung :
  // system_fehler > 0 : hoehere Werte in den Ausgabevariablen verbessern
  //                     Systemverhalten.
  // regel_fehler > 0  : Regel wird belohnt; d.h. naechste Ausgabe wird
  //                     groesser falls das noch mgl. ist.

  for(int rg_nr = 0; rg_nr < _anz_regeln; rg_nr++) {
    FuzzyTyp regel_fehler = _regeln[rg_nr]->sgn_steuer_werte() *
                            _regeln[rg_nr]->treffer_grad() *
                            _lern_faktor *
                            system_fehler;
/*
    int f;
    if(_regeln[rg_nr]->steuer_wert(0) > _ausgabe[0])
      f = 1;
    else
      f = -1;
    FuzzyTyp fehlerp = f *
                            _regeln[rg_nr]->treffer_grad() *
                            _lern_faktor *
                            system_fehler;
    FuzzyTyp fehlerk =_regeln[rg_nr]->sgn_steuer_werte() *
                            _regeln[rg_nr]->treffer_grad() *
                            _lern_faktor *
                            system_fehler;

*/

    // Praemisse justieren
    int fuz_nr;
    for(int pra_nr = 0; pra_nr < _anz_eingabe_var; pra_nr++)
      if((fuz_nr = _regeln[rg_nr]->pra(pra_nr)) != -1)
        _lingvarsatz->hol_lingvar(EingabeVar, pra_nr)->hol_fuzzyset(
          fuz_nr
        )->justiere(regel_fehler);
        //)->justiere(fehlerp);

    // Konklusion justieren
    for(int kon_nr = 0; kon_nr < _anz_ausgabe_var; kon_nr++)
      if((fuz_nr = _regeln[rg_nr]->kon(kon_nr)) != -1)
        _lingvarsatz->hol_lingvar(AusgabeVar, kon_nr)->hol_fuzzyset(
          fuz_nr
        )->justiere(-1 * regel_fehler);
        //)->justiere(-1 * fehlerk);
  }
}

void NeuroFuzController::_protokollieren(
       FuzzyTyp* eingaben, FuzzyTyp* ausgaben, FuzzyTyp fehler
     )
{
  if(_protokoll != nil) {
    // Ein/Ausgabe protokollieren
    for(int i = 0; i < _anz_eingabe_var; i++)
      _protokoll->zeig_eingabe_var(i, eingaben[i]);
    for(i = 0; i < _anz_ausgabe_var; i++)
      _protokoll->zeig_ausgabe_var(i, ausgaben[i]);

    // Fehler protokollieren
    _protokoll->zeig_fehler(fehler);

    // Regel mit max. Trefferquote protokollieren
    FuzzyTyp max_treff = 0;
    int max_regel      = -1;
    for(int rg_nr = 0; rg_nr < _anz_regeln; rg_nr++)
      if(_regeln[rg_nr]->treffer_grad() > max_treff) {
        max_treff = _regeln[rg_nr]->treffer_grad();
        max_regel = rg_nr;
      }
    if(max_regel != -1)
      _protokoll->zeig_regel(max_regel);

    // Aenderungen der Fuzzy-Sets protokollieren
    _protokoll->aktua_lingvarsatz();
  }
}


/*
 *------------------------------------------------------------------------------
 *---------------- Definition der Struktur : RgLernStruktur --------------------
 *------------------------------------------------------------------------------
 */
RgLernStruktur::RgLernStruktur()
{
  // Defaultbelegung
  schritte_phase1 = 100;
  schritte_phase2 = 100;
  nach_prozent    = true;
  prozent         = 97;
  nach_dauer      = true;
  dauer           = 80;
  min_treffergrad = 0.2;
  fuz_trainieren  = false;
}


/*
 *------------------------------------------------------------------------------
 *----------------- Definition der Klasse : RgLrnNeuroFuzCtrl ------------------
 *------------------------------------------------------------------------------
 */

RgLrnNeuroFuzCtrl::RgLrnNeuroFuzCtrl(
                     const LingVarSatz* steuer_var,
                     const LingVarSatz* fehler_var,
                     const RegelBasis* fehler_regeln,
                     RegelBasis* steuer_regeln,
                     RgLernStruktur* lern_strukt
                   )
  : NeuroFuzController(steuer_var, fehler_var, nil, fehler_regeln)
{
  _lern_strukt = *lern_strukt;
  _lern_strukt.schritte_phase1 = Max(1, _lern_strukt.schritte_phase1);
  _lern_strukt.schritte_phase2 = Max(1, _lern_strukt.schritte_phase2);

  _bereich    = new FuzzyTyp[_anz_eingabe_var + _anz_ausgabe_var];
  _regelbasis = steuer_regeln;

  int faktor = 1;
  for(int i = 0; i < steuer_var->anz_lingvars(AusgabeVar); i++)
    faktor *= steuer_var->hol_lingvar(AusgabeVar, i)->anz_fuzzysets() / 2;
  _rnd_grenze = 1 / (float) faktor;

  // Initialisieren der Zaehler der einzelnen Regeln
  for(i = 0; i < _anz_regeln; i++) {
    if(_lern_strukt.nach_prozent)
      _regeln[i]->zaehler1() = (_lern_strukt.schritte_phase1 / 100) *
                               _lern_strukt.prozent;
    if(_lern_strukt.nach_dauer)
      _regeln[i]->zaehler2() = _lern_strukt.dauer;
  }

  // Berechnen der Definitionsbereiche der Ausgabe-Variablen; diese dienen zum
  // Normieren der Differenzen in den Steuerwerten (siehe 'regeln_lern_phase2')

  // Initialisieren des Arrays '_bereich'. Zuerst die Def.bereiche der
  // Eingabe-Variablen dann die der Ausgabe-Variablen
  for(i = 0; i < _anz_eingabe_var; i++) {
    LingVar* var = _lingvarsatz->hol_lingvar(EingabeVar, i);
    _bereich[i] = var->max() - var->min();
  }
  for(i = 0; i < _anz_ausgabe_var; i++) {
    LingVar* var = _lingvarsatz->hol_lingvar(AusgabeVar, i);
    _bereich[_anz_eingabe_var + i] = var->max() - var->min();
  }
}


RgLrnNeuroFuzCtrl::~RgLrnNeuroFuzCtrl() { delete[] _bereich; }


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RgLrnNeuroFuzCtrl::berechne_steuerwerte(FuzzyTyp*)
 * Parameter : FuzzyTyp* - Messwerte
 * Zweck : berechnet Steuerwerte fuer uebergebene Masswerte.
 *         Solange die Anzahl der Lernschritte noch nicht 'verbraucht' ist,
 *         wird die RegelBasis trainiert; danach erscheint der
 *         'RgLrnNeuroFuzCtrl' als NeuroFuzController;
 *------------------------------------------------------------------------------
 */
FuzzyTyp* RgLrnNeuroFuzCtrl::berechne_steuerwerte(FuzzyTyp* eingaben)
{
  //-------------------------------------
  if(_lern_strukt.schritte_phase1 > 0) {
    _lern_strukt.schritte_phase1--;

    // Lernphase 1
    FuzzyTyp* ausgaben = FuzController::berechne_steuerwerte(eingaben);
    _regeln_lern_phase1(*_fehler_ctrl->berechne_steuerwerte(eingaben));

    // letzter Lernschritt ?
    if(_lern_strukt.schritte_phase1 == 0) {
      _erster_durchlauf = true;

      // Zaehler werden in der 2. Lernphase erneut benoetigt
      for(int i = 0; i < _anz_regeln; i++) {
        _regeln[i]->zaehler1() = 0;
        _regeln[i]->zaehler2() = 0;
      }
    }

    return ausgaben;
  }

  //-------------------------------------
  if(_lern_strukt.schritte_phase2 > 0) {
    _lern_strukt.schritte_phase2--;

    // Lernphase 2
    FuzzyTyp fehler = *_fehler_ctrl->berechne_steuerwerte(eingaben);
    if(!_erster_durchlauf) {

      if(_lern_strukt.fuz_trainieren) {
        _optimieren(fehler);
        if(_protokoll != nil)
          _protokoll->aktua_lingvarsatz();
      }

      _regeln_lern_phase2(fehler);
    } else
      _erster_durchlauf = false;

    // beim letzten Lernschritt RegelBasis erzeugen
    if(_lern_strukt.schritte_phase2 == 0) {
      for(int i = 0; i < _anz_regeln; i++)
        if(_regeln[i]->zaehler2() > 0)
          _regeln[i]->fehler() /= _regeln[i]->zaehler2();

     _erzeuge_regelbasis(_regelbasis);
      if(_protokoll != nil)
        _protokoll->init(_regelbasis);
    }

    return _rnd_steuerwerte(eingaben);
  }

  //-------------------------------------
  // Lernphasen beendet, ab jetzt Fuzzy-Sets optimieren
  if(_anz_regeln == 0) {
    for(int i = 0; i < _anz_ausgabe_var; _ausgabe[i++] = 0);
    return _ausgabe;
  } else
    return NeuroFuzController::berechne_steuerwerte(eingaben);
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RgLrnNeuroFuzCtrl::protokoll(Protokoll*)
 * Parameter : Protokoll* - zu benutzendes Protokoll
 * Zweck : Setzt Protokoll und initialisiert es mit zu benutzender RegelBasis.
 *------------------------------------------------------------------------------
 */
void RgLrnNeuroFuzCtrl::protokoll(Protokoll* protokoll)
{
  NeuroFuzController::protokoll(protokoll);

  // wenn Lernphasen beendet sind, Protokoll mit RegelBasis initialisieren
  if(_protokoll != nil && _lern_strukt.schritte_phase2 <= 0)
    _protokoll->init(_regelbasis);
}


/*
 *---------------------------- lokale Funktionen -------------------------------
 */

void RgLrnNeuroFuzCtrl::_regeln_lern_phase1(FuzzyTyp system_fehler)
{
  boolean regel_geloescht;
  int rg_nr = 0;

  while(rg_nr < _anz_regeln) {
    regel_geloescht = false;

    /*
     * Anmerkung : Wenn eine Regel benutzt wurde, wird geprueft, ob sie einen
     *             unsinnigen Steuerwert geliefert hat, ansonsten werden die
     *             verschiedenen Zaehler aktualisiert und ausgewertet.
     */

    // unsinnige Regeln loeschen
    if(_regel_unsinnig(rg_nr, system_fehler)) {
      _loesch_regel(rg_nr);
      regel_geloescht = true;
    } else {
      if(_regeln[rg_nr]->treffer_grad() > _lern_strukt.min_treffergrad) {

        // Zaehler2 zuruecksetzen
        if(_lern_strukt.nach_dauer)
          _regeln[rg_nr]->zaehler2() = _lern_strukt.dauer;
      } else {

        // Regeln nach 'Prozent' loeschen
        if(_lern_strukt.nach_prozent)
          if(--_regeln[rg_nr]->zaehler1() <= 0) {
            _loesch_regel(rg_nr);
            regel_geloescht = true;
          }

        // Regeln nach 'Nutzungsdauer' loeschen
        if(_lern_strukt.nach_dauer && !regel_geloescht)
          if(--_regeln[rg_nr]->zaehler2() <= 0) {
            _loesch_regel(rg_nr);
            regel_geloescht = true;
          }
      }
    }

    if(!regel_geloescht)
      rg_nr++;
  }
}

void RgLrnNeuroFuzCtrl::_regeln_lern_phase2(FuzzyTyp fehler)
{
  for(int var = 0; var < _anz_ausgabe_var; var++)
    for(int rg_nr = 0; rg_nr < _anz_regeln; rg_nr++)
      if(_regeln[rg_nr]->treffer_grad() > 0) {
        _regeln[rg_nr]->zaehler2()++;
        _regeln[rg_nr]->fehler() += Abs(fehler);
      }
}

boolean RgLrnNeuroFuzCtrl::_regel_unsinnig(int rg_nr, FuzzyTyp system_fehler)
{
  float grenze = 0.1;    // Toleranz

  // Regel ist unsinnig <=> ein Steuerwert ist unsinnig
  for(int var_nr = 0; var_nr < _anz_ausgabe_var; var_nr++)
    if(_regeln[rg_nr]->steuer_wert(var_nr) < 0 && system_fehler > grenze)
      return true;
    else
      if(_regeln[rg_nr]->steuer_wert(var_nr) > 0 && system_fehler < -1 * grenze)
        return true;
  return false;
}

FuzzyTyp* RgLrnNeuroFuzCtrl::_rnd_steuerwerte(FuzzyTyp* eingaben)
{
  FuzzyTyp wert;

  _eingabe->messwerte(eingaben);

  for(int i = 0; i < _anz_ausgabe_var; i++) {
    _treffer_summe[i] = 0;
    _ausgabe[i]       = 0;
  }

  int fuz_nr;
  FuzzyTyp gewicht;
  for(int regel = 0; regel < _anz_regeln; regel++) {

    // Treffergrade fuer die Regeln berechnen
    FuzzyTyp treffer = 1;
    for(int var = 0; var < _anz_eingabe_var; var++)
      if((fuz_nr = _regeln[regel]->pra(var)) != -1)
        treffer = _t_norm(treffer, _eingabe->fuzzy_messwert(var, fuz_nr));

    if(treffer > 0 && rnd() > _rnd_grenze)
       treffer = 0;

    _regeln[regel]->treffer_grad(treffer);

    // Steuerwerte fuer jede Ausgabevariable berechnen
    for(var = 0; var < _anz_ausgabe_var; var++)
      if((fuz_nr = _regeln[regel]->kon(var)) != -1) {
        if(treffer == 0) {
          wert    = 0;
          gewicht = 0;
        } else
          wert = _defuz(
                   treffer,
                   _lingvarsatz->hol_lingvar(
                     AusgabeVar, var
                   )->hol_fuzzyset(fuz_nr),
                   gewicht
                 );
        _regeln[regel]->steuer_wert(var, wert);  // Wert merken
        _ausgabe[var] += wert * gewicht;         // Werte aufaddieren
        _treffer_summe[var] += gewicht;          // Gesamtgewicht bilden
      }
  }

  for(int var = 0; var < _anz_ausgabe_var; var++)
    if(_treffer_summe[var] != 0)
      _ausgabe[var] /= _treffer_summe[var];      // Mittelwert bilden
    else
      _ausgabe[var] = 0;                         // sollte nicht sein

  return _ausgabe;
}

void RgLrnNeuroFuzCtrl::_loesch_regel(int nr)
{
  //die geloeschten Regeln stehen hinten im Array
  if(nr != _anz_regeln - 1) {
    RegelKnoten* merk_zeiger = _regeln[nr];
    _regeln[nr] = _regeln[_anz_regeln - 1];
    _regeln[_anz_regeln - 1] = merk_zeiger;
  }
  _anz_regeln--;
}

void RgLrnNeuroFuzCtrl::_erzeuge_regelbasis(RegelBasis* regelbasis)
{
  // alte Regeln loeschen, falls vorhanden
  while(regelbasis->anzahl_regeln() > 0)
    regelbasis->del_regel(0);

  // die Regeln, deren Zaehler im folgenden auf -1 gesetzt werden, werden nicht
  // beruecksichtigt, da eine Regel mit gleicher Praemisse und kleinerem
  // Fehler existiert
  for(int rg_nr = 0; rg_nr < _anz_regeln; rg_nr++)
    if(_regeln[rg_nr]->zaehler1() != -1) {
      int beste_regel = rg_nr;
      for(int i = rg_nr + 1; i < _anz_regeln; i++)
        if(_gleiche_gruppe(beste_regel, i) && _regeln[i]->zaehler1() != -1) {
          if(_regeln[beste_regel]->fehler() > _regeln[i]->fehler()) {
            _regeln[beste_regel]->zaehler1() = -1;
            beste_regel = i;
          } else
            _regeln[i]->zaehler1() = -1;
        }
      _regeln[beste_regel]->mach_regel(_lingvarsatz, regelbasis);
      _regeln[beste_regel]->zaehler1() = -1;
    }

  _init_regelbasis(regelbasis);
}

boolean RgLrnNeuroFuzCtrl::_gleiche_gruppe(int rg_nr1, int rg_nr2)
{
  for(int var_nr = 0; var_nr < _anz_eingabe_var; var_nr++)
    if(_regeln[rg_nr1]->pra(var_nr) != _regeln[rg_nr2]->pra(var_nr))
      return false;
  return true;
}
