/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <IV-look/telltale.h>
#include <IV-look/kit.h>
#include <InterViews/layout.h>
#include <InterViews/action.h>
#include <InterViews/border.h>
#include <InterViews/monoglyph.h>

#include "texte.h"
#include "protokoll.h"
#include "lern_ed.h"
#include "control_ed.h"

/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : ControllerEd ---------------------
 *------------------------------------------------------------------------------
 */

declareActionCallback(ControllerEd);
implementActionCallback(ControllerEd);

/*
 *------------------------------------------------------------------------------
 * Konstruktor fuer ControllerEd
 * Parameter : Wissensbank*       - zugrundeliegende Wissensbank
 *             ProtokollStruktur* - was soll im Protokoll stehen
 *             Action*            - Callback
 * Zweck : konstruieren der Regleroberflaeche mit Protokollbereich
 *------------------------------------------------------------------------------
 */
ControllerEd::ControllerEd(
  Wissensbank* wissensbank, ProtokollStruktur* prot_strukt, Action* ende_cb
)
  : NfcDialogFenster(nil)
{
  _lern_faktor     = 0.1;        // per Default : Lernfaktor = 0.1
  _regeln_lernen   = 0;          // per Default : keine Regeln erlernen
  _fuz_optimieren  = 1;          // per Default : Fuzzy-Sets optimieren
  _prot_einfrieren = false;      // per Default : Protokoll nicht einfrieren
  _aktiv           = false;
  _merk_anzahl1    = -1;
  _merk_anzahl2    = -1;

  _wissensbank = wissensbank;
  _protokoll   = new NfcProtokoll(_wissensbank, prot_strukt);
  _ende_cb     = ende_cb;
  if(_ende_cb != nil)
    Resource::ref(_ende_cb);

  Coord rand      = 10;
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  NfcKit* nkit    = NfcKit::instance();
  Action* cb;

  _ausgabe_feld = new NfcAusgabe(new AusgabeFeld(20, 10));

  wkit->begin_style("NfcInfoStyle","NfcInfoFont");

  /*
   *   Grafikblock fuer Abfrage von 'Regel-Basis erlernen ?' erstellen
   */
  TelltaleGroup* group = new TelltaleGroup();
  Button* bt1 = wkit->radio_button(
                  group, Txt(173), new NfcAction(&_regeln_lernen, 0, nil)
                );
  Button* bt2 = wkit->radio_button(
                  group, Txt(174), new NfcAction(&_regeln_lernen, 1, nil)
                );
  if(_regeln_lernen)
    bt2->state()->set(TelltaleState::is_chosen, true);
  else
    bt1->state()->set(TelltaleState::is_chosen, true);

  _lernphase1_ed = new EingabeFeld(
                     7, _lern_strukt.schritte_phase1, nil
                   );
  _lernphase2_ed = new EingabeFeld(
                     7, _lern_strukt.schritte_phase2, nil
                   );
  Glyph* block1 = lkit->vbox(
                    lkit->vcenter(lkit->hbox(bt1, lkit->hglue(0, fil, 0))),
                    lkit->vglue(rand, 0, 0),
                    lkit->hbox(
                      lkit->vcenter(bt2),
                      lkit->vcenter(lkit->hflexible(_lernphase1_ed, 0, 0)),
                      lkit->vcenter(wkit->label(Txt(197))),
                      lkit->vcenter(lkit->hflexible(_lernphase2_ed, 0, 0)),
                      lkit->vcenter(wkit->label(Txt(175))),
                      lkit->hglue(0, fil, 0)
                    )
                  );

  /*
   *   Grafikblock fuer Abfrage von 'Fuzzy-Sets optimieren ?' erstellen
   */
  group = new TelltaleGroup();
  Button* bt3 = wkit->radio_button(
                  group, Txt(176), new NfcAction(&_fuz_optimieren, 0, nil)
                );
  Button* bt4 = wkit->radio_button(
                  group, Txt(177), new NfcAction(&_fuz_optimieren, 1, nil)
                );
  if(_fuz_optimieren)
    bt4->state()->set(TelltaleState::is_chosen, true);
  else
    bt3->state()->set(TelltaleState::is_chosen, true);

  cb = new ActionCallback(ControllerEd)(this, &ControllerEd::_setz_faktor);
  _lernfaktor_ed = new EingabeFeld(4, _lern_faktor, cb);

  Glyph* block2 = lkit->vbox(
                    lkit->vcenter(lkit->hbox(bt3, lkit->hglue(0, fil, 0))),
                    lkit->vglue(rand, 0, 0),
                    lkit->hbox(
                      lkit->vcenter(bt4),
                      lkit->vcenter(lkit->hflexible(_lernfaktor_ed, 0, 0)),
                      lkit->vcenter(wkit->label(Txt(178))),
                      lkit->hglue(0, fil, 0)
                    )
                  );

  /*
   *   Grafikblock fuer 'Protokoll einfrieren'
   */
  cb = new ActionCallback(ControllerEd)(this,&ControllerEd::_stop_prot);
  Button* bt5 = wkit->check_box(Txt(179), cb);
  if(_prot_einfrieren)
    bt5->state()->set(TelltaleState::is_chosen, true);
  else
    bt5->state()->set(TelltaleState::is_chosen, false);

  /*
   *   Grafikblock fuer 'Regel-Lern Optionen'
   */
  cb = new ActionCallback(ControllerEd)(this, &ControllerEd::_hol_lern_strukt);
  _lern_option_bt = wkit->check_box(Txt(211), cb);

  wkit->end_style();

  // Start / Stop / Ende - Buttons erstellen
  Action* cb1 = new ActionCallback(ControllerEd)(this,&ControllerEd::_start);
  Action* cb2 = new ActionCallback(ControllerEd)(this,&ControllerEd::_stop);
  Action* cb3 = new ActionCallback(ControllerEd)(this,&ControllerEd::_ende);
  Glyph* buttons = lkit->hbox(
                     lkit->hglue(rand),
                     wkit->push_button(Txt(136), cb1), lkit->hglue(rand),
                     wkit->push_button(Txt(137), cb2), lkit->hglue(rand),
                     wkit->push_button(Txt(138), cb3), lkit->hglue(rand)
                   );

  InputHandler* steuerung = new InputHandler(
                              nil, WidgetKit::instance()->style()
                            );
  steuerung->body(
    lkit->center(
      nkit->text_drauf(Txt(134),
        lkit->margin(
          lkit->vbox(
            block1,
            lkit->vglue(2 * rand, 0, 0),
            block2,
            lkit->vglue(2 * rand, 0, 0),
            lkit->vcenter(lkit->hbox(bt5, lkit->hglue(0, fil, 0))),
            lkit->vglue(rand, 0, 0),
            lkit->vcenter(lkit->hbox(_lern_option_bt, lkit->hglue(0, fil, 0))),
            lkit->vglue(2 * rand, 0, 0),
            lkit->vcenter(buttons)
          ), rand / 2, 0, 0, rand, 0, 0, 0, 0, 0, rand, 0, 0
        ), true
      )
    )
  );

  // damit immer nur ein Cursor sichtbar ist, alle Eingabefelder verbinden
  steuerung->append_input_handler(_lernphase1_ed->input_handler());
  steuerung->append_input_handler(_lernphase2_ed->input_handler());
  steuerung->append_input_handler(_lernfaktor_ed->input_handler());

  body(
    lkit->margin(
      lkit->vbox(
        lkit->center(_protokoll),
        lkit->vglue(rand, 0, 0),
        lkit->center(
          lkit->hbox(
            lkit->hflexible(steuerung, 0, 0),
            lkit->hglue(rand, 0, 0),
            lkit->center(nkit->text_drauf(Txt(40), _ausgabe_feld, true))
          )
        )
      ), rand, rand
    )
  );
}


ControllerEd::~ControllerEd()
{
  if(_ende_cb != nil)
    Resource::unref(_ende_cb);
  if(_wissensbank->controller() != nil)
    _wissensbank->controller()->protokoll(nil);
}


/*
 *------------------------------ lokale Funktionen -----------------------------
 */

void ControllerEd::_lies_werte()
{
  // Verbindung zur Anwendung herstellen; D.h. : anhand eingelesener Messwerte
  // werden vom Controller berechnete Steuerwerte an die Anwendung uebergeben
  FuzzyTyp* messwerte = _wissensbank->anwendung()->messwerte();
  if(messwerte == nil)

    // Controll-Vorgang neu starten ( Bsp. Pendelsim. : Pendel kippte um )
    _wissensbank->controller()->reset();
  else {
    _wissensbank->anwendung()->steuerwerte(
      _wissensbank->controller()->berechne_steuerwerte(messwerte)
    );

    // Fuer den Fall, dass Regeln erlernt werden, die Anzahl der Lernschritte
    // herunterzaehlen
    if(_regeln_lernen) {
      if(_merk_anzahl1 > 0) {
        _merk_anzahl1--;
        _lernphase1_ed->setz_wert(_merk_anzahl1);
      } else
        if(_merk_anzahl2 > 0) {
          _merk_anzahl2--;
          _lernphase2_ed->setz_wert(_merk_anzahl2);
        } else
          if(_merk_anzahl2 == 0) {
            _lernphase1_ed->setz_wert(_lern_strukt.schritte_phase1);
            _lernphase2_ed->setz_wert(_lern_strukt.schritte_phase2);
            _merk_anzahl1 = -1;
            _merk_anzahl2 = -1;
          } else

            // Anzahl Schritte raufzaehlen, in der Fuzzy-Sets optimiert wurden
            _zaehler++;
    } else

      // Anzahl der Schritte raufzaehlen, in der Fuzzy-Sets optimiert wurden
      _zaehler++;
  }
}


void ControllerEd::_fehler()
{
  // in der Kommunikation mit der Anwendung ist ein Fehler aufgetreten
  _wissensbank->anwendung()->unlink();
  _ausgabe_feld->ausgeben(Warnung, Txt(157));
}

void ControllerEd::_start()
{
  // Wissensbank auf Konsistenz testen
  if(_wissensbank->konsistent(true, _ausgabe_feld)) {

    // Controller erzeugen
    if(_regeln_lernen) {
      if(!_lernphase1_ed->hol_wert(_lern_strukt.schritte_phase1))
        _ausgabe_feld->ausgeben(Fehler, Txt(198));
      else
        if(!_lernphase2_ed->hol_wert(_lern_strukt.schritte_phase2))
          _ausgabe_feld->ausgeben(Fehler, Txt(199));
        else {
          _merk_anzahl1 = _lern_strukt.schritte_phase1;
          _merk_anzahl2 = _lern_strukt.schritte_phase2;

          _wissensbank->controller(
            new RgLrnNeuroFuzCtrl(
              _wissensbank->steuer_var(),
              _wissensbank->fehler_var(),
              _wissensbank->fehler_regeln(),
              _wissensbank->steuer_regeln_ref(),
              &_lern_strukt
            )
          );
          _protokoll->init((RegelBasis*) nil);
        }
    } else {
      _wissensbank->controller(
        new NeuroFuzController(
          _wissensbank->steuer_var(),
          _wissensbank->fehler_var(),
          _wissensbank->steuer_regeln(),
          _wissensbank->fehler_regeln()
        )
      );
     _protokoll->init(_wissensbank->steuer_regeln());
   }

    // Controller initialisieren
    _wissensbank->controller()->protokoll(_protokoll);
    if(_fuz_optimieren)
      _wissensbank->controller()->lern_faktor(_lern_faktor);
    else
      _wissensbank->controller()->lern_faktor(0);

    // Controller mit Anwendung verbinden
    Action *ok_cb, *nook_cb;
    ok_cb   = new ActionCallback(ControllerEd)(this,&ControllerEd::_lies_werte);
    nook_cb = new ActionCallback(ControllerEd)(this,&ControllerEd::_fehler);
    _wissensbank->anwendung()->link(ok_cb, nook_cb);

    // urspruengliche Steuer-Variablen ins Protokoll schreiben
    _protokoll->datei_start(_lern_faktor ,_wissensbank->steuer_var());

    _aktiv   = true;
    _zaehler = 0;
    _ausgabe_feld->ausgeben(Meldung, Txt(194));
  }
}

void ControllerEd::_stop()
{
  // Controller stoppen und LingVarSatz uebenehmen

  if(_aktiv) {

    // Steuer-Variablen vom Controller uebernehmen und protokollieren
    if(_wissensbank->controller() != nil)
      _wissensbank->steuer_var(_wissensbank->controller()->lingvarsatz());
    _protokoll->datei_ende(_zaehler, _wissensbank->steuer_var());


    _aktiv = false;
    if(_wissensbank->anwendung() != nil)
      _wissensbank->anwendung()->unlink();
    _ausgabe_feld->ausgeben(Meldung, Txt(195));
  }
}

void ControllerEd::_ende()
{
  // Controller-Oberflaeche beenden

  _stop();
  if(_ende_cb != nil)
    _ende_cb->execute();
}

void ControllerEd::_stop_prot()
{
  // Protokoll einfrieren bzw. auftauen

  _protokoll->stop(_prot_einfrieren = !_prot_einfrieren);
}

void ControllerEd::_setz_faktor()
{
  // Controller mit Lernfaktor aus Eingabefeld initialisieren

  if(_wissensbank->controller() != nil)
    if(_fuz_optimieren) {
      float faktor;
      if(_lernfaktor_ed->hol_wert(faktor) && faktor >= 0 && faktor <= 1) {
        _lern_faktor = faktor;
        _wissensbank->controller()->lern_faktor(_lern_faktor);
      } else {
        _ausgabe_feld->ausgeben(Fehler, Txt(180));
        _lernfaktor_ed->setz_wert(_lern_faktor);
      }
    }
}

void ControllerEd::_hol_lern_strukt()
{
  // Button zuruecksetzen
  _lern_option_bt->state()->set(TelltaleState::is_chosen, false);

  // Popup - Fenster erstellen
  LernEd* lern_ed = new LernEd(WidgetKit::instance()->style());
  Resource::ref(lern_ed);

  // einblenden
  if(lern_ed->einblenden(window(), &_lern_strukt))
    _lern_strukt = *lern_ed->lern_struktur();
  Resource::unref(lern_ed);
}
