(************************************************************************************)
(* module name: Fuzzy                                                               *)
(* project: simulation of a fuzzy pathplanning algorithm                            *)
(* author: markus stadler, IIIC/8                                                   *)
(* first implementation: 5/31/91                                                    *)
(* last change:          7/12/91                                                    *)
(************************************************************************************)

IMPLEMENTATION MODULE Fuzzy;

FROM FuzzyScanner IMPORT SymbolType, Descriptor, IdentName, OpenFile, CloseFile, Get,
                         LineNo, RealNumber, Identifier;
FROM Conversions  IMPORT IntToString;
FROM String       IMPORT Equal, AppendCh, Append, Length;
FROM FileSystem   IMPORT File, Response, Lookup, Close, WriteChar;
FROM System       IMPORT Allocate, Deallocate;
FROM SYSTEM       IMPORT TSIZE;

(* definition of constants *)

CONST
  maxFuzzyVars   = 64;    (* max number of fuzzy variables        *)
  maxFuzzyVals   = 512;   (* max number of fuzzy values           *)
  maxValPerVar   = 32;    (* max number of values per variable    *)
  maxFuzzyRules  = 128;   (* max number of fuzzy rules            *)

(* definition of types *)

TYPE
  FuzzyVar           = POINTER TO FuzzyVarDesc;
  FuzzyVarDesc       = RECORD
                         (* varID is index of table array                            *)
                         varName:  IdentName;                (* Name of the variable *)
                         curVal:   REAL;            (* current value of the variable *)
                         lowerB:   REAL;                  (* lower boundary of range *)
                         upperB:   REAL;                  (* upper boundary of range *)
                         noVals:   INTEGER;                      (* number of values *)
                         valTable: ARRAY [0..maxValPerVar-1] OF INTEGER;
                                                                (* indices of values *)
                       END; (* RECORD *)

  OperatorType       = (opAND, opOR, opNOT, opIS);
  Operator           = POINTER TO OperatorDesc;
  OperatorDesc       = RECORD
                         CASE opType: OperatorType OF
                           opAND, opOR:  leftOp, rightOp:  Operator;
                         | opNOT:        onlyOp:           Operator;
                         | opIS:         varID, valID:     INTEGER;
                         END; (* CASE *)
                       END; (* RECORD *)

  RuleTableEntry     =  RECORD
                          varID:      INTEGER;  (* target of assignment *)
                          valID:      INTEGER;  (* value of assignment  *)
                          expression: Operator; (* logic expression     *)
                        END; (* RECORD *)
                        
(* definition of global variables *)

VAR
  noVars:    INTEGER;
  VarTable:  ARRAY [0..maxFuzzyVars-1] OF FuzzyVar;    
  noVals:    INTEGER;
  ValTable:  ARRAY [0..maxFuzzyVals-1] OF FuzzyVal;
  noRules:   INTEGER;
  RuleTable: ARRAY [0..maxFuzzyRules-1] OF RuleTableEntry;
  
  (* used by parser procedures *)
  symbol:    SymbolType;   (* last symbol read                  *)
  errorF:    File;         (* file   descriptor of error output *)
  varD,
  ruleD,
  currentD:  Descriptor;   (* scanner deskriptors *)
  parsingOK: BOOLEAN;
  
(* utility procedures *)

(* parser procedures ; used by LoadRuleBase ******************************************)

PROCEDURE ERROR(s: ARRAY OF CHAR); (*------------------------------------------------*)
  (* Writes a propriate error message to the error file and sets parsingOK to FALSE *)
VAR i: INTEGER;               (* index  *)
    n: ARRAY [0..9] OF CHAR;  (* to convert numbers to strings *)
    ok: BOOLEAN;
BEGIN
  IntToString(LineNo(currentD), 6, 6, 10, n, ok);  (* write LineNumber *)
  FOR i:= 0 TO Length(n)-1 DO
    WriteChar(errorF, n[i]);
  END; (* FOR *)
  WriteChar(errorF, " ");
  IF currentD = varD THEN                          (* var- or rule-file ? *)
    WriteChar(errorF, "V");
  ELSE
    WriteChar(errorF, "R");
  END;
  WriteChar(errorF, ":");
  WriteChar(errorF, " ");
  FOR i:= 0 TO HIGH(s) DO                          (* write message *)
    WriteChar(errorF, s[i]);
  END; (* FOR *)
  WriteChar(errorF, 15C);
  parsingOK:=FALSE;
END ERROR;
  
  
PROCEDURE Synchronize(s: SymbolType); (*---------------------------------------------*)
  (* read symbols until 's' or EofSy found *)
BEGIN
  WHILE (symbol # s) AND (symbol # EofSy) DO
    symbol:= Get(currentD);
  END
END Synchronize;

PROCEDURE ParseFileName(VAR fileName: ARRAY OF CHAR; VAR done: BOOLEAN); (*----------*)
  (* parsing a filename which will be returned in 'filename' *)
BEGIN
  fileName[0]:= 0C;
  IF symbol = ColonSy THEN                       (* diskette drive ? *)
    AppendCh(fileName, ":");
    symbol:= Get(currentD);
    IF symbol = IdentSy THEN                     (* name of diskette *)
      Append(fileName, Identifier);
    ELSE
      (* ERROR *)
      done:= FALSE;
      RETURN
    END; (* IF *)
    symbol:= Get(currentD);            
    IF symbol = ColonSy THEN                   (* colon after diskname necessary *)
      AppendCh(fileName, ":");
    ELSE
      (* ERROR *)
      done:= FALSE;
      RETURN
    END; (* IF *)
    symbol:= Get(currentD);
  END; (* IF *)
  IF symbol = IdentSy THEN                    (* could be file or folder name *)
    Append(fileName, Identifier);
  ELSE
    (* ERROR *)
    done:= FALSE;
    RETURN
  END; (* IF *)
  symbol:= Get(currentD);
  WHILE symbol = ColonSy DO                    (* append next folder name/file name *)
    AppendCh(fileName, ":");
    symbol:= Get(currentD);
    IF symbol = IdentSy THEN
      Append(fileName, Identifier);
    ELSE
      (* ERROR *)
      done:= FALSE;
      RETURN
    END; (* IF *)
    symbol:= Get(currentD);
  END; (* DO *)
  IF symbol = PeriodSy THEN                  (* filename with extension ? *)
    AppendCh(fileName, ".");
    symbol:= Get(currentD);
    IF symbol = IdentSy THEN                 (* extension *)
      Append(fileName, Identifier);
      symbol:= Get(currentD);
    ELSE
      (* ERROR *)
      done:= FALSE;
      RETURN
    END; (* IF *)
  END; (* IF *)
  done:= TRUE;
END ParseFileName;



PROCEDURE ParseVars; (*--------------------------------------------------------------*)
  (* parse variable declaration file   *)
  
  PROCEDURE ParseVarDecl; (*---------------------------------------------------------*)
    (* parse the declaration of a variable *)
  
    PROCEDURE ParseValDecl(v: FuzzyVar; ID: INTEGER); (*-----------------------------*)
      (* parse the declaration of a value *)
    
    VAR i: INTEGER;
    BEGIN
      IF symbol = ValueSy THEN
        symbol:= Get(currentD);
        IF symbol = IdentSy THEN
          IF (v^.noVals<maxValPerVar) AND (noVals<maxFuzzyVals) THEN
            (* if there is enough room, allocate space for val and read info *)
            INC(v^.noVals); INC(noVals);
            v^.valTable[v^.noVals-1]:= noVals-1;
            Allocate(ValTable[noVals-1], TSIZE(FuzzyValDesc));
            ValTable[noVals-1]^.valName:= Identifier;
            ValTable[noVals-1]^.varID:= ID;
            symbol:= Get(currentD);
            IF symbol = OpenParSy THEN
              symbol:= Get(currentD);
              IF symbol = RealNoSy THEN
                ValTable[noVals-1]^.noValPoints:= TRUNC(RealNumber);
                symbol:= Get(currentD);
                IF symbol = CloseParSy THEN
                  symbol:= Get(currentD);
                ELSE
                  ERROR(") expected");
                END;
              ELSE
                ERROR("Number expected");
                ValTable[noVals-1]^.noValPoints:= maxValPoints;
                symbol:= Get(currentD);
              END;               
            ELSE 
              ValTable[noVals-1]^.noValPoints:= maxValPoints;
            END;
            IF symbol = IsSy THEN
              (* read membership function *)
              i:= 0;
              symbol:= Get(currentD);
              WHILE (i<ValTable[noVals-1]^.noValPoints) AND (symbol = RealNoSy) DO
                ValTable[noVals-1]^.mu[i]:= RealNumber;
                INC(i);
                symbol:= Get(currentD);
              END; (* WHILE *)
              IF i< ValTable[noVals-1]^.noValPoints THEN
                ERROR("more Realnumbers expected ");
              END;
            ELSE
              ERROR("IS or ( expected ");
              Synchronize(ValueSy);
            END;
          ELSE
            ERROR("too many fuzzy values defined");
            Synchronize(ValueSy);
          END;
        ELSE
          ERROR("Identifier expected");
          Synchronize(ValueSy);
        END
      ELSE
        ERROR("VALUE expected");
        Synchronize(ValueSy);
      END
    END ParseValDecl;
    
  BEGIN (* ParseVarDecl *) (*---------------------------------------------------------*)
    IF symbol = LingvarSy THEN
      IF noVars < maxFuzzyVars THEN
        INC(noVars);
        Allocate(VarTable[noVars-1], TSIZE(FuzzyVarDesc));
        symbol:= Get(currentD);
        IF symbol = IdentSy THEN
          VarTable[noVars-1]^.varName:= Identifier;
          VarTable[noVars-1]^.noVals:= 0;
          symbol:= Get(currentD);
          IF symbol = OnSy THEN
            symbol:= Get(currentD);
            IF symbol=OpenBraceSy THEN
              symbol:= Get(currentD);
              IF symbol = RealNoSy THEN
                VarTable[noVars-1]^.lowerB:= RealNumber;
                symbol:= Get(currentD);
                IF symbol = CommaSy THEN
                  symbol:= Get(currentD);
                  IF symbol = RealNoSy THEN
                    VarTable[noVars-1]^.upperB:= RealNumber;
                    symbol:= Get(currentD);
                    IF symbol = CloseBraceSy THEN
                      symbol:= Get(currentD);
                      IF symbol = WithSy THEN
                        symbol:= Get(currentD);
                        WHILE symbol = ValueSy DO
                          ParseValDecl(VarTable[noVars-1], noVars-1);
                        END;
                        IF symbol = EndSy THEN
                          symbol:= Get(currentD);
                        ELSE
                          ERROR("END expected");
                          Synchronize(LingvarSy);
                        END
                      ELSE
                        ERROR("WITH expected");
                        Synchronize(LingvarSy);
                      END
                    ELSE
                      ERROR("] expected");
                      Synchronize(LingvarSy)
                    END
                  ELSE
                    ERROR("Number expected");
                    Synchronize(LingvarSy);
                  END
                ELSE
                  ERROR("Comma expected");
                  Synchronize(LingvarSy);
                END   
              ELSE
                ERROR("Number expected");
                Synchronize(LingvarSy);
              END
            ELSE
              ERROR("[ expected ");
              Synchronize(LingvarSy);
            END
          ELSE
            ERROR("ON expected");
            Synchronize(LingvarSy);
          END
        ELSE
          ERROR("Identifier expected");
          Synchronize(LingvarSy);
        END
      ELSE
        ERROR("to many fuzzy variables");
        Synchronize(LingvarSy);
      END
    ELSE
      ERROR("LINGVAR expected");
      Synchronize(LingvarSy);
    END;
  END ParseVarDecl;
  
BEGIN (* ParseVars *) (*--------------------------------------------------------------*)
  symbol:= Get(currentD);
  WHILE symbol # EofSy DO
    ParseVarDecl;
  END
END ParseVars;


PROCEDURE GetValID(name: IdentName;varID: INTEGER;VAR valID: INTEGER;VAR found: BOOLEAN);
  (* find the valID of a value (name) belonging to variable varID  *)
BEGIN
  valID:= 0;
  WHILE (valID<noVals) AND NOT (Equal(name, ValTable[valID]^.valName) AND
         (ValTable[valID]^.varID = varID)) DO
    INC(valID);
  END;
  found:= valID<noVals
END GetValID;


PROCEDURE ParseRule; (*---------------------------------------------------------------*)
  (* parse a fuzzy rule *)
  
  PROCEDURE ParseLogicExpr():Operator; (*---------------------------------------------*)
    (* parse a logic expression *)
  VAR
    p: Operator;
    found: BOOLEAN;
  BEGIN
    Allocate(p, TSIZE(OperatorDesc));
    IF symbol = IdentSy THEN                      (* IS-operation *)
      p^.opType:= opIS;
      GetVarID(Identifier, p^.varID, found);
      IF NOT found THEN
        ERROR("variable not defined");
        Deallocate(p);
        RETURN NIL
      ELSE
        symbol:= Get(currentD);
        IF symbol = IsSy THEN
          symbol:= Get(currentD);
          IF symbol = IdentSy THEN
            GetValID(Identifier, p^.varID, p^.valID, found);
            IF NOT found THEN
              ERROR("value not defined or belongs to another variable");
              Deallocate(p);
              RETURN NIL
            END
          ELSE
            ERROR("Identifier expected ");
            Deallocate(p);
            RETURN NIL
          END
        ELSE
          ERROR("IS expected ");
          Deallocate(p);
          RETURN NIL
        END
      END
    ELSIF symbol = NotSy THEN                    (* NOT-operation *)
      p^.opType:= opNOT;
      symbol:= Get(currentD);
      IF symbol = OpenParSy THEN
        symbol:= Get(currentD);
        p^.onlyOp:= ParseLogicExpr();  (* parse operand *)
        IF p^.onlyOp = NIL THEN
          Deallocate(p);
          RETURN NIL
        ELSE
          IF symbol # CloseParSy THEN
            ERROR(") expected");
            RETURN NIL
          END
        END
      ELSE
        ERROR("( expected");
        RETURN NIL
      END
    ELSIF symbol = OpenParSy THEN              (* AND/OR-operation *)
      symbol:= Get(currentD);
      p^.leftOp:= ParseLogicExpr();  (* parse left operand  *)
      IF p^.leftOp = NIL THEN
        Deallocate(p);
        RETURN NIL
      ELSE
        IF symbol = CloseParSy THEN
          symbol:= Get(currentD);    (* get kind of operation *)
          IF symbol = OrSy THEN
            p^.opType:= opOR
          ELSIF symbol = AndSy THEN
            p^.opType:= opAND
          ELSE
            ERROR("AND or OR expected ");
            Deallocate(p);
            RETURN NIL
          END;
          symbol:= Get(currentD);
          IF symbol = OpenParSy THEN
            symbol:= Get(currentD);
            p^.rightOp:= ParseLogicExpr();  (* parse right operand *)
            IF p^.rightOp = NIL THEN
              Deallocate(p);
              RETURN NIL
            ELSE
              IF symbol # CloseParSy THEN
                ERROR(") expected");
                RETURN NIL
              END
            END
          ELSE
            ERROR("( expected");
            RETURN NIL
          END
        ELSE
          ERROR(") expected");
          RETURN NIL
        END
      END
    ELSE
      Deallocate(p);
      RETURN NIL;
    END;
    symbol:= Get(currentD);
    RETURN p
  END ParseLogicExpr;
    
VAR 
  expr:  Operator;
  ID:    INTEGER;
  found: BOOLEAN;
  
BEGIN (* ParseRule *) (*------------------------------------------------------------*)
  IF noRules < maxFuzzyRules THEN
    IF symbol = IfSy THEN
      symbol:= Get(currentD);
      expr:= ParseLogicExpr();
      IF expr # NIL THEN
        INC(noRules);
          RuleTable[noRules-1].expression:= expr;
          IF symbol = ThenSy THEN
            symbol:= Get(currentD);
            IF symbol = IdentSy THEN
              GetVarID(Identifier, ID, found);    (* find ID of variable *)
              IF found THEN
                RuleTable[noRules-1].varID:= ID;
                symbol:= Get(currentD);
                IF symbol = BecomesSy THEN
                  symbol:= Get(currentD);
                  IF symbol = IdentSy THEN
                    (* find ID of value *)
                    GetValID(Identifier, RuleTable[noRules-1].varID, ID, found);  
                    IF found THEN
                      RuleTable[noRules-1].valID:= ID;
                      symbol:= Get(currentD);
                      IF symbol = SemicolonSy THEN
                        symbol:= Get(currentD);
                      ELSE
                        ERROR("; expected");
                        Synchronize(IfSy);
                      END
                    ELSE
                      ERROR("fuzzy value not declared or belonging to another variable");
                      Synchronize(IfSy);
                    END
                  ELSE
                    ERROR("Identifier expected");
                    Synchronize(IfSy);
                  END 
                ELSE
                  ERROR(":= expected");
                  Synchronize(IfSy);
                END
              ELSE
                ERROR("fuzzy variable not declared");
                Synchronize(IfSy);
              END             
            ELSE
              ERROR("Identifier expected");
              Synchronize(IfSy);
            END
          ELSE
            ERROR("THEN expected");
            Synchronize(IfSy);
          END
      ELSE
        ERROR("logical expression invalid");
        Synchronize(IfSy);
      END
    ELSE
      ERROR("IF expected");
      Synchronize(IfSy);
    END
  ELSE
    ERROR("too many rules defined");
    Synchronize(IfSy);
  END;
END ParseRule;

PROCEDURE SimplifyRuleBase; (* -------------------------------------------------------*)
  (* puts all rules with the same assignment to one rule where all the logic   *)
  (* expressions are connected by OR; so there is only one rule per assignment *)
VAR
  iRule, j:   INTEGER;
  p:          Operator;
BEGIN
  iRule:= 0;
  WHILE iRule<noRules DO
    j:= iRule+1;
    WHILE j<noRules DO    
      IF (RuleTable[iRule].varID = RuleTable[j].varID) AND
          (RuleTable[iRule].valID = RuleTable[j].valID) THEN
        Allocate(p, TSIZE(OperatorDesc));
        p^.opType:= opOR;
        p^.leftOp := RuleTable[iRule].expression;
        p^.rightOp:= RuleTable[j].expression;
        RuleTable[iRule].expression:= p;
        RuleTable[j]:= RuleTable[noRules-1];
        DEC(noRules);
      ELSE
        INC(j)
      END; (* IF *)
    END; (* WHILE *)
    INC(iRule);
  END; (* WHILE *)
END SimplifyRuleBase;

(* exported procedures *)
(*-----------------------------------------------------------------------------------*)

PROCEDURE LoadRuleBase(RuleFileName: ARRAY OF CHAR; VAR ok: BOOLEAN); (*-------------*)
  (* loads the fuzzy variable definitions and the rule base in file 'fileName' *)
  (* into memory; if an error occured, ok  becomes FALSE (TRUE otherwise)      *)
VAR
  varFileName: ARRAY [0..127] OF CHAR;
  done:        BOOLEAN;
BEGIN
  noVals:=0; noVars:=0; noRules:=0;  (* resetting current rule base *)
  parsingOK:= TRUE;
  Lookup(errorF, "FuzzyErrors.txt", TRUE);
  OpenFile(RuleFileName, ruleD,  done);
  currentD:= ruleD;  (* read in rule-file *)
  IF done THEN
    symbol:= Get(currentD);
    IF symbol = FuzzyVarsSy THEN
      symbol:= Get(currentD);
      IF symbol = InSy THEN
        symbol:= Get(currentD);
        ParseFileName(varFileName, done);
        IF done THEN
          IF symbol = SemicolonSy THEN
            OpenFile(varFileName, varD, done);
            currentD:=varD;  (* read in var-file *)
            IF done THEN
              ParseVars;
              currentD:=ruleD;  (* read in rule-file again *)
              symbol:= Get(currentD);
              WHILE symbol = IfSy DO
                ParseRule;
              END;
              IF symbol # EofSy THEN
                ERROR("end of file expected");
              END;
              CloseFile(varD);
            ELSE
              ERROR("could not open variable description file");
            END
          ELSE
            ERROR("; expected");
          END
        ELSE
          ERROR("FileName not valid");
        END
      ELSE
        ERROR("IN expected ");
      END
    ELSE
      ERROR("FUZZYVAR expected");
    END;
    CloseFile(ruleD)
  ELSE
    ERROR("could not open rule description file")
  END;
  Close(errorF);
  ok:= parsingOK;
  IF ok THEN SimplifyRuleBase END;
END LoadRuleBase;


PROCEDURE GetVarID(varName: ARRAY OF CHAR; VAR ID: INTEGER; VAR found: BOOLEAN); (*--*)
  (* returns the identification number of the fuzzy variable 'varName' in  'ID' *)
  (* if a fuzzy variable with this name does not exist, found becomes FALSE     *)
BEGIN
  ID:= 0;
  WHILE (ID<noVars) AND NOT Equal(varName, VarTable[ID]^.varName) DO
    INC(ID);
  END;
  found:= ID<noVars
END GetVarID;


PROCEDURE SetCurrentVal(varID: INTEGER; newVal: REAL); (*-----------------------------*)
  (* sets the current value/point of the fuzzy varialbe 'varID' to 'newVal' *)
BEGIN
  IF varID < noVars THEN
    VarTable[varID]^.curVal:= newVal
  END; (* IF *)
END SetCurrentVal;


PROCEDURE CalculateVar(varID: INTEGER; VAR resVal: FuzzyValDesc; VAR resPoint: REAL);
  (* calculates the resulting fuzzyvalue from the rulebase and the current value/point *)
  (* 'resVal' is the resulting fuzzy value, 'resPoint' the new value/point             *)
  
  PROCEDURE CalcLogicExpr(expr: Operator):REAL;
    (* returns the value of the logic expresseion 'expr' *)
    VAR
      a, b, min, max: REAL;
  BEGIN
    IF expr^.opType = opIS THEN
      RETURN GetMuOfVal(ValTable[expr^.valID], VarTable[expr^.varID]^.curVal)
    ELSIF expr^.opType = opAND THEN
      RETURN CalcLogicExpr(expr^.leftOp)*CalcLogicExpr(expr^.rightOp)
    ELSIF expr^.opType = opOR THEN
      a:= CalcLogicExpr(expr^.leftOp);
      b:= CalcLogicExpr(expr^.rightOp);
      IF a>b THEN
        RETURN a
      ELSE
        RETURN b
      END; (* IF *)
    ELSE  (* NOT *)
      RETURN 1.0-CalcLogicExpr(expr^.onlyOp);
    END; (* IF *)

 (*
    IF expr^.opType = opIS THEN
      RETURN GetMuOfVal(ValTable[expr^.valID], VarTable[expr^.varID]^.curVal)
    ELSIF expr^.opType = opNOT THEN
      RETURN 1.0-CalcLogicExpr(expr^.onlyOp);
    ELSE (* AND or OR *)
      a:= CalcLogicExpr(expr^.leftOp);
      b:= CalcLogicExpr(expr^.rightOp);
      IF a < b THEN
        min := a; max := b
      ELSE
        min := b; max := a
      END;
      IF expr^.opType = opAND THEN
        RETURN min
      ELSE (* expr^.opType = opOR *)
        RETURN max
      END
    END; (* IF *)
  *)
  END CalcLogicExpr;
 
VAR
  i, j:   INTEGER;
  factor: REAL;
  width:  REAL;
  lower:  REAL;
  value:  REAL;
  
BEGIN
  IF varID < noVars THEN
    resVal.varID:= varID;
    resVal.noValPoints:= maxValPoints;
    FOR j:= 0 TO maxValPoints-1 DO resVal.mu[j]:= 0.0 END;      (* erase mu-table *)
    FOR i:= 0 TO noRules-1 DO                                   (* check all rules *)
      IF RuleTable[i].varID = varID THEN    (* if rule determines varID then continue *)
        factor:= CalcLogicExpr(RuleTable[i].expression);
        lower:= VarTable[varID]^.lowerB;
        width:= (VarTable[varID]^.upperB - lower);
        FOR j:= 0 TO maxValPoints-1 DO
          value:= factor*GetMuOfVal(ValTable[RuleTable[i].valID],
                                    lower+(FLOAT(j)+0.5)/FLOAT(maxValPoints)*width);
          IF value > resVal.mu[j] THEN  (* take maximum *)
            resVal.mu[j]:= value
          END
        END; (* FOR *)
      END; (* IF *)
    END; (* FOR *)
    (* find center of mass *)
    DeFuzzyVal(resVal, resPoint);
  END; (* IF *) 
 END CalculateVar;


PROCEDURE DeFuzzyVal(VAR val: FuzzyValDesc; VAR point: REAL); (*-----------------------*)
  (* defuzzies the  fuzzy value 'val' and returns result in 'point' *)
VAR
  i:         INTEGER;
  totWeight: REAL;
  sum      : REAL;
  width    : REAL;
BEGIN
  sum:= 0.0;
  totWeight:= 0.0;
  width:= (VarTable[val.varID]^.upperB-VarTable[val.varID]^.lowerB)/
                                                            FLOAT(val.noValPoints);
  FOR i:= 1 TO val.noValPoints DO
    totWeight:= totWeight+val.mu[i-1];
    sum      := sum+val.mu[i-1]*(0.5+FLOAT(i-1));
  END; (* FOR *)
  IF totWeight#0.0 THEN
    point:= VarTable[val.varID]^.lowerB+sum*width/totWeight;
  ELSE
    point:= (VarTable[val.varID]^.lowerB+VarTable[val.varID]^.upperB)/2.0;
  END; (* IF *)
END DeFuzzyVal;


PROCEDURE GetLowerUpper(varID: INTEGER; VAR lowerB, upperB: REAL); (*-----------------*)
  (* returns the lower and the upper boundary of the interval of variable 'varID' *)
BEGIN
  IF varID < noVars THEN
    lowerB:= VarTable[varID]^.lowerB;
    upperB:= VarTable[varID]^.upperB;
  END; (* IF *)
END GetLowerUpper;


PROCEDURE GetMuOfVal(val: FuzzyVal; point: REAL):REAL; (*----------------------------*)
  (* returns the value of the membership function mu of a fuzzy value 'val' *)
  (* at position  'point'                                                   *)
VAR
  index: INTEGER;
  lower, upper: REAL;
BEGIN
  lower:= VarTable[val^.varID]^.lowerB;
  upper:= VarTable[val^.varID]^.upperB;
  IF point < lower THEN
    RETURN val^.mu[0]
  ELSIF point > upper THEN
    RETURN val^.mu[val^.noValPoints-1]
  ELSE
    index:= TRUNC((point-lower)/(upper-lower)*FLOAT(val^.noValPoints));
    IF index = val^.noValPoints THEN
      RETURN val^.mu[index-1]
    ELSE
      RETURN val^.mu[index]
    END
  END
END GetMuOfVal;

BEGIN
  noVars:= 0; noVals:= 0; noRules:= 0;  (* resetting  rulebase *)
END Fuzzy.