MODULE CreateLingVarTable;

(************************************************************)
(*  Author       : Nicola Voutsis        12.12.1991         *)
(*  Project      : roboTRAC                                 *)
(*  Compiler     : MacMETH 2.6                              *)
(*  Version      : 1.0                                      *)
(*  Last Update  : 19.02.1992   By :  N. Voutsis            *)
(*  Description  : allows the definition and creation of    *)
(*                 the LINGVAR-Tables for Fuzzy Logic       *)
(*                                                          *)
(*  (c) Institute of Robotics  , ETH - Zurich               *)
(************************************************************)


FROM FileSystem IMPORT Lookup, Close, ReadChar, WriteChar,
                       File, Response;

FROM FileUtil IMPORT GetFileName;

FROM Menu IMPORT SetMenu, GetMenuCmd, MenuRes;

FROM InOut IMPORT ReadString, WriteString, WriteLn, EOL, Read;


CONST N = 10;            (* max. number of points that define the fuzzy
                            functions                                   *)
      IdentLength = 60;
      neg  = TRUE;
      pos  = FALSE;
      zero = ORD("0");


TYPE SymbolType = (identsy, real, openbrace, closebrace, hiphen,
                   eofsy, error);
     PtType     = RECORD
                    x, y : REAL
                  END;
     IdentType  = ARRAY [0..IdentLength-1] OF CHAR;
                                         (* identifier for scanner *)


VAR ch         : CHAR;
    ident      : IdentType;
    number     : IdentType;
    realnumber : REAL;
    fin        : File;
    fout       : File;


PROCEDURE FileWriteString(text : ARRAY OF CHAR);
(* output whole text blanks included *)
VAR i : CARDINAL;
BEGIN
  FOR i := 0 TO HIGH(text) DO
    WriteChar(fout,text[i])
  END
END FileWriteString;


PROCEDURE FileWriteIdent(text : ARRAY OF CHAR);
(* output text up to the first blank *)
VAR i   : CARDINAL;
    len : CARDINAL;
BEGIN
  len := HIGH(text);
  i:=0;
  WHILE (i<len) AND (text[i]<>0C) AND (text[i]<>" ") DO
    WriteChar(fout,text[i]);
    INC(i)
  END
END FileWriteIdent;


PROCEDURE FileWriteReal(real : REAL; n : CARDINAL);
(* output real with n significant digits *)
VAR x : CARDINAL;
    i : CARDINAL;
BEGIN
  IF real < 0.0 THEN
    WriteChar(fout,"-");
    real := -real
  END;
  x := TRUNC(real);
  WriteChar(fout,CHR(zero+x)); WriteChar(fout,".");
  FOR i := 1 TO n DO
    real := (real-FLOAT(x))*10.0;
    x := TRUNC(real);
    WriteChar(fout,CHR(zero+x));
  END (* FOR *)
END FileWriteReal;


PROCEDURE GetChar(VAR f : File; VAR ch : CHAR);
BEGIN
  ReadChar(f,ch);
(*
  WriteString(ch)
*)
END GetChar;


PROCEDURE GetReal(VAR f : File; sign : BOOLEAN) : SymbolType;
VAR i    : CARDINAL;
    temp : REAL;
BEGIN
  i := 0;
  realnumber := 0.0;
  IF sign THEN
    number[i] := "-"; INC(i)
  END;
  WHILE (i<IdentLength) AND
        (ch>="0") AND (ch<="9") DO
     (* ch IN ["0".."9"] *)
    number[i] := ch; INC(i);
    realnumber := 10.0*realnumber+FLOAT(ORD(ch)-zero);
    GetChar(fin,ch)
  END;
  IF (i>=IdentLength) THEN
    WriteString("Error: not a real number !"); WriteLn;
    RETURN error
  ELSIF ch = "." THEN
    number[i] := ch; INC(i);
   GetChar(fin,ch)
  ELSE
    WriteString("Error: not a real number !"); WriteLn;
    RETURN error
  END; (* IF *)
  temp := 0.1;
  WHILE (i<IdentLength) AND
        (ch>="0") AND (ch<="9") DO
     (* ch IN ["0".."9"] *)
    number[i] := ch; INC(i);
    realnumber := realnumber+temp*FLOAT(ORD(ch)-zero);
    temp := temp / 10.0;
    GetChar(fin,ch)
  END;
  IF i<IdentLength THEN
    (* at least one digit after the period expected *)
    IF temp < 0.1 THEN (* expected digit exists *)
      number[i] := 0C;
      IF sign THEN realnumber := -realnumber END;
      RETURN real
    ELSE
      WriteString("Error: digit after period expected !"); WriteLn;
      RETURN error
    END
  ELSE (* truncate superflous digits after the period *)
    WHILE (ch>="0") AND (ch<="9") DO
       (* ch IN ["0".."9"] *)
      GetChar(fin,ch)
    END;
    WriteString("Info: digits truncated after period !"); WriteLn;
    IF sign THEN realnumber := -realnumber END;
    RETURN real
  END (* IF *)
END GetReal;


PROCEDURE GetNext():SymbolType;
VAR i : CARDINAL;
BEGIN
  LOOP
    IF fin.eof THEN RETURN eofsy END;
    CASE ch OF
      "(" : GetChar(fin,ch);
            IF ch = "*" THEN
              GetChar(fin,ch);
              (* comment *)
              LOOP
                WHILE (ch <> "*") AND (ch <> 0C) DO
                  GetChar(fin,ch)
                END;
                IF ch ="*" THEN
                  GetChar(fin, ch);
                  IF ch = ")" THEN
                    GetChar(fin,ch);
                    EXIT
                  END
                ELSE
                  WriteString("Error: unexpected end within comments !");
                  WriteLn;
                  RETURN error
                END (* IF *)
              END (* LOOP *)
            ELSE
              RETURN openbrace
            END (* IF *)
    | "-" : GetChar(fin,ch);
            IF (ch >= "0") AND (ch <= "9") THEN
              RETURN GetReal(fin,neg)
            ELSE
              RETURN hiphen
            END
    | ")" : GetChar(fin,ch);
            RETURN closebrace
    | "a".."z","A".."Z" :
            i := 0;
            WHILE (i<IdentLength) AND
                  (((ch>="a") AND (ch<="z")) OR
                   ((ch>="A") AND (ch<="Z")) OR
                   ((ch>="0") AND (ch<="9"))) DO
                               (* ch IN ["a".."z","A".."Z","0".."9"] *)
              ident[i] := ch; INC(i);
              GetChar(fin,ch)
            END;
            IF i<IdentLength THEN
              ident[i] := 0C;
              RETURN identsy
            ELSIF ((ch>="a") AND (ch<="z")) OR
                  ((ch>="A") AND (ch<="Z")) OR
                  ((ch>="0") AND (ch<="9")) THEN
               (* ch IN ["a".."z","A".."Z","0".."9"] *)
              WriteString("Error: Ident too long !"); WriteLn;
              RETURN error
            ELSE
              RETURN identsy
            END;
    | "0".."9" :
            RETURN GetReal(fin,pos)
    ELSE (* ignore rest *)
      GetChar(fin,ch)
    END (* CASE *)
  END (* LOOP *)
END GetNext;


PROCEDURE Parse;
CONST digits = 3;
      realsize = 7;
TYPE PointType = ARRAY [1..N] OF RECORD
                                 p     : PtType;
                                 index : CARDINAL
                               END;
VAR symbol : SymbolType;
    upperbound, lowerbound : REAL;
    point : PointType;     (* points that define the fuzzy function *)
    cnt   : CARDINAL;
    ok    : BOOLEAN;


    PROCEDURE Calculate(VAR ok : BOOLEAN);
    (* parameters
           upperbound (upper boundary of linguistic variable),
           lowerbound (lower boundary of linguistic variable),
           point      (points that define fuzzy function),
           fout       (output file) and
           cnt         are global to this procedure *)
    (* all values have to be within the boundaries:       *)
    (*   upperbound and lowerbound in x-direction         *)
    (*   0.0 and 1.0 in y-direction                       *) 
    (* this is already guaranteed in the Parse porocedure *)
    VAR values   : ARRAY [0..63] OF REAL;
        i, j, k  : CARDINAL;
        temp     : PtType;
        interval : REAL;
    BEGIN (* Calculate *)
      ok := TRUE;
      (* sort points, small values correspond with low indexes *)
      FOR i:=1 TO cnt-1 DO
        k := i;                (* find the smallest and      *)
        FOR j:= i+1 TO cnt DO  (* exchange it with the first *)
          IF point[j].p.x < point[k].p.x THEN k:=j END;
        END;
        temp := point[i].p; point[i].p := point[k].p; point[k].p := temp
      END;
      (* partitioning of boundary *)
      interval := (upperbound - lowerbound) / 64.0;
      (* transform x values into indexes between 0 and 63 *)
      point[1].index := TRUNC((point[1].p.x - lowerbound) / interval);
      FOR i := 2 TO cnt DO
        point[i].index := TRUNC((point[i].p.x - lowerbound) / interval);
        (* due to numerical uncertainties the index might become 64 *)
        (* this should be changed into 63 *)
        IF point[i].index = 64 THEN
          point[i].index := 63
        ELSIF point[i].index > 64 THEN
          WriteString("=== This is a programing error ==="); WriteLn;
          WriteString("the idea is to divide the boundary given by"); WriteLn;
          WriteString("'lowerbound' and 'upperbound' into 64 equal"); WriteLn;
          WriteString("parts to make them fit into an indextable"); WriteLn;
          WriteString("of the size [0..63] !"); WriteLn
        END;
        (* two indexes must not be equal *)
        ok := ok AND (point[i].index <> point[i-1].index)
      END;
      IF NOT ok THEN
        WriteString("Error: descrete points too close to each other !");
        WriteLn;
        RETURN
      END;
      (* fill up the values up to the first point 
         according to the first point *)
      FOR i := 0 TO point[1].index DO
        values[i] := point[1].p.y
      END;
      (* calculate a whole line *)
      FOR i := 1 TO cnt-1 DO
        (* calculate the values of the line *)
        FOR j := 0 TO point[i+1].index-point[i].index DO
          values[point[i].index+j] := point[i].p.y +
                                      (point[i+1].p.y-point[i].p.y) /
                                      (point[i+1].p.x-point[i].p.x) *
                                      interval * FLOAT(j);
        END (* FOR *)
      END; (* FOR *)
      (* fill up the values up to the end
         according to the last point *)
      FOR i := point[cnt].index TO 63 DO
        values[i] := point[cnt].p.y
      END;
      (* output data *)
      FOR i := 0 TO 63 DO
        IF (i MOD 8) = 0 THEN
          FileWriteString("          ");
        END; (* IF *)
        FileWriteReal(values[i],digits);
        FOR j := realsize TO digits+2 BY -1 DO
          WriteChar(fout," ");
        END; (* FOR *)
        IF ((i+1) MOD 8) = 0 THEN
          WriteChar(fout,EOL)
        END (* IF *)
      END (* FOR *)
    END Calculate;


BEGIN
  symbol := GetNext();
  (* repeatopen ident ... *)       (* name of the linguistic variable *)
  WHILE symbol = identsy DO
    FileWriteString("LINGVAR ");
    FileWriteIdent(ident);
    FileWriteString(" ON [");
    symbol := GetNext();
    (* ... real ... *)             (* lower bound *)
    IF symbol = real THEN
      FileWriteIdent(number);
      lowerbound := realnumber;
      symbol := GetNext()
    ELSE
      WriteString("Error: real number expected !"); WriteLn;
      RETURN
    END; (* IF *)
    WriteChar(fout,",");
    (* ... real ... *)             (* upper bound *)
    IF symbol = real THEN
      FileWriteIdent(number);
      upperbound := realnumber;
      IF upperbound < lowerbound THEN
        WriteString("Error: upper boundary has to be greater than lower boundary !");
        WriteLn;
        RETURN
      END; (* IF *)
      FileWriteString("] WITH");
      WriteChar(fout, EOL);
      symbol := GetNext()
    ELSE
      WriteString("Error: real number expected !"); WriteLn;
      RETURN
    END; (* IF *)
    (* ... openrepeat "-" ... *)   (* keyword "-" *)
    WHILE symbol = hiphen DO
      symbol := GetNext();
      (* ... ident ...*)           (* name of the linguistic value *)
      IF symbol = identsy THEN
        FileWriteString("    VALUE ");
        FileWriteIdent(ident);
        FileWriteString("( 64) IS");
        WriteChar(fout, EOL);
        symbol := GetNext();
      ELSE
        WriteString("Error: ident expected !"); WriteLn;
        RETURN
      END; (* IF *)

      (* read points vor the linguistic values *)
      cnt := 0;
      (* ... openrepeat"(" ... *)  (* keyword "(" *)
      WHILE symbol = openbrace DO
        symbol := GetNext();
        INC(cnt);
        IF cnt > N THEN
          WriteString("Error: too many points given !"); WriteLn;
          RETURN
        END;
        (* ... real ... *)         (* x value of point *)
        IF symbol = real THEN
          (* value must be within boundaries *)
          IF (realnumber < lowerbound) OR (realnumber > upperbound) THEN
            WriteString("Error: real number exceeds x-range !"); WriteLn;
            RETURN
          END;
          point[cnt].p.x := realnumber;
          symbol := GetNext()
        ELSE
          WriteString("Error: real number expected !"); WriteLn;
          RETURN
        END; (* IF *)
        (* ... real ... *)           (* y value of point *)
        IF symbol = real THEN
          IF realnumber > 1.0 THEN (* it can't be <0.0 *)
            WriteString("Error: real number exceeds y-range !"); WriteLn;
            RETURN
          END;
          point[cnt].p.y := realnumber;
          symbol := GetNext()
        ELSE
          WriteString("Error: real number expected !"); WriteLn;
          RETURN
        END; (* IF *)
        (* ... ")" ... *)            (* ")" for point *)
        IF symbol = closebrace THEN
          symbol := GetNext()
        ELSE
          WriteString("Error: ')' expected !"); WriteLn;
          RETURN
        END (* IF *)
      (* ... repeatclose ... *)
      END; (* WHILE *)

      (* calculate the 64 values *)
      (* the points are NOT checked ! *)
      Calculate(ok);
      IF NOT ok THEN RETURN END;

      (* ... repeatclose ... *)
    END; (* WHILE *)
    FileWriteString("END");
    WriteChar(fout, EOL)
  (* ... repeatclose ... *)
  END; (* WHILE *)
  IF symbol <> eofsy THEN
    WriteString("Error: eof missing !"); WriteLn;
    RETURN
  END
END Parse;


PROCEDURE CreateOutputFileName(infileName      : IdentType;
                               VAR outfileName : IdentType;
                               VAR ok          : BOOLEAN);
VAR i : CARDINAL;
BEGIN
  i:=0;
  WHILE (i<IdentLength) AND (infileName[i]<>".") AND
        (infileName[i]<>" ") AND (infileName[i]<>0C) DO
    outfileName[i] := infileName[i];
    INC(i)
  END; (* WHILE *)
  ok := (i<IdentLength-3);
  outfileName[i]:="."; INC(i);
  outfileName[i]:="V"; INC(i);
  outfileName[i]:="A"; INC(i);
  outfileName[i]:="R"; INC(i);
  IF i<IdentLength THEN outfileName[i]:=0C END
END CreateOutputFileName;


CONST FileMenu = 1;
      Open     = 1;
      Quit     = 2;
VAR ok         : BOOLEAN;
    infile     : IdentType;
    outfile    : IdentType;
    menuRes    : MenuRes;


BEGIN (* main part *)
  (* install menues *) 
  SetMenu(FileMenu,"File|Open/O|Quit/Q");
  
  LOOP
    GetMenuCmd(menuRes,ok);
    IF ok THEN
      CASE menuRes.menuCmd OF
        Open : GetFileName(infile,"",ok);
               IF ok THEN
                   CreateOutputFileName(infile, outfile, ok);
                   IF NOT ok THEN
					   WriteString("Filename "); WriteString(infile);
					   WriteString(" too long !"); WriteLn; WriteLn
				   ELSE
					   Lookup(fin, infile, FALSE);
					   IF fin.res = notdone THEN
						   WriteString("Input file "); WriteString(infile);
						   WriteString(" could not be opened !"); WriteLn;
						   WriteLn
					   ELSE
						   WriteString("Input file "); WriteString(infile);
						   WriteString(" opened !"); WriteLn;
						   Lookup(fout, outfile, TRUE);
						   IF fout.res = notdone THEN
							   WriteString("Output file "); WriteString(outfile);
							   WriteString(" could not be created !"); WriteLn
						   ELSE
							   WriteString("Output file "); WriteString(outfile);
							   WriteString(" created !"); WriteLn;
							   (* read lookahead character *)
							   GetChar(fin,ch);
							   (* write output file *)
							   Parse;
							   Close(fout);
							   WriteString("Ouput file "); WriteString(outfile);
							   WriteString(" closed !"); WriteLn
						   END; (* IF fout.res = notdone *)
						   Close(fin);
						   WriteString("Input file "); WriteString(infile);
						   WriteString(" closed !"); WriteLn; WriteLn;
					   END (* IF fin.res = notdone *)
				   END (* IF NOT ok *)
               END (* IF ok *)
      | Quit : EXIT
      END (* CASE *)
    END (* IF *)
  END (* LOOP *)
  
END CreateLingVarTable.

