function [R] = imply(dA,dB,n)
%IMPLY  This function returns fuzzy implication of two matrix of fuzzy
%       values A and B, based on selected method of implication.
%       With selecting variable N from the following list,different
%       methods of implication will be governed.
%
%                 [R] = IMPLY(A,B,N)
%
%       1: Mamdani representation (Min operation rule)
%       2: Early Zadeh representation (Max-Min operation rule)
%       3: Zadeh/Lukasiewicz representation (Arithmetic operation rule)
%       4: Kleene-Dienes representation (Bollean operation rule)
%       5: Kleene-Dienes-Lukasiewicz representation
%       6: Larsen representation (Product operation rule)
%       7: Standard sequence fuzzy implication
%       8: Goguen representation
%                                                      See also CRI.


% FISMAT: Fuzzy Inference Systems toolbox for MATLAB
% (c) A. Lotfi, University of Queensland (Email: lotfia@s1.elec.uq.oz.au)
% 13-10-93
% The program has been tested on MATLAB version 4.1, Sun workstation.

% Checking the inputs of function


one=ones(length(dA),length(dB));
R0=zeros(size(one));

       if nargin ~= 3
error('     Inputs of function is not correct  ') ;
       end


if (max(max(dA)) > 1) | (max(max(dB)) > 1) | (min(min(dA)) < 0) | (min(min(dB)) < 0)
	error('Grade of membership function should not be more than one or less than zero.')
end


[ma,na]=size(dA);[mb,nb]=size(dB);

if ma ~= mb,
	error('The number of rules in antecedent and consequent are not proper')
end

ind=1;

DA=dA;DB=dB;

for ind=1:ma

    dA=DA(ind,:);dB=DB(ind,:);

% * Mamdani representation (Min operation rule)
%      (u,v)=min ( (u) ;  (v) )                  uU ; vV
%     Am            S       R
    if n == 1

    	R=min(diag(dA)*one,one*diag(dB));

% * Early Zadeh representation (Max-Min operation rule)
%      (u,v)=max(min ( (u) ;  (v) ) ; 1- (u) )  uU ; vV
%     Ae                S       R           S

    elseif n == 2

      	for i=1:length(dA)
      	for j=1:length(dB)
 		R(i,j)=max(min(dA(i),dB(j)) , 1-dA(i) );
     	end
      	end
% * Zadeh/Lukasiewicz representation (Arithmetic operation rule)
%       (u,v)=min(1; (1- (u)+ (v) )              uU ; vV
%      Az                 S     R
    elseif n == 3
    	  for i=1:length(dA)
      	for j=1:length(dB)
 		R(i,j)=min(1, (1-dA(i)+dB(j)));
      	end
      	end
% * Kleene-Dienes representation (Bollean operation rule)
%       (u,v)=max((1- (u)) ;  (v) )              uU ; vV
%      Ad              S        R

    elseif n == 4
      	for i=1:length(dA)
      	for j=1:length(dB)
 		R(i,j)=max(1-dA(i),dB(j));
      	end
      	end
% * Kleene-Dienes-Lukasiewicz representation
%       (u,v)=1- (u)+ (v) (u)                   uU ; vV
%      Ak         S     R    S

    elseif n == 5
      	for i=1:length(dA)
      	for j=1:length(dB)
 		R(i,j)=1-dA(i)+dA(i)*dB(j) ;
      	end
      	end
% * Larsen representation (Product operation rule)
%       (u,v)= (u) (v)                           uU ; vV
%      Al       S    R

    elseif n == 6
      	for i=1:length(dA)
      	for j=1:length(dB)
 		R(i,j)=dA(i)*dB(j);
      	end
      	end
% * Standard sequence fuzzy implication
%       (u,v)=1    (u) <=  (v)                   uU ; vV
%      As           S        R
%              0    (u) >   (v)
%                   S        R
    elseif n == 7
      	for i=1:length(dA)
      	for j=1:length(dB)
    		if dA(i) <= dB(j)
 	R(i,j)=1;
    else
 R(i,j)=0;
    	end
      end
      end
% * Goguen representation
%       (u,v)=1           (u) <=  (v)            uU ; vV
%      Ag                  S        R
%               (v)
%              -----       (u) >   (v)
%               (u)       S        R

    elseif n == 8
      	for i=1:length(dA)
      	for j=1:length(dB)
    		if dA(i) <= dB(j)
 		R(i,j)=1;
 	        else
 		R(i,j)=dB(j)/dA(i) ;
    		end
      	end
      	end

    else
error('  Argument n is not correct  ');
    end

 R0=max(R0,R);

end

R=R0;
