//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: SDLProcessData.h,v 0.22 1994/07/22 09:21:42 cncl-adm Exp cncl-adm $
 *
 * Class: CNSDLProcessData --- ...
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#ifndef __CNSDLProcessData_h
#define __CNSDLProcessData_h

/* 
   This class contains data belonging to an SDL process. It contains its
   save list, timers, holds special PId's and provides functions to access
   them, holds the memory address of the process and something more. It is
   used in SDLManager to describe an process but not its behaviour which
   is described in (SDLProcess).
*/

#include <CNCL/Class.h>
#include <CNCL/Param.h>

#include <CNCL/Object.h>		// Base class

extern CNClassDesc CN_SDLPROCESSDATA;	// Class CNSDLProcessData description

#include <CNCL/DLList.h>
#include <CNCL/SDLTimer.h>


class CNSDLSignal;
class CNSDLProcess;

typedef int PId;
typedef int ProcessType;

/*
 * The class CNSDLProcessData
 */

class CNSDLProcessData : public CNObject
{
    friend class CNSDLManager;  // is allowed to set address, parent etc.
    friend class CNSDLTimer;    // needs save_list for reset
    
  public:
    PId self();  // special PIds
    PId parent(); 
    PId offspring();
    PId sender();
    
    void save(CNSDLSignal *sig);
   
    CNSDLProcess *address();  // get memory address
    
    ProcessType type(); // get process type
    int max_number();   // and max number of processes
    
    // Private members
  private:
    void parent(PId);    // set PIds (only SDLManager)
    void offspring(PId);
    void sender(PId);
    
    void clear_savelist();  // (only SDLManager)
             
    void address(CNSDLProcess*);
    
    PId p_self;       // own PId (for control only)
    PId p_parent;     // the creator (is zero when created at system start)
    PId p_offspring;  // the last child created by this process
    PId p_sender;     // sender of last received signal
    
    CNSDLProcess *p_address;  // memory address of process
    
    CNDLList *save_list;  // list of saved signals (pointer is NIL if no
                          // save-list is needed
    CNSDLTimer *p_timer;  // timers of process (pointer is NIL if no
                          // timer is used
    
    ProcessType p_type;   // process_type (used for signal handling)
    int p_number;         // maximum number of processes of this type
    
    /***** Constructors ******************************************************/
  public:
    CNSDLProcessData() {}				// Default constructor
    CNSDLProcessData(CNParam *param) {}			// CNParam constructor
    
    CNSDLProcessData(PId, ProcessType, int, bool=FALSE, bool=FALSE);
    
    /***** Member functions required by CNCL *********************************/
  public:
    virtual CNClassDesc class_desc() const	// CNClass description
    {
	return CN_SDLPROCESSDATA;
    };
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    {
	return desc == CN_SDLPROCESSDATA ? TRUE : CNObject::is_a(desc);
    };
        
    static CNSDLProcessData *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
        return (CNSDLProcessData *)obj;
#   else
        return (CNSDLProcessData *)( !obj || obj->is_a(CN_SDLPROCESSDATA)
               ? obj : fatal_type(obj->class_desc(), CN_SDLPROCESSDATA) );
#   endif
    }
    
    // CNObject creation
    static CNObject *new_object(CNParam *param = NIL)
    {return param ? new CNSDLProcessData(param) : new CNSDLProcessData; }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
    virtual void dump (ostream &strm = cout) const;
};



// now the inlines

// some special PId's
inline PId CNSDLProcessData::self()      { return p_self; }
inline PId CNSDLProcessData::parent()    { return p_parent; }
inline PId CNSDLProcessData::offspring() { return p_offspring; }
inline PId CNSDLProcessData::sender()    { return p_sender; }

inline CNSDLProcess *CNSDLProcessData::address() { return p_address; }

inline void CNSDLProcessData::parent(PId pa)    { p_parent = pa; }
inline void CNSDLProcessData::offspring(PId ch) { p_offspring = ch; }
inline void CNSDLProcessData::sender(PId sd)    { p_sender = sd; }

inline void CNSDLProcessData::address(CNSDLProcess *pr) { p_address = pr; }

inline ProcessType CNSDLProcessData::type() { return p_type; }
inline int CNSDLProcessData::max_number()   { return p_number; }


#endif /**__CNSDLProcessData_h**/
