/*
   Copyright (c) 1991 Jyh-Shing Roger Jang, Dept of EECS, U.C. Berkeley
   BISC (Berkeley Initiative on Soft Computing) group
   jang@eecs.berkeley.edu
   Permission is granted to modify and re-distribute this code in any manner
   as long as this notice is preserved.  All standard disclaimers apply.
*/

#include "standard.h"

/* Usage of create_cubic, create_matrix and create_array: 
   1. Declaration in the calling program:
	char ***create_cubic();
	char **create_matrix();
	char *create_array();
	 
   2. Examples of invocations:
	my_cubic = (float ***)create_cubic(row, col, height, sizeof(float));
	my_matrix = (int **)create_matrix(row, col, sizeof(int));
	my_array = (double *)create_array(row+col, sizeof(double));
*/

char ***
create_cubic(row, col, height, element_size)
int row, col, height, element_size;
{
	char ***cubic;
	int i, j;

	cubic = (char ***)malloc(sizeof(char **)*row);
	if (cubic == NULL) {
		printf("Error in create_cubic!\n");
		exit(1);
	}
	for (i = 0; i < row; i++) { 
		cubic[i] = (char **)malloc(sizeof(char *)*col);
		if (cubic[i] == NULL) {
			printf("Error in create_cubic!\n");
			exit(1);
		}
		for (j = 0; j < col; j++) { 
			cubic[i][j] = (char *)malloc(element_size*height);
			if (cubic[i][j] == NULL) {
				printf("Error in create_cubic!\n");
				exit(1);
			}
		}
	}
	return(cubic);
}

char **
create_matrix(row_n, col_n, element_size)
int row_n, col_n, element_size;
{
	char **matrix;
	int i;

	matrix = (char **) malloc(sizeof(char *)*row_n);
	if (matrix == NULL) {
		printf("Error in create_matrix!\n");
		exit(1);
	}
	for (i = 0; i < row_n; i++) { 
		matrix[i] = (char *) malloc(element_size*col_n);
		if (matrix[i] == NULL) {
			printf("Error in create_matrix!\n");
			exit(1);
		}
	}
	return(matrix);
}

/* the following uses calloc instead of malloc */
/*
char **
create_matrix(row_n, col_n, element_size)
int row_n, col_n, element_size;
{
	char **matrix;
	int i;

	matrix = (char **)calloc(row_n, sizeof(char *));
	if (matrix == NULL) {
		printf("Error in create_matrix!\n");
		exit(1);
	}
	for (i = 0; i < row_n; i++) { 
		matrix[i] = (char *) calloc(col_n, element_size);
		if (matrix[i] == NULL) {
			printf("Error in create_matrix!\n");
			exit(1);
		}
	}
	return(matrix);
}
*/

char *
create_array(array_size, element_size)
int array_size, element_size;
{
	char *array;

	array = (char *)malloc(array_size*element_size);
	if (array == NULL) {
		printf("Error in create_array!\n");
		exit(1);
	}
	return(array);
}

/* an friendly interface to fopen() */
FILE *
open_file(file, mode)
char *file, *mode;
{
	FILE *fp, *fopen();

	if ((fp = fopen(file, mode)) == NULL){
		printf("Cannot open '%s'.\n", file);
		exit(1);
	}
	return(fp);
}

void
free_cubic(cubic, row, col)
char ***cubic;
int row, col;
{
	int i, j;

	for (i = 0; i < row; i++) { 
		for (j = 0; j < col; j++) 
			if (cubic[i][j] == NULL) {
				printf("free_cubic: already free!\n");
				exit(1);
			} else {
				free(cubic[i][j]);
				cubic[i][j] = NULL;
			}

		if (cubic[i] == NULL) {
			printf("free_cubic: already free!\n");
			exit(1);
		} else {
			free(cubic[i]);
			cubic[i] = NULL;
		}
	}
	if (cubic == NULL) {
		printf("free_cubic: already free!\n");
		exit(1);
	} else {
		free(cubic);
		cubic = NULL;
	}
}

void
free_matrix(matrix, row_n)
char **matrix;
int row_n;
{
	int i;

	for (i = 0; i < row_n; i++) 
		if (matrix[i] == NULL) {
			printf("free_matrix: row %d is already free!\n", i);
			exit(1);
		} else {
			free(matrix[i]);
			matrix[i] = NULL;
		}

	if (matrix == NULL) {
		printf("free_matrix: given matrix is already free!\n");
		exit(1);
	} else {
		free(matrix);
		matrix = NULL;
	}
}

void
free_array(array)
char *array;
{
	if (array == NULL) {
		printf("free_array: already free!\n");
		exit(1);
	} else {
		free(array);
		array = NULL;
	}
}

void 
print_matrix(matrix, row_n, col_n)
double **matrix;
int row_n, col_n;
{
	int i, j;
	for (i = 0; i < row_n; i++) {
		for (j = 0; j < col_n; j++)
			printf("%lf ", matrix[i][j]);
		printf("\n");
	}
}

void
print_array(array, size)
double *array;
int size;
{
	int i;
	for (i = 0; i < size; i++)
		printf("%lf ", array[i]);
	printf("\n");
}

void 
write_matrix(matrix, row_n, col_n, file_name)
double **matrix;
int row_n, col_n;
char *file_name;
{
	FILE *fp;
	int i, j;

	fp = open_file(file_name, "w");
	for (i = 0; i < row_n; i++) {
		for (j = 0; j < col_n; j++)
			fprintf(fp, "%lf ", matrix[i][j]);
		fprintf(fp, "\n");
	}
}

/* write array to file 'file_name', using matlab format */
/* the matlab variable name is the baseanme of 'file_name' */
void
write_array(array, size, file_name)
double *array;
int size;
char *file_name;
{
	FILE *fp;
	int i;

	fp = open_file(file_name, "w");
	for (i = 0; i < size; i++) 
		fprintf(fp, "%d %lf\n", i+1, array[i]);
	fclose(fp);
}

/* return the pointer of the  baseanme in given path */
char *
basename(path)
char *path;
{
	char separator = '/';
	char *index;

	index = rindex(path, separator);
	if (index == NULL)
		return(path);	
	else
		return(index + 1);
}

/* get current directory */
/* same as getwd() */
void
get_pwd(pwd)
char *pwd;
{
	FILE *pwdpipe;
	FILE *popen();

	pwdpipe = popen("pwd", "r");
	if (pwdpipe == NULL) {
		printf("Cannot get pwd!\n");
		exit(1);
	}
	fscanf(pwdpipe, "%s", pwd);
	pclose(pwdpipe);
}

void
exit1(s)
char *s;
{
	printf("%s\n", s);
	exit(1);
}

/* matrix operations */

/* matrix plus matrix */
void
m_plus_m(m1, m2, row, col, out)
double **m1, **m2, **out;
int row, col;
{
	int i, j;
	for (i = 0; i < row; i++)
		for (j = 0; j < col; j++)
			out[i][j] = m1[i][j] + m2[i][j];
}

/* matrix minus matrix */
void
m_minus_m(m1, m2, row, col, out)
double **m1, **m2, **out;
int row, col;
{
	int i, j;
	for (i = 0; i < row; i++)
		for (j = 0; j < col; j++)
			out[i][j] = m1[i][j] - m2[i][j];
}

/* matrix times matrix */
void
m_times_m(m1, m2, row1, col1, col2, out)
double **m1, **m2, **out;
int row1, col1, col2;
{
	int i, j, k;
	for (i = 0; i < row1; i++)
		for (j = 0; j < col2; j++) {
			out[i][j] = 0;
			for (k = 0; k < col1; k++)
				out[i][j] += m1[i][k]* m2[k][j];
		}
}

/* scalar times matrix */
void
s_times_m(c, m, row, col, out)
double c;
double **m, **out;
int row, col;
{
	int i, j;
	for (i = 0; i < row; i++)
		for (j = 0; j < col; j++)
			out[i][j] = c*m[i][j];
}

/* matrix transpose */
void
m_transpose(m, row, col, m_t)
double **m, **m_t;
int row, col;
{
	int i, j;
	for (i = 0; i < row; i++)
		for (j = 0; j < col; j++)
			m_t[j][i] = m[i][j];
}

/* matrix L-2 norm */
double
m_norm(m, row, col)
double **m;
int row, col;
{
	int i, j;
	double total = 0;

	for (i = 0; i < col; i++)
		for (j = 0; j < row; j++)
			total += m[i][j]*m[i][j];
	return(sqrt(total));
}
