;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; wxShower.clp
;;
;; Implements the Fuzzy Shower example using wxFuzzyCLIPS
;; for the user interface to the shower
;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Global definitions used in the simulation
;;

(defglobal 

   ?*coldValvePos*     = 0.0
   ?*hotValvePos*      = 0.0
   ?*coldTemp*         = 5.0
   ?*hotTemp*          = 55.0
   ?*coldPress*        = 55.0
   ?*hotPress*         = 55.0
   ?*optimalTempMin*   = 34.0
   ?*optimalTempMax*   = 38.0
   ?*optimalFlowMin*   = 11.0
   ?*optimalFlowMax*   = 13.0
   ?*atmosphericPress* = 30.0
   ?*iterationFactor*  = 1.0
   ?*outFlow*          = 0.0
   ?*outTemp*          = 0.0
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; globals used by the interface
;;

(defglobal ?*main-frame* = 0)
(defglobal ?*panel* = 0)
(defglobal ?*canvas* = 0)
(defglobal ?*text-win* = 0)
(defglobal ?*hot-valve-slider* = 0)
(defglobal ?*cold-valve-slider* = 0)
(defglobal ?*hot-temp-slider* = 0)
(defglobal ?*cold-temp-slider* = 0)
(defglobal ?*hot-press-slider* = 0)
(defglobal ?*cold-press-slider* = 0)
(defglobal ?*outflow-slider* = 0)
(defglobal ?*outtemp-slider* = 0)
(defglobal ?*manual-auto-check-box* = 0)
(defglobal ?*manual-auto* = "manual")


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Deftemplates below define the fuzzy variables being used
;;
;;
;; Outputs of the shower
;;
;;   outTemp  -- output temperature
;;
;;  outFlow   -- flow out of the shower

(deftemplate outTemp
   5 65 Celcius
  ((none  (5 1) (5.1 0))
   (cold  (z 10 35))
   (OK    (pi  2 36))
   (hot   (s 37  60)) )
  ())

(deftemplate outFlow
   0 100 liters/minute
  ((none   (0 1) (0.05 0))
   (low    (z 3 11.5))
   (OK     (pi  1 12))
   (strong (s 12.5  25)) )
  () )
  
  
;; controls for hot and cold valve positions
;;


(deftemplate change_vc
   -1 1
  ((NB (-0.5 1) (-.25 0))
   (NM  (-.35 0) (-.3 1) (-.15 0))
   (NS (-.25 0) (-.15 1) (0 0))
   (Z (-.05 0) (0 1) (.05 0))
   (PS (0 0) (.15 1) (.25 0))
   (PM (.15 0) (.3 1) (.35 0))
   (PB (.25 0)(0.5 1)) )
  ( ; modifier definitions
;        (very         sqr)
;        (less         sqrt)
  )
)


(deftemplate change_vh
   -1 1
  ((NB (-0.5 1) (-.25 0))
   (NM  (-.35 0) (-.3 1) (-.15 0))
   (NS (-.25 0) (-.15 1) (0 0))
   (Z (-.05 0) (0 1) (.05 0))
   (PS (0 0) (.15 1) (.25 0))
   (PM (.15 0) (.3 1) (.35 0))
   (PB (.25 0)(0.5 1)) )
  ( ; modifier definitions
;        (very         sqr)
;        (less         sqrt)
  )
)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Some supporting functions that do simulation of the Shower
;;
;;
;; initShower - initializes certain global values and requests some
;;              initial values from the user and calls simulate to
;;              calc the out flow and temp and assert these as
;;              fuzzy facts to start the fuzzy rules going
;;
;; Simulate   - calculates the new outFlow and outTemp when changes
;;              to valve positions are made -- also asserts the
;;              fuzzified facts representing these new values --
;;              also checks for finished state (temp and flow in range
;;              or for certain error conditions
;;
;; calc-outFlow-and-outTemp  - calculate the value of the output flow
;;              and temperature when something changes
;;
;; shutdown-and-set-to-manual - shut off the system and set the control to
;;              manual mode so that the operator can try to fix the problem
;;


(deffunction calc-outFlow-and-outTemp ()
   (bind ?hotFlow (* ?*hotValvePos* (- ?*hotPress* ?*atmosphericPress*)))
   (bind ?coldFlow (* ?*coldValvePos* (- ?*coldPress* ?*atmosphericPress*)))
   (bind ?*outFlow* (+ ?hotFlow ?coldFlow))
   (if (= ?*outFlow* 0.0)
      then
         (bind ?*outTemp* 5.0)
      else
         (bind ?*outTemp* (/ (+ (* ?coldFlow ?*coldTemp*)
                                (* ?hotFlow ?*hotTemp*))
                             ?*outFlow*)
         )
   )
   (slider-set-value ?*outtemp-slider* (integer (+ 0.5 ?*outTemp*)))
   (slider-set-value ?*outflow-slider* (integer (+ 0.5 ?*outFlow*)))
)


(deffunction shutdown-and-set-to-manual ()
  (bind ?*coldValvePos* 0.0)
  (bind ?*hotValvePos* 0.0)
  (slider-set-value ?*cold-valve-slider* 0)
  (slider-set-value ?*hot-valve-slider*  0)
  (bind ?*manual-auto* "manual")
  (check-box-set-value ?*manual-auto-check-box* 0)
  (calc-outFlow-and-outTemp)
)


(deffunction Simulate (?coldValveChange ?hotValveChange)

  ;; Check for some rather bad situations where control is lost
  ;;
  (if (and (= ?*coldValvePos* 1.0) (> ?coldValveChange 0.0) (< (abs ?hotValveChange) 0.0001))
    then
      (frame-set-status-text ?*main-frame* 
         "*** Cold Water Problem: Shutting Down - Cold valve full open and trying to open more ***")
                       
      (bind ?*iterationFactor* 1.0)  ;; reset iteration Factor
;;      (halt)
      (shutdown-and-set-to-manual)
      (return)
  )
  (if (and (= ?*hotValvePos* 1.0) (> ?hotValveChange 0.0) (< (abs ?coldValveChange) 0.0001))
    then
      (frame-set-status-text ?*main-frame* 
         "*** Hot Water Problem: Shutting Down - Hot valve full open and trying to open more ***")
                       
      (bind ?*iterationFactor* 1.0)  ;; reset iteration Factor
;;      (halt)
      (shutdown-and-set-to-manual)
      (return)
  )
  (if (and (= ?*coldValvePos* 0.0) (< ?coldValveChange 0.0) (< (abs  ?hotValveChange) 0.0001)) 
    then
      (frame-set-status-text ?*main-frame* 
         "*** HOT TEMPERATURE PROBLEM: Shutting down - cannot get temperature high enough")
                       
      (bind ?*iterationFactor* 1.0)  ;; reset iteration Factor
;;      (halt)
      (shutdown-and-set-to-manual)
      (return)
  )
  (if (and (= ?*hotValvePos* 0.0) (< ?hotValveChange 0.0) (< (abs ?coldValveChange) 0.0001)) 
    then
      (frame-set-status-text ?*main-frame* 
         "*** HOT TEMPERATURE PROBLEM: Shutting down - cannot get temperature low enough")

      (bind ?*iterationFactor* 1.0)  ;; reset iteration Factor
;;      (halt)
      (shutdown-and-set-to-manual)
      (return)
  )
  
  ;; calc new hot and cold valve positions based on the recommendations
  ;; for change provided 
  ;; NOTE: we perform a scaling factor on the recommendations making the assumption
  ;;       that at high valve settings a change will produce less effect than
  ;;       if we are operating at a low valve setting; this is due to the effect
  ;;       of pressures -- low pressures will tend to put us at higher valve
  ;;       positions; of course in a REAL situation the actual operating conditions
  ;;       of the system would likely be known and we couuld adjust accordingly;
  ;;       One problem that can occur is that the adjustments cause the system
  ;;       to be too course in its adjustment and it makes a change to go up and then
  ;;       a symmetric change to go down -- it will then endlessly flip flop between
  ;;       a too high and a too low condition -- this is solved by reducing the scale
  ;;       factor after each iteration so the adjustments get smaller and smaller.
  ;;       Also note that we could also tune the fuzzy sets to reduce the range 
  ;;       of changes recommended.
  ;;       REMEMBER the conditions of the problem -- we do not know the temps or
  ;;       pressures of hot and cold supplies
  
  (if (or (<> ?coldValveChange 0.0) (<> ?hotValveChange 0.0))
     then
       (bind ?*coldValvePos* 
                 (max 0.0
                      (min 1.0 
                           (+ ?*coldValvePos* 
                              (* (* (max 0.1 ?*coldValvePos*) ?*iterationFactor*)
                                 ?coldValveChange
                              )
                           )
                      )
                 )
       )
       (bind ?*hotValvePos* 
                 (max 0.0
                      (min 1.0 
                           (+ ?*hotValvePos* 
                              (* (* (max 0.1 ?*hotValvePos*) ?*iterationFactor*)
                                 ?hotValveChange
                              )
                           )
                      )
                 )
       )
       (slider-set-value ?*cold-valve-slider* 
            (integer (+ 0.5 (* ?*coldValvePos*  100.0))))
       (slider-set-value ?*hot-valve-slider* 
            (integer (+ 0.5 (* ?*hotValvePos* 100.0))))
   )

   ;; calculate the new outputs - flow and temp
   (calc-outFlow-and-outTemp)
   
   ;; if both output flow and temp are within correct range control is
   ;; completed -- halt
   ;;
   
   (if (and (and (> ?*outFlow* ?*optimalFlowMin*) (< ?*outFlow* ?*optimalFlowMax*))
            (and (> ?*outTemp* ?*optimalTempMin*) (< ?*outTemp* ?*optimalTempMax*))
       )
      then    
        (frame-set-status-text ?*main-frame* 
             "Shower is under control!")
        (bind ?*iterationFactor* 1.0)  ;; reset iteration Factor
;;        (halt)
        (return)
   )

   (frame-set-status-text ?*main-frame* 
        "Fuzzy Control in effect - watch valve changes.")

   ;; iterationFactor reduces each cycle through
   (if (> ?*iterationFactor* 0.1)
     then
       (bind ?*iterationFactor* (* ?*iterationFactor* .96))
   )
   
   ;; now assert the fuzzified values of the Flow and Temp of Output
   (if (<= ?*outFlow* 0.0)
     then
       (assert (outFlow (0.0 1.0) (0.5 0.0)))
     else
       (if (>= ?*outFlow* 100.0)
         then
           (assert (outFlow (99.5 0.0) (100.0 1.0)))
         else
           (assert (outFlow ((max 0.0 (- ?*outFlow* 0.5)) 0.0) (?*outFlow* 1.0)
                            ((min 100.0 (+ ?*outFlow* 0.5)) 0.0) ))
       )
   )
   (if (<= ?*outTemp* 5.0)
     then
       (assert (outTemp (5.0 1.0) (5.1 0.0)))
     else
       (if (>= ?*outTemp* 65.0)
         then
           (assert (outTemp (64.9 0.0) (65.0 1.0)))
         else
           (assert (outTemp ((max 5.0 (- ?*outTemp* 0.1)) 0.0) (?*outTemp* 1.0)
                            ((min 65.0 (+ ?*outTemp* 0.1)) 0.0) ))
       )
   )
)




(deffunction initShower ()
   (slider-set-value  ?*cold-valve-slider* 
            (integer (+ 0.5 (* ?*coldValvePos* 100.0))))
   (slider-set-value  ?*hot-valve-slider*
            (integer (+ 0.5 (* ?*hotValvePos* 100.0))))
   (slider-set-value  ?*cold-temp-slider* ?*coldTemp*)
   (slider-set-value  ?*hot-temp-slider* ?*hotTemp*)
   (slider-set-value  ?*cold-press-slider* ?*coldPress*)
   (slider-set-value  ?*hot-press-slider* ?*hotPress*)
)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Rules that control the shower with fuzzy inferencing
;;

(defrule none_none "will jump start an off system"
   (outTemp none)
   (outFlow none)
   =>
   (assert (change_vh PS))
   (assert (change_vc PM))
)


(defrule cold_low
   (outTemp cold)
   (outFlow low)
   =>
   (assert (change_vh PB))
   (assert (change_vc Z))
)

(defrule cold_OK
    (outTemp cold)
    (outFlow OK)
   =>
    (assert (change_vh  PM))
    (assert (change_vc Z))
)

(defrule cold_strong
    (outTemp cold)
    (outFlow strong)
   =>
    (assert (change_vh Z))
    (assert (change_vc NB))
)

(defrule OK_low
   (outTemp OK)
   (outFlow low)
  =>
   (assert (change_vh  PS))
   (assert (change_vc  PS))
)

(defrule OK_strong
   (outTemp OK)
   (outFlow strong)
  =>
   (assert (change_vh  NS))
   (assert (change_vc NS))
)


(defrule hot_low
  (outTemp hot)
   (outFlow low)
  =>
   (assert (change_vh Z))
   (assert (change_vc  PB))
)

(defrule hot_OK
   (outTemp hot)
   (outFlow OK)
  =>
   (assert (change_vh NM))
   (assert (change_vc Z))
)

(defrule hot_strong
   (outTemp hot)
   (outFlow strong)
  =>
   (assert (change_vh NB))
   (assert (change_vc Z))
)

;; when all rules have fired and contributed to determination of the
;; control changes to be made then defuzzify the change values and
;; pass these new values to the simulate function

(defrule defuzzification-and-control
   (declare (salience -1))
   ?f1 <- (change_vc ?)
   ?f2 <- (change_vh ?)
   ?f3 <- (outTemp ?)
   ?f4 <- (outFlow ?)
 =>
   (bind ?coldChange (moment-defuzzify ?f1))
   (bind ?hotChange  (moment-defuzzify ?f2))
   (retract ?f1 ?f2 ?f3 ?f4)
   (Simulate ?coldChange ?hotChange)
)




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Functions that define the user interface to the Shower
;; Example
;;
;;
;; app-on-init   - start the application by executing this fcn
;;
;;


;;; Callback for Main Frame -- ?*main-frame*
;;; Sizing callback
(deffunction on-size (?id ?w ?h)
 (if (and (> ?id 0) (> ?*panel* 0) (> ?*canvas* 0)) then
;  (window-fit ?*panel*)
 )
)

;;; close callback
(deffunction on-close (?frame)
 (format t "Closing frame.%n")
 1
)

;;; menu callback
(deffunction on-menu-command (?frame ?id)
(printout t ?id "******" crlf)
 (if (eq ?id 201) ;; About selection
  then 
    (message-box "FuzzyCLIPS Shower Demo
by National Research Council of Canada" "About Shower Demo")
  else
    (if (eq ?id 1) ;; Quit selection
     then
       (on-close ?frame)
       (window-delete ?frame)
    )
 )
)

;;; Callbacks for sliders
(deffunction on-outtemp (?id) ;; cannot change this manually
  (slider-set-value ?*outtemp-slider* (integer (+ 0.5 ?*outTemp*)))
)

(deffunction on-outflow (?id) ;; cannot change this manually
  (slider-set-value ?*outflow-slider* (integer (+ 0.5 ?*outFlow*)))
)

(deffunction on-hot-temp (?id)
  (bind ?*hotTemp* (slider-get-value ?id))
  (if (< ?*hotTemp* 5) then 
      (bind ?*hotTemp* 5.0)
      (slider-set-value ?id 5)
  )
  (if (> ?*hotTemp* 100) then
      (bind ?*hotTemp* 100.0)
      (slider-set-value ?id 100)
  )
  (if (eq ?*manual-auto* "manual")
     then
       (calc-outFlow-and-outTemp)
     else
       (Simulate 0.0 0.0)
       (run)
  )
)

(deffunction on-cold-temp (?id)
  (bind ?*coldTemp* (slider-get-value ?id))
  (if (< ?*coldTemp* 5) then 
      (bind ?*coldTemp* 5.0)
      (slider-set-value ?id 5)
  )
  (if (> ?*coldTemp* 100) then
      (bind ?*coldTemp* 100.0)
      (slider-set-value ?id 100)
  )
  (if (eq ?*manual-auto* "manual")
     then
       (calc-outFlow-and-outTemp)
     else
       (Simulate 0.0 0.0)
       (run)
  )
)

(deffunction on-hot-press (?id)
  (bind ?*hotPress* (slider-get-value ?id))
  (if (< ?*hotPress* 30) then 
      (bind ?*hotPress* 30.0)
      (slider-set-value ?id 30)
  )
  (if (eq ?*manual-auto* "manual")
     then
       (calc-outFlow-and-outTemp)
     else
       (Simulate 0.0 0.0)
       (run)
  )
)

(deffunction on-cold-press (?id)
  (bind ?*coldPress* (slider-get-value ?id))
  (if (< ?*coldPress* 30) then 
      (bind ?*coldPress* 30.0)
      (slider-set-value ?id 30)
  )
  (if (eq ?*manual-auto* "manual")
     then
       (calc-outFlow-and-outTemp)
     else
       (Simulate 0.0 0.0)
       (run)
  )
)

(deffunction on-hot-valve (?id)
  (if (eq ?*manual-auto* "manual")
     then
       (bind ?*hotValvePos* (/ (float (slider-get-value ?id)) 100.0))
       (calc-outFlow-and-outTemp)
     else
       (slider-set-value ?id (integer (+ 0.5 (* ?*hotValvePos* 100))))
  )
)

(deffunction on-cold-valve (?id)
  (if (eq ?*manual-auto* "manual")
     then
       (bind ?*coldValvePos* (/ (float (slider-get-value ?id)) 100.0))
       (calc-outFlow-and-outTemp)
     else
       (slider-set-value ?id (integer (+ 0.5 (* ?*coldValvePos* 100))))
  )
)


;;; Callback for manual/auto check-box
(deffunction on-manual-auto (?id)
  (bind ?value (check-box-get-value ?id))
  (if (eq ?value 0)  ;; manual
     then
       (frame-set-status-text ?*main-frame* "Manual Mode: Select 'auto' check box for Fuzzy Control")
       (bind ?*manual-auto* "manual")
     else
       (frame-set-status-text ?*main-frame* "Auto Mode: Fuzzy Control in effect")
       (bind ?*manual-auto* "auto")
       (Simulate 0.0 0.0)
       (run)
  )
  
)

;;; painting the canvas -- shower graphic
;;; Painting callback
(deffunction on-paint (?id)
 (if (> ?id 0) then
  (bind ?dc (canvas-get-dc ?id))
  (bind ?red-pen (pen-create RED 8 wxSOLID))
  (dc-set-pen ?dc ?red-pen)
  (dc-draw-line ?dc 30.0 300.0 30.0 100.0)
  (dc-draw-line ?dc 30.0 100.0 110.0 100.0)
  (bind ?blue-pen (pen-create BLUE 8 wxSOLID))
  (dc-set-pen ?dc ?blue-pen)
  (dc-draw-line ?dc 190.0 300.0 190.0 100.0)
  (dc-draw-line ?dc 190.0 100.0 110.0 100.0)
  (bind ?green-pen (pen-create GREEN 8 wxSOLID))
  (dc-set-pen ?dc ?green-pen)
  (dc-draw-line ?dc 110.0 100.0 110.0 35.0)
  (dc-draw-line ?dc 110.0 35.0 160.0 35.0)
  (bind ?green-brush (brush-create GREEN wxSOLID))
  (dc-set-brush ?dc ?green-brush)
  (dc-set-pen ?dc ?green-pen)
  (dc-draw-polygon ?dc (create$ 145.0 35.0 165.0 15.0 165.0 55.0))
  (dc-draw-ellipse ?dc 105 95 10 10)
  (bind ?green-dashed-pen (pen-create GREEN 2 wxSHORT_DASH))
  (dc-set-pen ?dc ?green-dashed-pen) 
  (dc-draw-spline ?dc 
     (create$ 170.0 20.0 180.0 25.0 200.0 35.0 210.0 50.0)) 
  (dc-draw-spline ?dc 
     (create$ 170.0 30.0 180.0 35.0 200.0 45.0 210.0 60.0)) 
  (dc-draw-spline ?dc 
     (create$ 170.0 40.0 180.0 45.0 200.0 55.0 210.0 70.0)) 
  (dc-draw-spline ?dc 
     (create$ 170.0 50.0 180.0 55.0 200.0 65.0 210.0 80.0)) 
  (bind ?black-pen (pen-create BLACK 2 wxSOLID))
  (bind ?cyan-brush (brush-create CYAN wxSOLID))
  (dc-set-brush ?dc ?cyan-brush)
  (dc-set-pen ?dc ?black-pen)
  (dc-draw-polygon ?dc (create$ 5.0 160.0 55.0 190.0 55.0 160.0 5.0 190.0))
  (dc-draw-polygon ?dc (create$ 165.0 160.0 215.0 190.0 215.0 160.0 165.0 190.0))
 )
)


(deffunction app-on-init ()

  (bind ?*main-frame* (frame-create 0 "Fuzzy Shower Example" -1 -1 640 480))
  (frame-create-status-line ?*main-frame*)
  (frame-set-status-text ?*main-frame* "Initializing Fuzzy Shower Problem")
  (window-centre ?*main-frame* wxBOTH)

  (window-add-callback ?*main-frame* OnSize on-size)
  (window-add-callback ?*main-frame* OnClose on-close)
  (window-add-callback ?*main-frame* OnMenuCommand on-menu-command)

  ;;; Make a menu bar
  (bind ?file-menu (menu-create))
  (menu-append ?file-menu 1 "&Quit")

  (bind ?help-menu (menu-create))
  (menu-append ?help-menu 201 "&About")

  (bind ?menu-bar (menu-bar-create))
  (menu-bar-append ?menu-bar ?file-menu "&File")
  (menu-bar-append ?menu-bar ?help-menu "&Help")
  
  (frame-set-menu-bar ?*main-frame* ?menu-bar)

  ;;; Make a panel and panel items
  (bind ?*panel* (panel-create ?*main-frame* 0 0 640 460))
  (panel-set-label-position ?*panel* wxVERTICAL)

  ;; Make some sliders for temp, press, valves, and outputs
  (message-create ?*panel* "Hot Valve" 20 200)
  (bind ?*hot-valve-slider*
       (slider-create ?*panel* "on-hot-valve" ""
                      0 0 100 85 20 230)
  )
  (message-create ?*panel* "Cold Valve" 440 200)
  (bind ?*cold-valve-slider* 
       (slider-create ?*panel* "on-cold-valve" "" 
                      0 0 100 85 440 230)
  )
  (message-create ?*panel* "Hot Temp" 20 310)
  (bind ?*hot-temp-slider*
       (slider-create ?*panel* "on-hot-temp" ""
                      0 0 100 85 20 340)
  )
  (message-create ?*panel* "Cold Temp" 440 310)
  (bind ?*cold-temp-slider* 
       (slider-create ?*panel* "on-cold-temp" "" 
                      0 0 100 85 440 340)
  )
  (message-create ?*panel* "Hot Pressure" 20 90)
  (bind ?*hot-press-slider*
       (slider-create ?*panel* "on-hot-press" ""
                      55 30 90 85 20 120)
  )
  (message-create ?*panel* "Cold Pressure" 440 90)
  (bind ?*cold-press-slider* 
       (slider-create ?*panel* "on-cold-press" "" 
                      55 30 90 85 440 120)
  )
  (message-create ?*panel* "Shower Temp" 140 5)
  (bind ?*outtemp-slider*
       (slider-create ?*panel* "on-outtemp" ""
                      0 0 72 85 140 30)
  )
  (message-create ?*panel* "Shower Flow" 350 5)
  (bind ?*outflow-slider* 
       (slider-create ?*panel* "on-outflow" "" 
                      0 0 25 85 350 30)
  )

  ;; check-box for manual/auto mode
  (bind ?*manual-auto-check-box* 
       (check-box-create ?*panel* "on-manual-auto" "Auto" 20 30)
  )
  (check-box-set-value ?*manual-auto-check-box* 0)
  (bind ?*manual-auto* "manual")
  (frame-set-status-text ?*main-frame* "Manual Mode: Select 'auto' for Fuzzy Control")

  ;;; Make a canvas for displaying the graphic
  (bind ?*canvas* (canvas-create ?*main-frame* 210 80 220 270))
  (window-add-callback ?*canvas* OnPaint on-paint)

  (initShower)

  (window-show ?*main-frame* 1)

  ?*main-frame*
)

