;; Tests for the set-alpha-value and get-alpha-value functions
;;
;; First test out of range and other illegal arguments
;;
(clear)
(set-alpha-value -1.0)  ; float value too low
(set-alpha-value 2.0)   ; float value too high
(set-alpha-value -1)    ; integer value too low
(set-alpha-value 2)     ; integer value too high
(set-alpha-value one)   ; symbol not allowed
(set-alpha-value "two") ; string not allowed
(set-alpha-value 0.5 10); too many arguments
(set-alpha-value )      ; needs at least 1 argument
(get-alpha-value 1.0)   ; no arguments allowed
;;
;; legal values of alpha -- need to test that the setting is
;; correct first then to test that the alpha setting gives the
;; correct results when inferencing
(set-alpha-value 0.0)
(get-alpha-value)
(set-alpha-value 1.0)
(get-alpha-value)
(set-alpha-value 0.5)
(get-alpha-value)
(deftemplate temp
  0 100 C
  ( (low (10 1) (50 0))
    (ok  (20 0) (50 1) (80 0))
    (high (50 0) (90 1))
  )
)

(defrule test-alpha
   (temp low)
  =>
   (printout t "Rule fired ****" crlf)
)

(set-alpha-value 0.0)
(assert (temp (pi 0 30)))
(run) ; rule should fire with alpha 0.0
(retract *)
(set-alpha-value 0.5)
(assert (temp (pi 0 30)))
(run) ; rule should fire with alpha 0.5
(retract *)
(set-alpha-value 0.55)
(assert (temp (pi 0 30)))
(run) ; rule should NOT fire with alpha 0.55
(retract *)
(set-alpha-value 0.0)
(assert (temp (pi 0 49.9)))
(run) ; rule should fire with alpha 0.0
(retract *)
(set-alpha-value 0.0024)
(assert (temp (pi 0 49.9)))
(run) ; rule should fire with alpha 0.0024
(retract *)
(set-alpha-value 0.0026)
(assert (temp (pi 0 49.9)))
(run) ; rule should NOT fire with alpha 0.0026
(retract *)
(set-alpha-value 0.0)
(assert (temp (pi 0 50.0)))
(run) ; rule should NOT fire with alpha 0.0
(retract *)
(set-alpha-value 0.001)
(assert (temp (pi 0 50.0)))
(run) ; rule should NOT fire with alpha 0.001
(retract *)
(set-alpha-value 0.0001)
(assert (temp (pi 0 50.0)))
(run) ; rule should NOT fire with alpha 0.001
(set-alpha-value 0.0)
(clear)

;; Tests for the get-fuzzy-display-precision and
;; set-fuzzy-display-precision functions
;;
;; first test invalid arguments
(clear)
(set-fuzzy-display-precision )     ; needs at least 1 arg
(set-fuzzy-display-precision tt)   ; must be numeric not symbol
(set-fuzzy-display-precision "r")  ; must be numeric not string
(set-fuzzy-display-precision 1.0 4); no more than 1 arg
(get-fuzzy-display-precision 3)    ; no args allowed
;;
;; valid arguments
;;
(set-fuzzy-display-precision -1)   ; will set to min of 2
(get-fuzzy-display-precision)
(set-fuzzy-display-precision 25)   ; will set to max of 16
(get-fuzzy-display-precision)
(set-fuzzy-display-precision 2)    ; valid integer
(get-fuzzy-display-precision)
(set-fuzzy-display-precision (+ 1 4)); will set to 5
(get-fuzzy-display-precision)
(set-fuzzy-display-precision (- 8 3)); will set to 5
(get-fuzzy-display-precision)
;;
;; display some fuzzy sets in different precisions
;;
(deftemplate temp
  0 100 C
  ( (low (10 1) (50 0))
    (ok  (20 0) (50 1) (80 0))
    (high (50 0) (90 1))
  )
)

(set-fuzzy-display-precision 2)
(assert (temp very very low))
(facts)
(set-fuzzy-display-precision 16)
(facts)
(set-fuzzy-display-precision 8)
(facts)
(set-fuzzy-display-precision 4)
(facts)
(clear)

;; Tests for functions set-fuzzy-inference-type and
;; get-fuzzy-inference-type
;;
;; first test invalid function calls
;;
(clear)
(set-fuzzy-inference-type abc)      ; only max-min or max-prod
(set-fuzzy-inference-type 1.2)      ; only max-min or max-prod
(set-fuzzy-inference-type "t")      ; only max-min or max-prod
(set-fuzzy-inference-type max-min 2); only 1 argument allowed
(get-fuzzy-inference-type abc)      ; no args allowed
;;
;; valid inference requests
;;
(deftemplate temp1
  0 100 C
  ( (low (10 1) (50 0))
    (ok  (20 0) (50 1) (80 0))
    (high (50 0) (90 1))
  )
)
(deftemplate temp2
  0 100 C
  ( (low (10 1) (50 0))
    (ok  (20 0) (50 1) (80 0))
    (high (50 0) (90 1))
  )
)

(defrule test-maxmin-max-prod
   (temp1 ok)
  =>
   (bind ?t2fact (assert (temp2 ok)))
   (plot-fuzzy-value t * nil nil ?t2fact)
)

(assert (temp1 (pi 0 35)))
(set-fuzzy-inference-type max-min)
(run) ; should plot ok clipped at y=0.5
(facts)
(retract *)

(assert (temp1 (pi 0 35)))
(set-fuzzy-inference-type max-prod)
(run) ; should plot ok scaled by 0.5
(facts)
(set-fuzzy-inference-type max-min)
(clear)

;; Tests for user defined modifiers
;; Used in deftemplates, fuzzy patterns, asserts and fuzzy-modify
;;
;; first some invalid adds/removes of modifiers
(clear)
(add-fuzzy-modifier 23 sqrt)
(add-fuzzy-modifier mod2)
(add-fuzzy-modifier mod2 45.6)
(remove-fuzzy-modifier)
(remove-fuzzy-modifier mod33)
(remove-fuzzy-modifier (+ 3 4))
;;
;; valid user modifiers
;;
(add-fuzzy-modifier my-somewhat sqrt)
(deffunction most-extremely-fcn (?x)
    (** ?x 5)
)
(add-fuzzy-modifier most-extremely most-extremely-fcn)
(deftemplate temp
  0 100 C
  ( (low (10 1) (50 0))
    (high (50 0) (90 1))
    (sort-of-high my-somewhat high)
    (incredibly-low most-extremely low)
  )
)
(plot-fuzzy-value t * nil nil (assert (temp my-somewhat high)))
(retract *)
(plot-fuzzy-value t * nil nil (assert (temp most-extremely low)))
(retract *)
(defrule test-user-mods
   (temp most-extremely low)
  =>
   (printout t "Match against most-extremely low was made" crlf)
)
(assert (temp low))
(run)
(retract *)
(plot-fuzzy-value t * nil nil
     (fuzzy-modify (create-fuzzy-value temp high)
                   most-extremely))
(clear)

;; Tests for fuzzyvaluep function
;;
;; This function returns TRUE if its only argument is a FUZZY-VALUE
;; type

;; Errors is arguments

(fuzzyvaluep)

(fuzzyvaluep 1 2 3)

;; FALSE returns

(fuzzyvaluep 1) 
(fuzzyvaluep 2.45)
(fuzzyvaluep symbol)
(fuzzyvaluep "string")
(fuzzyvaluep (+ 2 3))
(fuzzyvaluep (assert (a fact)))

;; TRUE returns

(deftemplate temp 
 0 100 C
 ( (cold (z 30 50))
   (ok (pi 20 50))
   (hot (s 50 70))
 )
)

(deftemplate system
  (slot name)
  (slot temperature (type FUZZY-VALUE temp))
)

(retract *)
(fuzzyvaluep (get-fuzzy-slot (assert (temp cold))))
(retract *)
(fuzzyvaluep (create-fuzzy-value temp very hot))
(fuzzyvaluep (get-fuzzy-slot
                  (assert (system (name Sys-A) (temperature not hot)))
                  temperature
             )
)
(retract *)
(fuzzyvaluep (fuzzy-union (create-fuzzy-value temp cold)
                          (create-fuzzy-value temp hot)
             )
)

(clear)
