/*
  This set of procedures are used to provide an interface between 
  Clips and a simulation problem.  It worries about
  such things as:

	- handle requests from the interface (NeWS tnt) -- via the 'Specific' simulation
	- calls the actual simulation code for a 'specidic' simulation
	- keep track of simulated time
	- create events in an event queue and execute these 
	   events at the correct time

*/

#include "Simulate.h"
#include <stdio.h>
#ifdef SYSVREF
#include <stropts.h>
#include <poll.h>
#include <sys/types.h>
#include <time.h>
#else
#include <sys/time.h>
#endif
#include <sys/ioctl.h>
#include <signal.h>
#include <strings.h>

#include "router.h"
#include "clips.h"
#include "engine.h"
#include "agenda.h"


#define FALSE 0

#define TRUE 1


/*******************************************/
/*      Local globals                      */
/*******************************************/

static short simInited = FALSE;   /* to make sure we only init windows once! 
				     eg. after a second reset or a clear */

static struct eventQueue *eventQueueHdr = NULL;  /* points to first event in event queue */

static long   startTime=0;     /* time in millisecs since Jan 1, 1970 of start of simulation */
static long   initTime=0;        /* time (millisecs) at start of simulation (normally 0)
				  if 100 secs at start of simulation then we
				  need to know when time 0 would have been
				  -- stored in baseTime below */
static long   baseTime=0;      /* time 0 for this run relative to Jan 1, 1970
			            -- (startTime - initTime) */
static long   runstopTime=0;   /* time at which a run command stopped */
static char   fastTime;        /* true if time to advance as fast as possible, false if not */
static long lastDisplayedTime = -1; /* last time displayed in the UI (in seconds!!!) */


int compressionFactor=1;/* used to control speed of simulation - 1 is
		    wall clock time -- >1 is faster than real time */

static struct timeval tv;      /* to hold secs and usecs when gettimeofday called */
static struct timezone tz;     /* to hold timezone data when gettimeofday called  */


long getCurrentTime();         /* retrieves current Time in milliseconds */



/*******************************************/
/*      external globals                   */
/*******************************************/


/* Following routines must be supplied with the actual simulation being 
   implemented
*/

extern void eventSwitch();   /* routine to handle the events - simulation specific */
extern void simulateCycle(); /* user routine to be executed on each cycle --
	                       defined for specific simulation */
extern void initSpecificSimulation();  /* init routine for actual simulation */


/*********************************************************************
                 addEvent

        add an event to the event queue


**********************************************************************/

void addEvent( eventType, eventTime, eDataPtr )

     short eventType;
     long  eventTime;
     struct eventData *eDataPtr;
{
  struct eventQueue *qPtr;
  struct eventQueue *lastqPtr;
  struct eventQueue *newQentry;

  qPtr = eventQueueHdr;

  newQentry = (struct eventQueue *)malloc(sizeof(struct eventQueue));
  if (newQentry == NULL)
    {
      PrintCLIPS(WERROR,"Error in memory allocation for Event queue\n");
      return;
    }

  newQentry->eventType = eventType;
  newQentry->eventTime = eventTime;
  newQentry->eventDataPtr = eDataPtr;
  
  lastqPtr = NULL;
  
  while (qPtr != NULL)
    {
      if (qPtr->eventTime > eventTime) /* found place to put it */
	break;
      lastqPtr = qPtr;
      qPtr = qPtr->next;
    }

  if (lastqPtr == NULL)
    eventQueueHdr = newQentry;
  else
    lastqPtr->next = newQentry;
  
  newQentry->next = qPtr;
}





/*******************************************************************
                  initSimulation
  
*******************************************************************/


void initSimulation()
{
  struct eventQueue *eventPtr; /* ptr to queue of events */

  int n, i, j;

  void Simulate();
  void haltTimer();
  void continueTimer();

      if (simInited == FALSE)
	{
          AddRunFunction("Simulate",Simulate);
          AddRunStopFunction("haltTimer",haltTimer);
          AddRunStartFunction("continueTimer",continueTimer);
	}
      else
	{
          eventPtr = eventQueueHdr;
          eventQueueHdr = NULL;
          while (eventPtr != NULL)
	   { struct eventQueue *tmPtr;

	     tmPtr = eventPtr->next;
             	     free((char *)eventPtr->eventDataPtr);
	     free((char *)eventPtr);
	     eventPtr = tmPtr;
	   }
	}

      /* special call to routine to init the actual simulation */

      initSpecificSimulation(simInited);

      simInited = TRUE;
   
      startTime=0;     /* time (millisecs) since 1970/1/1 of start of simulation */
      initTime=0;      /* time (millisecs) at start of simulation (normally 0)
			  if 100 secs at start of simulation then we
			  need to know when time 0 would have been
				  -- stored in baseTime below */
      baseTime=0;      /* time 0 for this run relative to Jan 1, 1970
			            -- (startTime - initTime) */
      runstopTime=0;   /* time at which a run command stopped */
      fastTime=FALSE;  /* true if time to advance as fast as possible */
      lastDisplayedTime = -1; /* last time displayed in the UI */


      compressionFactor=1;/* used to control speed of simulation - 1 is
		    wall clock time -- >1 is faster than real time */

      gettimeofday(&tv,&tz);
      startTime = tv.tv_sec * 1000 + tv.tv_usec/1000;
      baseTime = startTime - initTime/compressionFactor; 
}



/*****************************************************************

    getCurrentTime

	returns current 'simulation' time in milliseconds 

*****************************************************************/

long getCurrentTime()
{
  gettimeofday(&tv,&tz);
  return( ((tv.tv_sec*1000+tv.tv_usec/1000)-baseTime) * compressionFactor);

}





/************************************************************************

          continueTimer

  - updates baseTime to reflect that the simulation was stopped

************************************************************************/

void continueTimer()
{
  baseTime += (getCurrentTime() - runstopTime)/compressionFactor;
}






/************************************************************************

          haltTimer

   - remembers when run command stops so simulation timer can 
     be stopped too

************************************************************************/

void haltTimer()
{
  runstopTime = getCurrentTime();
}





/************************************************************************

                   Simulate

      Simulate the events:


*************************************************************************/



void Simulate()
{
  struct eventQueue *eventPtr; 
  long eventTime;          /* time of event occurrence (millisecs) */
  long currentTime;        /* time from start of simulation (milliseconds) */
  VOID  *next_activation; /* ptr to activation list - if NULL agenda empty */
  int i;
  int interval;

  next_activation = GetNextActivation(NULL);

  eventPtr = eventQueueHdr;

  currentTime = getCurrentTime();
  
  /* if AGENDA is empty and fastTime TRUE advance simulation clock
     to time of next event else get the clock time  */
  if (fastTime && next_activation == NULL && eventPtr != NULL)
    {
      baseTime -= (eventPtr->eventTime - currentTime);
      currentTime = eventPtr->eventTime;
    }


  while (eventPtr != NULL) /* handle each event in the list */
    {
      eventTime = eventPtr->eventTime;

      /* if currentTime is >= next time for an event then do it */
      if (currentTime >= eventTime)
         {
	  /* must remove the event from the event queue at this point 
	     since another event may be added as this event is handled 
	  */

	  eventQueueHdr = eventPtr->next;

	  /* call actual simulation routine to handle the switch on the 
	     events for this specific simulation
	  */

	  eventSwitch( eventPtr );
	  
         }
      else
	break;

      /* free up the event removed from list and pick current 
	 head of the list */
      free((char *)eventPtr->eventDataPtr);
      free((char *)eventPtr);
      eventPtr = eventQueueHdr;

    }


  /* insert code here to call the actual simulation code that must be
     executed on each cycle of the inference engine
  */

  simulateCycle(eventPtr);

}




/**************************************************************

             show_event_queue

   will display the event queue contents on the
   display -- for debugging only

***************************************************************/


void show_event_queue()
{
  struct eventQueue *qPtr;
  char outstr[256];
  int i=1;

  qPtr = eventQueueHdr;

  while (qPtr != NULL)
    {
      sprintf(outstr,"EventQueue entry #%d, Type is %s, Time = %d\n",
	      i,eventNames[qPtr->eventType],qPtr->eventTime);
      PrintCLIPS("stdout",outstr);
      qPtr = qPtr->next;
      i++;
    }
}






/**************************************************************

             get_current_sim_time

     will return the current simulator time (milliseconds)

***************************************************************/



float get_current_sim_time()
{
  return((float)getCurrentTime());
}

