#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/signal.h>

#include <stropts.h>
#include <sys/ioctl.h>

#include "libaudio.h"
#include "audio_device.h"


#define	Error		(void) fprintf


/* Local variables */
char *prog = "Shower simulation";

static unsigned char	audio_buf[1024 * 64];		/* size should depend on sample_rate */


#define	MAX_GAIN		(100)		/* maximum gain */

/*
 * This defines the tolerable sample rate error as a ratio between the
 * sample rates of the audio data and the audio device.
 */
#define	SAMPLE_RATE_THRESHOLD	(.01)


static unsigned	Volume = ~0;			/* output volume */
static double		Savevol;		/* saved volume level */
static char		*Audio_dev = "/dev/audio";

static int		Audio_fd = -1;		/* file descriptor for audio device */
static Audio_hdr	Dev_hdr;		/* audio header for device */
static char		*Ifile;			/* current filename */
static Audio_hdr	File_hdr;		/* audio header for file */


/* Global variables */
extern int getopt();
extern int optind;
extern char *optarg;


static AudioOpened = FALSE;  /* TRUE when we get it open OK */


/* Open the audio device for our use */

int openAudio()
{
	int err;
	struct stat	st;

	AudioOpened = FALSE;

	/* Validate and open the audio device */
	err = stat(Audio_dev, &st);
	if (err < 0) {
		Error(stderr, "%s: cannot stat ", prog);
		perror(Audio_dev);
		return(1);
	}
	if (!S_ISCHR(st.st_mode)) {
		Error(stderr, "%s: %s is not an audio device\n", prog,
		    Audio_dev);
		return(2);
	}

	/* Try it quickly, first */
	Audio_fd = open(Audio_dev, O_WRONLY | O_NDELAY);
	if ((Audio_fd < 0) && (errno == EBUSY)) {
			Error(stderr, "%s: %s is busy\n", prog, Audio_dev);
			return(1);
		}
	if (Audio_fd < 0) {
		Error(stderr, "%s: error opening ", prog);
		perror(Audio_dev);
		return(3);
	}

	/* Get the device output encoding configuration */
	if (audio_get_play_config(Audio_fd, &Dev_hdr) != AUDIO_SUCCESS) {
		Error(stderr, "%s: %s is not an audio device\n",
		    prog, Audio_dev);
		return(4);
	}

	AudioOpened = TRUE;

	return(0);
}


/* close the audio device at end */

int closeAudio()
{
	if (AudioOpened == FALSE)
	   return(1);

	/*
	 * Though drain is implicit on close(), it's performed here
	 * for the sake of completeness, and to ensure that the volume
	 * is reset after all output is complete.
	 */
	(void) audio_drain(Audio_fd, FALSE);
	if (Volume != ~0) {
		(void) audio_set_play_gain(Audio_fd, &Savevol);
	}
	
	(void) close(Audio_fd);			/* close output */

	AudioOpened = FALSE;
	return(0);
}



/*
 * Play an audio file.
 */
int playAudio(Vol, filename)
	unsigned	Vol;		/* 0 to 100 */
	char		*filename;	/* audio file */
{
	int		i;
	int		cnt;
	int		err;
	int		ifd;
	int		stdinseen;
	double		vol;

	if (AudioOpened == FALSE)
	   return(1);

	Volume = Vol;

	/*  set the output volume now */
	if (Volume != ~0) {
		vol = (double) Volume / (double) MAX_GAIN;
		(void) audio_get_play_gain(Audio_fd, &Savevol);
		err = audio_set_play_gain(Audio_fd, &vol);
		if (err != AUDIO_SUCCESS) {
			Error(stderr,
			    "%s: could not set output volume for %s\n",
			    prog, Audio_dev);
			return(1);
		}
	}


       /* play the file */

	Ifile = filename;

	if ((ifd = open(Ifile, O_RDONLY, 0)) < 0) {
		Error(stderr, "%s: cannot open ", prog);
		perror(Ifile);
	}
	
	else  {

	  /* Check to make sure this is an audio file */
	  err = audio_read_filehdr(ifd, &File_hdr, (char *)NULL, 0);

	  if (err != AUDIO_SUCCESS) {
		Error(stderr, "%s: %s is not a valid audio file\n",
		    prog, Ifile);
	  }

	  else  {

	    /* Check the device configuration */
	    if (audio_cmp_hdr(&Dev_hdr, &File_hdr) != 0) {
		/*
		 * The device does not match the input file.
		 * Wait for any old output to drain, then attempt
		 * to reconfigure the audio device to match the
		 * input data.
		 */
		if (audio_drain(Audio_fd, FALSE) != AUDIO_SUCCESS) {
			Error(stderr, "%s: ", prog);
			perror("AUDIO_DRAIN error");
			return(1);
		}
		if (!reconfig())
			goto closeinput;
	    }

	    /*
	     * At this point, we're all ready to copy the data.
	     */
	    while ((cnt = read(ifd, (char *)audio_buf, sizeof (audio_buf))) >= 0) {
		/* If input EOF, write an eof marker */
		err = write(Audio_fd, (char *)audio_buf, cnt);

		if (err != cnt) { /* didn't write it all -- ERROR */
			Error(stderr, "%s: output error: ", prog);
			perror("");
			break;
		}
		if (cnt == 0)
			break;
	    }
	    if (cnt < 0) {
		Error(stderr, "%s: error reading ", prog);
		perror(Ifile);
	    }
	  }

closeinput:
		(void) close(ifd);		/* close input file */
	}

/*NOTREACHED*/
}


/*
 * Try to reconfigure the audio device to match the file encoding.
 * If this fails, we should attempt to make the input data match the
 * device encoding.  For now, we give up on this file.
 *
 * Returns TRUE if successful.  Returns FALSE if not.
 */
reconfig()
{
	int	err;
	char	msg[AUDIO_MAX_ENCODE_INFO];

	Dev_hdr = File_hdr;
	err = audio_set_play_config(Audio_fd, &Dev_hdr);

	switch (err) {
	case AUDIO_SUCCESS:
		return (TRUE);

	case AUDIO_ERR_NOEFFECT:
		/*
		 * Couldn't change the device.
		 * Check to see if we're nearly compatible.
		 * audio_cmp_hdr() returns >0 if only sample rate difference.
		 */
		if (audio_cmp_hdr(&Dev_hdr, &File_hdr) > 0) {
			double	ratio;

			ratio = (double) abs((int)
			    (Dev_hdr.sample_rate - File_hdr.sample_rate)) /
			    (double) File_hdr.sample_rate;
			if (ratio <= SAMPLE_RATE_THRESHOLD) {
								return (TRUE);
			}
			Error(stderr, "%s: %s sample rate %d not available\n",
			    prog, Ifile, File_hdr.sample_rate);
			return (FALSE);
		}
		(void) audio_enc_to_str(&File_hdr, msg);
		Error(stderr, "%s: %s encoding not available: %s\n",
		    prog, Ifile, msg);
		return (FALSE);

	default:
		Error(stderr, "%s: i/o error (set config)\n");
		exit(1);
/*NOTREACHED*/
	}
}


/* Parse an unsigned integer */
parse_unsigned(str, dst, flag)
	char		*str;
	unsigned	*dst;
	char		*flag;
{
	char		x;

	if (sscanf(str, "%u%c", dst, &x) != 1) {
		Error(stderr, "%s: invalid value for %s\n", prog, flag);
		return (1);
	}
	return (0);
}
