
#include <sys/types.h>
#include <sys/time.h>
#include <math.h>

#include "ShowerSimulate.h"
#include <stdio.h>
#include "clips.h"
#include "setup.h"
#include "factrhs.h"
#include "factmngr.h"
#include "agenda.h"

#include <strings.h>
#include <wire/wire.h>


#include "Simulate.h"

float	Tx;	/* [Celcius] water temperature at shower head */
float	Fx;	/* [litres/second] water flow at shower head*/

float	Fc;	/* [litres/second] cold water flow at cold valve output */
float	Fh;	/* [litres/second] hot  water flow at hot valve output*/

float	pc;	/* [kPa] cold water pressure at input to system */
float	ph;	/* [kPa] hot  water pressure at input to system */

float   th;	/* temperature of input hot water [ C ] */
float   tc;	/* temperature of input cold water [ C ] */

float   vh;     /* position of hot water valve (0 - closed  to 1 - fully open) */
float   vc;     /* position of cold water valve (0 - closed  to 1 - fully open) */

/* multiplier used with hot and cold valve changes recommended -- set to 1.0 when
   reach stable situation (flow and temp in range) and decreased on each 
   iteration of the inference which calculates new hot and cold valve changes
   until stable again - this allows for a sensitivity adjustment in situations
   that cause the adjustments to oscillate in and out of range
*/
static float Mfactor = 1.0;   

static void    fuzzyfy_Fx_Tx();

void    calc_Fx_Tx();
void	setShowerTempColor();
void	setShowerFlowColor();

static int TempOK();    /* 1 if temperature is in range */
static int FlowOK();    /* 1 if flow is in range */

int SystemProblem = NoProblem;  /* set when problem with Hot or Cold water */
int SystemMode = AutoMode;      /* tells if in auto or manual mode */

extern void	addEvent();   /* used to add events to the event Queue */
extern int      UI_init();    /* to init the user interface from GNT */

extern int	playAudio();   /* to play an audio message */
extern int	openAudio();   /* open the audio device */

extern void StopLoggingCallback();

extern long getCurrentTime();  /* retrieves current Time in milliseconds */


extern FILE *logFile;     /* file handle for the logging file */
extern FILE *statFile;    /* file handle for the stat file */

extern int logFileIndex;   /* index into arrays below beginning with lf */
extern int statFileIndex;  /* index into arrays below beginning with sf */

extern long  lfCurrTime[];
extern float  lfTx[], lfFx[], lfVh[], lfVc[];
extern float  lfPh[], lfPc[], lfTh[], lfTc[];

extern long sfStartTime[], sfEndTime[];  /* start/end time of T or P going out of range */
extern int    sfNumMoves[];  /* number of moves made during out of range period */
extern float sfT[], sfF[];   /* temp and flow at start of out of range period */



extern wire_Wire wire_Current();

static int valvePositionsDetermined = TRUE; /* used to make simulate waits till 
					     CLIPS does reasoning to determine
				             valve positions */


#define flushAudioFilename  "flush.au"




/*
	showCurrenttime

    called to update display of the time for the simulation

*/

static char stringTime[9] = "00:00:00";


void showCurrenttime( t )

    long t; /* time in seconds */
{	
	int hr, min, sec;
	
	sec = t % 60;
	t = t / 60;
	min = t % 60;
	hr = t / 60;  

	stringTime[7] = (char)((sec % 10) + '0');
	stringTime[6] = (char)((sec / 10) + '0');

	stringTime[4] = (char)((min % 10) + '0');
	stringTime[3] = (char)((min / 10) + '0');

	stringTime[1] = (char)((hr % 10) + '0');
	stringTime[0] = (char)((hr % 100) / 10 + '0');

	ps_setCurrenttime( stringTime );
}





/*
	eventSwitch

    routine to handle any events in the event Queue
*/

void eventSwitch( eventPtr )

     struct eventQueue *eventPtr; 
{
	struct eventData *eData1;
	long curTime;

	float pressureChange;

	curTime = getCurrentTime();

	  switch (eventPtr->eventType)
	    {
	    case EVENT_changePressureHot:

		/* pressure of the Hot water input is to change */
		pressureChange = eventPtr->eventDataPtr->floatValue;
		ph = ph + pressureChange;
		if (ph < 0.0) 
			{ pressureChange = pressureChange-ph;
			  ph = 0.0; 
			}

		switch (eventPtr->eventDataPtr->intValue)
		  {
			case WashingMachineStart:
				ps_disableWashingMachineChoice();
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -pressureChange;
				eData1->intValue = WashingMachineCycle1End;
				addEvent( EVENT_changePressureHot, curTime+120000L, eData1 );
			break;

			case WashingMachineCycle1End:
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -6.0;
				eData1->intValue = WashingMachineCycle2Start;
				addEvent( EVENT_changePressureHot, curTime+240000L, eData1 );
			break;

			case WashingMachineCycle2Start:
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -pressureChange;
				eData1->intValue = WashingMachineCycle2End;
				addEvent( EVENT_changePressureHot, curTime+120000L, eData1 );
			break;

			case WashingMachineCycle2End:
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->intValue = WashingMachineEnd;
				addEvent( EVENT_changePressureHot, curTime+300000L, eData1 );
			break;

			case WashingMachineEnd:
				ps_enableWashingMachineChoice();
			break;

			case DishwasherStart:
				ps_disableDishwasherChoice();
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -pressureChange;
				eData1->intValue = DishwasherCycle1End;
				addEvent( EVENT_changePressureHot, curTime+30000L, eData1 );
			break;

			case DishwasherCycle1End:
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -5.0;
				eData1->intValue = DishwasherCycle2Start;
				addEvent( EVENT_changePressureHot, curTime+180000L, eData1 );
			break;

			case DishwasherCycle2Start:
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -pressureChange;
				eData1->intValue = DishwasherCycle2End;
				addEvent( EVENT_changePressureHot, curTime+30000L, eData1 );
			break;

			case DishwasherCycle2End:
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->intValue = DishwasherEnd;
				addEvent( EVENT_changePressureHot, curTime+300000L, eData1 );
			break;

			case DishwasherEnd:
				ps_enableDishwasherChoice();
			break;
		  }

		calc_Fx_Tx();

		ps_setHotPressureGuage(ph);
		ps_setShowerFlowGuage(Fx);
		ps_setShowerTempGuage(Tx);
		setShowerTempColor();
		setShowerFlowColor();
	      break;

	    case EVENT_changePressureCold:

		/* pressure of the Cold water input is to change */
		pressureChange = eventPtr->eventDataPtr->floatValue;
		pc = pc + pressureChange;
		if (pc < 0.0) 
			{ pressureChange = pressureChange-pc;
			  pc = 0.0; 
			}

		switch (eventPtr->eventDataPtr->intValue)
		  {
			case Toilet1FlushStart:
				ps_disableToilet1Choice();
				playAudio((unsigned)15, flushAudioFilename );
				/* set time and pressure change for the end of flush */
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -pressureChange;
				eData1->intValue = Toilet1FlushEnd;
				addEvent( EVENT_changePressureCold, curTime+15000L, eData1 );
			break;

			case Toilet1FlushEnd:
				ps_enableToilet1Choice();
			break;

			case Toilet2FlushStart:
				ps_disableToilet2Choice();
				playAudio((unsigned)20, flushAudioFilename );
				/* set time and pressure change for the end of flush */
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -pressureChange;
				eData1->intValue = Toilet2FlushEnd;
				addEvent( EVENT_changePressureCold, curTime+15000L, eData1 );
			break;

			case Toilet2FlushEnd:
				ps_enableToilet2Choice();
			break;

			case GardenHoseON:
				ps_changeGardenHoseChoice( "on");
			break;

			case GardenHoseOFF:
				ps_changeGardenHoseChoice( "off");
			break;

			case WashingMachineStart:  /* Hot handles disabling of menu item */
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -pressureChange;
				eData1->intValue = WashingMachineCycle1End;
				addEvent( EVENT_changePressureCold, curTime+120000L, eData1 );
			break;

			case WashingMachineCycle1End:
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -3.0;
				eData1->intValue = WashingMachineCycle2Start;
				addEvent( EVENT_changePressureCold, curTime+240000L, eData1 );
			break;

			case WashingMachineCycle2Start:
				eData1 = (struct eventData *)malloc(sizeof(struct eventData));
				eData1->floatValue = -pressureChange;
				eData1->intValue = WashingMachineCycle2End;
				addEvent( EVENT_changePressureCold, curTime+120000L, eData1 );
			break;

			case WashingMachineCycle2End:
				/* nothing to do here -- handled by Hot -- 
				   selects WashingMachineEnd to occur */
			break;

			case WashingMachineEnd: /* Should NEVER happen!!! */
				ps_enableWashingMachineChoice();
			break;

			case DishwasherStart:
				ps_disableDishwasherChoice();
			break;

			case DishwasherEnd:
				ps_enableDishwasherChoice();
			break;
		  }

		calc_Fx_Tx();

		ps_setColdPressureGuage(pc);
		ps_setShowerFlowGuage(Fx);
		ps_setShowerTempGuage(Tx);
		setShowerTempColor();
		setShowerFlowColor();

	      break;

	    case EVENT_updateClock:  /* update the clock display to next second */
     		showCurrenttime(curTime/1000); /* note: pass second not millisec */
		/* add clock event at next second */
      		addEvent( EVENT_updateClock, eventPtr->eventTime+1000L, NULL);
	      break;

	    case EVENT_moveHotValve:  /* hot valve move done at this time */
	    case EVENT_moveColdValve:  /* cold valve move done at this time */
	    case EVENT_moveHotValve_setProblem:  /* hot valve move done at this time and set SysProblem */
	    case EVENT_moveColdValve_setProblem:  /* cold valve move done at this time and set SysProblem */
			
 		/* count the number of 'moves' of valves made to solve out of range problem */
		/*  ( each move of a valve increments the count!)  */

		sfNumMoves[statFileIndex]++;
		
		if (eventPtr->eventType == EVENT_moveHotValve || eventPtr->eventType == EVENT_moveHotValve_setProblem)
		     vh = eventPtr->eventDataPtr->floatValue;
		else
		     vc = eventPtr->eventDataPtr->floatValue;

		calc_Fx_Tx();

   		/* note: assign SystemProblem after calc_Fx_Tx so that we don't
       		     cancel the problem till later -- see calc_Fx_Tx
   		*/
		if (eventPtr->eventType == EVENT_moveColdValve_setProblem || eventPtr->eventType == EVENT_moveHotValve_setProblem)
		           {
   			SystemProblem = eventPtr->eventDataPtr->intValue;

   			/* note that we got the advice from CLIPS and acted on it so we can
			    ask for further advice if necessary */
			valvePositionsDetermined = TRUE;
		           }

 		/*   Set all the sliders and guages etc. to correct values */

		ps_setColdValveSlider(vc);
		ps_setHotValveSlider(vh);
		ps_setShowerFlowGuage(Fx);
		ps_setShowerTempGuage(Tx);
		setShowerTempColor();
		setShowerFlowColor();

		psio_flush(wire_PSoutput(wire_Current()));         
	      break;

	    }   /* end of switch */


	psio_flush(wire_PSoutput(wire_Current()));

}


int TempHot()   /* 1 if temp above accepted range else 0 */
{
  return( Tx >= T_OPT+T_OPT_range );
}

int TempCold()   /* 1 if temp below accepted range else 0 */
{
  return( Tx <= T_OPT-T_OPT_range );
}

static int TempOK()    /* 1 if temperature is in range else 0 */
{
   return(fabs((double)(Tx - T_OPT)) < T_OPT_range);
}


static int FlowOK()    /* 1 if flow is in range else 0  */
{
   return(fabs((double)(Fx - F_OPT)) < F_OPT_range);
}


int FlowLow()    /* 1 if flow is in below range */
{
   return( Fx <= F_OPT - F_OPT_range);
}

int FlowHigh()    /* 1 if flow is in above range */
{
   return( Fx >= F_OPT + F_OPT_range);
}




/*
	simulateCycle

    called on each cycle of the inference engine

*/

static struct timeval no_delay = { 0L, 0L };

static struct timeval delay = { 0L, 0L };

static int OutOfRange = FALSE;  /* true while T or P remain out of range */

void simulateCycle(eventPtr)
 
     struct eventQueue *eventPtr; 
 {
       VOID *next_activation; /* ptr to activation list - 
		                if NULL agenda empty */
 	/* handle any waiting callbacks */

	wire_Notify(&no_delay); /* must try once so that wire is flushed!! */

	while (wire_WouldNotify(wire_Current()))
	     {
	      wire_Notify((struct timeval*)NULL);
	     }

	/* check the output temperature and flow to see if in range and
	   if not then assert facts to cause the CLIPS program to determine
	   the adjustments required for the valve settings.

	   Note that if there is a system problem that prevents us from
	   attaining our desired goals or if we have asked for a recommendation
	   and haven't yet received it then don't ask for a recommendation.
	*/

	if ( SystemProblem == NoProblem &&
	     SystemMode == AutoMode &&
	     valvePositionsDetermined == TRUE 
	   )
	   {
	     if ( !TempOK() || !FlowOK() ) 
	       {
	         fuzzyfy_Fx_Tx();
	     
	         if (!OutOfRange && statFileIndex < MaxStatFileEntries-1)
	            { /* if exceed number of entries in array should force logging off!  but we don't! */
			statFileIndex++;
			OutOfRange = TRUE;
			sfStartTime[statFileIndex] = getCurrentTime();
			sfEndTime[statFileIndex] = sfStartTime[statFileIndex];
			sfT[statFileIndex] = Tx;
			sfF[statFileIndex] = Fx;
			sfNumMoves[statFileIndex] = 0;
	            }
	       }
	     else
	       {  /* turn off indicator that we are out of range if not already off */
                  Mfactor = 1.0;
	          if (OutOfRange)
	             {
		      OutOfRange = FALSE;
		      sfEndTime[statFileIndex] = getCurrentTime();
	             }
	        }
	    }

	/* if nothing on agenda then wait for up to 1 second for a Notify 
	   NOTE: cannot do this if there are functions other than
	         Simulate on the list of functions to be done each
	         cycle of CLIPS!!! Just put ALL cycle stuff in here!

	   Could wait till next event on Queue or forever if nothing on
	   Queue BUT should always be a clock event to update to next second
        */

       next_activation = GetNextActivation(NULL);

       if (next_activation == NULL)  /* agenda empty? */
         { 
	    long timeDifference = 0;
	    long currentTime; /* current time in milliseconds */

	    currentTime = getCurrentTime();
	 
	    /* queue empty? -- should never be empty since always a clock event */
	    if (eventPtr != NULL)
		{
		  if ((timeDifference = eventPtr->eventTime - currentTime - 1) < 0)
			timeDifference = 0;
		  
		}
	    
	    delay.tv_usec =  timeDifference * 1000;

	    wire_Notify(&delay);
         }
}



/* routines to support setting of color of guages etc. */

static char *colorArray[3] = { "Red", "Blue", "Green" };


void	setShowerTempColor()

{
	float diff;
	static int lastColorIndex= -1;
	int colorIndex; 

	diff = Tx - T_OPT;
	if (diff > 2.0) 
	    colorIndex = 0;   /* RED */
	else if (diff < -2.0)
	    colorIndex = 1;  /* BLUE */
	else
	    colorIndex = 2;  /* GREEN */

        if (lastColorIndex != colorIndex)
	    ps_setShowerTempColor(colorArray[colorIndex]);

	lastColorIndex = colorIndex;
}

void	setShowerFlowColor()

{
	float diff;
	static int lastColorIndex= -1;
	int colorIndex; 

	diff = Fx - F_OPT;
	if (diff > 1.0) 
	    colorIndex = 0;   /* RED */
	else if (diff < -1.0)
	    colorIndex = 1;  /* BLUE */
	else
	    colorIndex = 2;  /* GREEN */

        if (lastColorIndex != colorIndex)
	    ps_setShowerFlowColor(colorArray[colorIndex]);

	lastColorIndex = colorIndex;
}




/* Set to AUTO mode --- system controls valves */

void setAutoMode()
{
	ps_setAutoMode();
	SystemMode = AutoMode;
	SystemProblem = NoProblem;
	
}


/* Set to MANUAL mode --- USER controls valves */

void setManualMode()
{
	ps_setManualMode();
	SystemMode = ManualMode;	
}



 
/*
	initSpecificSimulation

    routine to initialize the specific simulation

*/

void		initSpecificSimulation( initedAlready )

      short initedAlready;  /* true if already inited once */
  {
        Mfactor = 1.0;  /* used to adjust suggested hot/cold valve changes over time */   
	vc = 0.0;	/* cold water valve closed */
	vh = 0.0;	/* hot water valve closed */

	/* get pressure of hot and cold water inputs from the 
	   initial settings on the slider
	*/
	pc = PC;	/* cold water initial pressure*/
	ph = PH;	/* hot water initial pressure */

	th = TH;	/* Hot water initial temp */
	tc = TC;	/* Cold water initial temp */

	calc_Fx_Tx();

	/* start the user interface and set all guages and sliders and
	   messages accordingly
	*/

        if (!initedAlready)
	  {
	   UI_init();

	   openAudio();
	  }

	ps_setColdPressureGuage(pc);
	ps_setHotPressureGuage(ph);
	ps_setColdTempSlider(tc);
	ps_setHotTempSlider(th);
	ps_setColdValveSlider(vc);
	ps_setHotValveSlider(vh);
	ps_setShowerFlowGuage(Fx);
	ps_setShowerTempGuage(Tx);
	setShowerTempColor();
	setShowerFlowColor();
	ps_changeGardenHoseChoice( "off");
	ps_enableToilet1Choice();
	ps_enableToilet2Choice();
	ps_changeGardenHoseChoice("off");
	ps_enableDishwasherChoice();
	ps_enableWashingMachineChoice();
	setManualMode();

	if (logFile != NULL)
	   {
	    StopLoggingCallback();
	   }
	ps_enableStartLoggingChoice();
	ps_disableStopLoggingChoice();
	ps_setDefaultStartLoggingChoice();

	psio_flush(wire_PSoutput(wire_Current()));

	valvePositionsDetermined = TRUE;

	addEvent( EVENT_updateClock, 1L, NULL );  /* add clock event at 1 second */


  }




/*         calcValveMoveTime

	calculate the time to move a valve thru a given distance

	Simple model -- a fixed time for setting up and a variable time
	depending on the distance to move as a part of the time to move
	from 0 to 1 (full off to full on)

	return time in milliseconds
*/

long calcValveMoveTime( delta )

   float delta;
{
   return(  ValveMoveSetupTime + (long)(delta * ValveFullMoveTime)  );
}



/*
	SetValvePositions


    routine to set the valve positions as requested by the CLIPS program

*/

void SetValvePositions()
{
   /* there are 2 arguments passed with the call to this routine:

	1. hot valve position  (0 to 1)
	2. cold valve position  (0 to 1)

        these are changes to make to the current valve settings NOT
	the absolute positions of the valve

   */

   char string[256];

   int theProblem = SystemProblem;

   long curTime; 

   struct eventData *eData1, *eData2;
   float next_vc, next_vh;
   float delta_vc , delta_vh;
   long timeToMoveVc=0, timeToMoveVh=0;

   float delta_vh_suggested;    /* hot water valve position requested */
   float delta_vc_suggested;    /* cold water valve position requested */

   /* get the values suggested by CLIPS */
   delta_vh_suggested = (float)RtnDouble(1);
   delta_vc_suggested = (float)RtnDouble(2);

   curTime = getCurrentTime();

/* update the display of the valve positions and determine the
      new outputs of flow and temperature

	*** this is a percentage change in the value of the
	    valve position in the direction of the sign of
	    the suggested change.
	    Must try variations on this to get good results!!!!
   */

   /* first check for a few conditions that suggest that we cannot reach
      our goals and if so disable the automatic control and put on manual
   */

   if (vh == 1.0 && delta_vh_suggested > 0.0001 && 
         (     (delta_vc_suggested == 0.0)
             ||  (delta_vc_suggested < -0.0001 && (TempHot() || FlowLow()))  
             ||  (delta_vc_suggested > 0.0001 && (TempCold() || FlowHigh()))
          )
       )
	theProblem = HotWaterProblem;
   else
      if (vc == 1.0 && delta_vc_suggested > 0.0001 && 
            (     (delta_vh_suggested == 0.0)
              ||  (delta_vh_suggested > 0.0001 && (TempHot() || FlowHigh()))  
              ||  (delta_vh_suggested < -0.0001 && (TempCold() || FlowLow()))
            )
          )
  	theProblem = ColdWaterProblem;
     else
           if (vc == 0.0 && delta_vc_suggested < -0.0001 && !TempOK())
	theProblem = HotWaterTempProblem; 
           else
                if (vh == 0.0 && delta_vh_suggested < -0.0001 && !TempOK())
	theProblem = ColdWaterTempProblem; 

   if (theProblem != NoProblem)
      /* set system to manual mode or stop trying to solve problem */
      { 
	char *probStr;
	char *probStr2;
	
	next_vc = vc;   /* when problem encountered usually do nothing, so... */
	next_vh = vh;  /* vc and vh are left as they are  */

	probStr2 = "Control paused till Flow/Temp change or AUTO reset.";

	switch (theProblem)
	  {
		case HotWaterProblem:
		   probStr = "Can't increase flow of Hot Water.";
		   break;

		case ColdWaterProblem:
		   probStr = "Can't increase flow of Cold Water.";
		   if (Tx >= TScalding) /* shut off system if water very hot */
		      {
		         probStr2= "Shutting off to prevent scalding and setting MANUAL mode";
       		         setManualMode();
		         next_vc = 0.0; next_vh = 0.0;
		      }
		   break;

		case HotWaterTempProblem:
		   probStr = "Cold Water off and still too cold. Hot water below desired temperature.";
		   break;

		case ColdWaterTempProblem:
		   probStr = "Hot Water off and still too Hot. Cold Water Temp too high";
		   if (Tx >= TScalding) /* shut off system if water very hot */
		      {
		         probStr2= "Shutting off to prevent scalding and setting MANUAL mode";
       		         setManualMode();
		         next_vc = 0.0; next_vh = 0.0;
		      }
		   break;
	  }
	sprintf(string,"System Problem: %s\n\t%s\n",probStr,probStr2);  fflush(stdout);
        PrintCLIPS("stdout",string);

      }
   else     
      { /*  NO Problem encountered */

        /* set the valve positions  
                 -- note: the   vc = vc + vc*delta_vc_suggested*Mfactor
                          formula does NOT work when valve 
                          position close to zero so adjust!! 
       */

        if (vc == 0.0 && vh == 0.0 ) /* startup position */
	   { next_vc = 0.1; next_vh = 0.1; }
        else	
           { if ( TempOK() )
	        next_vc = vc + vc * delta_vc_suggested * Mfactor;
             else
	      {  /* need to adjust for the lower valve positions or can never move from a 0 position! */
	        float modifier = vc;
	        if (modifier < 0.5) modifier = 0.9*modifier + 0.05;
	        next_vc = vc + delta_vc_suggested * modifier * Mfactor;
	      }
             if (next_vc < 0.001) next_vc = 0.0;
             if (next_vc > 1.0) next_vc = 1.0;

             if ( TempOK() )
                        next_vh = vh + vh * delta_vh_suggested * Mfactor;
             else
	      {  /* need to adjust for the lower valve positions or can never move from a 0 position! */
	        float modifier = vh;
	        if (modifier < 0.5) modifier = 0.9*modifier + 0.05;
	        next_vh = vh + delta_vh_suggested * modifier * Mfactor;
	      }
             if (next_vh < 0.001) next_vh = 0.0;
             if (next_vh > 1.0) next_vh = 1.0;
            }
      }

     if (Mfactor > 0.1) Mfactor = Mfactor * 0.96;
     
     delta_vc = next_vc - vc;
     delta_vh = next_vh - vh;

     if (delta_vc != 0.0)
	{
	  /* generate an event at future time when the valve movement will be completed */
	  eData1 = (struct eventData *)malloc(sizeof(struct eventData));
	  eData1->floatValue = next_vc;
	  eData1->intValue = theProblem;
	  timeToMoveVc = calcValveMoveTime( fabs(delta_vc) );
	}

     if (delta_vh != 0.0)
	{
	  /* generate an event at future time when the valve movement will be completed */
	  eData2 = (struct eventData *)malloc(sizeof(struct eventData));
	  eData2->floatValue = next_vh;
	  eData2->intValue = theProblem;
	  timeToMoveVh = calcValveMoveTime( fabs(delta_vh) );
	}

    if (delta_vh == 0.0 && delta_vc == 0.0)
	{
	   /* nothing to do since no change requested -- just set the problem */
	   SystemProblem = theProblem;
   
   	  /* note that we got the advice from CLIPS and acted on it so we can
	      ask for further advice if necessary */
	  valvePositionsDetermined = TRUE;

	 /* if have a problem nothing we can do -- stop recording attempts to fix system */
	  if (SystemProblem != NoProblem && OutOfRange) 
	             {
	                OutOfRange = FALSE;  /* still out of range really but nothing we can do! */
	                sfEndTime[statFileIndex] = getCurrentTime();
	             }
	}
    else if (delta_vh == 0.0)
	  addEvent( EVENT_moveColdValve_setProblem, curTime+timeToMoveVc, eData1 );
    else if (delta_vc == 0.0)
	  addEvent( EVENT_moveHotValve_setProblem, curTime+timeToMoveVh, eData2 );
    else
           {
               /* both to be moved -- move cold first if Hot to increase and hot first if hot to decrease */
               /* assumes that we are only allowed to move the valves one at a time! If we can move them 
                   at the same time then we must order the events according to which takes longest to do and 
                   the one that takes longest must set SysProblem !! */
#if ValvesMoveOneAtaTime 
	if (delta_vh > 0.0)
	     {
	        addEvent( EVENT_moveColdValve, curTime+timeToMoveVc, eData1 );
	        addEvent( EVENT_moveHotValve_setProblem, curTime+timeToMoveVh+timeToMoveVc, eData2 );
	     }
	else
	     {
	        addEvent( EVENT_moveHotValve, curTime+timeToMoveVh, eData2 );
	        addEvent( EVENT_moveColdValve_setProblem, curTime+timeToMoveVh+timeToMoveVc, eData1 );
	     }
#else
	if (timeToMoveVh > timeToMoveVc)
	     {
	        addEvent( EVENT_moveColdValve, curTime+timeToMoveVc, eData1 );
	        addEvent( EVENT_moveHotValve_setProblem, curTime+timeToMoveVh, eData2 );
	     }
	else
	     {
	        addEvent( EVENT_moveHotValve, curTime+timeToMoveVh, eData2 );
	        addEvent( EVENT_moveColdValve_setProblem, curTime+timeToMoveVc, eData1 );
	     }
#endif
             }
 

}



/*
	calc_Fx_Tx

    routine to calculate the Flow and Temp at output when any of valve 
    positions, input temps or input pressures change

*/

void calc_Fx_Tx()
{
	float Last_Fx = Fx;
	float Last_Tx = Tx;

  	/* calculate parameters */
	Fc = vc * (pc - PX);
	Fh = vh * (ph - PX);
	if (Fc < 0.0) Fc = 0.0;  /* when pc < PX then no flow */
	if (Fh < 0.0) Fh = 0.0;  /* when ph < PX then no flow */
	Fx = Fc + Fh;
	if (Fx != 0.0)
	   Tx = (Fc*tc + Fh*th) / Fx;
	else
	   Tx = 0.0;

	if (SystemProblem != NoProblem && (Last_Fx != Fx || Last_Tx != Tx) )
	   { /* change in system: pressures or temps */
		SystemProblem = NoProblem; /* clear any problem */
	   }

	/* if 'logging' is ON then save the changed values in arrays (write out to the file later
                            when logging stopped  */

	if (logFile != NULL)
	   {
	        if (logFileIndex < MaxLogFileEntries-1);
	           {
		logFileIndex++;
		lfCurrTime[logFileIndex] = getCurrentTime();
		lfTx[logFileIndex] = Tx;
		lfFx[logFileIndex] = Fx;
		lfVh[logFileIndex] = vh;
		lfVc[logFileIndex] = vc;
		lfPh[logFileIndex] = ph;
		lfPc[logFileIndex] = pc;
		lfTh[logFileIndex] = th;
		lfTc[logFileIndex] = tc;
	           }
	   }
}


static void fuzzyfy_Fx_Tx( )
  {
   float Fx_begin, Fx_end, Tx_begin, Tx_end;
   char string[250];
   struct fact *factPtr;

   /* note that we will now wait for the advice from CLIPS */
   valvePositionsDetermined = FALSE;

   /* get Tx & Fx within 0 and 100 */
   if (Tx > 100.0) Tx = 100.0;
   else if (Tx < 0.0) Tx = 0.0;
   if (Fx > 100.0) Fx = 100.0;
   else if (Fx < 0.0) Fx = 0.0;

   Tx_begin = Tx - .5;
   Tx_end   = Tx + .5;
   Fx_begin = Fx - .5;
   Fx_end   = Fx + .5;
   if (Tx_begin < 0.0) {Tx_begin = 0.0; }
   if (Tx_end > 100.0) { Tx_end = 100.0; }
   if (Fx_begin < 0.0) {Fx_begin = 0.0; }
   if (Fx_end > 100.0) { Fx = 100.0; }

   if (Tx_begin == Tx)
        sprintf(string,"(Tx (%f 1)  (%f 0))",Tx,Tx_end);  /* consecutive numbers cannot be the same! */
   else if (Tx == Tx_end)
        sprintf(string,"(Tx (%f 0) (%f 1))",Tx_begin,Tx);
   else
        sprintf(string,"(Tx (%f 0) (%f 1) (%f 0))",Tx_begin,Tx,Tx_end);

   factPtr = StringToFact(string);
   if (factPtr == NULL)
     {
       printf("**** Error in fuzzyfy_Fx_Tx routine (ShowerSimulate.c)\n");
       printf("     trying to assert '%s'/n", string);
       fflush(stdout);
       exit(1);
     }
   Assert((VOID *)factPtr);    

   if (Fx_begin == Fx)
       sprintf(string,"(Fx (%f 1) (%f 0))",Fx,Fx_end);  /* consecutive numbers cannot be the same! */
   else if (Fx == Fx_end)
       sprintf(string,"(Fx (%f 0) (%f 1))",Fx_begin,Fx);
   else
       sprintf(string,"(Fx (%f 0) (%f 1) (%f 0))",Fx_begin,Fx,Fx_end); 

   factPtr = StringToFact(string);
   if (factPtr == NULL)
     {
       printf("**** Error in fuzzyfy_Fx_Tx routine (ShowerSimulate.c)\n");
       printf("     trying to assert '%s'/n", string);
       fflush(stdout);
       exit(1);
     }
   Assert((VOID *)factPtr);    

  }



