   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 6.00  05/12/93            */
   /*                                                     */
   /*          DEFTEMPLATE CONSTRUCTS-TO-C MODULE         */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*      Brian L. Donnell                                     */
/*      Bob Orchard (NRCC - Nat'l Research Council of Canada)*/
/*                  (Fuzzy reasoning extensions)             */
/*                  (certainty factors for facts and rules)  */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _TMPLTCMP_SOURCE_

#include "setup.h"

#if DEFTEMPLATE_CONSTRUCT && CONSTRUCT_COMPILER && (! RUN_TIME)

#define SlotPrefix() ArbitraryPrefix(DeftemplateCodeItem,2)

#if FUZZY_DEFTEMPLATES
/* add 3 more files to store the Fuzzy template definitions */
#define LvUniversePrefix()  ArbitraryPrefix(DeftemplateCodeItem,3)
#define PrimaryTermPrefix() ArbitraryPrefix(DeftemplateCodeItem,4)
#define ModifierPrefix()    ArbitraryPrefix(DeftemplateCodeItem,5)
#endif

#include <stdio.h>
#define _CLIPS_STDIO_

#include "conscomp.h"
#include "factcmp.h"
#include "cstrncmp.h"
#include "tmpltdef.h"

#include "tmpltcmp.h"

#if FUZZY_DEFTEMPLATES
#include "fuzzyval.h"
#include "fuzzylv.h"
#include "dffnxcmp.h"
#include "prntutil.h"
#endif


/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if ANSI_COMPILER
   static int                     ConstructToCode(char *,int,FILE *,int,int);
   static VOID                    SlotToCode(FILE *,struct templateSlot *,int,int,int);
   static VOID                    DeftemplateModuleToCode(FILE *,struct defmodule *,int,int,int);
#if FUZZY_DEFTEMPLATES
   static VOID                    DeftemplateToCode(FILE *,struct deftemplate *,
                                                 int,int,int,int,int,int);
   static VOID                    CloseDeftemplateFiles(FILE *,FILE *,FILE *,FILE *,FILE *,FILE *,int);
#else
   static VOID                    DeftemplateToCode(FILE *,struct deftemplate *,
                                                 int,int,int,int);
   static VOID                    CloseDeftemplateFiles(FILE *,FILE *,FILE *,int);
#endif
#if FUZZY_DEFTEMPLATES
   static VOID                    LvUniverseToCode(FILE *,struct fuzzyLv *,
                                                 int,int,int,int,int,int,int,int);
   static VOID                    primaryTermToCode(FILE *,struct primary_term *,
                                                 int,int,int *, int);
   static VOID                    modifierToCode(FILE *,struct modifier *,
                                                 int,int,int *, int);
#endif
#else
   static int                     ConstructToCode();
   static VOID                    SlotToCode();
   static VOID                    DeftemplateModuleToCode();
   static VOID                    DeftemplateToCode();
   static VOID                    CloseDeftemplateFiles();
#if FUZZY_DEFTEMPLATES
   static VOID                    LvUniverseToCode();
   static VOID                    primaryTermToCode();
   static VOID                    modifierToCode();
#endif
#endif

/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/

   static struct CodeGeneratorItem *DeftemplateCodeItem;

/*********************************************************/
/* DeftemplateCompilerSetup: Initializes the deftemplate */
/*   construct for use with the constructs-to-c command. */
/*********************************************************/
globle VOID DeftemplateCompilerSetup()
  {
#if FUZZY_DEFTEMPLATES
DeftemplateCodeItem = AddCodeGeneratorItem("deftemplate",0,NULL,NULL,ConstructToCode,6);
#else
DeftemplateCodeItem = AddCodeGeneratorItem("deftemplate",0,NULL,NULL,ConstructToCode,3);
#endif
  }
  
/***************************************************************/
/* ConstructToCode:  Produces all the deftemplate definitions. */
/***************************************************************/
static int ConstructToCode(fileName,fileID,headerFP,imageID,maxIndices)
  char *fileName;
  int fileID;
  FILE *headerFP;
  int imageID;
  int maxIndices;
  {
   int fileCount = 1;
   struct defmodule *theModule;
   struct deftemplate *theTemplate;
   struct templateSlot *slotPtr;
   int slotCount = 0, slotArrayCount = 0, slotArrayVersion = 1;
   int moduleCount = 0, moduleArrayCount = 0, moduleArrayVersion = 1;  
   int templateCount = 0, templateArrayCount = 0, templateArrayVersion = 1;
   FILE *slotFile = NULL, *moduleFile = NULL, *templateFile = NULL;
#if FUZZY_DEFTEMPLATES
   int lvUniverseArrayCount = 0, lvUniverseArrayVersion = 1;
   int primaryTermArrayCount = 0, primaryTermArrayVersion = 1;
   int modifierArrayCount = 0, modifierArrayVersion = 1;
   FILE *lvUniverseFile = NULL, *primaryTermFile = NULL, *modifierFile = NULL;
   struct fuzzyLv *lvPtr;
   struct primary_term *primaryTermPtr;
   struct modifier *modifierPtr;
#endif

   /*==================================================*/
   /* Include the appropriate deftemplate header file. */
   /*==================================================*/
   
   fprintf(headerFP,"#include \"tmpltdef.h\"\n");
   
   /*=============================================================*/
   /* Loop through all the modules, all the deftemplates, and all */
   /* the deftemplate slots writing their C code representation   */
   /* to the file as they are traversed.                          */
   /*=============================================================*/
   
   theModule = (struct defmodule *) GetNextDefmodule(NULL);
   
   while (theModule != NULL)
     {           
      SetCurrentModule((VOID *) theModule);
            
      moduleFile = OpenFileIfNeeded(moduleFile,fileName,fileID,imageID,&fileCount,
                                    moduleArrayVersion,headerFP,
                                    "struct deftemplateModule",ModulePrefix(DeftemplateCodeItem),
                                    CLIPS_FALSE,NULL);
                                    
      if (moduleFile == NULL)
        {
#if FUZZY_DEFTEMPLATES
         CloseDeftemplateFiles(moduleFile,templateFile,slotFile,
			                   lvUniverseFile,primaryTermFile,modifierFile,maxIndices);
#else
         CloseDeftemplateFiles(moduleFile,templateFile,slotFile,maxIndices);
#endif
         return(0);
        }
        
      DeftemplateModuleToCode(moduleFile,theModule,imageID,maxIndices,moduleCount);
      moduleFile = CloseFileIfNeeded(moduleFile,&moduleArrayCount,&moduleArrayVersion,
                                     maxIndices,NULL,NULL);

      theTemplate = (struct deftemplate *) GetNextDeftemplate(NULL);

      while (theTemplate != NULL)
        {
         templateFile = OpenFileIfNeeded(templateFile,fileName,fileID,imageID,&fileCount,
                                         templateArrayVersion,headerFP,
                                         "struct deftemplate",ConstructPrefix(DeftemplateCodeItem),
                                         CLIPS_FALSE,NULL);
         if (templateFile == NULL)
           {
#if FUZZY_DEFTEMPLATES
            CloseDeftemplateFiles(moduleFile,templateFile,slotFile,
			                       lvUniverseFile,primaryTermFile,modifierFile,maxIndices);
#else
            CloseDeftemplateFiles(moduleFile,templateFile,slotFile,maxIndices);
#endif
            return(0);
           }
           
#if FUZZY_DEFTEMPLATES
         DeftemplateToCode(templateFile,theTemplate,imageID,maxIndices,
                        moduleCount,slotCount,
						lvUniverseArrayCount, lvUniverseArrayVersion);
#else
         DeftemplateToCode(templateFile,theTemplate,imageID,maxIndices,
                        moduleCount,slotCount);
#endif
         templateCount++;
         templateArrayCount++;
         templateFile = CloseFileIfNeeded(templateFile,&templateArrayCount,&templateArrayVersion,
                                          maxIndices,NULL,NULL);
                              
#if FUZZY_DEFTEMPLATES
         /* write out the fuzzyLv and universe structs together in 1 struct */
         lvPtr = theTemplate->fuzzyTemplate;
		 if (lvPtr != NULL)
		    {
              lvUniverseFile = OpenFileIfNeeded(lvUniverseFile,fileName,fileID,imageID,&fileCount,
                                        lvUniverseArrayVersion,headerFP,
                                       "struct LvUniverse",LvUniversePrefix(),CLIPS_FALSE,NULL);
              if (lvUniverseFile == NULL) 
                {
                 CloseDeftemplateFiles(moduleFile,templateFile,slotFile,
			                         lvUniverseFile,primaryTermFile,modifierFile,maxIndices);
                 return(0);	
			    }
              LvUniverseToCode(lvUniverseFile,lvPtr,imageID,maxIndices,
			                   lvUniverseArrayCount,lvUniverseArrayVersion,
							   primaryTermArrayCount,primaryTermArrayVersion,
							   modifierArrayCount,modifierArrayVersion);
              lvUniverseArrayCount++;
              lvUniverseFile = CloseFileIfNeeded(lvUniverseFile,&lvUniverseArrayCount,&lvUniverseArrayVersion,
                                         maxIndices,NULL,NULL);
			  
			  /* now write out the primaryTerm list*/
			  primaryTermPtr = lvPtr->primary_term_list;
              primaryTermFile = OpenFileIfNeeded(primaryTermFile,fileName,fileID,imageID,&fileCount,
                                        primaryTermArrayVersion,headerFP,
                                       "struct primary_term",PrimaryTermPrefix(),CLIPS_FALSE,NULL);
              if (primaryTermFile == NULL) 
                {
                 CloseDeftemplateFiles(moduleFile,templateFile,slotFile,
			                         lvUniverseFile,primaryTermFile,modifierFile,maxIndices);
                 return(0);	
			    }
              primaryTermToCode(primaryTermFile,primaryTermPtr,imageID,maxIndices,
							   &primaryTermArrayCount,primaryTermArrayVersion);
              primaryTermFile = CloseFileIfNeeded(primaryTermFile,&primaryTermArrayCount,
			                            &primaryTermArrayVersion,maxIndices,NULL,NULL);
			  
			  
			  /* now write out the modifier list*/
			  modifierPtr = lvPtr->modifier_list;
			  if (modifierPtr != NULL)
			    {
                  modifierFile = OpenFileIfNeeded(modifierFile,fileName,fileID,imageID,&fileCount,
                                        modifierArrayVersion,headerFP,
                                       "struct modifier",ModifierPrefix(),CLIPS_FALSE,NULL);
                  if (modifierFile == NULL) 
                    {
                     CloseDeftemplateFiles(moduleFile,templateFile,slotFile,
			                         lvUniverseFile,primaryTermFile,modifierFile,maxIndices);
                     return(0);	
			        }
                  modifierToCode(modifierFile,modifierPtr,imageID,maxIndices,
							     &modifierArrayCount,modifierArrayVersion);
                  modifierFile = CloseFileIfNeeded(modifierFile,&modifierArrayCount,
				                         &modifierArrayVersion,maxIndices,NULL,NULL);
                }
			  
			  
			} /* end of if (lvUniversePtr != NULL) */
#endif

         slotPtr = theTemplate->slotList;
         while (slotPtr != NULL)
           {
            slotFile = OpenFileIfNeeded(slotFile,fileName,fileID,imageID,&fileCount,
                                        slotArrayVersion,headerFP,
                                       "struct templateSlot",SlotPrefix(),CLIPS_FALSE,NULL);
            if (slotFile == NULL) 
              {
#if FUZZY_DEFTEMPLATES
               CloseDeftemplateFiles(moduleFile,templateFile,slotFile,
			                         lvUniverseFile,primaryTermFile,modifierFile,maxIndices);
#else
               CloseDeftemplateFiles(moduleFile,templateFile,slotFile,maxIndices);
#endif
               return(0);
              }
              
            SlotToCode(slotFile,slotPtr,imageID,maxIndices,slotCount);
            slotCount++;
            slotArrayCount++;
            slotFile = CloseFileIfNeeded(slotFile,&slotArrayCount,&slotArrayVersion,
                                         maxIndices,NULL,NULL);
            slotPtr = slotPtr->next;
           }
           
         theTemplate = (struct deftemplate *) GetNextDeftemplate(theTemplate);
        }
        
      theModule = (struct defmodule *) GetNextDefmodule(theModule);
      moduleCount++;
      moduleArrayCount++;

     }
        
#if FUZZY_DEFTEMPLATES
   CloseDeftemplateFiles(moduleFile,templateFile,slotFile,
			             lvUniverseFile,primaryTermFile,modifierFile,maxIndices);
#else
   CloseDeftemplateFiles(moduleFile,templateFile,slotFile,maxIndices);
#endif

   return(1);
  }

/************************************************************/
/* CloseDeftemplateFiles: Closes all of the C files created */
/*   for deftemplates. Called when an error occurs or when  */
/*   the deftemplates have all been written to the files.   */
/************************************************************/
#if FUZZY_DEFTEMPLATES
static VOID CloseDeftemplateFiles(moduleFile,templateFile,slotFile,
			             lvUniverseFile,primaryTermFile,modifierFile,maxIndices)
#else
static VOID CloseDeftemplateFiles(moduleFile,templateFile,slotFile,maxIndices)
#endif
  FILE *moduleFile, *templateFile, *slotFile; 
#if FUZZY_DEFTEMPLATES
  FILE *lvUniverseFile, *primaryTermFile, *modifierFile;
#endif
  int maxIndices;  
  {
   int count = maxIndices;
   int arrayVersion = 0;
   
   if (slotFile != NULL) 
     {
      count = maxIndices;
      slotFile = CloseFileIfNeeded(slotFile,&count,&arrayVersion,maxIndices,NULL,NULL);
     }
     
   if (templateFile != NULL) 
     {
      count = maxIndices;
      templateFile = CloseFileIfNeeded(templateFile,&count,&arrayVersion,maxIndices,NULL,NULL);
     }
     
   if (moduleFile != NULL) 
     {
      count = maxIndices;
      moduleFile = CloseFileIfNeeded(moduleFile,&count,&arrayVersion,maxIndices,NULL,NULL);
     }

#if FUZZY_DEFTEMPLATES
   if (lvUniverseFile != NULL) 
     {
      count = maxIndices;
      lvUniverseFile = CloseFileIfNeeded(lvUniverseFile,&count,&arrayVersion,maxIndices,NULL,NULL);
     }

   if (primaryTermFile != NULL) 
     {
      count = maxIndices;
      primaryTermFile = CloseFileIfNeeded(primaryTermFile,&count,&arrayVersion,maxIndices,NULL,NULL);
     }

   if (modifierFile != NULL) 
     {
      count = maxIndices;
      modifierFile = CloseFileIfNeeded(modifierFile,&count,&arrayVersion,maxIndices,NULL,NULL);
     }

#endif
  }
    
	
#if FUZZY_DEFTEMPLATES

/* generate code for the fuzzy deftemplate definitions */

/************************************************************/
/* LvUniverseToCode:                                        */
/************************************************************/
#if IBM_TBC
#pragma argsused
#endif
static VOID LvUniverseToCode(lvUniverseFile,lvPtr,imageID,maxIndices,
			             lvUniverseArrayCount,lvUniverseArrayVersion,
					     primaryTermArrayCount,primaryTermArrayVersion,
					     modifierArrayCount,modifierArrayVersion)
  FILE *lvUniverseFile;
  struct fuzzyLv *lvPtr;
  int imageID;
  int maxIndices;
  int lvUniverseArrayCount, lvUniverseArrayVersion;
  int primaryTermArrayCount,primaryTermArrayVersion;
  int modifierArrayCount,modifierArrayVersion;
  {
#if MAC_MPW
#pragma unused(maxIndices)
#endif
    fprintf(lvUniverseFile, "{{(struct universe *)&%s%d_%d[%d].u, &%s%d_%d[%d], ",
	        LvUniversePrefix(), imageID, lvUniverseArrayVersion, lvUniverseArrayCount,
			PrimaryTermPrefix(), imageID, primaryTermArrayVersion, primaryTermArrayCount);
	if (lvPtr->modifier_list != NULL)
	   fprintf(lvUniverseFile, "&%s%d_%d[%d]},",
			   ModifierPrefix(), imageID, modifierArrayVersion, modifierArrayCount);
	else
	   fprintf(lvUniverseFile, "NULL},");
	fprintf(lvUniverseFile, "{%s, %s, ", 
	        FloatToString(lvPtr->u_ptr->from), FloatToString(lvPtr->u_ptr->to));
	PrintSymbolReference(lvUniverseFile,lvPtr->u_ptr->units);
	fprintf(lvUniverseFile, "}}");
  }


/************************************************************/
/* primaryTermToCode:                                       */
/************************************************************/
static VOID primaryTermToCode(primaryTermFile,primaryTermPtr,imageID,maxIndices,
							  primaryTermArrayCount,primaryTermArrayVersion)
  FILE *primaryTermFile;
  struct primary_term *primaryTermPtr;
  int imageID;
  int maxIndices;
  int *primaryTermArrayCount,primaryTermArrayVersion;
  {
    int count, arrayVersion;
    struct primary_term *nextPtr;
	
    while (primaryTermPtr != NULL)
       {
         nextPtr = primaryTermPtr->next;
         if ((nextPtr == NULL) && (*primaryTermArrayCount >= maxIndices))
           {
             count = 0;
             arrayVersion = primaryTermArrayVersion+1;
           }
         else
           {
             count = *primaryTermArrayCount+1;
             arrayVersion = primaryTermArrayVersion;
           }
	  
         fprintf(primaryTermFile,"{");  
         PrintFuzzyValueReference( primaryTermFile, primaryTermPtr->fuzzy_value_description);
         if (nextPtr != NULL)
            fprintf(primaryTermFile,",&%s%d_%d[%d]}",
                    PrimaryTermPrefix(), imageID, arrayVersion, count);
         else
            fprintf(primaryTermFile, ",NULL}");
	     
         *primaryTermArrayCount += 1;
         primaryTermPtr = nextPtr;
		 
         if (primaryTermPtr != NULL)
         fprintf(primaryTermFile,",");
       }
  }


/************************************************************/
/* modifierToCode:                                          */
/************************************************************/
 static VOID modifierToCode(modifierFile,modifierPtr,imageID,maxIndices,
							modifierArrayCount,modifierArrayVersion)
  FILE *modifierFile;
  struct modifier *modifierPtr;
  int imageID;
  int maxIndices;
  int *modifierArrayCount,modifierArrayVersion;
  {
    int count, arrayVersion;
    struct modifier *nextPtr;
	
    while (modifierPtr != NULL)
       {
         nextPtr = modifierPtr->next;
         if ((nextPtr == NULL) && (*modifierArrayCount >= maxIndices))
           {
             count = 0;
             arrayVersion = modifierArrayVersion+1;
           }
         else
           {
             count = *modifierArrayCount+1;
             arrayVersion = modifierArrayVersion;
           }
	  
         fprintf(modifierFile,"{"); 
         PrintSymbolReference(modifierFile,modifierPtr->name);
         fprintf(modifierFile,","); 
         PrintFunctionReference(modifierFile,modifierPtr->function);
         fprintf(modifierFile,","); 
         PrintDeffunctionReference( modifierFile, modifierPtr->deffunction, ImageID,MaxIndices);

         if (nextPtr != NULL)
            fprintf(modifierFile,",&%s%d_%d[%d]}",
                    ModifierPrefix(), imageID, count, arrayVersion);
         else
            fprintf(modifierFile, ",NULL}");

         *modifierArrayCount += 1;
         modifierPtr = nextPtr;
		 
         if (modifierPtr != NULL)
            fprintf(modifierFile,",");
       }
  }


#endif	
	
	
	
/*************************************************************/
/* DeftemplateModuleToCode: Writes the C code representation */
/*   of a single deftemplate module to the specified file.   */
/*************************************************************/
#if IBM_TBC
#pragma argsused
#endif
static VOID DeftemplateModuleToCode(theFile,theModule,imageID,maxIndices,moduleCount)
  FILE *theFile;
  struct defmodule *theModule;
  int imageID;
  int maxIndices;
  int moduleCount;
  {   
#if MAC_MPW
#pragma unused(moduleCount)
#endif
   fprintf(theFile,"{"); 
   
   ConstructModuleToCode(theFile,theModule,imageID,maxIndices,
                                  DeftemplateModuleIndex,ConstructPrefix(DeftemplateCodeItem));
      
   fprintf(theFile,"}"); 
  }
  
/************************************************************/
/* DeftemplateToCode: Writes the C code representation of a */
/*   single deftemplate construct to the specified file.    */
/************************************************************/
#if FUZZY_DEFTEMPLATES
static VOID DeftemplateToCode(theFile,theTemplate,imageID,maxIndices,
                           moduleCount,slotCount,
						   lvUniverseArrayCount, lvUniverseArrayVersion)
#else
static VOID DeftemplateToCode(theFile,theTemplate,imageID,maxIndices,
                           moduleCount,slotCount)
#endif
  FILE *theFile;
  struct deftemplate *theTemplate;
  int imageID;
  int maxIndices;
  int moduleCount;
  int slotCount;
#if FUZZY_DEFTEMPLATES
  int lvUniverseArrayCount;
  int lvUniverseArrayVersion;
#endif
  {
   /*====================*/
   /* Deftemplate Header */
   /*====================*/
   
   fprintf(theFile,"{");
            
   ConstructHeaderToCode(theFile,&theTemplate->header,imageID,maxIndices,
                                  moduleCount,ModulePrefix(DeftemplateCodeItem),
                                  ConstructPrefix(DeftemplateCodeItem));
   fprintf(theFile,","); 
    
   /*===========*/
   /* Slot List */
   /*===========*/

   if (theTemplate->slotList == NULL)
     { fprintf(theFile,"NULL,"); }
   else
     {
      fprintf(theFile,"&%s%d_%d[%d],",SlotPrefix(),
                 imageID,
                 (slotCount / maxIndices) + 1,
                 slotCount % maxIndices);
     }

   /*==========================================*/
   /* Implied Flag, Watch Flag, In Scope Flag, */
   /* Number of Slots, and Busy Count.         */
   /*==========================================*/
   
   fprintf(theFile,"%d,0,0,%d,%ld,",theTemplate->implied,theTemplate->numberOfSlots,theTemplate->busyCount);

   /*=================*/
   /* Pattern Network */
   /*=================*/
   
   if (theTemplate->patternNetwork == NULL)
     { fprintf(theFile,"NULL"); }
   else
     { FactPatternNodeReference(theTemplate->patternNetwork,theFile,imageID,maxIndices); }

#if FUZZY_DEFTEMPLATES

   if (theTemplate->fuzzyTemplate == NULL)
     { fprintf(theFile,",NULL"); }
   else
     { fprintf(theFile, ",(struct fuzzyLv *)&%s%d_%d[%d]",LvUniversePrefix(),imageID,
	           lvUniverseArrayVersion,lvUniverseArrayCount); 
	 }
	 
#endif /* FUZZY_DEFTEMPLATES */
   
   fprintf(theFile,"}");
  }
  
/*****************************************************/
/* SlotToCode: Writes the C code representation of a */
/*   single deftemplate slot to the specified file.  */
/*****************************************************/
static VOID SlotToCode(theFile,theSlot,imageID,maxIndices,slotCount)
  FILE *theFile;
  struct templateSlot *theSlot;
  int imageID;
  int maxIndices;
  int slotCount;
  {
   /*===========*/
   /* Slot Name */
   /*===========*/
   
   fprintf(theFile,"{");
   PrintSymbolReference(theFile,theSlot->slotName);
   
   /*=============================*/
   /* Multislot and Default Flags */
   /*=============================*/
   
   fprintf(theFile,",%d,%d,%d,%d,",theSlot->multislot,theSlot->noDefault,
                                   theSlot->defaultPresent,theSlot->defaultDynamic);
   
   /*=============*/
   /* Constraints */
   /*=============*/
               
   PrintConstraintReference(theFile,theSlot->constraints,imageID,maxIndices);
      
   /*===============*/
   /* Default Value */
   /*===============*/
   
   fprintf(theFile,",");
   PrintHashedExpressionReference(theFile,theSlot->defaultList,imageID,maxIndices);
   fprintf(theFile,",");

   /*===========*/
   /* Next Slot */
   /*===========*/
   
   if (theSlot->next == NULL)
     { fprintf(theFile,"NULL}"); }
   else
     {
      fprintf(theFile,"&%s%d_%d[%d]}",SlotPrefix(),imageID,
                               ((slotCount+1) / maxIndices) + 1,
                                (slotCount+1) % maxIndices);
     }
  }
        
/*********************************************************/
/* DeftemplateCModuleReference:    */
/*********************************************************/
globle VOID DeftemplateCModuleReference(theFile,count,imageID,maxIndices)
  FILE *theFile;
  int count;
  int imageID;
  int maxIndices;
  {
   fprintf(theFile,"MIHS &%s%d_%d[%d]",ModulePrefix(DeftemplateCodeItem),
                      imageID,
                      (count / maxIndices) + 1,
                      (count % maxIndices));
  }

/*********************************************************/
/* DeftemplateCConstructReference:    */
/*********************************************************/
globle VOID DeftemplateCConstructReference(theFile,vTheTemplate,imageID,maxIndices)
  FILE *theFile;
  VOID *vTheTemplate;
  int imageID;
  int maxIndices;
  {  
   struct deftemplate *theTemplate = (struct deftemplate *) vTheTemplate;

   if (theTemplate == NULL)
     { fprintf(theFile,"NULL"); }
   else
     {
      fprintf(theFile,"&%s%d_%ld[%ld]",ConstructPrefix(DeftemplateCodeItem),
                      imageID,
                      (theTemplate->header.bsaveID / maxIndices) + 1,
                      theTemplate->header.bsaveID % maxIndices);
     }

  } 
#endif


