   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 6.00  05/12/93            */
   /*                                                     */
   /*            DEFTEMPLATE BSAVE/BLOAD MODULE           */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*      Brian L. Donnell                                     */
/*      Bob Orchard (NRCC - Nat'l Research Council of Canada)*/
/*                  (Fuzzy reasoning extensions)             */
/*                  (certainty factors for facts and rules)  */
/*                  (extensions to run command)              */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define  _TMPLTBIN_SOURCE_

#include "setup.h"

#if DEFTEMPLATE_CONSTRUCT && (BLOAD || BLOAD_ONLY || BLOAD_AND_BSAVE) && (! RUN_TIME)

#include <stdio.h>
#define _CLIPS_STDIO_

#include "clipsmem.h"
#include "bload.h"
#include "bsave.h"
#include "factbin.h"
#include "cstrnbin.h"
#include "factmngr.h"
#include "tmpltpsr.h"
#include "tmpltdef.h"

#include "tmpltbin.h"


#if FUZZY_DEFTEMPLATES
#include "fuzzylv.h"
#include "fuzzyval.h"
#include "dffnxbin.h"
#endif


/****************************************/
/* GLOBAL INTERNAL VARIABLE DEFINITIONS */
/****************************************/

   globle struct deftemplate HUGE_ADDR        *DeftemplateArray;
   
/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/

   static long                                 NumberOfDeftemplates;
   static long                                 NumberOfTemplateSlots;
   static long                                 NumberOfTemplateModules;
#if FUZZY_DEFTEMPLATES
   static long                                 NumberOfFuzzyTemplates;
   static long                                 NumberOfFuzzyPrimaryTerms;
   static long                                 NumberOfFuzzyModifiers;
#endif
   static struct templateSlot HUGE_ADDR       *SlotArray;
   static struct deftemplateModule HUGE_ADDR  *ModuleArray;
#if FUZZY_DEFTEMPLATES

   struct LvPlusUniverse
     {
	   struct fuzzyLv lv;
	   struct universe u;
	 };
	 
   static struct LvPlusUniverse HUGE_ADDR     *LvPlusUniverseArray;
   static struct primary_term   HUGE_ADDR     *PrimaryTermArray;
   static struct modifier       HUGE_ADDR     *ModifierArray;
#endif

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if ANSI_COMPILER
#if BLOAD_AND_BSAVE
   static VOID                    BsaveFind(void);
   static VOID                    BsaveStorage(FILE *);
   static VOID                    BsaveBinaryItem(FILE *);
#endif
   static VOID                    BloadStorage(void);
   static VOID                    BloadBinaryItem(void);
   static VOID                    UpdateDeftemplateModule(VOID *,long);
   static VOID                    UpdateDeftemplate(VOID *,long);
   static VOID                    UpdateDeftemplateSlot(VOID *,long);
#if FUZZY_DEFTEMPLATES
   static VOID                    UpdateLvPlusUniverse(VOID *,long);
   static VOID                    UpdateFuzzyPrimaryTerms(VOID *,long);
   static VOID                    UpdateFuzzyModifiers(VOID *,long);
#endif
   static VOID                    ClearBload(void);
#else
#if BLOAD_AND_BSAVE
   static VOID                    BsaveFind();
   static VOID                    BsaveStorage();
   static VOID                    BsaveBinaryItem();
#endif
   static VOID                    BloadStorage();
   static VOID                    BloadBinaryItem();
   static VOID                    UpdateDeftemplateModule();
   static VOID                    UpdateDeftemplate();
   static VOID                    UpdateDeftemplateSlot();
#if FUZZY_DEFTEMPLATES
   static VOID                    UpdateLvPlusUniverse();
   static VOID                    UpdateFuzzyPrimaryTerms();
   static VOID                    UpdateFuzzyModifiers();
#endif
   static VOID                    ClearBload();
#endif

/***********************************************/
/* DeftemplateBinarySetup: Installs the binary */
/*   save/load feature for deftemplates.       */
/***********************************************/
globle VOID DeftemplateBinarySetup()
  {
#if BLOAD_AND_BSAVE
   AddBinaryItem("deftemplate",0,BsaveFind,NULL,
                             BsaveStorage,BsaveBinaryItem,
                             BloadStorage,BloadBinaryItem,
                             ClearBload);
#endif
#if (BLOAD || BLOAD_ONLY)
   AddBinaryItem("deftemplate",0,NULL,NULL,NULL,NULL,
                             BloadStorage,BloadBinaryItem,
                             ClearBload);
#endif
  }

#if BLOAD_AND_BSAVE

/****************************************************************/
/* BsaveFind: Find expressions function for deftemplate bsaves. */
/****************************************************************/
static VOID BsaveFind()
  {
   struct deftemplate *theDeftemplate;
   struct templateSlot *tsPtr;
   struct defmodule *theModule;

   if (Bloaded())
     {
      SaveBloadCount(NumberOfDeftemplates);
      SaveBloadCount(NumberOfTemplateSlots);
      SaveBloadCount(NumberOfTemplateModules);
#if FUZZY_DEFTEMPLATES
      SaveBloadCount(NumberOfFuzzyTemplates);
      SaveBloadCount(NumberOfFuzzyPrimaryTerms);
      SaveBloadCount(NumberOfFuzzyModifiers);
#endif
     }
     
   NumberOfDeftemplates = 0;
   NumberOfTemplateSlots = 0;
   NumberOfTemplateModules = 0;
#if FUZZY_DEFTEMPLATES
   NumberOfFuzzyTemplates = 0;
   NumberOfFuzzyPrimaryTerms = 0;
   NumberOfFuzzyModifiers = 0;
#endif
   
   theModule = (struct defmodule *) GetNextDefmodule(NULL);
   while (theModule != NULL)
     {
      SetCurrentModule((VOID *) theModule);
      NumberOfTemplateModules++;
      theDeftemplate = (struct deftemplate *) GetNextDeftemplate(NULL);
      while (theDeftemplate != NULL)
        {
         MarkConstructHeaderNeededItems(&theDeftemplate->header,
                                        NumberOfDeftemplates++);
#if FUZZY_DEFTEMPLATES
         if (theDeftemplate->fuzzyTemplate != NULL)
		 {
		   struct primary_term *ptptr;
		   struct modifier *mptr;
		   struct fuzzy_value *fvptr;
		   
		   /* fuzzy template -- deal with the fuzzyLv (linguistic value)
		      which contains the universe of discourse, the list of primary 
			  terms and the list of modifiers
			  Count the number of fuzzy lv's and the number of primary terms
			  and the number of modifiers ... also mark the fuzzy values and
			  symbols (modifier names, fv names, unit names) as 'needed' for bsave
		   */
		   NumberOfFuzzyTemplates++;
		   if (theDeftemplate->fuzzyTemplate->u_ptr->units != NULL)
		      theDeftemplate->fuzzyTemplate->u_ptr->units->neededSymbol = CLIPS_TRUE;
			  
		   ptptr = theDeftemplate->fuzzyTemplate->primary_term_list;
		   while (ptptr != NULL)
		     {
			   NumberOfFuzzyPrimaryTerms++;
			   ptptr->fuzzy_value_description->neededFuzzyValue = CLIPS_TRUE;
			   fvptr = ((FUZZY_VALUE_HN *)ptptr->fuzzy_value_description)->contents;
			   fvptr->name->neededSymbol = CLIPS_TRUE;
			   ptptr = ptptr->next;
			 }
		   mptr = theDeftemplate->fuzzyTemplate->modifier_list;
		   while (mptr != NULL)
		     {
			   NumberOfFuzzyModifiers++;
			   mptr->name->neededSymbol = CLIPS_TRUE;
			   mptr = mptr->next;
			 }
		 }
         else
         {
#endif
         tsPtr = theDeftemplate->slotList;
         while (tsPtr != NULL)
           {
            NumberOfTemplateSlots++;
            tsPtr->slotName->neededSymbol = CLIPS_TRUE;
            tsPtr = tsPtr->next;
           }
#if FUZZY_DEFTEMPLATES
         }
#endif
         theDeftemplate = (struct deftemplate *) GetNextDeftemplate(theDeftemplate);
        }
        
      theModule = (struct defmodule *) GetNextDefmodule(theModule);
     }
  }

/*********************************************************/
/* BsaveStorage: Writes out the storage requirements for */
/*   all deftemplate data structures to the beginning of */
/*   the binary file.                                    */
/*********************************************************/
static VOID BsaveStorage(fp)
  FILE *fp;
  {
   unsigned long space;

   space = sizeof(long) * 3;
   GenWrite(&space,(unsigned long) sizeof(long int),fp);
   GenWrite(&NumberOfDeftemplates,(unsigned long) sizeof(long int),fp);
   GenWrite(&NumberOfTemplateSlots,(unsigned long) sizeof(long int),fp);
   GenWrite(&NumberOfTemplateModules,(unsigned long) sizeof(long int),fp);
#if FUZZY_DEFTEMPLATES
   GenWrite(&NumberOfFuzzyTemplates,(unsigned long) sizeof(long int),fp);
   GenWrite(&NumberOfFuzzyPrimaryTerms,(unsigned long) sizeof(long int),fp);
   GenWrite(&NumberOfFuzzyModifiers,(unsigned long) sizeof(long int),fp);
#endif
  }

/***********************************************/
/* BsaveBinaryItem: Writes out all deftemplate */
/*   structures to the binary file.            */
/***********************************************/
static VOID BsaveBinaryItem(fp)
  FILE *fp;
  {
   unsigned long space;
   struct deftemplate *theDeftemplate;
   struct bsaveDeftemplate tempDeftemplate;
   struct templateSlot *tsPtr;
   struct bsaveTemplateSlot tempTemplateSlot;
   struct bsaveDeftemplateModule tempTemplateModule;
   struct defmodule *theModule;
   struct deftemplateModule *theModuleItem;
#if FUZZY_DEFTEMPLATES
   long lastNumberOfFuzzyPrimaryTerms;
   long lastNumberOfFuzzyModifiers;
#endif   
   /*=============================================*/
   /* Determine the total amount of space needed. */
   /*=============================================*/
   
   space = (NumberOfDeftemplates * sizeof(struct bsaveDeftemplate)) +
           (NumberOfTemplateSlots * sizeof(struct bsaveTemplateSlot)) +
#if FUZZY_DEFTEMPLATES
           (NumberOfFuzzyTemplates * sizeof(struct bsaveLvPlusUniverse)) +
           (NumberOfFuzzyPrimaryTerms * sizeof(struct bsaveFuzzyPrimaryTerm)) +
           (NumberOfFuzzyModifiers * sizeof(struct bsaveFuzzyModifier)) +
#endif
           (NumberOfTemplateModules * sizeof(struct bsaveDeftemplateModule));
   GenWrite(&space,(unsigned long) sizeof(unsigned long int),fp);
   
   /*===================================================*/
   /* Write out each deftemplate module data structure. */
   /*===================================================*/

   NumberOfDeftemplates = 0;
#if FUZZY_DEFTEMPLATES
   NumberOfFuzzyTemplates = 0;
   NumberOfFuzzyPrimaryTerms = 0;
   NumberOfFuzzyModifiers = 0;
   lastNumberOfFuzzyPrimaryTerms = 0;
   lastNumberOfFuzzyModifiers = 0;
#endif
   theModule = (struct defmodule *) GetNextDefmodule(NULL);
   while (theModule != NULL)
     {
      SetCurrentModule((VOID *) theModule);
      
      theModuleItem = (struct deftemplateModule *) 
                      GetModuleItem(NULL,FindModuleItem("deftemplate")->moduleIndex);
      AssignBsaveDefmdlItemHdrVals(&tempTemplateModule.header,
                                           &theModuleItem->header);
      GenWrite(&tempTemplateModule,(unsigned long) sizeof(struct bsaveDeftemplateModule),fp);
      theModule = (struct defmodule *) GetNextDefmodule(theModule);
     }
     
   /*============================================*/
   /* Write out each deftemplate data structure. */
   /*============================================*/

   NumberOfTemplateSlots = 0;
   theModule = (struct defmodule *) GetNextDefmodule(NULL);
   while (theModule != NULL)
     {
      SetCurrentModule((VOID *) theModule);
      
      theDeftemplate = (struct deftemplate *) GetNextDeftemplate(NULL);
      while (theDeftemplate != NULL)
        {
         AssignBsaveConstructHeaderVals(&tempDeftemplate.header,
                                          &theDeftemplate->header);
         tempDeftemplate.implied = theDeftemplate->implied;
         tempDeftemplate.numberOfSlots = theDeftemplate->numberOfSlots;
         tempDeftemplate.patternNetwork = BsaveFactPatternIndex(theDeftemplate->patternNetwork);

#if FUZZY_DEFTEMPLATES
         /* a -1 in fuzzyTemplateList slot of tempDeftemplate indicates
		    NOT a fuzzy deftemplate -- other values give index into the 
			LvPlusUniverseArray for resolving ptr when read back in
	     */
         if (theDeftemplate->fuzzyTemplate == NULL)
             tempDeftemplate.fuzzyTemplateList = -1L;
         else 
		     tempDeftemplate.fuzzyTemplateList = NumberOfFuzzyTemplates;
#endif
         if (theDeftemplate->slotList != NULL)
           { tempDeftemplate.slotList = NumberOfTemplateSlots; }
         else tempDeftemplate.slotList = -1L;

         GenWrite(&tempDeftemplate,(unsigned long) sizeof(struct bsaveDeftemplate),fp);

#if FUZZY_DEFTEMPLATES
         if (theDeftemplate->fuzzyTemplate != NULL)
		    NumberOfFuzzyTemplates++;
		 else
         /* only count standard template slots NOT fuzzy ones as well */
#endif
            NumberOfTemplateSlots += theDeftemplate->numberOfSlots;
			 

         theDeftemplate = (struct deftemplate *) GetNextDeftemplate(theDeftemplate);
        }
      
      theModule = (struct defmodule *) GetNextDefmodule(theModule);
     }


#if FUZZY_DEFTEMPLATES
   /* At this point the bsave templates are written out .. now append  
	  to that for any fuzzy deftemplates
			    
	     1. write out the Lv plus universe of discourse
	     2. write out the list of primary terms
	     3. write out the list of modifiers
				  
	  all LvPlusUniverse structs written, then all primary terms then
	  all modifiers
   */

   if (NumberOfFuzzyTemplates > 0)
    {
     theModule = (struct defmodule *) GetNextDefmodule(NULL);
     while (theModule != NULL)
       {
        SetCurrentModule((VOID *) theModule);
   
        theDeftemplate = (struct deftemplate *) GetNextDeftemplate(NULL);
        while (theDeftemplate != NULL)
          {
           if (theDeftemplate->fuzzyTemplate != NULL)
		     {
			   struct bsaveLvPlusUniverse bLVFU;
			   struct fuzzyLv *flvptr = theDeftemplate->fuzzyTemplate;
			   struct primary_term *ptptr;
			   struct modifier *mptr;
			   
			   ptptr = flvptr->primary_term_list;
			   while (ptptr != NULL)
			     { /* count the primary terms */
				   NumberOfFuzzyPrimaryTerms++;
				   ptptr = ptptr->next;
				 }
			   mptr = flvptr->modifier_list;
			   while (mptr != NULL)
			     { /* count the modifiers */
				   NumberOfFuzzyModifiers++;
				   mptr = mptr->next;
				 }
				 
               /* -1 means no primary terms or no modifiers */
			   bLVFU.ptPtr = (NumberOfFuzzyPrimaryTerms == lastNumberOfFuzzyPrimaryTerms) ? 
			                                               -1L : lastNumberOfFuzzyPrimaryTerms;
			   bLVFU.modPtr = (NumberOfFuzzyModifiers == lastNumberOfFuzzyModifiers) ? 
			                                               -1L : lastNumberOfFuzzyModifiers;
			   bLVFU.from = flvptr->u_ptr->from;
			   bLVFU.to = flvptr->u_ptr->to;
			   if (flvptr->u_ptr->units == NULL) /* may not be a unit specifier */
			     bLVFU.unitsName = -1L;
			   else
			     bLVFU.unitsName = (long)flvptr->u_ptr->units->bucket;
               GenWrite(&bLVFU,(unsigned long) sizeof(struct bsaveLvPlusUniverse),fp);
			   
			   lastNumberOfFuzzyPrimaryTerms = NumberOfFuzzyPrimaryTerms;
			   lastNumberOfFuzzyModifiers = NumberOfFuzzyModifiers;
             }

           theDeftemplate = (struct deftemplate *) GetNextDeftemplate(theDeftemplate);
          }
        
        theModule = (struct defmodule *) GetNextDefmodule(theModule);
       }

     theModule = (struct defmodule *) GetNextDefmodule(NULL);
     while (theModule != NULL)
       {
        SetCurrentModule((VOID *) theModule);
   
        theDeftemplate = (struct deftemplate *) GetNextDeftemplate(NULL);
        while (theDeftemplate != NULL)
          {
           if (theDeftemplate->fuzzyTemplate != NULL)
		     {
			   struct fuzzyLv *flvptr = theDeftemplate->fuzzyTemplate;
			   struct primary_term *ptptr;
               struct bsaveFuzzyPrimaryTerm bFPT;
			  
			   ptptr = flvptr->primary_term_list;
			   while (ptptr != NULL)
			     { /* write out the primary terms */
				   bFPT.fuzzyValue = (long)ptptr->fuzzy_value_description->bucket;
                   if (ptptr->next != NULL)
				      bFPT.next = 0L;
                   else 
				      bFPT.next = -1L;

                   GenWrite(&bFPT,(unsigned long) sizeof(struct bsaveFuzzyPrimaryTerm),fp);
				   ptptr = ptptr->next;
				 }
			  }
  
           theDeftemplate = (struct deftemplate *) GetNextDeftemplate(theDeftemplate);
          }
        
        theModule = (struct defmodule *) GetNextDefmodule(theModule);
       }

     theModule = (struct defmodule *) GetNextDefmodule(NULL);
     while (theModule != NULL)
       {
        SetCurrentModule((VOID *) theModule);
     
        theDeftemplate = (struct deftemplate *) GetNextDeftemplate(NULL);
        while (theDeftemplate != NULL)
          {
           if (theDeftemplate->fuzzyTemplate != NULL)
		     {
			   struct fuzzyLv *flvptr = theDeftemplate->fuzzyTemplate;
			   struct modifier *mptr;
			   struct bsaveFuzzyModifier bFM;
			  
			   mptr = flvptr->modifier_list;
			   while (mptr != NULL)
			     { /* write out the modifiers */
				   bFM.modifierName = (long)mptr->name->bucket;
				   if (mptr->function == NULL)
				      bFM.modifierFunction = -1L;
				   else
				      bFM.modifierFunction = (long)mptr->function->bsaveIndex;
				   if (mptr->deffunction == NULL)
				      bFM.modifierDeffunction = -1L;
				   else
				      bFM.modifierDeffunction = 
					     (long)((struct constructHeader *)mptr->deffunction)->bsaveID;

                   if (mptr->next != NULL)
				      bFM.next = 0L;
                   else 
				      bFM.next = -1L;

                   GenWrite(&bFM,(unsigned long) sizeof(struct bsaveFuzzyModifier),fp);
				   mptr = mptr->next;
				 }
			  }

           theDeftemplate = (struct deftemplate *) GetNextDeftemplate(theDeftemplate);
          }
        
        theModule = (struct defmodule *) GetNextDefmodule(theModule);
       }
     }
#endif


   /*==============================================*/
   /* Write out each template slot data structure. */
   /*==============================================*/

   theModule = (struct defmodule *) GetNextDefmodule(NULL);
   while (theModule != NULL)
     {
      SetCurrentModule((VOID *) theModule);
   
      theDeftemplate = (struct deftemplate *) GetNextDeftemplate(NULL);
      while (theDeftemplate != NULL)
        {
         tsPtr = theDeftemplate->slotList;
         while (tsPtr != NULL)
           {
            tempTemplateSlot.constraints = ConstraintIndex(tsPtr->constraints); 
            tempTemplateSlot.slotName = (long) tsPtr->slotName->bucket;
            tempTemplateSlot.multislot = tsPtr->multislot;
            tempTemplateSlot.noDefault = tsPtr->noDefault;
            tempTemplateSlot.defaultPresent = tsPtr->defaultPresent;
            tempTemplateSlot.defaultDynamic = tsPtr->defaultDynamic;
            tempTemplateSlot.defaultList = HashedExpressionIndex(tsPtr->defaultList);

            if (tsPtr->next != NULL) tempTemplateSlot.next = 0L;
            else tempTemplateSlot.next = -1L;

            GenWrite(&tempTemplateSlot,(unsigned long) sizeof(struct bsaveTemplateSlot),fp);
            tsPtr = tsPtr->next;
           }

         theDeftemplate = (struct deftemplate *) GetNextDeftemplate(theDeftemplate);
        }
        
      theModule = (struct defmodule *) GetNextDefmodule(theModule);
     }
     
   if (Bloaded())
     {
      RestoreBloadCount(&NumberOfDeftemplates);
      RestoreBloadCount(&NumberOfTemplateSlots);
      RestoreBloadCount(&NumberOfTemplateModules);
#if FUZZY_DEFTEMPLATES
      RestoreBloadCount(&NumberOfFuzzyTemplates);
      RestoreBloadCount(&NumberOfFuzzyPrimaryTerms);
      RestoreBloadCount(&NumberOfFuzzyModifiers);
#endif
     }
  }
 
#endif /* BLOAD_AND_BSAVE */

/***********************************************/
/* BloadStorage: Allocates the storage for the */
/*   deftemplates used by this binary image.   */
/***********************************************/
static VOID BloadStorage()
  {
   unsigned long int space;

   /*===========================================*/
   /* Determine the number of items to be read. */
   /*===========================================*/
   
   GenRead(&space,(unsigned long) sizeof(unsigned long int));
   GenRead(&NumberOfDeftemplates,(unsigned long) sizeof(long int));
   GenRead(&NumberOfTemplateSlots,(unsigned long) sizeof(long int));
   GenRead(&NumberOfTemplateModules,(unsigned long) sizeof(long int));
#if FUZZY_DEFTEMPLATES
   GenRead(&NumberOfFuzzyTemplates,(unsigned long) sizeof(long int));
   GenRead(&NumberOfFuzzyPrimaryTerms,(unsigned long) sizeof(long int));
   GenRead(&NumberOfFuzzyModifiers,(unsigned long) sizeof(long int));
#endif
   
   /*==============================================*/
   /* Read the deftemplate module data structures. */
   /*==============================================*/
   
   if (NumberOfTemplateModules == 0)
     {
      DeftemplateArray = NULL;
      SlotArray = NULL;
      ModuleArray = NULL;
#if FUZZY_DEFTEMPLATES
      LvPlusUniverseArray = NULL;
      PrimaryTermArray = NULL;
      ModifierArray = NULL;
#endif
     }
     
   space = NumberOfTemplateModules * sizeof(struct deftemplateModule);
   ModuleArray = (struct deftemplateModule HUGE_ADDR *) genlongalloc(space);
   
   /*=======================================*/
   /* Read the deftemplate data structures. */
   /*=======================================*/
   
   if (NumberOfDeftemplates == 0)
     {
      DeftemplateArray = NULL;
      SlotArray = NULL;
#if FUZZY_DEFTEMPLATES
      LvPlusUniverseArray = NULL;
      PrimaryTermArray = NULL;
      ModifierArray = NULL;
#endif
      return;
     }

   space = NumberOfDeftemplates * sizeof(struct deftemplate);

   DeftemplateArray = (struct deftemplate HUGE_ADDR *) genlongalloc(space);

#if FUZZY_DEFTEMPLATES
   /*==============================================*/
   /* Get space for the LvPlusUniverse structures. */
   /*==============================================*/
   
   if (NumberOfFuzzyTemplates == 0)
     {
      LvPlusUniverseArray = NULL;
     }
   else
     {
      space = NumberOfFuzzyTemplates * sizeof(struct LvPlusUniverse);
   
      LvPlusUniverseArray = (struct LvPlusUniverse HUGE_ADDR *) genlongalloc(space);
     }

   /*============================================*/
   /* Get space for the primary term structures. */
   /*============================================*/
   
   if (NumberOfFuzzyPrimaryTerms == 0)
     {
      PrimaryTermArray = NULL;
     }
   else
     {
      space = NumberOfFuzzyPrimaryTerms * sizeof(struct primary_term);
   
      PrimaryTermArray = (struct primary_term HUGE_ADDR *) genlongalloc(space);
     }

   /*============================================*/
   /* Get space for the Modifier structures.     */
   /*============================================*/
   
   if (NumberOfFuzzyModifiers == 0)
     {
      ModifierArray = NULL;
     }
   else
     {
      space = NumberOfFuzzyModifiers * sizeof(struct modifier);
   
      ModifierArray = (struct modifier HUGE_ADDR *) genlongalloc(space);
     }
#endif  /* FUZZY_DEFTEMPLATES */

   /*============================================*/
   /* Read the deftemplate slot data structures. */
   /*============================================*/
   
   if (NumberOfTemplateSlots == 0)
     {
      SlotArray = NULL;
      return;
     }

   space =  NumberOfTemplateSlots * sizeof(struct templateSlot);
   SlotArray = (struct templateSlot HUGE_ADDR *) genlongalloc(space);
  }

/*******************************************************/
/* BloadBinaryItem: Loads and updates the deftemplates */
/*   used by this binary image.                        */
/*******************************************************/
static VOID BloadBinaryItem()
  {
   unsigned long int space;

   GenRead(&space,(unsigned long) sizeof(unsigned long int));
   BloadandRefresh(NumberOfTemplateModules,(unsigned) sizeof(struct bsaveDeftemplateModule),
                   UpdateDeftemplateModule);
   BloadandRefresh(NumberOfDeftemplates,(unsigned) sizeof(struct bsaveDeftemplate),
                   UpdateDeftemplate);
#if FUZZY_DEFTEMPLATES
   BloadandRefresh(NumberOfFuzzyTemplates,(unsigned) sizeof(struct bsaveLvPlusUniverse),
                   UpdateLvPlusUniverse);
   BloadandRefresh(NumberOfFuzzyPrimaryTerms,(unsigned) sizeof(struct bsaveFuzzyPrimaryTerm),
                   UpdateFuzzyPrimaryTerms);
   BloadandRefresh(NumberOfFuzzyModifiers,(unsigned) sizeof(struct bsaveFuzzyModifier),
                   UpdateFuzzyModifiers);
#endif  /* FUZZY_DEFTEMPLATES */
   BloadandRefresh(NumberOfTemplateSlots,(unsigned) sizeof(struct bsaveTemplateSlot),
                   UpdateDeftemplateSlot);
  }
  
/*************************************************/
/* UpdateDeftemplateModule: Updates pointers in  */
/*   bloaded deftemplate module data structures. */
/*************************************************/
static VOID UpdateDeftemplateModule(buf,obji)
  VOID *buf;
  long obji;
  {
   struct bsaveDeftemplateModule *bdmPtr;
   
   bdmPtr = (struct bsaveDeftemplateModule *) buf;
   UpdateDefmoduleItemHeader(&bdmPtr->header,&ModuleArray[obji].header,
                             (int) sizeof(struct deftemplate),
                             (VOID *) DeftemplateArray);
  }
  
/**************************************************/
/* UpdateDeftemplate: Updates pointers in bloaded */
/*   deftemplate data structures.                 */
/**************************************************/
static VOID UpdateDeftemplate(buf,obji)
  VOID *buf;
  long obji;
  {
   struct deftemplate *theDeftemplate;
   struct bsaveDeftemplate *bdtPtr;
   
   bdtPtr = (struct bsaveDeftemplate *) buf;
   theDeftemplate = (struct deftemplate *) &DeftemplateArray[obji];

   UpdateConstructHeader(&bdtPtr->header,&theDeftemplate->header,
                         (int) sizeof(struct deftemplateModule),(VOID *) ModuleArray,
                         (int) sizeof(struct deftemplate),(VOID *) DeftemplateArray);
   
   if (bdtPtr->slotList != -1L)
     { theDeftemplate->slotList = (struct templateSlot *) &SlotArray[bdtPtr->slotList]; }
   else
     { theDeftemplate->slotList = NULL; }

#if FUZZY_DEFTEMPLATES
   if (bdtPtr->fuzzyTemplateList != -1L)
     { theDeftemplate->fuzzyTemplate = (struct fuzzyLv *) &LvPlusUniverseArray[bdtPtr->fuzzyTemplateList]; }
   else
     { theDeftemplate->fuzzyTemplate = NULL; }
#endif

   if (bdtPtr->patternNetwork != -1L)
     { theDeftemplate->patternNetwork = (struct factPatternNode *) BloadFactPatternPointer(bdtPtr->patternNetwork); }
   else
     { theDeftemplate->patternNetwork = NULL; }
     
   theDeftemplate->implied = bdtPtr->implied;
#if DEBUGGING_FUNCTIONS
   theDeftemplate->watch = WatchFacts;
#endif
   theDeftemplate->inScope = CLIPS_FALSE;
   theDeftemplate->numberOfSlots = bdtPtr->numberOfSlots;
  }


#if FUZZY_DEFTEMPLATES
/******************************************************/
/* UpdateLvPlusUniverse: Updates pointers in bloaded  */
/*   LvPlusUniverse slot structures.                  */
/******************************************************/
static VOID UpdateLvPlusUniverse(buf,obji)
  VOID *buf;
  long obji;
  {
    struct LvPlusUniverse *lvpuPtr;
    struct bsaveLvPlusUniverse *blvpuPtr;
	
	blvpuPtr = (struct bsaveLvPlusUniverse *) buf;
	lvpuPtr = (struct LvPlusUniverse *) &LvPlusUniverseArray[obji];

    lvpuPtr->lv.u_ptr = (struct universe *)&lvpuPtr->u;
	
	if (blvpuPtr->ptPtr != -1L)
	  lvpuPtr->lv.primary_term_list = (struct primary_term *) &PrimaryTermArray[blvpuPtr->ptPtr];
    else
	  lvpuPtr->lv.primary_term_list = NULL;
	  
	if (blvpuPtr->modPtr != -1L)
	  lvpuPtr->lv.modifier_list = (struct modifier *) &ModifierArray[blvpuPtr->modPtr];
    else
	  lvpuPtr->lv.modifier_list = NULL;
	  
	lvpuPtr->u.from = blvpuPtr->from;
	lvpuPtr->u.to = blvpuPtr->to;
	if (blvpuPtr->unitsName == -1L)
	   lvpuPtr->u.units = NULL;	  
	else
           {
	     lvpuPtr->u.units = SymbolArray[blvpuPtr->unitsName];
	     IncrementSymbolCount(lvpuPtr->u.units);
           }
  }



/*********************************************************/
/* UpdateFuzzyPrimaryTerms: Updates pointers in bloaded  */
/*   primary term data structures.                       */
/*********************************************************/
static VOID UpdateFuzzyPrimaryTerms(buf,obji)
  VOID *buf;
  long obji;
  {
    struct primary_term *ptPtr;
    struct bsaveFuzzyPrimaryTerm *bptPtr;
	
	bptPtr = (struct bsaveFuzzyPrimaryTerm *) buf;
	ptPtr = (struct primary_term *) &PrimaryTermArray[obji];

    ptPtr->fuzzy_value_description = (FUZZY_VALUE_HN *) FuzzyValueArray[bptPtr->fuzzyValue];
	IncrementFuzzyValueCount(ptPtr->fuzzy_value_description);
	IncrementSymbolCount(ptPtr->fuzzy_value_description->contents->name);
	
	if (bptPtr->next == -1L)
	  ptPtr->next = NULL;
	else
	  ptPtr->next = (struct primary_term *) &PrimaryTermArray[obji+1];
  }



/******************************************************/
/* UpdateFuzzyModifiers: Updates pointers in bloaded  */
/*   modifier data structures.                        */
/******************************************************/
static VOID UpdateFuzzyModifiers(buf,obji)
  VOID *buf;
  long obji;
  {
    struct modifier *mPtr;
    struct bsaveFuzzyModifier *bmPtr;
	
	bmPtr = (struct bsaveFuzzyModifier *) buf;
	mPtr = (struct modifier *) &ModifierArray[obji];

    mPtr->name = SymbolArray[bmPtr->modifierName];
	IncrementSymbolCount(mPtr->name);

    mPtr->function = FunctionPointer(bmPtr->modifierFunction);
	   
    mPtr->deffunction = DeffunctionPointer(bmPtr->modifierDeffunction);
	   
	if (bmPtr->next == -1L)
	  mPtr->next = NULL;
	else
	  mPtr->next = (struct modifier *) &ModifierArray[obji+1];
  }
#endif /* FUZZY_DEFTEMPLATES */


/******************************************************/
/* UpdateDeftemplateSlot: Updates pointers in bloaded */
/*   deftemplate slot data structures.                */
/******************************************************/
static VOID UpdateDeftemplateSlot(buf,obji)
  VOID *buf;
  long obji;
  {
   struct templateSlot *tsPtr;
   struct bsaveTemplateSlot *btsPtr;

   btsPtr = (struct bsaveTemplateSlot *) buf;
   tsPtr = (struct templateSlot *) &SlotArray[obji];

   tsPtr->slotName = SymbolPointer(btsPtr->slotName);
   IncrementSymbolCount(tsPtr->slotName);
   tsPtr->defaultList = HashedExpressionPointer(btsPtr->defaultList);
   tsPtr->constraints = ConstraintPointer(btsPtr->constraints);
   
   tsPtr->multislot = btsPtr->multislot;
   tsPtr->noDefault = btsPtr->noDefault;
   tsPtr->defaultPresent = btsPtr->defaultPresent;
   tsPtr->defaultDynamic = btsPtr->defaultDynamic;
   
   if (btsPtr->next != -1L)
     { tsPtr->next = (struct templateSlot *) &SlotArray[obji + 1]; }
   else
     { tsPtr->next = NULL; }
  }
  
/******************************************************/
/* ClearBload: Clear function for deftemplate bloads. */
/******************************************************/
static VOID ClearBload()
  {
   unsigned long int space;
   int i;
   
   for (i = 0; i < NumberOfDeftemplates; i++)
     { UnmarkConstructHeader(&DeftemplateArray[i].header); }
     
   for (i = 0; i < NumberOfTemplateSlots; i++)
     { DecrementSymbolCount(SlotArray[i].slotName); }

#if FUZZY_DEFTEMPLATES
   for (i = 0; i < NumberOfFuzzyTemplates; i++)
     { 
       if (LvPlusUniverseArray[i].u.units != NULL)
         DecrementSymbolCount(LvPlusUniverseArray[i].u.units);
     }

   for (i = 0; i < NumberOfFuzzyPrimaryTerms; i++)
     {
       DecrementSymbolCount(PrimaryTermArray[i].fuzzy_value_description->contents->name);
       DecrementFuzzyValueCount(PrimaryTermArray[i].fuzzy_value_description);
     }

   for (i = 0; i < NumberOfFuzzyModifiers; i++)
     { DecrementSymbolCount(ModifierArray[i].name); }
#endif

   space =  NumberOfTemplateModules * sizeof(struct deftemplateModule);
   if (space != 0) genlongfree((VOID *) ModuleArray,space);
   
   space = NumberOfDeftemplates * sizeof(struct deftemplate);
   if (space != 0) genlongfree((VOID *) DeftemplateArray,space);

#if FUZZY_DEFTEMPLATES
   space = NumberOfFuzzyTemplates * sizeof(struct LvPlusUniverse);
   if (space != 0) genlongfree((VOID *) LvPlusUniverseArray,space);

   space = NumberOfFuzzyPrimaryTerms * sizeof(struct primary_term);
   if (space != 0) genlongfree((VOID *) PrimaryTermArray,space);

   space = NumberOfFuzzyModifiers * sizeof(struct modifier);
   if (space != 0) genlongfree((VOID *) ModifierArray,space);

#endif

   space =  NumberOfTemplateSlots * sizeof(struct templateSlot);
   if (space != 0) genlongfree((VOID *) SlotArray,space);
   
#if (! BLOAD_ONLY)
   CreateImpliedDeftemplate(AddSymbol("initial-fact"),CLIPS_FALSE);
#endif
  }
  
/************************************************************/
/* BloadDeftemplateModuleReference: Returns the deftemplate */
/*   module pointer for using with the bload function.      */
/************************************************************/
globle VOID *BloadDeftemplateModuleReference(index)
  int index;
  {
   return ((VOID *) &ModuleArray[index]);
  }
  
#endif /* DEFTEMPLATE_CONSTRUCT && (BLOAD || BLOAD_ONLY || BLOAD_AND_BSAVE) && (! RUN_TIME) */


