   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 6.00  05/12/93            */
   /*                                                     */
   /*             FUZZY RHS PARSER MODULE                 */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*      Brian L. Donnell                                     */
/*      Bob Orchard (NRCC - Nat'l Research Council of Canada)*/
/*                  (Fuzzy reasoning extensions)             */
/*                  (certainty factors for facts and rules)  */
/*                  (extensions to run command)              */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/


/******************************************************************
    Fuzzy Logic Module

    This file contains the following categories of functions:
    
    functions to parse fuzzy facts asserted on rhs
    allocation and de-allocation of arrays from CLIPS memory

 ******************************************************************/
 
#define _FUZZYRHS_SOURCE_


#include "setup.h"


#include <stdio.h>
#define _CLIPS_STDIO_

#include <math.h>
#include <string.h>

#if FUZZY_DEFTEMPLATES
#include "fuzzyrhs.h"
#include "fuzzyval.h"
#include "fuzzylv.h"
#include "fuzzyutl.h"
#include "fuzzypsr.h"
#endif

#if FUZZY_DEFTEMPLATES || CERTAINTY_FACTORS
#include "symbol.h"
#include "expressn.h"
#include "evaluatn.h"
#include "exprnops.h"
#include "exprnpsr.h"
#include "extnfunc.h"
#include "tmpltdef.h"
#include "pprint.h"
#include "scanner.h"
#include "clipsmem.h"
#include "constant.h"
#include "prntutil.h"
#include "router.h"
#endif



#if FUZZY_DEFTEMPLATES




/******************************************************************
    Global Internal Function Declarations
    
    Defined in fuzzyrhs.h
 ******************************************************************/



/******************************************************************
    Local Internal Function Declarations
 ******************************************************************/
 
#if ANSI_COMPILER
  static struct fuzzy_value      *fuzzyOrExpression(char *readSource,
                                               struct token *tempToken,
                                               struct fuzzyLv *lvp,
                                               int *error,
                                               int NOTbracketFlag);
  static struct fuzzy_value      *fuzzySimpleExpression(char *readSource,
                                               struct token *tempToken,
                                               struct fuzzyLv *lvp,
                                               int *error);
  static VOID                  ModifyFuzzyValue(struct modifier *mptr,
                                                struct fuzzy_value *elem);
  static VOID                    modifyExpandedFuzzyValue(struct fuzzy_value *fv,
                                                 struct modifier *mptr);
  static VOID                  expand_set(struct fuzzy_value *fv);
  static struct fuzzy_value      *PrimaryTerm(char *readSource,
                                               struct token *tempToken,
                                               struct fuzzyLv *lvp,
                                               int *error);
  static char                   *modifyName(char *str1, char *str2);
  static struct modifier      *FindModifier(struct fuzzyLv *lvp, VOID *mod_name);
  static struct primary_term  *FindPrimaryTerm(struct fuzzyLv *lvp, VOID *pt_name);
  static struct expr           *assertParseFuzzySet(char *readSource,
                                                   struct token *tempToken,
                                                   int  *error,
                                                   struct universe *u,
                                                   int constantsOnly,
                                                   int *onlyConstantsFound);
  static struct expr           *assertParseStandardSet(char *readSource,
                                                      struct token *tempToken,
                                                      int  *error,
                                                      struct universe *u,
                                                      int constantsOnly,
                                                      int *onlyConstantsFound,
                                                      int function_type);
  static struct expr          *assertParseSingletonSet(char *readSource,
                                                      struct token *tempToken,
                                                      int  *error,
                                                      struct universe *u,
                                                      int constantsOnly,
                                                      int *onlyConstantsFound);
  static struct fuzzy_value   *convertStandardSet(struct expr *top, int *error);
  static struct fuzzy_value   *convertSingletonSet(struct expr *top, int *error);
  static VOID                   expressionToFloat(struct expr *test_ptr,
                                                  double *result,
                                                  int *error);
  static VOID                   expressionToInteger(struct expr *test_ptr,
                                                  int *result,
                                                  int *error);

#else
  static struct fuzzy_value      *fuzzyOrExpression();
  static struct fuzzy_value      *fuzzySimpleExpression();
  static VOID                     ModifyFuzzyValue();
  static VOID                     modifyExpandedFuzzyValue();
  static VOID                     expand_set();
  static struct fuzzy_value      *PrimaryTerm();
  static char                    *modifyName();
  static struct modifier         *FindModifier();
  static struct primary_term     *FindPrimaryTerm();
  static struct expr             *assertParseFuzzySet();
  static struct expr             *assertParseStandardSet();
  static struct expr             *assertParseSingletonSet();
  static struct fuzzy_value      *convertStandardSet();
  static struct fuzzy_value      *convertSingletonSet();
  static VOID                     expressionToFloat();
  static VOID                     expressionToInteger();

#endif


   
/******************************************************************
    Global External Variable Declarations
 ******************************************************************/



/******************************************************************
    Global Internal Variable Declarations
 ******************************************************************/



/************************************************************************
    FUNCTIONS that Copy and Compact Fuzzy Values

 ************************************************************************/
 

/*************************************************************/ 
/* CopyFuzzyValue:                                           */
/* copies a Fuzzy Value structure                            */
/*************************************************************/
globle struct fuzzy_value *CopyFuzzyValue(fv)
  struct fuzzy_value *fv;
{
    int i, num;
    struct fuzzy_value *new_fv;
   
    new_fv = get_struct(fuzzy_value);
    new_fv->name = fv->name;
    num = fv->n;
    
    new_fv->n = num;
    new_fv->maxn = num;
    new_fv->x = FgetArray(num);
    new_fv->y = FgetArray(num);
    
    for ( i = 0; i < num; i++)
     {
        new_fv->x[i] = fv->x[i];
        new_fv->y[i] = fv->y[i];
     }
    return (new_fv);
}



/*************************************************************/ 
/* CompactFuzzyValue:                                        */
/* compacts the membership values of a Fuzzy Value structure */
/* so that n and maxn are the same                           */
/*************************************************************/
globle VOID CompactFuzzyValue(fv)
  struct fuzzy_value *fv;
{
    int i, num, maxnum;
    double *xptr, *yptr;   
    
    num = fv->n;
    maxnum = fv->maxn;
    
    if (maxnum == num) return;
    
    fv->maxn = num;
    xptr = FgetArray(num);
    yptr = FgetArray(num);
    
    for ( i = 0; i < num; i++)
     {
        xptr[i] = fv->x[i];
        yptr[i] = fv->y[i];
     }
    
    FrtnArray(fv->x, maxnum);
    FrtnArray(fv->y, maxnum);
    
    fv->x = xptr;
    fv->y = yptr;
}




/************************************************************************
    FUNCTIONS THAT PARSE ASSERT ARGUMENTS for FUZZY FACTS

 ************************************************************************/
 
 
 
/***************************************************************/
/* ParseAssertFuzzyFact:                                       */
/* Returns assert argument in expr format.                     */ 
/* Error flag is set to true if an error occurs.               */
/*                                                             */
/* NOTE: don't use endType arg since it appears that it is     */
/*       ALWAYS RPAREN -- if this changes then should change   */
/*       this code to use endType rather than RPAREN           */
/***************************************************************/

#if IBM_TBC
#pragma argsused
#endif

globle struct expr *ParseAssertFuzzyFact(readSource,tempToken,
                                         error,endType,
                                         constantsOnly,theDeftemplate)
  char *readSource;
  struct token *tempToken;
  int *error;
  int endType;
  int constantsOnly; /* TRUE if only Constants allowed in fuzzy set descriptions */
  struct deftemplate *theDeftemplate;
{
#if MAC_MPW
#pragma unused(endType)
#endif
    struct fuzzyLv *lvp = theDeftemplate->fuzzyTemplate;
       struct expr *next_one, *temp;
       struct fuzzy_value *fv;
    int onlyConstantsFound; /* TRUE if find only Constants in fuzzy set description */
    
    *error = CLIPS_FALSE;

    /*=======================================*/
    /* Put a space between the template name */
    /* and the fuzzy set definition.         */
    /*=======================================*/

    SavePPBuffer(" ");
    GetToken(readSource,tempToken);
    
    if ((tempToken->type == LPAREN) ||
        ((tempToken->type == SYMBOL) &&
        (strcmp(ValueToString(tempToken->value),"#") == 0))
       )
         /* fuzzy set specified - # is optional so expect  '# (' or  '('  */
         {
          if (tempToken->type == SYMBOL)
            {  
              SavePPBuffer(" ");
              /* assertParseFuzzySet is expecting to see a LPAREN as current token */
              GetToken(readSource,tempToken);
            }

            next_one = assertParseFuzzySet(readSource, tempToken, error,
                                         lvp->u_ptr, constantsOnly, &onlyConstantsFound);
          if (*error == CLIPS_TRUE)
           {
            return(NULL);
           }
         else
           {            
            if (onlyConstantsFound)
              {
                fv = getConstantFuzzyValue (next_one, error);
                if (*error == CLIPS_TRUE)
                  {
                   ReturnExpression(next_one);
                   return(NULL);
                  }
                temp = get_struct(expr);
                temp->argList = NULL;
                temp->nextArg = NULL;
                temp->type = FUZZY_VALUE;
                temp->value = (VOID *)AddFuzzyValue(fv);
                /* AddFuzzyValue makes a copy of fv .. so return it */
                rtnFuzzyValue(fv);
                ReturnExpression(next_one);
                  return(temp);
              }
            else
             {
                return(next_one);
             }
            }        
       }
       else if (tempToken->type != SYMBOL)
         {
            SyntaxErrorMessage("Fuzzy Expression (Primary Term or modifier expected)");
            *error = CLIPS_TRUE;
            return(NULL);
         }
       else if ((fv = ParseFuzzyValue(readSource,tempToken,lvp,error))
             == NULL)
       {
           *error = CLIPS_TRUE;
          return(NULL);
       }
       else
       {
           next_one = get_struct(expr);
           next_one->argList = NULL;
           next_one->nextArg = NULL;
           next_one->type = FUZZY_VALUE;
         next_one->value = (VOID *)AddFuzzyValue(fv);
         /* AddFuzzyValue makes a copy of fv .. so return it */
         rtnFuzzyValue(fv);
           return(next_one);
       } 
}    

/***************************************************************/
/* ParseFuzzyValue:                                            */
/* Parses fuzzy expression with fuzzy terms and modifiers ONLY */
/*       eg.  very cold                                        */
/* Returns fuzzy value structure.                              */
/* Error flag is set to true if an error occurs.               */
/***************************************************************/
globle struct fuzzy_value *ParseFuzzyValue(readSource,tempToken,lvp,error)
  char *readSource;
  struct token *tempToken;
  struct fuzzyLv *lvp;
  int *error;
{
   struct fuzzy_value *fv;
   char *tmpstr;
   
     if (tempToken->type != SYMBOL)
       {
         SyntaxErrorMessage("Fuzzy Expression (NOT, Primary Term or Modifier expected)");
         *error = CLIPS_TRUE;
         return(NULL);
       }

     if (strcmp(ValueToString(tempToken->value),"not") == 0 ||
         strcmp(ValueToString(tempToken->value),"NOT") == 0 )
     {
      SavePPBuffer(" ");
      GetToken(readSource,tempToken);
      if (tempToken->type != SYMBOL)
        {
          SyntaxErrorMessage("Fuzzy Expression ([, Primary Term or Modifier expected)");
          *error = CLIPS_TRUE;
          return(NULL);
        }
      if (strcmp(ValueToString(tempToken->value),"[") == 0)
        {/* if have NOT and [ then can have OR's else can't have OR's */
         SavePPBuffer(" ");
         GetToken(readSource,tempToken);
         if (tempToken->type != SYMBOL)
           {
             SyntaxErrorMessage("Fuzzy Expression (Primary Term or Modifier expected)");
             *error = CLIPS_TRUE;
             return(NULL);
           }
         fv = fuzzyOrExpression(readSource,tempToken,lvp,error,CLIPS_TRUE);
         if (*error == CLIPS_TRUE)  
            {
             return(NULL);
            }
         fcompliment(fv);   /* fcompliment done in_place so copy not made */
         tmpstr = modifyName("NOT [",ValueToString(fv->name));
         fv->name = (SYMBOL_HN *)AddSymbol(tmpstr);
         rm(tmpstr, strlen(tmpstr)+1);
        }
      else
        { /* NO OR's allowed */
         fv = fuzzySimpleExpression(readSource,tempToken,lvp,error);
         if (*error == CLIPS_TRUE)  
           {
             return(NULL);
           }
         if (tempToken->type != RPAREN)  
           {
             *error = CLIPS_TRUE;
             SyntaxErrorMessage("Fuzzy Expression ( ')' expected)");
             rtnFuzzyValue(fv);
             return(NULL);
           }
         PPBackup(); PPBackup();
         SavePPBuffer(")");

         fcompliment(fv);   /* fcompliment done in_place so copy not made */
         tmpstr = modifyName("NOT",ValueToString(fv->name));
         fv->name = (SYMBOL_HN *)AddSymbol(tmpstr);
         rm(tmpstr, strlen(tmpstr)+1);
        }
     }
   else
     {
      fv = fuzzyOrExpression(readSource,tempToken,lvp,error,CLIPS_FALSE);
      if (*error == CLIPS_TRUE)  
        {
          return(NULL);
        }
     }
   
   return(fv);
}

/***************************************************************/
/* fuzzyOrExpression                                           */
/*                                                             */
/* Parses fuzzy expression (possibly with OR operators)        */
/*       eg. very cold  OR  very hot                           */
/***************************************************************/
static struct fuzzy_value *fuzzyOrExpression(readSource,tempToken,lvp,
                                           error,NOTbracketFlag)
  char *readSource;
  struct token *tempToken;
  struct fuzzyLv *lvp;
  int *error;
  int NOTbracketFlag;
{  
   struct fuzzy_value *w,*fv, *tmpfv;
   char *tmpstr;
   char *tmpstr2;

   w = fuzzySimpleExpression(readSource,tempToken,lvp,error);
   if (*error == CLIPS_TRUE)  
      { 
        return(NULL);
      }
   if (tempToken->type == RPAREN)
     {
       if (NOTbracketFlag == CLIPS_TRUE) /* if TRUE expecting a ] */
          {
            *error = CLIPS_TRUE;
            SyntaxErrorMessage("Fuzzy Expression (expecting ']')");
            rtnFuzzyValue(w);
            return(NULL);
          }
       PPBackup(); PPBackup();
       SavePPBuffer(")");
       return(w);
     }
   else if (tempToken->type == SYMBOL)
     {
       if (strcmp(ValueToString(tempToken->value),"]") == 0)
         {
           if (NOTbracketFlag == CLIPS_FALSE) /* if TRUE expecting a ] */
             {
               *error = CLIPS_TRUE;
               SyntaxErrorMessage("Fuzzy Expression (unexpected ']')");
               rtnFuzzyValue(w);
               return(NULL);
             }
           GetToken(readSource,tempToken);
           if (tempToken->type != RPAREN) /* ] must be followed by a ) */
             {
               *error = CLIPS_TRUE;
               SyntaxErrorMessage("Fuzzy Expression (expecting ')')");
               rtnFuzzyValue(w);
               return(NULL);
             }
           tmpstr = modifyName(ValueToString(w->name),"]");
           w->name = (SYMBOL_HN *)AddSymbol(tmpstr);
           rm(tmpstr, strlen(tmpstr)+1);
           return(w);
         }
       else if (strcmp(ValueToString(tempToken->value),"OR") == 0 ||
                strcmp(ValueToString(tempToken->value),"or") == 0)
         {
           SavePPBuffer(" ");
           GetToken(readSource,tempToken);
           if (tempToken->type != SYMBOL)
             {
               SyntaxErrorMessage("Fuzzy Expression (Primary Term or Modifier expected)");
               *error = CLIPS_TRUE;
               rtnFuzzyValue(w);
               return(NULL);
             }
           fv = fuzzyOrExpression(readSource,tempToken,lvp,
                                  error,NOTbracketFlag);
           if (*error == CLIPS_TRUE)  
             {
               return(NULL);
             }
           tmpfv = funion(w,fv);  /* OR is union of fuzzy sets */
           tmpstr = modifyName("OR",ValueToString(fv->name));
           tmpstr2 = modifyName(ValueToString(w->name), tmpstr);
           tmpfv->name = (SYMBOL_HN *)AddSymbol(tmpstr2);
           rtnFuzzyValue(w);
           rtnFuzzyValue(fv);
           rm(tmpstr2, strlen(tmpstr2)+1);      
           rm(tmpstr, strlen(tmpstr)+1);
           return(tmpfv);
          }
     }
   SyntaxErrorMessage("Fuzzy Expression ( ']', ')', or 'OR' expected)");
   *error = CLIPS_TRUE;
   rtnFuzzyValue(w);
   return(NULL);
}

/***************************************************************/
/* fuzzySimpleExpression                                       */
/* Parses fuzzy simple expression                              */
/***************************************************************/
static struct fuzzy_value *fuzzySimpleExpression(readSource,tempToken,lvp,
                                                 error)
  char *readSource;
  struct token *tempToken;
  struct fuzzyLv *lvp;
  int *error;
{  
   struct modifier *mptr;
   struct fuzzy_value *fvptr;

   mptr = FindModifier(lvp,tempToken->value);
   if (mptr != NULL)
     {
      SavePPBuffer(" ");
      GetToken(readSource,tempToken);
      if (tempToken->type != SYMBOL)
        {
          SyntaxErrorMessage("Fuzzy Expression (Primary Term or Modifier expected)");
          *error = CLIPS_TRUE;
          return(NULL);
        }
      fvptr = fuzzySimpleExpression(readSource,tempToken,lvp,error);
      if (*error == CLIPS_TRUE) 
        { 
          return(NULL);
        }
        
      ModifyFuzzyValue(mptr,fvptr);
      return(fvptr);
     }

   return( PrimaryTerm(readSource,tempToken,lvp,error) );
}



/***************************************************************/
/* ModifyFuzzyValue:                                           */
/*                                                             */
/* Takes a fuzzy value and modifies it according by operating  */
/* on each value of the fuzzy set with the function identified */
/* by the modifier                                             */
/*                                                             */
/* Modifies fuzzy_value without making a new one.              */
/***************************************************************/
static VOID ModifyFuzzyValue(mptr, fv)
  struct modifier *mptr;
  struct fuzzy_value *fv;
{
   char *tmpstr;

   if (fv == NULL)  /*should never happen!! */
     {  return; }

   /* modify the name eg. cold --> very cold */
   tmpstr = modifyName(ValueToString(mptr->name),ValueToString(fv->name));
   fv->name = (SYMBOL_HN *)AddSymbol(tmpstr);
   rm(tmpstr, strlen(tmpstr)+1);

   /* expand and modify fuzzy set */
   expand_set(fv);  /* expands the set without making copy of the fuzzy value */
   modifyExpandedFuzzyValue(fv,mptr);

}

/********************************************************************
    expand_set(fv)

    expands a fuzzy value with more points interpolated as necessary
    between segment endpoints such that the maximum difference in
    height between points in the new set is "YSPACING"
    
    NOTE: replaces the x and y arrays - does NOT create a new fuzzy value

    called before modifyExapndedFuzzyValue() in order to maintain 
    accuracy of f(x) ... for example if we have a fuzzy set that 
    consists of the 3 points
    
        (0,0) (5, 1) (10,0)
        
    and we apply the sqrt function to this fuzzy set then we get
    the same 3 points again (since we are storing pts and assuming
    that they are connected by a straight line). To get a better
    description of the function we want we need to make sure there
    are some other points between the extreme values of 0 and 1
    for the membership value.
 ********************************************************************/
 

/* YSPACING used to determine max Y difference between points when interpolating */
#define YSPACING 0.1



static VOID expand_set(fv)
  struct fuzzy_value *fv;
{
    int i, j, count;
    int num=fv->n;
    int newmaxn;
    int subdiv;
    double xdelta, ydelta, rise, run;
    double lastx, lasty, nextx, nexty, newx, newy;
    double *newxArray, *newyArray;

    newmaxn = num + ( floor(1.0/YSPACING) - 1 ) * ( num - 1 );
    newxArray = FgetArray ( newmaxn );
    newyArray = FgetArray ( newmaxn );
    
    newxArray[0] = fv->x[0];
    newyArray[0] = fv->y[0];
    
    count = 1;
    
    lastx = fv->x[0];
    lasty = fv->y[0];
    
    for ( i = 1; i < num; i++ )
    {
      nextx = fv->x[i];
      nexty = fv->y[i];

      rise = fabs ( nexty - lasty );
      subdiv = ceil ( rise / YSPACING );
      
      if (subdiv > 1)
        {
          run = nextx - lastx;
          xdelta = run / subdiv;
          ydelta = rise / subdiv;
          newx = lastx;
          newy = lasty;
          if ( nexty < lasty )
                ydelta = -ydelta;
                
          for ( j = 1; j < subdiv; j++ )
            {
                newx += xdelta;
                newy += ydelta;
                newxArray[count] = newx;
                newyArray[count] = newy;
                count++;
            }
        }
      lastx = newxArray[count] = nextx;
      lasty = newyArray[count] = nexty;
      count++;
    }
    
    FrtnArray(fv->x, fv->maxn);
    FrtnArray(fv->y, fv->maxn);
     fv->n = count;
    fv->maxn = newmaxn;
    fv->x = newxArray;
    fv->y = newyArray;
    
    /* does compacting in place (ie. no copy) */
    CompactFuzzyValue(fv);
}
              
/************************************************/
/* modifyExpandedFuzzyValue                     */
/*                                              */
/* Apply the modifier function to each value in */
/* the fuzzy set. To do this we must call the   */
/* function for each value after setting up the */
/* argument to the function as expected by the  */
/* function (expects that CurrentExpression is  */
/* pointing to an FCALL or PCALL expression     */
/* with args)                                   */
/************************************************/

static VOID modifyExpandedFuzzyValue(fv,mptr)
  struct fuzzy_value *fv;
  struct modifier *mptr;
{ 
       int i;
    int num=fv->n;
       double (*fp)(VOID_ARG);
    struct expr *tmpExpr;
    DATA_OBJECT result;
    BOOLEAN IsStandardFunction, res;
    DEFFUNCTION *deffun;
    
    /* CurrentExpression points to current expression to be evaluated
       so we set it up temporarily to point to a dummy expression 
       holding our fuzzy set y values (1 at a time) to be modified.
       CurrentExpression is a GLOBAL value!!
    */
    tmpExpr = CurrentExpression;
    /* this expr is where the FCALL/PCALL would normally be */
    CurrentExpression = get_struct(expr); 

    IsStandardFunction = (mptr->function == NULL) ? CLIPS_FALSE : CLIPS_TRUE;
    
    if (IsStandardFunction)
      { /* system defined or user defined internal function */
        fp = (double (*) (VOID_ARG)) (mptr->function->functionPointer);
      }
    else
      { /* user defined 'deffunction' */
        deffun = mptr->deffunction;
      }

    for (i=0; i<num; i++)
     {    
        double newval;
        
        CurrentExpression->argList = GenConstant(FLOAT, (VOID *)AddDouble(fv->y[i]));
        
        if (IsStandardFunction)
           newval = (*fp) ();
        else
          {
            res = (*PrimitivesArray[PCALL]->evaluateFunction)(deffun,&result);
            if (!res || (result.type != FLOAT))
              {
                SyntaxErrorMessage("Fuzzy Modifier function must accept FLOAT arg and return FLOAT)");
                break;               
              }
            newval = ValueToDouble(result.value);
          }
        
        /* membership values must lie between 0 and 1 */
        if (newval > 1.0) 
           newval = 1.0;
        else if (newval < 0.0)
           newval = 0.0;
        
        fv->y[i] = newval;
        rtn_struct(expr, CurrentExpression->argList);
     }

    /*DO NOT use ReturnExpression here! Haven't set up ptrs in this expression */
    rtn_struct(expr, CurrentExpression); 
    CurrentExpression = tmpExpr;
}

/***************************************************************/
/* PrimaryTerm:                                                */
/* Returns assert argument in fuzzy_value format.                */
/* Error flag is set to true if an error occurs.               */
/***************************************************************/
static struct fuzzy_value *PrimaryTerm(readSource, tempToken, lvp, error)
  char *readSource;
  struct token *tempToken;
  struct fuzzyLv *lvp;
  int *error;
{
   struct primary_term *pt;
   struct fuzzy_value *fv;

   if ((pt = FindPrimaryTerm(lvp,tempToken->value)) == NULL)
     {
       *error = CLIPS_TRUE;
       SyntaxErrorMessage("Fuzzy Expression (expecting a Primary Term or Modifier)");
       return(NULL);
     }
   else
     {
       fv = CopyFuzzyValue((struct fuzzy_value *) ValueToFuzzyValue(pt->fuzzy_value_description));
       SavePPBuffer(" ");
       GetToken(readSource,tempToken);
       return(fv);
     }
}

/*************************************************************/
/* modifyName:                                              */
/*************************************************************/
static char *modifyName(str1,str2)
  char *str1, *str2;
{
   char *temp;
   int str1len = strlen(str1);

   temp = gm2(sizeof(char) * (str1len + strlen(str2) + 2));
   strcpy(temp,str1);
   temp[str1len]   = ' ';
   temp[str1len+1] = '\0';
   strcpy(temp+str1len+1, str2);
   return(temp);
}
   

/*************************************************************/
/* FindModifier: Searches for modifier.                      */
/* Returns a pointer to the modifier if found,               */
/* otherwise NULL.                                           */
/*************************************************************/
static struct modifier *FindModifier(lvp,mod_name)
  struct fuzzyLv *lvp;
  VOID *mod_name;
{
   struct modifier *mptr;

   mptr = lvp->modifier_list;
   while (mptr != NULL)
     {
      if (mptr->name == (SYMBOL_HN *)mod_name)
        {  return(mptr);  }

      mptr = mptr->next;
     }
   return(NULL);
}

/*************************************************************/
/* FindPrimaryTerm: Searches for primary term.               */
/* Returns a pointer to the primary term if found,           */
/* otherwise NULL.                                           */
/*************************************************************/
static struct primary_term *FindPrimaryTerm(lvp,pt_name)
  struct fuzzyLv *lvp;
  VOID *pt_name;
{
   struct primary_term *ptptr;

   ptptr = lvp->primary_term_list;
   while (ptptr != NULL)
     {
      struct fuzzy_value *fvptr = (struct fuzzy_value *)
                                  ValueToFuzzyValue(ptptr->fuzzy_value_description);
      if (fvptr->name == (SYMBOL_HN *)pt_name)
        {  return(ptptr);  }

      ptptr = ptptr->next;
     }
   return(NULL);
}
  
  
  

/******************************************************************
    assertParseFuzzySet()

    Returns fuzzy set in expanded expr format for assert
    command.  Parses constant, variable, and function type 
    fuzzy set parameters.
    
    Parses assert of fuzzy sets of the form
    
        (fuzzyvar # (1 0) (5 1) (7 0) )  -- a singleton set OR
        (fuzzyvar # (z 4 8) )            -- a standard set
        
        NOTE: the # is optional !!! This routine expects that the
              current token should be a LPAREN [ '(' ]
        
        numeric values may be expressions if constantsOnly is FALSE
        
        sets onlyConstantsFound to TRUE if no expressions used
        
 ******************************************************************/

static struct expr *assertParseFuzzySet(readSource,tempToken,error,u,constantsOnly,onlyConstantsFound)
  char *readSource;
  struct token *tempToken;
  int  *error;
  struct universe *u;
  int constantsOnly;
  int *onlyConstantsFound;
{
      struct expr *parse_result;
      int function_type = -1;
      
      if (tempToken->type == LPAREN)
        {
          if (u == NULL) /* this should never happen */
            {
              *error = CLIPS_TRUE;
              SyntaxErrorMessage("Universe of discourse undefined for Fuzzy Variable");
              return(NULL);
            }  
      
          GetToken(readSource,tempToken);
          if (tempToken->type == SYMBOL) /* check for the s, z, or PI functions */
            {
              char *tokenStr = ValueToString(tempToken->value);
              
              if (strcmp(tokenStr,"S") == 0 ||
                  strcmp(tokenStr,"s") == 0)
                {
                 function_type = S_FUNCTION;
                }
              else if (strcmp(tokenStr,"Z") == 0 ||
                       strcmp(tokenStr,"z") == 0)
                {
                 function_type = Z_FUNCTION;
                }
              else if (strcmp(tokenStr,"PI") == 0 ||
                       strcmp(tokenStr,"pi") == 0 ||
                       strcmp(tokenStr,"Pi") == 0)
                {
                 function_type = PI_FUNCTION;
                }
             }
          if (function_type != -1)
            {
              parse_result = assertParseStandardSet(readSource,tempToken,error,u,constantsOnly,
                                                    onlyConstantsFound,function_type);
            }
          else
            {
              parse_result = assertParseSingletonSet(readSource,tempToken,error,u,
                                                     constantsOnly, onlyConstantsFound );
            }
         }  
      else
        {
          *error = CLIPS_TRUE;
          SyntaxErrorMessage("Expecting Fuzzy Set description");
        }
      if (*error == TRUE)
        {
         return(NULL);
        }
        
      return(parse_result);
}


/**********************************************************************
    assertParseStandardSet()
    
    Parses fuzzy sets of the form   (fuzzyvar (S 4 8))
    
    Function_type has aleady been determined as one of PI_FUNCTION, 
    S_FUNCTION, or Z_FUNCTION
 **********************************************************************/
static struct expr *assertParseStandardSet(readSource,tempToken,error,u,constantsOnly,
                                           onlyConstantsFound,function_type)
  char *readSource;
  struct token *tempToken;
  int  *error;
  struct universe *u;
  int constantsOnly;
  int *onlyConstantsFound;
  int function_type;
{
       struct expr *top, *from, *to, *arg1, *arg2;
    *onlyConstantsFound = FALSE;
        
    SavePPBuffer(" ");

    top = get_struct(expr);
    top->value  = NULL;
    top->type = function_type; /* standard fuzzy function */
    top->nextArg = NULL;
        
    from = GenConstant(FLOAT, (VOID *)AddDouble((double)u->from));
    top->argList = from;

    to = GenConstant(FLOAT, (VOID *)AddDouble((double)u->to));
    from->nextArg = to;
        
    /* get first parameter */ 
    GetToken(readSource, tempToken);
    SavePPBuffer(" ");
    
    arg1 = tokenToFloatExpression ( readSource, tempToken, error, constantsOnly);
    if ( *error == CLIPS_TRUE)
      {
        ReturnExpression(top);
        return(NULL);
      }
    else
      {
        to->nextArg = arg1;    
      }

    /* get 2nd parameter */
    GetToken(readSource, tempToken);   
    arg2 = tokenToFloatExpression ( readSource, tempToken, error, constantsOnly);
    if ( *error == CLIPS_TRUE)
      {
        ReturnExpression(top);
        return(NULL);
      }
    else
      {
        arg1->nextArg = arg2;
      }
        
    GetToken(readSource,tempToken);
    if (tempToken->type == RPAREN)
      {    
        if (arg1->type == FLOAT && arg2->type == FLOAT )
          { 
            *onlyConstantsFound = CLIPS_TRUE; 
          }
        GetToken(readSource,tempToken);
        return(top);
      }
    else
      {
        *error = CLIPS_TRUE;
        SyntaxErrorMessage("Expecting ')' ");
        ReturnExpression(top);
        return(NULL);
      }
}

/**********************************************************************
    assertParseSingletonSet()

    Parses fuzzy sets of form   (fuzzyvar # (0 0) (5 .5) (7 1) (12 0))
    
 **********************************************************************/
static struct expr *assertParseSingletonSet(readSource,tempToken,error,u,
                                            constantsOnly,onlyConstantsFound)
  char *readSource;
  struct token *tempToken;
  int  *error;
  struct universe *u;
  int constantsOnly;
  int *onlyConstantsFound;
{
   
    int  count;    /* number of (x,y) pairs input */
    struct expr *top, *first, *next, *from, *to, *countExpr;

    *onlyConstantsFound = TRUE;
    
    /********************************************************
     Initialize start of linked list and assign first element.
     Token should be x coordinate.
     ********************************************************/
    first  = tokenToFloatExpression (readSource, tempToken, error, constantsOnly);
    if (*error == CLIPS_TRUE)
    {
        SyntaxErrorMessage("Error in parsing Fuzzy Set x coordinate");
        return(NULL);
    }
    next = first;
    if (next->type != FLOAT)
        *onlyConstantsFound = CLIPS_FALSE;

    count = 0;

    while (CLIPS_TRUE)
      {
        /*************************************************
        Get the next token, which should be y coordinate
        *************************************************/
        SavePPBuffer(" ");
        GetToken(readSource,tempToken);
        next->next_arg = tokenToFloatExpression(readSource, tempToken, error, constantsOnly);
        if (*error == CLIPS_TRUE)
          {
            SyntaxErrorMessage("Error in parsing Fuzzy Set y coordinate");
            ReturnExpression(first);
            return(NULL);
          }
        next = next->nextArg;
        if (next->type != FLOAT)
            *onlyConstantsFound = FALSE;
 
        /*********************************************************************
        Get the next token, which should be closing bracket for the(x,y) pair
        ********************************************************************/
        GetToken(readSource,tempToken);

        if (tempToken->type == RPAREN)
          {
            count++;
            SavePPBuffer(" ");
          }     
        else
          {
             *error = CLIPS_TRUE;
             SyntaxErrorMessage("Expected ')' ");
            ReturnExpression(first);
            return(NULL);
          }
        /**************************************************************
        Get the next token, which should be either a closing bracket
        indicating the end of the set, or an opening bracket indicating
        the start of another (x,y) pair.
        *************************************************************/
        GetToken(readSource,tempToken);
        if ((tempToken->type == RPAREN) || (tempToken->type == STOP))
          {
            top = get_struct(expr);
            top->type = SINGLETON_EXPRESSION;
            top->value = NULL;
            top->nextArg = NULL;

            from = GenConstant(FLOAT, (VOID *)AddDouble((double)u->from));
            top->argList = from;

            to = GenConstant(FLOAT, (VOID *)AddDouble((double)u->to));
            from->nextArg = to;
            
            /* put the count of the x,y pairs as 3rd arg in list of args */
            countExpr = GenConstant(INTEGER,(VOID *)AddLong((long)count));
            to->nextArg = countExpr;
            countExpr->nextArg = first;
        
               return(top);
          }
          else if (tempToken->type != LPAREN)
          {
             *error = CLIPS_TRUE;
             SyntaxErrorMessage("Expected '(' ");
            ReturnExpression(first);
             return(NULL);
             }
            
        /************************************************
        Get next token, which should be x coordinate.
        ************************************************/
        SavePPBuffer(" ");
        GetToken(readSource, tempToken);
        next->next_arg = tokenToFloatExpression (readSource, tempToken, error, constantsOnly);
        if (*error)
          {
            SyntaxErrorMessage("Error in parsing Fuzzy Set x coordinate");
            ReturnExpression(first);
            return(NULL);
          }
        
        next = next->nextArg;
        if (next->type != FLOAT)
                *onlyConstantsFound = CLIPS_FALSE;

    } /* end of while (CLIPS_TRUE)  */
}


/**********************************************************************
    getConstantFuzzyValue()

    Given a singleton or standard fuzzy value in expanded expr format,
    this function evaluates the parameter expressions and returns 
    a fuzzy value structure.
 **********************************************************************/
globle struct fuzzy_value *getConstantFuzzyValue( top, error )
  struct expr *top;
  int *error;
{
    struct fuzzy_value *new_fv = NULL;
    
    if (top->type == PI_FUNCTION ||
        top->type == Z_FUNCTION  ||
        top->type == S_FUNCTION)
      {
        new_fv = convertStandardSet ( top, error );
        if (*error)
            return (NULL);
      }
    else if (top->type == SINGLETON_EXPRESSION)
      {
        new_fv = convertSingletonSet ( top, error );
        if (*error == CLIPS_TRUE)
            return (NULL);
        
      }
    return(new_fv);
}

/**********************************************************************
    convertStandardSet()

    Evaluates parameters of a standard fuzzy value and returns a 
    fuzzy value structure.
    
    Assumes that top->type is one of PI_FUNCTION, S_FUNCTION, or
    Z_FUNCTION and that top->argList points to 4 arguments
    
       arg 1    from value (universe lower limit)
       arg 2    to   value (universe upper limit)
       arg 3    alpha value
       arg 4    value depends on function type
 **********************************************************************/
static struct fuzzy_value *convertStandardSet( top, error )
  struct expr *top;
  int *error;
{
    struct expr *next;
    struct fuzzy_value *fv;
    double xtolerance;
    
    double from, to, alfa, beta, gamma;
    int function_type = top->type;

    /* get the 1st parameter  - from */
    next = top->argList;
    expressionToFloat(next, &from, error);
    if (*error == CLIPS_TRUE)   /* should never happen */
        return(NULL);

    /* get the 2nd parameter  - to */
    next = next->nextArg;
    expressionToFloat(next, &to, error);
    if (*error == CLIPS_TRUE)   /* should never happen */
        return(NULL);

    /* get the 3rd parameter  - alpha */
    next = next->nextArg;    
    expressionToFloat(next, &alfa, error);
    if (*error == CLIPS_TRUE)
        return(NULL);
    
    /* We want to allow specifications for Standard functions and singleton sets
       to NOT be thrown out with an error message due to a floating point
       representation problem so we will allow the points to be considered
       within the universe of discourse (range) if very close.

       FuzzyTolerance is defined in constant.h
    */
    xtolerance = ((to - from) >= 1.0) ? FuzzyTolerance : (to-from) * FuzzyTolerance;

    if (function_type == PI_FUNCTION)
      {
        if (alfa <= 0.0)
          {
            *error = CLIPS_TRUE;
            PrintCLIPS(WERROR,"PI function 1st parameter must be positive\n");
            return(NULL);
          }
        else 
          { 
            beta = alfa; 
          }
      }
    else if (alfa < from)
      {
        if (from-alfa > xtolerance)
          {
            *error = CLIPS_TRUE;
            PrintCLIPS(WERROR,"S or Z function 1st parameter out of range (too small)\n");
            return(NULL);
          }
        alfa = from;
      }
     else if (alfa > to)
      {
        if (alfa-to < xtolerance)
          {
            *error = CLIPS_TRUE;
            PrintCLIPS(WERROR,"S or Z function 1st parameter out of range (too large)\n");
            return(NULL);
          }
        alfa = to;
      }
       
    /* get the 4th parameter */
    next = next->next_arg;
    
    expressionToFloat(next, &gamma, error);
    if (*error == CLIPS_TRUE)
        return(NULL);

    if  (function_type == PI_FUNCTION)
      { 
        if ((gamma > to) || (gamma < from))
           {
             *error = CLIPS_TRUE;
             PrintCLIPS(WERROR,"PI function produces x values out of range\n");
             return(NULL);
           }    
        else if ((gamma - beta) < from)
           {
             if (from - (gamma - beta) > xtolerance)
               {
                 *error = CLIPS_TRUE;
                 PrintCLIPS(WERROR,"PI function produces x values too small\n");
                 return(NULL);
               }
              beta = gamma - from;
           }    
        else if ((gamma + beta) > to)
           {
             if (gamma + beta - to > xtolerance)
               {
                 *error = CLIPS_TRUE;
                 PrintCLIPS(WERROR,"PI function produces x values too large\n");
                 return(NULL);
               }
              beta = to - gamma;
           }    
      }
    else if (gamma <= alfa)
      {
        *error = CLIPS_TRUE;
        PrintCLIPS(WERROR,"S or Z function 2nd parameter must be > 1st parameter\n");
        return(NULL);
      }
    else if (gamma > to)
      {
        if (gamma-to > xtolerance)
          {
            *error = CLIPS_TRUE;
            PrintCLIPS(WERROR,"S or Z function 2nd parameter out of range (too large)\n");
            return(NULL);
          }
        gamma = to;
      }
    
    /* Construct the fuzzy value -- Get_S_Z_or_PI_FuzzyValue in fuzzypsr.c */

    fv = Get_S_Z_or_PI_FuzzyValue(alfa, beta, gamma, function_type);
           
    return (fv);
}


/**********************************************************************
    convertSingletonSet()

    Evaluates parameters of a singleton fuzzy set and returns a 
    fuzzy set structure.
    Assumes that top points to an argList that has:
    
        1st arg is from value (universe lower limit)
        2nd arg is to   value (universe upper limit)
        3rd arg is n    (number of fuzzy set pairs) (will be a constant integer)
        4th, 5th args are 1st x,y pair
        6th, 7th args are 2nd x,y pair
        etc.
 **********************************************************************/
static struct fuzzy_value *convertSingletonSet( top, error )
  struct expr *top;
  int *error;
{
    struct expr *next;
    struct fuzzy_value *fv;
    int num, i, numpairs_retrieved;
    double newx, newy, previous;
    double from, to;
    double xtolerance;
    
    /* get the 1st parameter  - from */
    next = top->argList;
    expressionToFloat(next, &from, error);
    if (*error == CLIPS_TRUE)   /* should never happen */
        return(NULL);

    /* get the 2nd parameter  - to */
    next = next->nextArg;
    expressionToFloat(next, &to, error);
    if (*error == CLIPS_TRUE)   /* should never happen */
        return(NULL);

    /**************************************************************
     Initialize the fuzzy set - 3rd parameter gives required size
     **************************************************************/
    next = next->nextArg;
    expressionToInteger(next, &num, error);
    if (*error == CLIPS_TRUE)   /* should never happen */
        return(NULL);

    fv = get_struct(fuzzy_value);
    fv->name = (SYMBOL_HN *)AddSymbol("???");

    fv->n = num;
    fv->maxn = num;
    fv->x = FgetArray(num);
    fv->y = FgetArray(num);

    previous = from - 1.0;
    next = next->nextArg;
    
    /* We want to allow specifications for Standard functions and singleton sets
       to NOT be thrown out with an error message due to a floating point
       representation problem so we will allow the points to be considered
       within the universe of discourse (range) if very close.

       FuzzyTolerance is defined in constant.h
    */
    xtolerance = ((to - from) >= 1.0) ? FuzzyTolerance : (to-from) * FuzzyTolerance;

    i = 0; /* index into the array for storing values */
    numpairs_retrieved = 0; /* count of the pairs obtained from the expression */
    
    while ( numpairs_retrieved < num && next != NULL )
     {
        /************************************************
         Next Expression should be x coordinate.
         ************************************************/
        expressionToFloat(next, &newx, error);
        if (*error == CLIPS_FALSE)
          {
            if (newx > to)
              {
                if (newx-to > xtolerance)
                  {
                    *error = CLIPS_TRUE;
                    PrintCLIPS(WERROR,"X coordinate of Fuzzy Set out of range (too large)\n");
                  }
                newx = to;
              }
            else if (newx < from)
              {
                if (from-newx > xtolerance)
                  {
                    *error = CLIPS_TRUE;
                    PrintCLIPS(WERROR,"X coordinate of Fuzzy Set out of range (too small)\n");
                  }
                newx = from;
              }
            if (newx <= previous)
              {
                *error = CLIPS_TRUE;
                PrintCLIPS(WERROR,"(x,y) pairs should be in increasing x order in Fuzzy Set\n");
                PrintCLIPS(WERROR,"      and successive x values must not be the same\n");
              }
          }
        if (*error == CLIPS_TRUE)
          {
            rtnFuzzyValue(fv);
            return(NULL);
          }
        fv->x[i] = newx;
          previous = newx;
        
        /***********************************************
         Next Expression should be y coordinate.
         ***********************************************/
        next = next->nextArg;
        if ( next == NULL )
        {    
            *error = CLIPS_TRUE;
            PrintCLIPS(WERROR,"Y coordinate of fuzzy set missing (possible internal error)\n");
            rtnFuzzyValue(fv);
            return(NULL);
        }
        expressionToFloat(next, &newy, error);
        if (*error == CLIPS_FALSE)
          {
              if (newy < 0.0)
              {
                if (newy < -FuzzyTolerance)
                  {
                    *error = CLIPS_TRUE;
                    PrintCLIPS(WERROR,"Fuzzy membership value (y coordinate) must be >= 0.0\n");
                  }
                newy = 0.0;
              }
              if (newy > 1.0)
              {
                if (newy-1.0 > FuzzyTolerance)
                  {
                    *error = CLIPS_TRUE;
                    PrintCLIPS(WERROR,"Fuzzy membership value (y coordinate) must be <= 1.0\n");
                  }
                newy = 1.0;
              }
          }
        if (*error == CLIPS_TRUE)
          {
            rtnFuzzyValue(fv);
            return(NULL);
          }
          
        /* when adding a y value make sure it is NOT the same as 1st value if
           it is the second one in the array OR that it is not same as previous 2
           if it is past the 3rd pair (ie. i > 2)
        */
        if ( (i == 1 && newy == fv->y[0]) ||
             (i > 2 && newy == fv->y[i-1] && newy == fv->y[i-2]) )
          {
            i--;
            fv->x[i] = newx;
          }
           
        fv->y[i] = newy;
        next = next->nextArg;
        i++;
        numpairs_retrieved++;
      }
    if ( numpairs_retrieved != num || next != NULL)
      {
        /* impossible error ??? */
        *error = CLIPS_TRUE;
        PrintCLIPS(WERROR,"Fuzzy set - incorrect number of (x,y) pairs - internal error\n");
        rtnFuzzyValue(fv);
          return(NULL);
      }

    /* finally if more than 2 pairs check that the last 2 are not the same  -- if so
       just discard the last one */
    if (i > 2 && fv->y[i-1] == fv->y[i-2])
       i--;
      
    fv->n = i;

    if ((num - i) > 5)
       CompactFuzzyValue( fv );
    
    return(fv);
}



/**********************************************************************
    expressionToFloat()

    Given an expr structure which has been formed by tokenToFloatExpression(),
    this function evaluates the expression and returns the float value.
 **********************************************************************/
static VOID expressionToFloat(exprPtr, result, error)
  struct expr *exprPtr;
  double *result;
  int *error;
{

  DATA_OBJECT exprValue;
  int type;
  
  EvaluateExpression( exprPtr, &exprValue );
  type = exprValue.type;
  
  if (type == INTEGER)
     *result = (double) DOToLong(exprValue);
     
  else if (type == FLOAT)
     *result = DOToDouble(exprValue);
     
  else 
    {
      *error = CLIPS_TRUE;
      PrintCLIPS(WERROR,"Fuzzy set value (expecting FLOAT value)\n");
    }
}





/**********************************************************************
    expressionToInteger()

    Given an expr structure which has been formed by tokenToFloatExpression(),
    this function evaluates the expression and returns the integer value.
 **********************************************************************/
static VOID expressionToInteger(exprPtr, result, error)
  struct expr *exprPtr;
  int *result;
  int *error;
{

  DATA_OBJECT exprValue;
  int type;
  
  EvaluateExpression( exprPtr, &exprValue );
  type = exprValue.type;
  
  if (type == INTEGER)
     *result = (int) DOToLong(exprValue);
     
  else 
    {
      *error = CLIPS_TRUE;
      PrintCLIPS(WERROR,"Fuzzy set internal evaluation (expecting int value)\n");
    }
}




/************************************************************************
    FUNCTIONS FOR ALLOCATING AND DE-ALLOCATING MEMORY 
************************************************************************/

/************************************************************************
    FgetArray(length)
 
    Allocates memory for a floating point array of size "length" and
    returns a pointer to the allocated array.
 ************************************************************************/
globle double *FgetArray ( length )
  int length;
{
    double *p;

    /* Use CLIPS memory management function gm1() */
    p = (double *) gm1 ( length * (sizeof(double)) );
    return (p);
}

/************************************************************************
    FrtnArray(p, length )
    
    Deallocates memory of floating point array p of size "length".
 ************************************************************************/
globle VOID FrtnArray ( p, length )
  double *p;
  int length;
{
    rm ( p, length * (sizeof(double)) );    /* a CLIPS memory function */
}

/************************************************************************
    IgetArray(length)
 
    Allocates memory for an integer array of size "length" and
    returns a pointer to the allocated array.
 ************************************************************************/
globle int *IgetArray ( length )
  int length;
{
    int *p;

    /* Use CLIPS memory management function gm1() */
    p = (int *) gm1 ( length * (sizeof(int)) );
    return (p);
}

/************************************************************************
    IrtnArray(p, length )
    
    Deallocates memory of integer array p of size "length".
 ************************************************************************/
globle VOID IrtnArray ( p, length )
  int *p;
  int length;
{
    rm ( p, length * (sizeof(int)) );    /* a CLIPS memory function */
}

/************************************************************************
    CgetArray(length)
 ************************************************************************/
globle char *CgetArray ( length )
  int length;
{
    char *p = NULL;
    p = (char *) gm1 ( length * (sizeof (char)) );
    return (p);
}

/************************************************************************
    CrtnArray ( p, length )
 ************************************************************************/
globle VOID CrtnArray ( p, length )
char *p;
int length;
{
    rm ( p, length * (sizeof (char) ) );
}



#endif  /* FUZZY_DEFTEMPLATES */


/***********************************************

 Following routine used by FUZZY and CERTAINTY 
 FACTOR code

***********************************************/

#if FUZZY_DEFTEMPLATES || CERTAINTY_FACTORS



/****************************************************************
    tokenToFloatExpression()

    Given the next token, this function
    parses constants, variables and functions and returns
    appropriate expr structures
 ****************************************************************/
globle struct expr *tokenToFloatExpression ( readSource, tempToken, error, constantsOnly )
  char *readSource;
  struct token *tempToken;
  int  *error;
  int constantsOnly;
{
    struct expr *result=NULL;
    int exprType;

    if (tempToken->type == FLOAT || tempToken->type == INTEGER)    
      {
       /******************************************************
        Constant - FLOAT or INTEGER allowed
        ******************************************************/
           if (tempToken->type == INTEGER)
            result = GenConstant(FLOAT,
                                 (VOID *) AddDouble((double) ValueToLong(tempToken->value)));
         else
            result = GenConstant(FLOAT,tempToken->value);
            
         return(result);

         }
        
   /*=============================================================*/
   /* If an equal sign or left parenthesis was parsed, then parse */
   /* a function which is to be evaluated to determine the        */
   /* value. The equal sign corresponds to the return value       */
   /* constraint which can be used in LHS fact patterns. The      */
   /* equal sign is no longer necessary on either the LHS or RHS  */
   /* of a rule to indicate that a function is being evaluated to */
   /* determine its value either for assignment or pattern        */
   /* matching.                                                   */
   /*=============================================================*/
   
   if ((tempToken->type == SYMBOL) ? 
       (strcmp(ValueToString(tempToken->value),"=") == 0) : 
       (tempToken->type == LPAREN))
     {
      if (constantsOnly)
        {
         SyntaxErrorMessage("Constants Only Allowed");
         *error = CLIPS_TRUE;
         return(NULL);
        }

#if ! RUN_TIME
      if (tempToken->type == LPAREN) 
         result = Function1Parse(readSource);
      else 
         result = Function0Parse(readSource);
#endif
      if (result == NULL)
        {
         *error = CLIPS_TRUE;
        }
      else
        {
          exprType = ExpressionFunctionType(result);
          if (GetStaticConstraintChecking() == CLIPS_TRUE)
            if ((result->type == FCALL) && 
                 exprType != 'd' &&
                  exprType != 'f' &&
                  exprType != 'n' &&
                 exprType != 'i' 
               )
              {
               SyntaxErrorMessage("Expected numeric result from function");
               *error = CLIPS_TRUE;
               ReturnExpression(result);
               return( NULL );
              }
         }
       
      return(result);
     }

   /*========================================*/
   /* Variables are also allowed as RHS      */
   /* values under some circumstances.       */
   /*========================================*/
   
   if ((tempToken->type == SF_VARIABLE) 
#if DEFGLOBAL_CONSTRUCT
       || (tempToken->type == GBL_VARIABLE) 
#endif
      )
     {
      if (constantsOnly)
        {
         *error = CLIPS_TRUE;
         return(NULL);
        }

      return(GenConstant(tempToken->type,tempToken->value));
     }

   /*==========================================================*/
   /* If none of the other case have been satisfied, then the  */
   /* token parsed is not appropriate for a numeric expression */
   /*==========================================================*/
   
   SyntaxErrorMessage("Numeric expression expected");
   *error = CLIPS_TRUE;
   return(NULL);

}



#endif  /*  FUZZY_DEFTEMPLATES || CERTAINTY_FACTORS  */
