   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 6.00  05/12/93            */
   /*                                                     */
   /*             FUZZY REASONING PARSER MODULE           */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*      Brian L. Donnell                                     */
/*      Bob Orchard (NRCC - Nat'l Research Council of Canada)*/
/*                  (Fuzzy reasoning extensions)             */
/*                  (certainty factors for facts and rules)  */
/*                  (extensions to run command)              */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/


/******************************************************************
    Fuzzy Logic Module

    This file contains the following categories of functions:
    
    parsing of fuzzy value definition in deftemplate statement

 ******************************************************************/
 
#define _FUZZYPSR_SOURCE_

#include <string.h>


#include "setup.h"


#if FUZZY_DEFTEMPLATES

#include <stdio.h>
#define _CLIPS_STDIO_

#include "fuzzypsr.h"
#include "fuzzyrhs.h"
#include "fuzzyval.h"
#include "fuzzylv.h"

#include "extnfunc.h"
#include "prntutil.h"
#include "scanner.h"
#include "constant.h"
#include "pprint.h"
#include "symbol.h"
#include "clipsmem.h"
#include "dffnxfun.h"
#include "exprnops.h"
#include "modulutl.h"
#include "moduldef.h"
#include "evaluatn.h"


/******************************************************************
    Global Internal Function Declarations
    
    Defined in fuzzypsr.h
 ******************************************************************/



/******************************************************************
    Local Internal Function Declarations
 ******************************************************************/
 
#if ANSI_COMPILER

#if (! RUN_TIME) && (! BLOAD_ONLY)
  static struct universe      *parseUniverse(char *read_source, struct token *inputToken, 
                                             int *DeftemplateError);
  static struct primary_term  *parsePrimaryTermList(char *in_file, struct token *inputToken, 
                                             int *DeftemplateError, struct universe *u);
  static struct primary_term  *parsePrimaryTerm(char *readSource, 
                                                struct token *inputToken, int *DeftemplateError,
                                                struct universe *u);
  static struct fuzzy_value      *parseTemplateFuzzyValue(char *readSource, struct token *inputToken, 
                                             int  *DeftemplateError, struct universe *u);
  static struct fuzzy_value   *parseSingletonFuzzyValue(char *readSource, 
                                                      struct token *inputToken,
                                                      int  *DeftemplateError,
                                                      struct universe *u);
  static struct fuzzy_value      *parseStandardFuzzyValue(char *readSource, 
                                                     struct token *inputToken,
                                                     int  *DeftemplateError,
                                                     struct universe *u);
  static struct modifier      *parseModifierList(char *readSource, struct token *inputToken, int *DeftemplateError);
  static struct modifier      *parseModifier(char *readSource, struct token *inputToken,
                                             int *DeftemplateError);
#endif
  static VOID                  rtnModifierList(struct modifier *m);
  static VOID                  rtnPrimaryTermList(struct primary_term *pt);
  static VOID                  rtnUniverse(struct universe *u);

#else

#if (! RUN_TIME) && (! BLOAD_ONLY)
  static struct universe      *parseUniverse();
  static struct primary_term  *parsePrimaryTermList();
  static struct primary_term  *parsePrimaryTerm();
  static struct fuzzy_value   *parseTemplateFuzzyValue();
  static struct fuzzy_value   *parseSingletonFuzzyValue();
  static struct fuzzy_value   *parseStandardFuzzyValue();
  static struct modifier      *parseModifierList();
  static struct modifier      *parseModifier();
#endif
  static VOID                  rtnModifierList();
  static VOID                  rtnPrimaryTermList();
  static VOID                  rtnUniverse();

#endif




/******************************************************************
    Local Internal Variable Declarations
 ******************************************************************/ 

   
   
/******************************************************************
    Global External Variable Declarations
 ******************************************************************/
  


/******************************************************************
    Global Internal Variable Declarations
 ******************************************************************/


   globle double S_array[ArraySIZE];
   globle double Z_array[ArraySIZE];
   globle double PI_array[ArraySIZE];


/**********************************************************************
    FUNCTIONS to Initialize arrays for the S, Z and PI fuzzy functions
 **********************************************************************/


 
/*********************** !!!!!!!!!!!!!!!!!!!!!! **************

 NOTE: 
 
 PI function is s(x, gamma-beta, gamma-beta/2.0, gamma) if x < gamma
            and 1.0 - s(x, gamma, gamma+beta/2,0, gamma+beta) if x > gamma
            and 1.0 if x = gamma
           
 Z function is 1.0 - s(x, alpha, beta, gamma)

**************************************************************/
        


/***********************************/
/* S - Standard_function:          */
/***********************************/
globle double sFunction(x,alfa,beta,gamma)
  double x;
  double alfa;
  double beta;
  double gamma;
  {
   register double gMinusa = gamma - alfa;
   register double gMinusaSqr = gMinusa * gMinusa;
   
   if (x <= alfa)
     { return(0.0); }
   else if (x <= beta)
     { return(2.0*(x-alfa)*(x-alfa)/(gMinusaSqr)); }
   else if (x < gamma)
     { return(1.0 - 2.0*(x-gamma)*(x-gamma)/(gMinusaSqr)); }
   else
     { return(1.0); }     
  }



/**********************************************************/
/* Init_S_Z_PI_yvalues:                                   */
/*                                                        */
/* We can do this in advance since the y values for any   */
/* S, Z, or PI function is the same due to the nature of  */
/* the functions. Just use a 0 to 1 range n calculations  */
/**********************************************************/

globle VOID Init_S_Z_PI_yvalues()
   {
     int i;
     int ArraySzBy2 = ArraySIZE/2;
     int ArraySzMinus1 = ArraySIZE - 1;
     
     /* PI y values stored in PI_array */
     
     PI_array[0] = 0.0;
     PI_array[ArraySzBy2] = 1.0;
     PI_array[ArraySzMinus1] = 0.0;
     
     for ( i=1; i < ArraySzBy2; i++)
        {
          PI_array[i] = sFunction((double)i/(double)ArraySzBy2, 0.0, 0.5, 1.0);
        }
     for ( i = ArraySzBy2+1; i < ArraySzMinus1; i++)
        {
          PI_array[i] = PI_array[ArraySzMinus1-i]; /* symmetry let's us do this */
        }
        
     /* S y values stored in S_array */
     
     S_array[0] = 0.0;
     S_array[ArraySzMinus1] = 1.0;
     
     for ( i=1; i < ArraySzMinus1; i++)
        {
          S_array[i] = sFunction((double)i/(double)(ArraySzMinus1), 0.0, 0.5, 1.0);
        }     
     
     /* Z y values stored in Z_array */

     for ( i=0; i < ArraySIZE; i++)
        {
          Z_array[i] = S_array[ArraySzMinus1-i]; /* symmetry of the functions */
        }     
   }



/**********************************************************/
/* Get_S_Z_or_PI_FuzzyValue:                              */
/*                                                        */
/* Depending on the type of function requested get the    */
/* x and y values for the S, PI or Z function             */
/**********************************************************/

globle struct fuzzy_value *Get_S_Z_or_PI_FuzzyValue(alfa, beta, gamma, function_type)

  double alfa;
  double beta;
  double gamma;
  int   function_type;
{
    struct fuzzy_value *fv;
    double h, deltah;
    double gPlusb;
    double gMinusb;
    int i;
    
    /* Construct the fuzzy value */

    fv = get_struct(fuzzy_value);
       
    fv->name = (SYMBOL_HN *)AddSymbol("???");
    fv->maxn = ArraySIZE;
    fv->n = ArraySIZE;
    fv->x = FgetArray( fv->maxn );
    fv->y = FgetArray( fv->maxn );
              
    if  (function_type == PI_FUNCTION)
      {
    gPlusb = gamma + beta;
    gMinusb = gamma - beta;
              
        deltah = beta / (double)(ArraySIZE/2);
    fv->x[0] = gMinusb;
    fv->y[0] = 0.0;
    fv->x[ArraySIZE/2] = gamma;
        fv->y[ArraySIZE/2] = 1.0;
        fv->x[ArraySIZE-1] = gPlusb;
        fv->y[ArraySIZE-1] = 0.0;
              
        for ( i=1, h=deltah; i < ArraySIZE/2; i++)
          {
            fv->x[i] = gMinusb + h;
            fv->y[i] = PI_array[i];
            h = h + deltah;
          }
        for ( i = ArraySIZE/2+1, h = deltah; i < ArraySIZE-1; i++)
           {
             fv->x[i] = gamma + h;
             fv->y[i] = PI_array[i];
             h = h + deltah;
           }
      }
    else
      { /* s or z function */
        double h, deltah;
        
        fv->x[0] = alfa;
        fv->x[8] = gamma;
        if (function_type == S_FUNCTION)
          { fv->y[0] = 0.0; fv->y[ArraySIZE-1] = 1.0; }
        else
          { fv->y[0] = 1.0; fv->y[ArraySIZE-1] = 0.0; }
        
        deltah = (gamma - alfa) / (double)(ArraySIZE-1);
        for (i=1, h = deltah; i < ArraySIZE-1; i++)
           {
             fv->x[i] = alfa + h;
         fv->y[i] = (function_type == S_FUNCTION) ? S_array[i] :
                                        Z_array[i];
             h = h + deltah;
           }
       }
    
    return (fv);

}


#if (! RUN_TIME) && (! BLOAD_ONLY)

/********************************************************************
    FUNCTIONS THAT PARSE A FUZZY DEFTEMPLATE 
 ********************************************************************/
 
/*****************************************************************/
/* ParseFuzzyTemplate:  The purpose of this function is to parse */
/*   the deftemplate statement for fuzzy variables.              */ 
/* The list of primary terms and a list of modifiers and the     */
/* universe of discourse are saved.                              */
/*****************************************************************/
globle struct fuzzyLv *ParseFuzzyTemplate(readSource, inputToken, DeftemplateError)
  char *readSource;
  struct token *inputToken;
  int *DeftemplateError;
  {
   struct primary_term *temp_pt = NULL;
   struct modifier     *temp_mod = NULL;
   struct fuzzyLv      *new_lv = NULL;
   struct universe     *u = NULL;


   /*========================================*/
   /* Parse the universe of discourse        */
   /*========================================*/

   if (inputToken->type == FLOAT || inputToken->type == INTEGER)  
     {
      u = parseUniverse(readSource, inputToken, DeftemplateError);
      if (*DeftemplateError == CLIPS_TRUE) 
          { 
            return(NULL); 
          }
     }

   /*========================================*/
   /* Check that next token is a '('.        */
   /*========================================*/
   
   if (inputToken->type != LPAREN) 
     {
      SyntaxErrorMessage("Deftemplate (Expecting Fuzzy Term List)");
      *DeftemplateError = CLIPS_TRUE;
      /* need to return the universe of discourse struct */
      rtnUniverse(u);
      return(NULL);
     }
   else
     {
      temp_pt = parsePrimaryTermList(readSource, inputToken, DeftemplateError, u);
      if (*DeftemplateError == CLIPS_TRUE) 
           { 
             /* need to return the universe of discourse struct */
               rtnUniverse( u);
             return(NULL); 
           }
     }
   GetToken(readSource,inputToken);
   if (inputToken->type == LPAREN)
     {
      PPBackup();
      PPCRAndIndent();
      SavePPBuffer(" (");
      temp_mod = parseModifierList(readSource, inputToken, DeftemplateError);
      if (*DeftemplateError == CLIPS_TRUE) 
         { 
           /* must return universe of discourse and primary term structs */
           rtnUniverse(u);
           rtnPrimaryTermList(temp_pt);
           return(NULL);
         }
      SavePPBuffer("\n");
      GetToken(readSource,inputToken);
     }
   else
     {
      PPBackup();       
      SavePPBuffer("\n)"); /* expecting the current token is ')' */
     }

   if (inputToken->type != RPAREN)
     {
      SyntaxErrorMessage("Deftemplate (Closing ')' for deftemplate expected)");
      *DeftemplateError = CLIPS_TRUE;
      /* must return universe of discourse and primary term structs */
      rtnUniverse( u);
      rtnPrimaryTermList(temp_pt);
      return(NULL);
     }

   SavePPBuffer("\n");

   new_lv = get_struct(fuzzyLv);
   new_lv->u_ptr = u;
   new_lv->primary_term_list = temp_pt;
   new_lv->modifier_list = temp_mod;

   return(new_lv);       
  }


/*****************************************************/
/* parseUniverse:                                    */
/*                                                   */
/*  Must return a NULL ptr if error parsing universe */
/*  Must free any allocated structs if error         */
/*                                                   */
/*****************************************************/
static struct universe *parseUniverse(readSource, inputToken, DeftemplateError)
  char *readSource;
  struct token *inputToken;
  int *DeftemplateError;
  {
   double f, t;  /* from and to values */
   struct universe *u; /* universe of discourse */
   
   SavePPBuffer(" ");
   
   /* inputToken has an integer or a float at this point */
   f = (inputToken->type == FLOAT) ? ValueToDouble(inputToken->value) : 
                                        (double)ValueToInteger(inputToken->value);
   GetToken(readSource,inputToken);
   if (inputToken->type != FLOAT && inputToken->type != INTEGER) 
     {
      SyntaxErrorMessage("Deftemplate: Number Expected ('to' part of Universe)");
      *DeftemplateError = CLIPS_TRUE;
      return(NULL);
     }
   
   t = (inputToken->type == FLOAT) ? ValueToDouble(inputToken->value) : 
                                        (double)ValueToInteger(inputToken->value);
   if (f > t) 
     {
      SyntaxErrorMessage("Deftemplate: Invalid interval for Universe of Discourse");
      *DeftemplateError = TRUE;
      return(NULL);
     }
     
   u = get_struct(universe);
   u->from = f;
   u->to = t;
   u->units = NULL;
   
   SavePPBuffer(" ");
   GetToken(readSource,inputToken);
   if ((inputToken->type == STRING) || (inputToken->type ==  SYMBOL))
     {
       u->units = (SYMBOL_HN *) inputToken->value;
       PPCRAndIndent();
       SavePPBuffer(" ");
       GetToken(readSource,inputToken);
     }
   else
     {
       PPBackup();
       PPCRAndIndent();
       SavePPBuffer(" ");
       SavePPBuffer(inputToken->printForm);
     }
   return(u);
  }


/*****************************************************/
/* parsePrimaryTermList:                             */
/*                                                   */
/*  Must return a NULL ptr if error parsing          */
/*        PRIMARY TERMS                              */
/*  Must free any allocated structs if error parsing */
/*  Already have the opening '(' of the list         */
/*  Should eat up the closing ')' of list            */
/*****************************************************/
static struct primary_term *parsePrimaryTermList(readSource, inputToken, DeftemplateError, u)
  char *readSource;
  struct token *inputToken;  
  int  *DeftemplateError;
  struct universe *u;
  {
   struct primary_term *last_one, *next_one, *assert_list;

   last_one = assert_list = NULL;
   
   /* expect to see a set of terms, each enclosed in ()  */
    while ((next_one = parsePrimaryTerm(readSource, inputToken,
                                   DeftemplateError, u)) != NULL)
     {        
      if (last_one == NULL)
        { assert_list = next_one; }
      else
        { last_one->next = next_one; }
      last_one = next_one;     
     }
          
   if (*DeftemplateError || (assert_list == NULL))
     { /* need to free any allocated primary term structs */
       if (assert_list == NULL)
          {
            *DeftemplateError = CLIPS_TRUE;
            SyntaxErrorMessage("Deftemplate (At least one primary term must be defined)");
          }
       else
          { /* need to free any allocated primary term structs */
            rtnPrimaryTermList(assert_list);
          }
       return(NULL);
     }

   return(assert_list);
  }
  
/***************************************************************/
/* parsePrimaryTerm:                                           */
/* Returns NULL if ) is first token.                           */
/* Returns primary term.                                       */
/* DeftemplateError flag is set to true if an error occurs.    */
/***************************************************************/
static struct primary_term *parsePrimaryTerm(readSource,inputToken,DeftemplateError,u)
  char *readSource;
  struct token *inputToken;
  int *DeftemplateError;
  struct universe *u; 
{
   struct primary_term *ptr;
   SYMBOL_HN *pt_name;
   struct fuzzy_value *fuzzy_value_dsc;
   
   /*==========================================================*/
   /* Get the opening parenthesis of the primary term pattern. */
   /*==========================================================*/
   
   GetToken(readSource,inputToken);
      
   if (inputToken->type == RPAREN) 
     {
       PPBackup();
       PPBackup();
       SavePPBuffer(" )");
       return(NULL);
     }
      
   if (inputToken->type != LPAREN)
     {
       SyntaxErrorMessage("Deftemplate (Expected primary term)");
       *DeftemplateError = TRUE;
       return(NULL);
     }
   
   /*================================================================*/
   /* Get the primary term name.                                     */
   /*================================================================*/
     
   GetToken(readSource,inputToken);
   
   if (inputToken->type == SYMBOL)
     {
      pt_name = (SYMBOL_HN *)inputToken->value;
     }
   else 
     {
      SyntaxErrorMessage("Deftemplate (Expected primary term name)");
      *DeftemplateError = CLIPS_TRUE;
      return(NULL);
     }
     
   /*====================================*/
   /* Get description of the fuzzy value */
   /*====================================*/
   
   SavePPBuffer(" ");
   fuzzy_value_dsc = parseTemplateFuzzyValue(readSource,inputToken,DeftemplateError,u);

   if (fuzzy_value_dsc != NULL && inputToken->type == RPAREN)
     {
      ptr = get_struct(primary_term);
      fuzzy_value_dsc->name = pt_name;
      ptr->fuzzy_value_description = (FUZZY_VALUE_HN *)AddFuzzyValue(fuzzy_value_dsc);
      /* AddFuzzyValue makes a copy of the Fuzzy Value so we need to return this one */
      rtnFuzzyValue(fuzzy_value_dsc);
      ptr->next = NULL;
      PPCRAndIndent();
      SavePPBuffer("  ");
      return(ptr);
     }
   else
     {
      *DeftemplateError = CLIPS_TRUE;
      rtnFuzzyValue(fuzzy_value_dsc);
      SyntaxErrorMessage("Deftemplate (expected ')' )");
      return(NULL);
     }
  }

/***********************************/    
/* parseTemplateFuzzyValue:        */
/***********************************/
static struct fuzzy_value *parseTemplateFuzzyValue(readSource,inputToken,DeftemplateError,u)
  char *readSource;
  struct token *inputToken;
  int  *DeftemplateError;
  struct universe *u;
{
   struct fuzzy_value *fv_ptr = NULL;

      GetToken(readSource,inputToken);
      
      if (inputToken->type == LPAREN)
         {
           if (u == NULL)
              {
                *DeftemplateError = CLIPS_TRUE;
                SyntaxErrorMessage("Deftemplate (Missing universe of discourse description)");
                return(NULL);
              }        
           GetToken(readSource,inputToken);
           if (inputToken->type == FLOAT || inputToken->type == INTEGER)
              { fv_ptr = parseSingletonFuzzyValue(readSource,inputToken,DeftemplateError,u); }
           else
              { fv_ptr = parseStandardFuzzyValue(readSource,inputToken,DeftemplateError,u); }
         }  
      else
        {
          *DeftemplateError = CLIPS_TRUE;
          SyntaxErrorMessage("Deftemplate (Fuzzy set description expected)");
        }
        
      if (*DeftemplateError == CLIPS_TRUE)
        {
          rtnFuzzyValue(fv_ptr);
          return(NULL);
        }
      else     
        { return(fv_ptr); }
  }


/***********************************/    
/* parseSingletonFuzzyValue:      */
/***********************************/
static struct fuzzy_value *parseSingletonFuzzyValue(readSource,inputToken,DeftemplateError,u)
  char *readSource;
  struct token *inputToken;
  int  *DeftemplateError;
  struct universe *u;
  {
   
    struct fuzzy_value *fv;
    double  previous;
    int i, count, maxlength = 10, increment = 10;
    double newx, newy, *tempx, *tempy;
    double xtolerance;

    fv = get_struct(fuzzy_value);
    fv->name = NULL;
    fv->x = FgetArray ( maxlength );
    fv->y = FgetArray ( maxlength );
    fv->maxn = maxlength;
      
    /* We want to allow specifications for Standard functions and singleton sets
       to NOT be thrown out with an error message due to a floating point
       representation problem so we will allow the points to be considered
       within the universe of discourse (range) if very close.

       FuzzyTolerance is defined in constant.h
    */
    xtolerance = ((u->to - u->from) >= 1.0) ? FuzzyTolerance : (u->to - u->from) * FuzzyTolerance;

    previous =  u->from - 1.0;
    count = 0;
    
    while (inputToken->type == FLOAT || inputToken->type == INTEGER)
         {
          /************************************************
           Token should be x coordinate.
          ************************************************/
          newx = (inputToken->type == FLOAT) ? (double)ValueToDouble(inputToken->value) : 
                                               (double)ValueToInteger(inputToken->value);
          if (newx < u->from)
            {
              if (u->from - newx > xtolerance)
                {
                 *DeftemplateError = CLIPS_TRUE;
                 SyntaxErrorMessage("Deftemplate (X value out of range (too small))");
                 rtnFuzzyValue(fv);
                 return(NULL);
                }
              newx = u->from;
            }
          else if (newx > u->to)
            {
              if (newx - u->to > xtolerance)
                {
                 *DeftemplateError = CLIPS_TRUE;
                 SyntaxErrorMessage("Deftemplate (X value out of range (too large))");
                 rtnFuzzyValue(fv);
                 return(NULL);
                }
              newx = u->to;
            }
          if (newx <= previous)
            {
             *DeftemplateError = CLIPS_TRUE;
             SyntaxErrorMessage("Deftemplate (Singletons should be in increasing order)");
             rtnFuzzyValue(fv);
             return(NULL);
            }
          SavePPBuffer(" ");
          previous = newx;
      
          /*************************************************
           Get the next token, which should be y coordinate
          *************************************************/
          GetToken(readSource,inputToken);
        
          if (inputToken->type == FLOAT || inputToken->type == INTEGER)
            { newy = (inputToken->type == FLOAT) ? (double)ValueToDouble(inputToken->value) : 
                                                   (double)ValueToInteger(inputToken->value);
            }
          else
            {
              *DeftemplateError = CLIPS_TRUE;
              SyntaxErrorMessage("Deftemplate (Number expected)");
              rtnFuzzyValue(fv);
              return(NULL);
            }
          if (newy < 0.0)
            {
              if (newy < -FuzzyTolerance)
                {
                  *DeftemplateError = CLIPS_TRUE;
                  SyntaxErrorMessage("Deftemplate (Membership value must be >= 0.0)");
                  rtnFuzzyValue(fv);
                  return(NULL);
                }
              newy = 0.0;
            }
          else if (newy > 1.0)
            {
              if (newy-1.0 > FuzzyTolerance)
                {
                  *DeftemplateError = CLIPS_TRUE;
                  SyntaxErrorMessage("Deftemplate (Membership must be <= 1.0)");
                  rtnFuzzyValue(fv);
                  return(NULL);
                }
              newy = 1.0;
            }
          /*************************************************
           Get the next token, which should be closing bracket
          *************************************************/
          GetToken(readSource,inputToken);
          if (inputToken->type == RPAREN)
           {
            /*************************************************
             Store new (x,y) in arrays re-allocating if necessary.
            *************************************************/
            if ( count == maxlength )
              {
                tempx = FgetArray ( maxlength + increment );
                tempy = FgetArray ( maxlength + increment );
                for (i=0; i<maxlength; i++ )
                 {
                    tempx[i] = fv->x[i];
                    tempy[i] = fv->y[i];
                 }
                FrtnArray(fv->x, maxlength);
                FrtnArray(fv->y, maxlength);
                fv->x = tempx;
                fv->y = tempy;
                maxlength += increment;
                fv->maxn = maxlength;
              }
            fv->x[count] = newx;
            fv->y[count] = newy;
            count++;
           }     
          else
           {
             *DeftemplateError = CLIPS_TRUE;
             SyntaxErrorMessage("Deftemplate ( ')' expected)");
             rtnFuzzyValue(fv);
             return(NULL);
           }
          SavePPBuffer(" ");
         
          /************************************************
           Get the next token, which should be either a closing bracket
           indicating the end of the set, or an opening bracket indicating
           the start of another (x,y) pair.
          ***********************************************/
          GetToken(readSource, inputToken);
          if ((inputToken->type == RPAREN) || (inputToken->type == STOP))
            {
              fv->n = count;
              return(fv);
            }
          else if (inputToken->type != LPAREN)
            {
              *DeftemplateError = CLIPS_TRUE;
              SyntaxErrorMessage("Deftemplate ( '(' expected)");
              rtnFuzzyValue(fv);
              return(NULL);
            }
          else    /* Get next token, which should be x coordinate */
           { 
              GetToken(readSource, inputToken);
           }
         }
     *DeftemplateError = CLIPS_TRUE;
     SyntaxErrorMessage("Deftemplate (Number expected)");
     rtnFuzzyValue(fv);
     return(NULL);
  }

/***********************************/
/* parseStandardFuzzyValue:       */    
/***********************************/
static struct fuzzy_value *parseStandardFuzzyValue(readSource,inputToken,DeftemplateError,u)
  char *readSource;
  struct token *inputToken;
  int  *DeftemplateError;
  struct universe *u;
  {
    struct fuzzy_value *fv;
    double alfa, beta, gamma;
    char *name;
    int z_function=CLIPS_FALSE, s_function=CLIPS_FALSE, pi_function=CLIPS_FALSE;
    double xtolerance;
    
    /* get the name of the standard fuction */
    if ((inputToken->type != SYMBOL) && (inputToken->type != STRING))
      {
        *DeftemplateError = CLIPS_TRUE;
         SyntaxErrorMessage("Deftemplate (Fuzzy standard function expected)");
         return(NULL);
      }
       
    name = ValueToString(inputToken->value);
    /* classify the function */
    if ( (strcmp(name,"S")==0) || (strcmp(name,"s")==0) )
        s_function = CLIPS_TRUE;
    else if ( (strcmp(name,"Z")==0) || (strcmp(name,"z")==0) )
        z_function = CLIPS_TRUE;
    else if ( (strcmp(name, "PI") == 0) || (strcmp(name, "pi") == 0) )
        pi_function = CLIPS_TRUE;
        
    /* check for valid function */
    if ( !s_function && !z_function && !pi_function)
      {
         *DeftemplateError = CLIPS_TRUE;
         SyntaxErrorMessage("Deftemplate (Fuzzy standard function name expected)");
         return(NULL);
      }

    SavePPBuffer(" ");
              
    /* We want to allow specifications for Standard functions and singleton sets
       to NOT be thrown out with an error message due to a floating point
       representation problem so we will allow the points to be considered
       within the universe of discourse (range) if very close.

       FuzzyTolerance is defined in constant.h
    */
    xtolerance = ((u->to - u->from) >= 1.0) ? FuzzyTolerance : (u->to - u->from) * FuzzyTolerance;

    /* get first parameter */  
    GetToken(readSource, inputToken);
    if (inputToken->type != FLOAT && inputToken->type != INTEGER)
       {
         *DeftemplateError = CLIPS_TRUE;
         SyntaxErrorMessage("Deftemplate (Number expected)");
         return(NULL);
       }
    SavePPBuffer(" ");
    alfa = (inputToken->type == FLOAT) ? (double)ValueToDouble(inputToken->value) : 
                                         (double)ValueToInteger(inputToken->value);
    if  (pi_function)
      {
        if (alfa <= 0.0)
           {
              *DeftemplateError = CLIPS_TRUE;
              SyntaxErrorMessage("Deftemplate (PI function 1st parameter must be positive)");
               return(NULL);
           }
        else { beta = alfa; }
      }
    else if (alfa < u->from)
      {
        if (u->from - alfa > xtolerance)
          {
           *DeftemplateError = CLIPS_TRUE;
            SyntaxErrorMessage("Deftemplate (s or z function 1st parameter out of range (too small))");
            return(NULL);
          }
        alfa = u->from;
      }
    else if (alfa > u->to)
      {
        if (alfa - u->to > xtolerance)
          {
           *DeftemplateError = CLIPS_TRUE;
            SyntaxErrorMessage("Deftemplate (s or z function 1st parameter out of range (too large))");
            return(NULL);
          }
        alfa = u->to;
      }

    /* get 2nd parameter */   
    GetToken(readSource, inputToken);
    if (inputToken->type != FLOAT && inputToken->type != INTEGER)
       {
         *DeftemplateError = CLIPS_TRUE;
         SyntaxErrorMessage("Deftemplate (Number expected for standard function parameter)");
         return(NULL);
       }
    SavePPBuffer(" ");
    gamma = (inputToken->type == FLOAT) ? (double)ValueToDouble(inputToken->value) : 
                                            (double)ValueToInteger(inputToken->value);
    if (pi_function)
       { 
         if ((gamma > u->to) || (gamma < u->from))
           {
             *DeftemplateError = CLIPS_TRUE;
             SyntaxErrorMessage("Deftemplate (pi function produces x values out of range)");
             return(NULL);
           }
         else if ((gamma - beta) < u->from)
           {
             if (u->from - (gamma - beta) > xtolerance)
               {
                 *DeftemplateError = CLIPS_TRUE;
                 SyntaxErrorMessage("Deftemplate (pi function produces x values too small)");
                 return(NULL);
               }
             beta = gamma - u->from;
           }
         else if ((gamma + beta) > u->to)
           {
             if (gamma + beta - u->to > xtolerance)
               {
                 *DeftemplateError = CLIPS_TRUE;
                 SyntaxErrorMessage("Deftemplate (pi function produces x values too large)");
                 return(NULL);
               }
             beta = u->to - gamma;
           }
        }
     else if (gamma <= alfa)
        {
          *DeftemplateError = CLIPS_TRUE;
          SyntaxErrorMessage("Deftemplate (s or z function 2nd parameter must be > 1st parameter)");
          return(NULL);
        }
     else if (gamma > u->to)
        {
         if (gamma - u->to > xtolerance)
           {
            *DeftemplateError = CLIPS_TRUE;
            SyntaxErrorMessage("Deftemplate (S or Z function 2nd parameter out of range (too large))\n");
            return(NULL);
          }
         gamma = u->to;
        }

     GetToken(readSource, inputToken);
     if (inputToken->type == RPAREN)
        {
          int ftype;
          
          if (s_function)       ftype = S_FUNCTION;
          else if (pi_function) ftype = PI_FUNCTION;
          else                  ftype = Z_FUNCTION;
           
          fv = Get_S_Z_or_PI_FuzzyValue(alfa, beta, gamma, ftype);

          GetToken(readSource, inputToken);
          return(fv);
        }
     else
        {
          *DeftemplateError = CLIPS_TRUE;
          SyntaxErrorMessage("Deftemplate ( ')' expected)");
          return(NULL);
        }
   }
   
   
   


/***************************************************************/
/* parseModifierList:                                          */
/***************************************************************/
static struct modifier *parseModifierList(readSource, inputToken, DeftemplateError)
  char *readSource;
  struct token *inputToken;
  int  *DeftemplateError;
  {
   struct modifier *last_one, *next_one, *assert_list;

   last_one = assert_list = NULL;
   
   while((next_one = parseModifier(readSource,inputToken,
                                    DeftemplateError)) != NULL)
     {
      if (last_one == NULL)
        { assert_list = next_one; }
      else
        { last_one->next = next_one; }
      last_one = next_one;
      
     }
          
   if (*DeftemplateError)
     {
      rtnModifierList(assert_list);
      return(NULL);
     }
     
   if (assert_list != NULL)
     {
       PPBackup();
       PPBackup();
       SavePPBuffer(" )");
     }
   
   return(assert_list);
  }


/***************************************************************/
/* parseModifier:                                              */
/* Returns NULL if ) is first token.                           */
/* Returns modifier.                                           */
/* Error flag is set to true if an error occurs.               */
/***************************************************************/
static struct modifier *parseModifier(readSource,tempToken, DeftemplateError)
  char *readSource;
  struct token *tempToken;
  int *DeftemplateError; 
{
   struct modifier *ptr;
   struct symbolHashNode *mod_name;
   struct FunctionDefinition *fun;
   DEFFUNCTION *deffun;
   
   /*==========================================================*/
   /* Get the opening parenthesis of the modifier pattern.     */
   /*==========================================================*/
   
   GetToken(readSource,tempToken);
   if (tempToken->type == RPAREN) 
     {
       return(NULL);
     }
      
   if (tempToken->type != LPAREN)
      {
        SyntaxErrorMessage("Deftemplate (Expected modifier pattern to begin with a '(' )");
        *DeftemplateError = CLIPS_TRUE;
         return(NULL);
      }
   
   /*================================================================*/
   /* Get the modifier name.                                          */
   /*================================================================*/
  
  
   GetToken(readSource,tempToken);
   if (tempToken->type == RPAREN)
     { return(NULL); }
   
   if (tempToken->type ==  SYMBOL)
     {
       mod_name = (SYMBOL_HN *)tempToken->value;
     }
   else 
     {
       SyntaxErrorMessage("Deftemplate (Expected fuzzy modifier name)");
       *DeftemplateError = CLIPS_TRUE;
       return(NULL);
     }
     
    SavePPBuffer(" ");
  
   /*==================================*/
   /* Get description of the modifier. */
   /*==================================*/
   
   if (FindModuleSeparator(ValueToString(tempToken->value)) )
     {
       IllegalModuleSpecifierMessage();
       *DeftemplateError = CLIPS_TRUE;
       return(NULL);
     }
   
   GetToken(readSource,tempToken);
   if ((tempToken->type ==  SYMBOL) || (tempToken->type == STRING)) 
     {
#if DEFFUNCTION_CONSTRUCT
      deffun = (DEFFUNCTION *)LookupDeffunctionInScope(ValueToString(tempToken->value));
      
      if (deffun == NULL)
        {
#endif
         /* should be name of a function of 1 arg of type float, returns float */
         fun = FindFunction(ValueToString(tempToken->value));
          if (fun == NULL)
            {
              *DeftemplateError = CLIPS_TRUE;
              SyntaxErrorMessage("Deftemplate (undefined function)");
              return(NULL);
            }
          if ((strcmp(fun->restrictions,"11f") != 0 &&
               strcmp(fun->restrictions,"11d") != 0 &&
               strcmp(fun->restrictions,"11n") != 0) 
               ||
              (fun->returnValueType != 'f' && fun->returnValueType != 'd')) 
             {
               *DeftemplateError = CLIPS_TRUE;
               SyntaxErrorMessage("Deftemplate (modifier function must have float arg and return type)");
               return(NULL);               
             }
          else
             {
               ptr = get_struct(modifier);
               ptr->name = mod_name;
               ptr->function = fun;
#if DEFFUNCTION_CONSTRUCT
               ptr->deffunction = NULL;
#endif
               ptr->next = NULL;
             }
        }
#if DEFFUNCTION_CONSTRUCT
      else
        {    
          struct expr *top, *temp;
          BOOLEAN res;
          DATA_OBJECT result;
          
          if (deffun->minNumberOfParameters != 1 || deffun->maxNumberOfParameters != 1)
             {
               *DeftemplateError = CLIPS_TRUE;
               SyntaxErrorMessage("Deftemplate (modifier function requires exactly 1 arg)");
               return(NULL);               
             }
          /* try out the deffunction and make sure it accepts a float arg and returns a float */
          top = GenConstant(PCALL, (VOID *)deffun);
          top->argList = GenConstant(FLOAT, (VOID *)AddDouble(0.0));
          temp = CurrentExpression;
          CurrentExpression = top;
          res = (*PrimitivesArray[PCALL]->evaluateFunction)(deffun,&result);
          if (!res || (result.type != FLOAT))
             {
               *DeftemplateError = CLIPS_TRUE;
               SyntaxErrorMessage("Deftemplate (modifier function must accept and return FLOAT)");
               CurrentExpression = temp;
               ReturnExpression(top);
               return(NULL);               
             }
          CurrentExpression = temp;
          ReturnExpression(top);
          ptr = get_struct(modifier);
          ptr->name = mod_name;
          ptr->function = NULL;
          ptr->deffunction = deffun;
          ptr->next = NULL;    
        }
#endif
     }
   else
     {
       SyntaxErrorMessage("Deftemplate (expecting function name)");
       *DeftemplateError = CLIPS_TRUE;
       return(NULL);
     }
     
   GetToken(readSource,tempToken);
   if (tempToken->type == RPAREN)
     {
      PPCRAndIndent();
      SavePPBuffer("  ");
      return(ptr);
     }
   else
     {
      rtnModifierList(ptr);
      *DeftemplateError = CLIPS_TRUE;
      SyntaxErrorMessage("Deftemplate ( ')' expected)");
      return(NULL);
     } 
  }

#endif /* (! RUN_TIME) && (! BLOAD_ONLY)  */


/************************************************************************
    FUNCTIONS FOR MAINTAINING AND ACCESSING LV'S
 ************************************************************************/

/******************************************************************/
/* InstallFuzzyTemplate:                                          */
/* DeinstallFuzzyTemplate:                                        */
/*                                                                */
/*                                                                */
/******************************************************************/
globle VOID InstallFuzzyTemplate(fzTemplate)
  struct fuzzyLv *fzTemplate;
{
   struct primary_term *pt;
   struct modifier *mod;

   if (fzTemplate != NULL)
      {
       if (fzTemplate->u_ptr->units != NULL)
          IncrementSymbolCount(fzTemplate->u_ptr->units);
          
       pt = fzTemplate->primary_term_list;
       while (pt != NULL)
          {
            InstallFuzzyValue((VOID *)pt->fuzzy_value_description);
            pt = pt->next;
          }
       mod = fzTemplate->modifier_list;
       while (mod != NULL)
          {
            IncrementSymbolCount(mod->name);
            if (mod->deffunction != NULL)
               (*PrimitivesArray[PCALL]->incrementBusyCount)((VOID *)mod->deffunction);
            mod = mod->next;
          }
      }
  
}


globle VOID DeinstallFuzzyTemplate(fzTemplate)
  struct fuzzyLv *fzTemplate;
{
   struct primary_term *pt, *this_pt;
   struct modifier *mod, *this_mod;

   if (fzTemplate != NULL)
      {
       if (fzTemplate->u_ptr->units != NULL)
         {
           DecrementSymbolCount(fzTemplate->u_ptr->units);
         }
       rtnUniverse(fzTemplate->u_ptr);
          
       pt = fzTemplate->primary_term_list;
       while (pt != NULL)
          {
            DeinstallFuzzyValue((VOID *)pt->fuzzy_value_description);
            this_pt = pt;
            pt = pt->next;
            rtn_struct(primary_term, this_pt);
          }
          
       mod = fzTemplate->modifier_list;
       while (mod != NULL)
          {
            DecrementSymbolCount(mod->name);
            if (mod->deffunction != NULL)
               (*PrimitivesArray[PCALL]->decrementBusyCount)((VOID *)mod->deffunction);
            this_mod = mod;
            mod = mod->next;
            rtn_struct(modifier,this_mod);
          }
        
       /* return the FuzzyLv struct */
       rtn_struct(fuzzyLv, fzTemplate);
      }
}



/******************************************************************/
/* rtnUniverse:                                                   */
/*                                                                */
/*                                                                */
/******************************************************************/
static VOID rtnUniverse(u)
  struct universe *u;
  {
    if (u != NULL)
      {
       rtn_struct(universe, u);
      }
  }


/******************************************************************/
/* rtnPrimaryTermList:                                            */
/*                                                                */
/******************************************************************/
static VOID rtnPrimaryTermList(pt)
  struct primary_term *pt;
  {
   struct primary_term *this_one;
   
   while (pt != NULL)
     {
      this_one = pt;
      pt = pt->next;
      rtn_struct(primary_term,this_one);
     }
  }

/******************************************************************/
/* rtnModifierList:                                               */
/*                                                                */
/******************************************************************/
static VOID rtnModifierList(m)
  struct modifier *m;
  {
   struct modifier *this_one;
   
   while (m != NULL)
     {
      this_one = m;
      m = m->next;
      rtn_struct(modifier,this_one);
     }
  }



  
/******************************************************************/
/* InstallFuzzyValue                                              */
/*                                                                */
/* DeinstallFuzzyValue                                            */
/*                                                                */
/* 1 arg to each -- a (VOID *) which is really a ptr to a         */
/*                  FUZZY_VALUE_HN (hash node)                    */
/******************************************************************/


globle VOID InstallFuzzyValue(fv)
  VOID *fv;
{
   struct fuzzy_value *fvptr;
   
   if (fv != NULL)
     {
      fvptr= (struct fuzzy_value *)ValueToFuzzyValue(fv);
   
      if (fvptr != NULL)
        {
         if (fvptr->name != NULL)
             IncrementSymbolCount(fvptr->name);
        }
      IncrementFuzzyValueCount(fv);
     } 
}
           
globle VOID DeinstallFuzzyValue(fv)
  VOID *fv;
{
   struct fuzzy_value *fvptr;

   if (fv != NULL)
     {
      fvptr= (struct fuzzy_value *)ValueToFuzzyValue(fv);
   
       if (fvptr != NULL)
         {
           if (fvptr->name != NULL)
              DecrementSymbolCount(fvptr->name);
         }
      DecrementFuzzyValueCount(fv);
     } 
}
           
           

/******************************************************************/
/* rtnFuzzyValue                                                  */
/*                                                                */
/******************************************************************/

globle VOID rtnFuzzyValue(fv)
  struct fuzzy_value *fv;
{
    if (fv != NULL)
      {
         FrtnArray ( fv->x, fv->maxn );
         FrtnArray ( fv->y, fv->maxn );
         rtn_struct(fuzzy_value, fv);
      }
}




#endif  /* FUZZY_DEFTEMPLATES */
