;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:RPG; Base:10; -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   ut-rand
;;; Short Desc: Simple statistics and random functions
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   03.07.91 - ThE
;;; Author:     ThE
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------

;;; ==========================================================================
;;; PACKAGE DECLARATIONS (should go into package pail-lib)
;;; ==========================================================================

(in-package :rpg)

(eval-when (load) (format *verbose* "~&ut-rand~30T1.0 "))

(eval-when (compile load eval)
  (export '(randomize random-seed *random-state*
	    list-statistics list-range list-sum list-sum*2 
	    list-average list-stddev list-z-transform
	    random-seed-demo random-test
	    )))

(defvar *random-state* nil "Global variable to hold random-seed")


;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================
;;;
;;; Provide random functions and simple statistics for numerical lists
;;; --------------------------------------------------------------------------
;;; Calling Instructions: 
;;; (list-statistics  <list of numbers>)		simple stats
;;; (list-range       <list of numbers>)		(minimum maximum)
;;; (list-sum         <list of numbers>)		sum of numbers
;;; (list-sum*2       <list of numbers>)		squared sum of numbers
;;; (list-average     <list of numbers>)		average
;;; (list-stddev      <list of numbers>)		standard deviation
;;; (list-z-transform <list of numbers>)		list of z-transforms
;;; (randomize <lowest-integer> <highest-integer>)      random function
;;; (random-seed)                                       reusing the same seed
;;; --------------------------------------------------------------------------
;;; Demo & Misc:
;;; (random-seed-demo)                          
;;;    shows how to use (random-seed)
;;;
;;; (random-test <iterations> [highest] [lowest] [t/nil])
;;;    allows simple testing of randomization functions
;;; --------------------------------------------------------------------------

;;; ==========================================================================
;;; SIMPLE STATISTICS (MIN MAX AVERAGE STDDEV)
;;; ==========================================================================

;;; A convenient function which returns some statistics
;;; (and avoids multiple traversing of the list)
;;; If just one measurement is needed, use the simpler versions below.
(defun list-statistics (list)
  "Compute all statistics for a numerical list.
  Returns a list: (Minimum Maximum Average Standard-Deviation)."
  ;; for efficiency, everything is done locally within this function.
  (let ((n (length list)) 
	(min most-positive-fixnum)
	(max most-negative-fixnum)
	(sum 0)	(sum2 0) )
    ;; now loop trough all elements of a row
    (dolist (x list)
      (when (< x min) (setf min x)) 	; minimum
      (when (> x max) (setf max x)) 	; maximum
      (setf sum (+ sum x)) 		; summing up
      (setf sum2 (+ sum2 (* x x))) 	; quadratic sum
      )
    (list min max 			; range
	  (/ sum (float n)) 		; average
	  (sqrt (/ (- sum2 (/ (* sum sum) n)) n)) ; standard deviation
	  ) ))

;;; Example usage:
;;; (list-statistics  '(1 2 3 4 5 6 7 8 9 10))

;;; ==========================================================================
;;; SIMPLE STATISTCIS (single functions) 
;;; ==========================================================================
;;;
;;; IMPLEMENTATION NOTE:
;;; In some inplementations, long lists (e.g. with length > 100)
;;; cannot be tackled with the usual (apply #'min list)
;;; (especially with non-compiled functions),
;;; so we need our own functions.


(defun list-range (list)
  "Find the minimum and the maximum in a list."
  (let ((min most-positive-fixnum)
	(max most-negative-fixnum))
    (dolist (x list (list min max))
      (when (< x min) (setf min x))
      (when (> x max) (setf max x)) )))

;;; Example usage:
;;; (list-range  '(1 2 3 4 5 6 7 8 9 10))


(defun list-sum (list)
  "Sum the values of all all elements."
  (let ((sum 0))
    (dolist (x list sum)
      (setf sum (+ sum x)))))

;;; Example usage:
;;; (list-sum '(1 2 3 4 5 6 7 8 9 10))


(defun list-sum*2 (list)
  "Sum the squares of all element of the list."
  (let ((sum 0))
    (dolist (x (mapcar 
		 (function (lambda (x)
			     (* x x )))
		 list) sum)
      (setf sum (+ sum x)))))

;;; Example usage:
;;; (list-sum*2 '(1 2 3 4 5 6 7 8 9 10))

(defun list-average (list)
  "Calculate the arithmetic average of a row in a two-dimensional array."
  (/ (list-sum list)
     (float (length list))))

;;; Example usage:
;;; (list-average '(1 2 3 4 5 6 7 8 9 10))


(defun list-stddev (list)
  "Standard deviation of all elements in a list."
  (let ((n (length list))
	(sum 0)	(sum2 0))
    ;; now loop trough all elements
    (dolist (x list (sqrt (/ (- sum2 (/ (* sum sum) n)) n)))
      (setf sum (+ sum x)) 		; summing up
      (setf sum2 (+ sum2 (* x x))) 	; quadratic sum
      )))

;;; Example usage:
;;; (list-stddev '(1 2 3 4 5 6 7 8 9 10))


(defun list-variance (list)
  "Variance of all elements in a row of an array."
  (expt (list-stddev list) 2))

;;; Example usage:
;;; (list-variance '(1 2 3 4 5 6 7 8 9 10))

;;; ==========================================================================
;;; STANDARIZATION OF VALUES (Z-TRANSFORMATION)
;;; ==========================================================================

(defun list-z-transform (list)
  "Transform a list into Z-Values (Average=0, Std deviation=1)."
  ;; for efficiency, everything is done locally within this function.
  (let ((n (length list)) (new '())
			  (sum 0) (sum2 0) 
			  (average 0) (stddev 0))
    ;; now loop trough all elements of a row
    (dolist (x list)
      (setf sum (+ sum x)) 		; summing up
      (setf sum2 (+ sum2 (* x x))) 	; quadratic sum
      )
    (setf stddev  (sqrt (/ (- sum2 (/ (* sum sum) n)) n)))
    (setf average (/ sum (float n)))
    (dolist (x list (reverse new))
      (push (/ (- x average) stddev) new)) ))

;;; Example usage:
;;; (setf l0 '(3.3 1.7 2.0 4.0 1.3 2.0 3.0 2.7 3.7 2.3 1.7 2.3))
;;; (setf l1 (list-z-transform l0))
;;; now try the following:
;;; (list-average l0)    ; 2.5
;;; (list-average l1)    ; 0.0 (more or less)
;;; (list-stddev  l0)    ; 0.812
;;; (list-stddev  l1)    ; 1.0 (more or less)

;;; --------------------------------------------------------------------------
;;; RANDOM FUNCTIONS
;;; --------------------------------------------------------------------------
;;; To avoid problems with implementation differences, a set of function
;;; to generate random numbers is provided.
;;; (For information about the concepts see CLRM2 365ff.)
;;;
;;; The function RANDOMIZE returns an integer between (and including)
;;; lower and upper boundaries. 
;;; The function RND-STAT tests a randomization function by running
;;; it several times and keeping track of lowest and highest values
;;; and by calculating the average.

(defun randomize (min max)
  "Produces a random integer in the range between min and max (incl.)"
  (cond ((not (and (numberp min)
		   (numberp max)))
	 (error "Both arguments to the function RANDOMIZE should be numbers."))
	((<= max min)
	 (cerror 
	   "Take a random number between 0 and 1.0"
	   "The arguments to the function RANDOMIZE are not correct:~%~
	   The first argument (in your case ~s) should be GREATER than ~
	   the second (~s).~%"
	   min max )
	 (random 1.0))
	 (t
	  (+ min 
	     (if (or (floatp max)(floatp min))
		 (random (abs (- max min)))
	       (random (1+ (abs (- max min)))))
	     ) )))

;;; Usage and test examples for RANDOMIZE:
;;; (randomize 0 0)    returns an error
;;; (randomize 0 nil)    returns an error
;;; (randomize 0 10)    
;;; (randomize 0 1.0)    
;;; (randomize 10 100)
;;; (randomize -10 10)    
;;; (randomize -100 -10)    
;;; (randomize -1.0 1.0)    


;;; The function RANDOM-SEED creates a random-state and writes in 
;;; on the file SEED.RND, where it may be retrieved later on
;;; (this is useful for testing under stable conditions).
;;;
;;; The function RANDOM-SEED-DEMO demonstrates the use of (rand-seed)

(defun random-seed ()
  "Returns the (implementation dependent) random state 
  which may be used as a stable seed for testing.
  Use (setf *random-state* (random-seed)) for this.
  If the seed doesn't exist, the respective state will be created
  and stored in the file 'seed.rnd'"
  (unless (probe-file "seed.rnd") 	; Already there?
    (with-open-file (out-stream "seed.rnd" :direction :output
				:if-exists nil	:if-does-not-exist :create)
      (format out-stream "~s" 
	      (make-random-state t)))) 	; Disregard compiler warning!
  (with-open-file (in-stream "seed.rnd" :direction :input
			     :if-does-not-exist :error)
    (read in-stream)))

(defun random-seed-demo ()
  (let ((seq '()))
    (format t "~%Setting random state: (setf *random-state* (random-seed))")
    (setf *random-state* (random-seed))
    (format t "~%Random state: ~s" *random-state*)
    (format t "~%1st 10 randoms: ~{~a, ~}"
	    (dotimes (i 10 seq) (setf seq (cons (randomize 0 100) seq))))
    (format t "~%Resetting random state: (setf *random-state* (random-seed))")
    (setf seq '()) (setf *random-state* (random-seed))
    (format t "~%2nd 10 randoms: ~{~a, ~}"
	    (dotimes (i 10 seq) (setf seq (cons (randomize 0 100) seq))))
    :DONE))

;;; (random-seed-demo)


;;;;; This function produces a test-sequence of random numbers
;;; and returns statistical data on it
(defun random-test (n &optional  (low 0) (high 1000) (old-seed nil))
  (let ((liste '()))
    (cond ((not old-seed)
	   (format t "~&NOT using random seed.~%"))
	  (t
	   (format t "~&Using old random seed (if possible).~%")
	   (setf *random-state* (random-seed)) ))
    (princ "Iterations                ") (princ n) (princ " ")
    ;; here goes the random function
    (dotimes (i n)
      (if (zerop (mod i (/ n 10)))
	  (princ "."))
      (push (randomize low high) liste))
    (princ " done")(terpri) 
    (princ "Computing statistics    ") 
    (let ((stat (list-statistics liste)))
      (terpri)
      (format t 
	      "~%DESIRED:  Average = ~9,3F; Lowest = ~5D, Highest = ~5D~%"
	      (list-average (list low high)) low high)
      (format t 
	      "~&ACHIEVED: Average = ~9,3F; Lowest = ~5D, Highest = ~5D~%"
	      (third stat) (first stat) (second stat)  )
      (format t 
	      "          Stddev  = ~9,3F~%" (fourth stat)) ) ))

;;; Example usage:
;;; (random-test 1000 0 10)
;;; (random-test 1000 -10 10 t)
;;; (random-test 1000 -10 -1)
;;; (random-test 1000 0.0 1.0 t)

;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================

