;;; -*- Mode: LISP; Package: PAIL-LIB; Syntax: Common-lisp; -*-
;;;
;;; ************************************************************************
;;; PORTABLE AI LAB - UNI ZH
;;; ************************************************************************
;;;
;;; Filename:   rules.cl
;;; Short Desc: functions for rules
;;; Version:    1.0
;;; Last Mod:   23. 5.91 SK
;;; Status:     Review
;;; Author:     SK
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; ------------------------------------------------------------------------
;;; Change History: 
;;; 11.11.90 SK some pAIL standards applied
;;; 23. 5.91 SK ready for review  
;;; 5.6.91 DTA Methods added to support rule menus
;;; ========================================================================

(in-package :pail-lib)


(export '(rule name-part if-part then-part doc-part date-part
          print-rule
          make-rule 
          rule-set name-part rule-set-part
          print-rule-set
          get-rule
          sexpr-to-rule
          rule-to-sexpr
	  rule-instance preconditions conclusions bindings 
	  working-memory working-table assertions trace-info
	  find-rule
	  rule-display help-button eblist
          ))

(defvar rules::*known-rulesets* nil)
(defvar rules::*known-wm* nil)


;;; ========================================================================
;;; RULE CLASS
;;; ========================================================================

(defclass rule ()
  ((name-part        :accessor name-part
                     :initarg :name-part
                     :initform nil
                     :type symbol) ;string?
   (if-part          :accessor if-part
                     :initarg :if-part
                     :initform nil
                     :type list)
   (then-part        :accessor then-part
                     :initarg :then-part
                     :initform nil
                     :type list)
   (type-part        :accessor type-part
                     :initarg :type-part
                     :initform nil
                     :type list) ;(nil forward backward)
   (doc-part         :accessor doc-part
                     :initarg :doc-part
                     :initform nil
                     :type string) ;(author, comments)
   (date-part        :accessor date-part
                     :initarg :date-part
                     :initform nil; (date-and-time-string)
                     :type string) ;"30-Apr-91 17:30:20"
   )
  (:default-initargs 
    :doc-part "No documentation")
  (:documentation "The foundation of the rule class"))


(defmethod dump-editable ((rule rule)   )
  (concatenate 'string
			(format nil "~%")
			(format nil "(~s" (name-part rule))
			(format nil "~%")
			(format nil "    (IF   ~a)"
				(let ((sofar "("))
				  (loop for fact in (if-part rule) do
					(setf sofar (concatenate 'string sofar
								 (format nil "~%~11t~s" fact))))
				  (concatenate 'string sofar ")")))
			(format nil "~%")
			(format nil "    (THEN ~a)"
				(if (eq (car (then-part rule)) 'dump::and)
				    (let ((sofar "(and"))
				  (loop for fact in (cdr (then-part  rule)) do
					(setf sofar (concatenate 'string sofar
								 (format nil "~%~11t~s" fact))))
				  (concatenate 'string sofar ")"))
				  (format nil "~%~11t~s" (then-part rule))))
			(format nil "~%")
			(format nil "    (TYPE ~s)" (type-part rule))
			(format nil "~%")
			(format nil "    (DOC ~s)" (doc-part rule))
			(format nil "~%")
			(format nil "    (DATE ~s))" (date-part rule))))


(defmethod read-instance ((rule pail-lib::rule) desc name)
  (progn 
    (setf (name-part rule) (car desc))
    (setf (if-part rule) (cadr (nth 1 desc)))
    (setf (then-part rule) (cadr (nth 2 desc)))
    (setf (type-part rule) (cadr (nth 3 desc)))
    (setf (doc-part rule) (cadr (nth 4 desc)))
    (setf (date-part rule) (cadr (nth 5 desc)))
    rule))

(defun make-rule (rulename &key if then type)
  (make-instance 'rule
                 :name-part rulename
                 :if-part if
                 :then-part then
                 :type-part type))

(defclass rule-instance ()
  ((rule             :accessor rule
                     :initarg :rule
                     :initform nil
                     :type rule) 
   (preconditions    :accessor preconditions
                     :initarg :preconditions
                     :initform nil
                     :type list)
   (conclusions      :accessor conclusions
                     :initarg :conclusions
                     :initform nil
                     :type list)
   (bindings         :accessor bindings
                     :initarg :bindings
                     :initform nil
                     :type list)
   (type-part        :accessor type-part
                     :initarg :type-part
                     :initform nil
                     :type list) ;(nil forward backward)
   (doc-part         :accessor doc-part
                     :initarg :doc-part
                     :initform nil
                     :type list)) ;(author, comments)
  (:documentation "The foundation of the rule class"))



(defclass rule-set ()
  ((name-part        :accessor name-part
                     :initarg :name-part
                     :initform nil
                     :type symbol) ;string?
   (rule-set-part    :accessor rule-set-part
                     :initarg :rule-set-part
                     :initform nil
                     :type rule))
  (:documentation "The rule-set class"))


(defmethod dump-editable ((rule-set rule-set))
  (eval `(concatenate 'string "(" ,@(loop for rule in (rule-set-part rule-set)
   collect (dump-editable rule)) ")")
  ))


(defmethod read-instance ((rule-set rule-set) desc name)
  (progn
    (setf (name-part rule-set) name)
    (setf (rule-set-part rule-set)
      (loop for rule-desc in desc collect (let
					      ((rule (make-instance 'rule)))
					    (read-instance rule rule-desc name))))
    
    rule-set))



(defclass rule-fired (rule)
  ((asserted-by      :accessor asserted-by
                     :initarg :asserted-by
                     :initform nil
                     :type symbol) ;(rule-name user fact)
   (localvalues      :accessor localvalues
                     :initarg :localvalues
                     :initform nil
                     :type list))
  (:documentation "The instantiated and fired rule class"))


(defmethod print-object ((rule rule) stream)
  (if ;(setq *readable* t)
;    (setq *readable* nil)
    *readable*
    (print-rule rule stream)
    (print-unreadable-object 
      (rule stream :type t :identity t))))


#|
(defmethod print-rule ((rule rule) stream &key)
  (format stream "(~%~a~%~2T(IF   ~a)~%~2T(THEN ~a)~%~2T(TYPE ~a)~%~2T(DOC  ~a)~%~2T(DATE ~a))" 
          (name-part rule)
          (if-part rule)
          (then-part rule)
          (type-part rule)
          (doc-part rule)
          (date-part rule)))

|#

(defmethod print-rule ((rule rule) &optional (stream t) (level 0))
  (format stream "~a" (concatenate 'string
			(format nil "~%")
			(dotimes (dummy level) (format nil "  "))
			(format nil "(~a" (name-part rule))
			(format nil "~%")
			(dotimes (dummy level) (format nil "  "))
			(format nil "    (IF   ~a)"
				(let ((sofar "("))
				  (loop for fact in (if-part rule) do
					(setf sofar (concatenate 'string sofar
								 (format nil "~%~11t~a" fact))))
				  (concatenate 'string sofar ")")))
			
			(format nil "~%")
			(dotimes (dummy level) (format nil "  "))
			(format nil "    (THEN ~a)" (then-part rule)))))
;(print-rule (get-rule *rule-set* 'rule-b))


(defmethod find-rule (name (rules rule-set))
  (do ((tail (rule-set-part rules) (cdr tail)))
	   ((equal (symbol-name (name-part (car tail)))
		   name) tail)))


(defmethod initialize-instance :after ((set rule-set) &rest junk)
  (declare (ignore junk))
  (setf rules::*known-rulesets* (cons set rules::*known-rulesets*)))

(defmethod change-type ((a rule-set) (b rule-set)) a)


(defmethod print-object ((rule-set rule-set) stream)
  (if					;(setq *readable* t)
;					(setq *readable* nil)
      *readable*
      (print-rule-set rule-set stream)
    (print-unreadable-object 
	(rule-set stream :type t :identity t))))


(defmethod print-rule-set ((rule-set rule-set) &optional (stream t))
  (format stream "~%~a:" (concatenate 'string (rectify-string (name-part rule-set))
				      (if (null (rule-set-part rule-set)) 
					  (format nil "~%empty)")
					(let ((all ""))
					  (dolist (rule (rule-set-part rule-set) all)
					    (setq all (format nil "~a~%~a" all (print-rule rule nil)))))))))



(defmethod get-rule ((rule-set rule-set) rule-name)
  (let ((rule nil))
    (dolist (rule-obj (rule-set-part rule-set))
      (if (eq rule-name (name-part rule-obj))
        (setf rule rule-obj))) rule))

(defun sexpr-to-rule (rulename sexpr)
  (make-rule rulename
             :if (cadr sexpr)
             :then (car sexpr)))

(defmethod rule-to-sexpr ((rule rule))
  (append (list (then-part rule))
          (if-part rule)))


(defclass working-memory ()
	  ((name-part  :accessor name-part
		       :initarg :name-part
		       :initform nil
		       :type string)
	   (assertions :accessor assertions
		       :initarg :assertions
		       :initform '(()()()())
		       :type list)
	   (trace-info :accessor trace-info
		       :initarg :trace-info
		       :initform nil
		       :type list))
  (:documentation "Working memory for both forward and backward chainers"))

(defclass working-table (table)
	  ((trace-info :accessor trace-info
		       :initarg :trace-info
		       :initform nil
		       :type list))
  (:documentation "Working memory for both forward and backward chainers"))


(defmethod initialize-instance :after ((mem working-memory) &rest junk)
  (declare (ignore junk))
  (setf rules::*known-wm* (cons mem rules::*known-wm*)))


(defmethod dump-editable ((w pail-lib::working-memory))
  (let ((sofar "("))
    (loop for fact in (pail-lib::assertions w) do
	(setf sofar (concatenate 'string sofar (format nil "~s~%" fact))))
    (concatenate 'string sofar ")")))

(defmethod read-instance ((item pail-lib::working-memory) desc name)
  (progn
    (setf (assertions item) desc)
    (setf (name-part item) name)
    item))

(defmethod print-object ((w pail-lib::working-memory) (s stream))
  (format s "~a" (pail-lib::assertions w) s))


(defmethod change-type ((a working-memory) (b working-memory)) (make-instance 'working-memory
								 :name-part (name-part a)
								 :assertions (copy-tree (assertions a)))) 


;(defmethod change-type ((a working-memory) (b working-memory)) a)

#| ---------------------------------------------------------------------------
(setf x (sexpr-to-rule 'x '((SAFE-TO-STACK ?VAR1 ?VAR2) 
                            ((VOLUME ?VAR1 ?V1) 
                             (DENSITY ?VAR1 ?D1) 
                             (ISA ?VAR2 TABLE) 
                             (LISP (< (LISP (* ?V1 ?D1)) 5))))))
(if-part x)

(then-part x)

(rule-to-sexpr x)
--------------------------------------------------------------------------- |#

;;;--- Tests

;(mapcar #'(lambda (rule-obj) (name-part rule-obj)) 
;        (rule-set-part *rule-set*))

;(if-part (get-rule *rule-set* 'rule-a))

(defclass rule-display (display)
	  ((help-button :accessor help-button
			:initarg :help-button
			:initform nil)
	   (eblist :accessor eblist
		   :initarg :eblist
		   :initform nil)))

(defmethod close-display :before ((disp rule-display))
  (close-display (technical-window (help-button disp)))
  (close-display (general-window (help-button disp)))
  (loop for d in (eblist disp) do (close-display d)))




;;; ==========================================================================
;;; * END OF FILE *
;;; ==========================================================================


