;;; Copyright (c) 1990 Franz Inc, Berkeley, Ca.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; Franz Incorporated provides this software "as is" without
;;; express or implied warranty.
;;;

(unless (find-package :cw-examples)
  (defpackage :cw-examples (:use :common-lisp :cw) (:nicknames :cwex)))
(in-package :cw-examples)

(export '(*static-pann-bar-height*
	  make-static-pann-bar reattach-static-pann-bar))

;; Static pann bar example.
;; Creates a static pann bar on a window, itself consisting of a
;; common window, with all code written at the user level.

;; NOTE:  Middle-bar dragging is not implemented.

(defparameter *static-pann-bar-height* 20)

(defparameter *static-pann-bar-method-wrappers*
    '(((:repaint) repaint-static-pann-bar)
      ((:expose :expose-notify) expose-static-pann-bar)
      ((:bury :bury-notify) bury-static-pann-bar)
      ((:deactivate) deactivate-static-pann-bar)
      ((:flush :flush-notify) flush-static-pann-bar)
      ((:move :move-notify :reshape :reshape-notify)
       reattach-static-pann-bar)))

(defun make-static-pann-bar (window)
  (let ((bar-window (make-window-stream :activate-p t
				 :title nil
				 :borders (window-stream-borders window)
				 :parent (window-stream-parent window)
				 :bottom (- (window-stream-bottom window)
					  *static-pann-bar-height*)
				 :left (window-stream-left window)
				 :height *static-pann-bar-height*
				 :width (window-stream-width window)))
	)
    (setf (window-stream-get bar-window :panned-window) window)    
    (setf (window-stream-get window :pann-bar) bar-window)
    (setf (window-stream-button bar-window) '(static-pann-bar-button))
    (setf (window-stream-repaint bar-window) '(static-pann-bar-repaint))
    (disable-window-stream-extent-scrolling window :vertical nil)
    
    (dolist (methods *static-pann-bar-method-wrappers*)
      (dolist (method (car methods))
	(modify-window-stream-method window method :after (cadr methods))))
    (repaint bar-window)))

(defvar *scratch-mouse-state* (make-mouse-state))
(defparameter *static-pann-bar-right-margin* 10)

(defun static-pann-bar-button (bar-window mouse-state &rest ignore)
  (declare (ignore ignore))
  (let ((window (window-stream-get bar-window :panned-window)))
    (unless (and window (not (eq (window-stream-status window) :flushed)))
      (flush bar-window)
      (format t "~%Pann bar's window no longer existed.~%")
      (return-from static-pann-bar-button))
    (let ((button-state (mouse-state-button-state mouse-state))
	  (extent-left (window-stream-extent-left window))
	  (extent-width (window-stream-extent-width window))
	  (window-width (window-stream-inner-width window))
	  (bar-window-width (window-stream-inner-width bar-window)))

      ;; Loop to keep panning if user is holding middle button down.
      (do* ((last-x-pos nil x-pos)
	    (first-time-or-middle-button-down-p
	     t
	     (eq (mouse-state-button-state 
		  (get-mouse-state bar-window *scratch-mouse-state*))
		 *middle-button-down*))
	    (x-pos (position-x (mouse-state-position mouse-state))
		   (position-x (mouse-state-position *scratch-mouse-state*))))
	  ((not first-time-or-middle-button-down-p))

	;; Hack --- if middle-click near right, act as if clicked exactly
	;; at right, so moving to right is easy with a single middle click.
	(when (and (eq button-state *middle-button-event*)
		   (< (- bar-window-width x-pos)
		      *static-pann-bar-right-margin*))
	  (setq x-pos bar-window-width))

	;; If user is holding middle button down but not moving, just sleep.
	(if (and last-x-pos (= last-x-pos x-pos))
	    (sleep .1)

	  ;; Pann the window, depending on which button was clicked and
	  ;; where the pann bar window was clicked.
	  (scroll window
		  (make-region
		   :bottom (window-stream-y-offset window)
		   :height (window-stream-inner-height window)
		   :left

		   ;; Don't allow panning outside the window's extent.
		   (max extent-left
			(min (- (+ extent-left extent-width)
				window-width)

			     ;; Absolute panning with the middle button.
			     (if (eq button-state *middle-button-event*)
				 (+ extent-left
				    (- (round (* (/ extent-width
						    bar-window-width)
						 x-pos))
				       window-width))

			       ;; Relative panning with the other buttons.
			       (+ (window-stream-x-offset window)
				  (if (eq button-state *left-button-event*)
				      (- x-pos bar-window-width)
				    (- bar-window-width x-pos))))))

		   :width (window-stream-inner-width window))))))))

(defun static-pann-bar-repaint (bar-window &rest ignore)
  (declare (ignore ignore))
  (let* ((window (window-stream-get bar-window :panned-window))
	 (extent-left (window-stream-extent-left window))	 
	 (extent-width (window-stream-extent-width window))	 
	 (x-offset (window-stream-x-offset window))
	 (bar-window-width (window-stream-inner-width bar-window))
	 (scale-factor (/ bar-window-width extent-width))
	 bar-left bar-width old-bar-left old-bar-width
	 source-height source-width draw-height draw-width move-left-p
	 )

    ;; Don't redraw pann bar if it hasn't moved.
    (when (and (= extent-left
		  (or (window-stream-get bar-window :extent-left) 0))
	       (= extent-width
		  (or (window-stream-get bar-window :extent-width) 0))
	       (= x-offset
		  (or (window-stream-get bar-window :x-offset) 0))
	       (= bar-window-width
		  (or (window-stream-get bar-window :window-width) 0)))
      (return-from static-pann-bar-repaint))
    (setf (window-stream-get bar-window :extent-left) extent-left)
    (setf (window-stream-get bar-window :extent-width) extent-width)
    (setf (window-stream-get bar-window :x-offset) x-offset)
    (setf (window-stream-get bar-window :window-width) bar-window-width)

    (setq old-bar-left (window-stream-get bar-window :bar-left)
	  old-bar-width (window-stream-get bar-window :bar-width)
	  bar-left (round (* (- x-offset extent-left) scale-factor))
	  bar-width (max 3 (round (* (window-stream-inner-width window)
				      scale-factor)))
	  source-height (bitmap-height *grey-bitmap*)
	  source-width (bitmap-width *grey-bitmap*)
	  draw-height (window-stream-inner-height bar-window))
    (setf (window-stream-get bar-window :bar-left) bar-left)
    (setf (window-stream-get bar-window :bar-width) bar-width)

    ;; If we know where the pann bar is now, just erase a bit off one
    ;; end and draw a bit onto the other end to prevent flashing.
    (cond #| ((and old-bar-width old-bar-left
		(= old-bar-width bar-width))
	   (setq draw-width (min (abs (- old-bar-left bar-left))
				  bar-width)
		 move-left-p (< bar-left old-bar-left))
	   (clear-rectangle-xy bar-window 0 (if move-left-p
						(- (+ old-bar-left bar-width)
						   draw-width)
					      old-bar-left)
			       draw-height draw-width)
	   (draw-filled-rectangle-xy bar-window 0 (if move-left-p
						      bar-left
						    (- (+ bar-left bar-width)
						       draw-width))
				     draw-height draw-width
				     :texture *grey-bitmap*)) |#

	  ;; Otherwise just redraw the whole pann bar.
	  (t
	   (clear bar-window)
	   (draw-filled-rectangle-xy
	    bar-window bar-left 0 bar-width draw-height
	    :texture *grey-bitmap*)))))

(defvar *static-pann-bar-last-window* nil)
(defvar *static-pann-bar-last-region* nil)

(defun reattach-static-pann-bar (window &rest ignore)
  (declare (ignore ignore))
  (let ((bar-window (window-stream-get window :pann-bar))
	(region (window-stream-region window))
	bar-height)

    ;; Ignore the move/reshape notification from the window manager
    ;; if it passes the same window and region as the previous notification.
    ;; Use this to get rid of most of the redundancy.
    (when (and *static-pann-bar-last-window*
	       (eq *static-pann-bar-last-window* window)
	       (region= *static-pann-bar-last-region* region))
      (return-from reattach-static-pann-bar))
    (setq *static-pann-bar-last-window* window
	  *static-pann-bar-last-region* region)
    
    ;; Get rid of this cached value so the bar will have to be redrawn
    ;; completely next time, instead of the fancy incremental draw.
    (setf (window-stream-get bar-window :bar-width) nil)

    ;; Move/shape the bar window to fit the application window.
    (when (and bar-window
	       (not (eq (window-stream-status bar-window) :flushed)))
      (setq bar-height (window-stream-height bar-window))
      (setf (window-stream-region bar-window)
	(make-region :bottom (- (window-stream-bottom window) bar-height)
		     :left (window-stream-left window)
		     :height *static-pann-bar-height*
		     :width (window-stream-width window)))
      (repaint bar-window))))

(defun region= (r1 r2)
  (and (= (region-bottom r1)(region-bottom r2))
       (= (region-left r1)(region-left r2))
       (= (region-height r1)(region-height r2))
       (= (region-width r1)(region-width r2))))

;; method wrappers to keep the pann bar attached to its window.
(defun repaint-static-pann-bar (window &rest ignore)
  (declare (ignore ignore))
  (bring-pann-bar-along window 'repaint))
(defun expose-static-pann-bar (window &rest ignore)
  (declare (ignore ignore))
  (bring-pann-bar-along window 'expose))
(defun bury-static-pann-bar (window &rest ignore)
  (declare (ignore ignore))
  (bring-pann-bar-along window 'bury))
(defun deactivate-static-pann-bar (window &rest ignore)
  (declare (ignore ignore))
  (bring-pann-bar-along window 'deactivate))
(defun flush-static-pann-bar (window &rest ignore)
  (declare (ignore ignore))
  (bring-pann-bar-along window 'flush))
  
(defun bring-pann-bar-along (window method)  
  (let ((bar-window (window-stream-get window :pann-bar)))
    (when (and bar-window
	       (not (eq (window-stream-status bar-window) :flushed)))
      (funcall method bar-window))))

;(format t "~%Call (make-static-pann-bar some-window) to create a pann bar.")

