;;; -*- Mode: LISP; Package: GTRE; Syntax: Common-lisp;                 -*-
;;;
;;; **********************************************************************
;;;
;;; PORTABLE AI LAB - EPFL
;;;
;;; **********************************************************************
;;;
;;; Filename:   jtms.cl
;;; Short Desc: JTMS
;;; Version:    1.0
;;; Status:     experimental
;;; Last Mod:   07.01.92 - Fatma Fekih-Ahmed
;;; Author:     Ken Forbus
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; -----------------------------------------------------------------------
;;; RCS $Log$
;;; -----------------------------------------------------------------------


;;; =======================================================================
;;; PACKAGE DECLARATIONS
;;; =======================================================================
 

;;; justification-based truth maintenence system
;;
;; copyright, 1986, kenneth d. forbus, university of illinois
;;
;;Last modification: 19.08.91 FFA
;;
;; this version is very simple and monotonic.

(in-package :gtre)


;;;user interface:
;;
;; see the following "user hooks" part of the program.
;;;

;;;global variables:

(defvar *contra-call* nil)

;;*node-counter* provides a unique name for nodes

(defvar *node-counter* 0)      

;;*just-counter* provides a unique name for justifications.
;;an advantage of it is that it provides a simple sort predicate when needed.

(defvar *just-counter* 0)     

;;*beg-queue* is the list of nodes that have just been retracted. these nodes
;;are examined to see if alternate support can be found for them.

(defvar *beg-queue* nil)      

;;*node* is the list of all tms nodes

(defvar *node* nil)            

;;*just* is the list of all justifications

(defvar *just* nil)            

;; debugging flag to trace internals of the tms's operation

(defvar *debug-tms* nil)       

;; list of nodes representing contradictions. such nodes must never be :in
;;and, when they are, a contradiction is signalled.

(defvar *contradictions* nil)

;;for external systems

(defvar *disable-contradiction-checking* nil)

;;;structures.

(defstruct (tms-node :named (:print-function
			      (lambda (n st ignore)
				(format st "<tms node ~d>"
					(tms-node-index n)))))

  ;;index is an integer serving as unique name for the node
  
  (index 0)
  
  ;;datum is supplied by the inference engine. for simple demonstration systems
  ;;one should put something printable in here. for tre-like systems, a pointer
  ;;to the assertion object goes here.

  (datum nil)           

  ;; status represents current belief status of the node. :in indicates the 
  ;; node is believed, and :out indicates that the node is not believed. any
  ;; other value indicates a bug.

  (status ':out)
  
  ;; support is nil if the node is labelled :out. if the node is derived, 
  ;; then it contains the justifications currently providing support.
  ;; if the node is an enabled assumption, this field contains the symbol 
  ;; :enabled-assumption.
  
  (support nil) 
  
  ;; justifications corresponds to the set of justifications which 
  ;; could provide a support for this node.

  (justifications nil) 
                        
  ;; consequences contains the justifications that use the node as an 
  ;; antecedent.

  (consequences nil)    

  ;;mark holds a marker for sweep algorithms used in finding the 
  ;;assumptions underlying a node.

  (mark nil)            
                        
  ;;if non-nil, contradictory? indicates that belief in this node represents 
  ;;a contradiction. these nodes are handled specially in the jtms.

  (contradictory? nil)

  ;;the in-rules field contains the rules that should be triggered when
  ;;node goes in.
  
  (in-rules nil)

  ;;the out-rules field contains rules that should be triggered when node
  ;;goes out.
  
  (out-rules nil)

  ;; this is true for nodes that are not roots of trees
  (not-root nil)

  )
                        
;;;the justification structure implements horn clauses used to represent
;;;relationships between beliefs in the jtms.

(defstruct (justification :named (:print-function
				   (lambda (j st ignore)
				     (format st "<just ~d>"
					     (justification-index j)))))
  ;; index is an integer for unique name

  (index 0)

  ;;type is an inference-engine supplied description of the justification
  ;;this is usually only supplied for generating explanations. while this 
  ;;information is preserved within the jtms, it is not used by any of the 
  ;;algorithms.

  type

  ;;consequence holds the node which this justification can support.

  consequence

  ;;antecedents contains the nodes which must be believed in order for this 
  ;;justification to provide support for its consequence.

  ;;Il faudrait partager les antecedents en IN-LIST et OUT-LIST

  antecedents)

(proclaim '(special *contradiction-handler*))
(setq *contra-assumptions* nil)
(defvar *tms-node-printer* #'(lambda (n) 
			       (format nil "~a" (tms-node-datum n))))

;;jtms-init initializes the jtms. this function simply clears the indices
;;the beg-queue and the list of contradictory nodes.


(defun jtms-init ()
  (setq *beg-queue* nil)
  (setq *node* nil)
  (setq *just* nil)
  (setq *just-counter* 0)
  (setq *node-counter* 0)
  (setq *contradictions* nil)
  (setq *tms-node-printer*
	#'(lambda (n)
	      (format nil "~a" (assertion-lisp-form (tms-node-datum n)))))
  (setq *contradiction-handler* #'ask-user-handler)
  (setq *node* nil))

;;;; user hooks

;;both in-node? and out-node? functions are used to query the state of a node.
;;they can be used directly, but more often, are called by problem-solver level
;;programs

(defun in-node? (node) (eq (tms-node-status node) ':in))

(defun out-node? (node) (eq (tms-node-status node) ':out))

;;install-tms-node creates a jtms node. the function takes keyword arguments
;;to specify whether a node is a contradiction or an assumption. if the node is
;;an assumption, then it is initially disabled.

(defun install-tms-node (datum &aux node) 
  (setq node (make-tms-node :index (incf *node-counter*)
			    :datum datum))
  (push node *node*)
  node)

;;install-ground-node causes truth-maintenance processing while
;;install-tms-node does not. the sequence is the following:
;;
;;  1) make this particular node believed. (make-node-in)
;;
;;  2) compute what other nodes must be believed as a consequence of believing
;;     this node. (propagate-inness)
;;
;;  3) see if any contradictions result. (check-for-contradictions)

(defun install-ground-node (node &optional (reason 'user)) 
  (if *debug-tms* 
      (let ((*readable* t))
	(format-display *debug-tms* "~% *** Making ~a a ground-node (~a)." 
			(funcall *tms-node-printer* node) reason)))
  (make-node-in node reason)
  (propagate-inness node)
  (check-for-contradictions)
  )

;;install-justification creates a justification, adds it to the node's 
;;potential justifiers.  if the supporters are in and the conseq isn't
;;currently supported then install appropriate support. since the consequence
;;of this justification may now be supported, the same sequence of processing 
;;occurs as in install-premise. the first two steps are encapsulated in the
;;function install-support.

;;note:
;;notice that calling install-justification twice with the same arguments
;;will result in the creation of distinct data structures that represent
;;the same justification. while no errors will result, duplicate justifications
;;decreases efficiency. well-designed problem-solvers will prevent this from 
;;happening away.

(defun install-justification (type conseq supporters &aux just)
  (setq just (make-justification :index (incf *just-counter*)
				 :type type
				 :consequence conseq
				 :antecedents supporters))
  (dolist (supporter supporters)
    (setf (tms-node-not-root supporter) t))
  (push just (tms-node-justifications conseq))
  (dolist (node supporters) (push just (tms-node-consequences node)))
  (push just *just*)
  (if *debug-tms* 
      (let ((*readable* t))
	(format-display *debug-tms* 
	   "~% --> Justifying ~a ~% with index ~a~%    by~%    ~a ~%    using~%    ~a~%" 
			  (funcall *tms-node-printer* conseq)
			  (justification-index just)
			  type
			  (mapcar *tms-node-printer* supporters))))
  (if (check-justification just) (install-support conseq just))
  (check-for-contradictions))

;;install-contradiction is converts a non-contradiction node to a contradiction
;;node. it adds the node to the contradiction list for the jtms which is always
;;checked at the conclusion of any operation which can change node labels. as
;;this function is called on a node that may already have justifications, this
;;invokes contadiction checking.

(defun install-contradiction (node) 
  (setf (tms-node-contradictory? node) t)
  (push node *contradictions*)
  (check-for-contradictions))

;;;; support for adding justifications

;;check-justification returns non-nil when the given justifications can supply
;;new support for its consequent node. this occurs exactly when the node isn't
;;already known and when the justification is satisfied.

(defun check-justification (just)
  (and (out-node? (justification-consequence just))
       (justification-satisfied? just)))

;;justification-satisfied checks whether a justification is satisfied, in which
;;case, all corresponding antecedents are :in.

(defun justification-satisfied? (just) 
  (every #'in-node? (justification-antecedents just)))

;;install-support makes a particular node :in and calls propagate-inness to
;;determine consequences.

(defun install-support (conseq just)
  ;;first make the node in, then propagate the consequences.
  (make-node-in conseq just)
  (propagate-inness conseq))

;;propagate-inness works by checking the justifications which use the newly
;;believed node, checking each to see if they can lend new support. when they
;;can, that is invoked recursively. notice that check-justification will only 
;;return non-nil when a node is out, and this procedure can only make nodes 
;; :in. consequently, this propagation must halt on any finite network of nodes
;;and justifications.

(defun propagate-inness (node) 
  (if *debug-tms* 
      (let ((*readable* t))
	(format-display *debug-tms* "~%   * Propagating belief in ~a.." 
			(funcall *tms-node-printer* node))))
  (do ((queue (tms-node-consequences node)
		    (append (cdr queue) new))
       (new nil nil))
      ((null queue))
    (when (check-justification (car queue))
      (make-node-in (justification-consequence (car queue))
		    (car queue))
      (setq new (tms-node-consequences
		  (justification-consequence (car queue)))))))

;;make-node-in simply marks a node :in.

(defun make-node-in (conseq just)
  (if *debug-tms* 
      (let ((*readable* t))
	(format-display *debug-tms* "~%   ** Believing ~a via ~a."
			(funcall *tms-node-printer* conseq) just)))
  (setf (tms-node-status conseq) ':in)
  (setf (tms-node-support conseq) just) 
  (dolist (in-rule (tms-node-in-rules conseq))
    (enqueue in-rule))
  (setf (tms-node-in-rules conseq) nil))

;;; premise manipulations

(defun ground-node? (node) 
  (and (in-node? node)
       (not (typep (tms-node-support node) 'justification))))

(defun premise-node? (node)
  (and (ground-node? node) (eq (tms-node-support node) 'god)))

(defun assumption-node? (node)
  (and (ground-node? node)
       (not (eq  (tms-node-support node) 'god))))


;;ground-for-node computes the set of ground support (premises and assumptions)
;;which form the basis for belief in the given node. it operates by searching
;;backwards through the antecedents of justifications until ground nodes are
;;found.

(defun ground-for-node (node) 
  (do ((nodes (list node) (append (cdr nodes) new))
       (new nil nil)
       (marker (list nil))
       (ground nil))
      ((null nodes) ground)
    (unless (eq (tms-node-mark (car nodes)) marker)
      (if (ground-node? (car nodes))
	  (push (car nodes) ground)
	  (if (in-node? (car nodes))
	      (setq new (justification-antecedents
			 (tms-node-support (car nodes))))
	      nil))
      (setf (tms-node-mark (car nodes)) marker))))

;;premises-node prints out the ground support of a node. when the node itself
;;is a premise node, it's printed out. when it's :out, it isn't a premise by
;;definition. the last case consists in determining the node's premises using
;;get-premises and print them out.
;
;(defun premises-node (node) 
;  (cond ((premise? node)
;	 (format t "~%  ~a, from ~a" (funcall *tms-node-printer* node)
;		 (tms-node-support node)))
;	((out-node? node) (format t "~%~a is out."
;			     (funcall *tms-node-printer* node)))
;	(t (dolist (node (get-premises node))
;	     (format t "~%  ~a, from ~a" 
;		     (funcall *tms-node-printer* node)
;		     (tms-node-support node)))))
;  node)

;;; forgetting requires retracting all the consequences of
;;; a node, and then looking for alternate support.

;;retract-assumption has the following basic sequence of operations:
;;
;;  1) make the particular node to be disbelieved.
;;     (make-node-out)
;;
;;  2) forget all nodes which depend on this particular node. 
;;     (propagate-outness)
;;
;;  3) attempt to find alternate support for nodes which have been retracted.
;;     (beg-inness)

(defun retract-assumption (node) 
  (when (assumption-node? node)
    (if *debug-tms* 
	(let ((*readable* t))
	  (format-display *debug-tms* "~%  ## Retracting assumption ~a."
			  (funcall *tms-node-printer* node))))
    (make-node-out node)
    (propagate-outness node)
    (beg-inness)))

;;make-node-out simply clears the status of a node
;;(symmetric with make-node-in)

(defun make-node-out (node)
  (if *debug-tms* 
      (let ((*readable* t))
	(format-display *debug-tms* "~% ### Retracting belief in ~a."
			(funcall *tms-node-printer* node))))
  (setf (tms-node-support node) nil)
  (setf (tms-node-status node) ':out)
  (dolist (out-rule (tms-node-out-rules node))
    (enqueue out-rule))
  (setf (tms-node-out-rules node) nil))

;;propagate-outness works by recursively retracting all justifications which
;;relied on the given node. the search starts with the justifications which 
;;mention the node being forgotten as an antecedent. each justification in the 
;;queue is tested to see if it was the source of support for its consequence.
;;if it was, that node in turn is forgotten and each justification which uses 
;;it as an antecedent is queued for examination.

(defun propagate-outness (node)
  (if *debug-tms* 
      (let ((*readable* t))
	(format-display *debug-tms* "~%   propagating disbelief in ~a."
			(funcall *tms-node-printer* node))))
  (do ((js (tms-node-consequences node) (append (cdr js) new))
       (new nil nil)
       (conseq nil))
      ((null js))
    ;;for each justification using the node, check to see if
    ;;it supports some other node.  if so, forget that node,
    ;;queue up the node to look for other support, and recurse
    (setq conseq (justification-consequence (car js)))
    (when (eq (tms-node-support conseq) (car js)) 
      (make-node-out conseq)
      (push conseq *beg-queue*)
      (setq new (tms-node-consequences conseq)))))

;;as each node is forgotten, it's placed on the *beg-queue*. beg-inness 
;;attempts to find alternate support for everything on this queue. basically, 
;;each justification for a particular node (the contents of node's 
;;justification field) are tested to see if they can supply support. if so,
;;install-support is called to provide that new support. recall that this new 
;;support can propagate, and so some nodes on the beg queue may have received
;;alternate support before being examined by beg-inness. that's why in-node?
;;is used to filter queue entries.

(defun beg-inness ()
  (format t "~%   looking for alternate supports.")
  (do ((node (pop *beg-queue*) (pop *beg-queue*)))
      ((null node))
    (unless (in-node? node)
      (dolist (just (tms-node-justifications node))
	(when (check-justification just)
	  (install-support (justification-consequence just)
				 just)
	  (return just))))))

;;;; interogatives

;;why-node shows the status and source of support for a node.

(defun why-node (node)
  (cond ((premise-node? node)
	 (format t "~%~a is a premise (~a)."
		 (funcall *tms-node-printer* node)
		 (tms-node-support node)))
	((assumption-node? node)
	 (format t "~%~a is an assumption <~a>"
	         (funcall *tms-node-printer*  node)
	         (tms-node-support node)))
	((in-node? node)
	 (format t "~%~a is in via ~a on"
		 (funcall *tms-node-printer* node)
		 (justification-type (tms-node-support node)))
	 (dolist (anode (justification-antecedents
			  (tms-node-support node)))
	   (format t "~%  ~a" (funcall *tms-node-printer* anode))))
	((out-node? node)
	 (format t "~%~a is out." (funcall *tms-node-printer* node)))
	(t (format t "~%~a has impossible belief status!"
		   (funcall *tms-node-printer* node))
	   (error "~%why-node")))
  node)

;;why-nodes uses why-node to show the state of the entire jtms database.

(defun why-nodes ()
  (dolist (node *node*) (why-node node)))

;;;note:
;;
;;why-nodes is useful for debugging small examples, but typically will not
;;be used in more complex systems. why-node itself, however, is designed for
;;use with external systems.

;;;; contradiction processing

;;*contra-assumptions* contains the subset of the ground support for the 
;;contradiction that consists of assumptions, as opposed to premises.

(proclaim '(special *contra-assumptions* *disable-contradiction-checking*))

;;check-for-contradiction detects contradictions by seeing if any contradictory
;;node is :in. if one of them is, the ground support for that node is computed.
;;recall that premises are considered to be "unchanging truth" and can't be
;;retracted. 

;;Assumptions, on the other hand, can be changed. the premises are removed 
;;from the ground support found, leaving the set of assumptions underlying 
;;the contradiction. these assumptions are stored in *contra-assumptions* 
;;for debugging and easy access.

;;If a contradiction involves no assumptions, then the premises given to 
;;the JTMS are wrong and an error message says "There is a flaw in the universe".

;;Whenever a contradiction is found, ask-user-handler is called to allow the user
;;to retract one among the contradictory assumptions, or ask-user-handler 
;;retracts automatically a single assumption node in contradiction with
;;premise nodes.
 
(defun check-for-contradictions ()
  (unless *disable-contradiction-checking*
  (do ((done? nil))
      (done?)
    (setq done? t)
    (dolist (cnode *contradictions*)
      (when (in-node? cnode)
        (setq done? nil)
        (setq *contra-assumptions*
          (remove-if #'premise-node? (ground-for-node cnode)))
        (unless *contra-assumptions*
          (error "~%there is a flaw in the universe ~a"
                  (funcall *tms-node-printer* cnode)))
	(setf *contra-call* t)
        (funcall *contradiction-handler* cnode *contra-assumptions*))))))

(defun ask-user-handler (contra-node ignore)
  (declare (ignore ignore))
  (cond ((null (cdr *contra-assumptions*))
	 ;;when there is only one element in *contra-assumptions*
	 ;;the other contradictory nodes must be premisses (justified
	 ;;by GOD and are not retractable!! That's why TMS has to 
	 ;;retract the single *contra-assumptions* element automatically!!)
	 (retract-assumption (car *contra-assumptions*)))
	(t
	 (let ((contradiction-browser
		(make-instance 'pail-lib:jtms-contradiction-browser
		  :title "Contradiction found"
		  :roots (list contra-node))))
	   (mp:process-wait
	    "Waiting for user to resolve contradiction"
	    #'pail-lib:contradiction-resolved-p
	    contradiction-browser)))))

(defvar *contradiction-handler* #'ask-user-handler)

;;;===============================================================
;;; END OF FILE                             
;;;===============================================================
