v ;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:PAIL-LIB; Base:10; -*-
;;; ***********************************************************************
;;;
;;;                       EXPLANATION BASED LEARNING BROWSER
;;;
;;; ***********************************************************************
;;;
;;; Filename:   tms-browser.cl
;;; Short Desc: Browsers to display TMS output.
;;; Version:    0.1
;;; Status:     Experimental
;;; Last Mod:   10.02.92 15:00:01 Fatma FEKIH-AHMED
;;; Author:     Dean ALLEMANG
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; -----------------------------------------------------------------------
;;;
;;; Modification : Le 1993/01/29  par Farra Wadah 
;;;  * la methode "construct-just-tree"
;;;  * la methode "label"
;;;  * la methode "construct-tms-tree"
;;;
;;; -----------------------------------------------------------------------
;;;
;;; =======================================================================
 
;;; DESCRIPTION
;;; -----------------------------------------------------------------------
;;; Specializes the general browser class for use by EBG.  This
;;; includes defining CLOS objects to correspond to the TMS objects.
;;; -----------------------------------------------------------------------
 
(in-package :pail-lib)

(defmethod gtre-ground-nodes ((this tms-browser))
  (let ((assertions gtre::*assert*) (in-nodes '()))
    (dolist (assertion assertions (nreverse in-nodes))
      (when (gtre::ground-node? (gtre::assertion-tms-node assertion))
	(push assertion in-nodes)))))

(defmethod gtre-out-ground-nodes ((this tms-browser))
  (let ((assertions gtre::*assert*) 
	(out-nodes '()))
    (dolist (assertion assertions (nreverse out-nodes))
      (let ((node (gtre::assertion-tms-node assertion)))
	(when (and (gtre::out-node? node)
		   (null (gtre::tms-node-justifications node)))
	  (push assertion out-nodes))))))

(defun assert-new? (item)
  (let ((name (ask "Please type the new assertion here:~%"))
	)
    (unless (string-equal name "")
      (with-input-from-string (stream name)
	(setf item 
	  (intern-all (read stream) :gtre-example))
	)
      (gtre::assert! item))
    ))

(defun make-assert-menu (assertions callback doc-string)
  (if (endp assertions)
      (list
       (list "New?" 
	     #'(lambda (item browser)
		 (declare (ignore browser))
		 (assert-new? item))
	     "Adds a new assertion to the database."))
    (cons
     (list "New?" 
	   #'(lambda (item browser)
	       (declare (ignore browser))
	       (assert-new? item))
	   "Adds a new assertion to the database.")    
     (mapcar #'(lambda (assertion)
		 (list (write-to-string
			(gtre::assertion-lisp-form assertion)
			:length 3
			:level 3
			:escape nil)
		       #'(lambda (item browser)
			   (declare (ignore item browser))
			   (funcall callback assertion))
		       doc-string))
	     assertions))))
  
(defun make-retract-menu (assertions callback doc-string)
  (if (endp assertions)
      (list "No corresponding nodes" 
	    #'(lambda (item browser)
		(declare (ignore item browser))) "")
    (mapcar #'(lambda (assertion)
		(list (write-to-string
		       (gtre::assertion-lisp-form assertion)
		       :length 3
		       :level 3
		       :escape nil)
		      #'(lambda (item browser)
			  (declare (ignore item browser))
			  (funcall callback assertion))
		      doc-string))
	    assertions)))
 
(defmethod show-database ((item tms-tree) (this tms-browser))
  (let ((*readable* t))
    (format-display gtre::*debug-tms* "--content of database--~%")
    (dolist (class gtre::*tre-class*)
      (dolist (assertion (gtre::tre-class-facts class))
	(format-display gtre::*debug-tms* (data-status assertion))))
    (format-display gtre::*debug-tms* "~%--end of database--~%")))

(defun data-status (assertion)
  (concatenate 'string (write-to-string (gtre::assertion-lisp-form assertion)
					:length 5
					:level 5
					:escape nil)
	       " is "
	       (write-to-string (gtre::tms-node-status
				 (gtre::assertion-tms-node assertion))
				:escape nil)
	       " --> support: "
	       (write-to-string (gtre::tms-node-support
				 (gtre::assertion-tms-node assertion))
				:escape nil)
	       ))

#|  |#

(defmethod shared-initialize :after ((b tms-browser) slot-names &key roots)
  (with-slots (starting-tree)
      b
    (setf starting-tree
      (mapcar #'(lambda (node) (construct-tms-tree node b)) roots))))

(defmethod initialize-instance :after ((b tms-browser)
				       &rest ignore)
  (declare (ignore ignore))
  (with-slots (left-menu)
     b
    (let ((retract-menu
	   (make-instance 'menu
	     :items (make-retract-menu
		     (gtre-ground-nodes b)
		     #'(lambda (assertion)
			 (let ((node (gtre::assertion-tms-node assertion)))
			   (gtre::make-node-out node)
			   (gtre::propagate-outness node))
			 (gtre::beg-inness)
			 (recompute-labels b))
		     "Retract this ground node")
	     :query "Please choose a ground node to retract"))
	  (assert-menu
	   (make-instance 'menu
	     :items (make-assert-menu
		     (gtre-out-ground-nodes b)
		     #'(lambda (assertion)
			 (gtre::assert! (gtre::assertion-lisp-form assertion))
			 (recompute-labels b))
		     "Assert this node")
	     :query "Please choose a node to assert")
	   ))
      (setf left-menu
	(make-instance
	    'menu
	  :items
	  `( 
            ;("Assert-menu" ,(options assert-menu) "Shows retracted nodes")
	    ("Assert this node" assert-node "Asserts this node.")
	    ("Help" help "Documentation about the JTMS/JTRE browser")
	    ("Inspect" inspect-tree
		       "Calls the inspector on the selected item.")
	    ("Recompute" recompute-me "Recomputes the browser")
	    ("Recompute labels" recompute-labels-me
				"Recomputes browser labels.")
            ;("Retract-menu" ,(options retract-menu) "Shows retractable nodes")
	    ("Retract this node" retract-node "Retracts this node")
	    ("Show database" show-database 
			     "Prints the content of the database in Verbose mode")
	    ))))))

(defmethod retract-node ((item tree) (b tms-browser))
  (let* ((node (content item)) (just (gtre::tms-node-support node)))
    (cond ((gtre::in-node? node)
	   (cond ((and (gtre::assumption-node? node) (equal 'gtre::USER just))
		  (gtre::retract-assumption node)
		  (recompute-labels b))
		 (t (documentation-print  "THIS NODE IS NOT AN  ASSUMPTION"))))
	   (t (documentation-print  "THIS NODE IS NOT IN.")))))

(defmethod pool-enter ((item tms-tree) (b tms-browser))
  (put-pool *pail-pool* item))

(defmethod assert-node ((item tree) (b tms-browser))
  (let* ((node (content item)) (supp (gtre::tms-node-support node))
	 (justifs (gtre::tms-node-justifications node)))
    (cond ((gtre::out-node? node)
	   (cond ((and (null supp) (null justifs))
		  (gtre::install-support node 'gtre::USER)
		  (recompute-labels b))
		 (t (documentation-print  "THIS NODE IS NOT AN  ASSUMPTION"))))
	  (t (documentation-print  "THIS NODE IS NOT OUT.")))))

(defmethod recompute-labels-me ((item tree) (b tms-browser))
  (recompute-labels b))

(defmethod help-window ((b tms-browser))
  (setf (help-window b) (make-instance  'help-display
			 :width 520
			 :button-region t
			 :title "JTMS browser help"
			 :filename (add-path "tms-browser.asc" pail::*gtrepath*))))

(defmethod help ((item tms-tree) (b tms-browser))
  (setf (filename (help-window b)) (add-path "tms-browser.asc" pail::*gtrepath*)))

(defmethod help ((item just-tree) (b tms-browser))
  (setf (filename (help-window b)) (add-path "tms-browser.asc" pail::*gtrepath*)))

(defmethod make-help-window ((item just-tree) (b tms-browser))
  (gin::expose 
   (if (or (not (help-window item)) (not (status (help-window item))))
       (setf (help-window item)
	 (make-instance 'help-display :width 520 
		        :left 20 
		       :button-region t 
		       :title (format nil "JTMS browser Documentation") 
		       :filename (add-path "tms-browser.asc" pail::*gtrepath*)))
     (help-window item))))
 
(defmethod make-help-window ((item tms-tree) (b tms-browser))
  (gin::expose 
   (if (or (not (help-window item)) (not (status (help-window item))))
       (setf (help-window item)
	 (make-instance  'help-display :width 520 
		       :left 20 
		       :button-region t 
		       :title (format nil "JTMS browser Documentation") 
		       :filename (add-path "tms-browser.asc" pail::*gtrepath*)))
     (help-window item))))

(defmethod label ((tree tms-tree))
  (let ((fmt-string 
	 (if (member (content tree) gtre::*contra-assumptions*)
	     "**~a**" "~a")))
	(if (equal :in (gtre::tms-node-status (content tree)))
	(setq fmt-string (format nil "<~a>" fmt-string))
        (setq fmt-string (format nil ">~a<" fmt-string)))
	(format nil fmt-string (gtre::assertion-lisp-form 
				(gtre::tms-node-datum (content tree))))))

;;; =========================================================================
;;;
;;; un noeud de justification a le nom qui est dans le champs type
;;; les noeuds IN-LIST et OUT-LIST sont des noeuds de type justification
;;; mais leur nom reste IN-LIST et OUT-LIST .
;;;
;;; =========================================================================

(defmethod label ((tree just-tree))
  (if (eq (content tree ) 'in-list) (format nil "IN-LIST") 
    (if (eq (content tree ) 'out-list) (format nil "OUT-LIST") 
    (let ((in-just (gtre::justification-satisfied? (content tree)))
	(just (gtre::justification-type (content tree))))
    (if in-just
	(format nil "<~a>" just)
      (format nil ">~a<" just))))))

;;; ========================================================================

(defmethod change-type ((tree tms-tree) (e ebg-tree)) (build-explanation tree))

(defmethod build-explanation ((tree tms-tree) )
  (if (null (descendants tree))
      (make-instance 'ebg-tree
	:fact (gtre::assertion-lisp-form (gtre::tms-node-datum (content tree)))
	:descendants nil
	:content nil)
    (build-explanation (car (descendants tree)))))

(defmethod build-explanation ((just just-tree) )
  (make-instance 'ebg-tree
    :content (gtre::justification-type (content just))
    :descendants (let ((result nil))
		   (dolist (tms (descendants just) (reverse result))
		     (push (build-explanation tms ) result)))))


(defmethod find-rule (label (rule-set rule-set))
  (do ((rule (rule-set-part rule-set) (cdr rule)))
      ((or (null rule)
	   (equal (symbol-name (name-part (car rule))) label))
       (car rule))))
;;; ==================================================================
;;;
;;; La construction de l'arbre de justification et fait pour les 
;;; noueds IN et les noueds OUT . c'est-a-dire que la justification
;;; courante est prise dans la liste de justifications "justifications"
;;;
;;; ==================================================================

(defmethod construct-tms-tree (node (b tms-browser))
  (if (or (null (gtre::tms-node-justifications node))
	  (equal (gtre::tms-node-justifications node) 'gtre::user)
	  (equal (gtre::tms-node-justifications node) 'gtre::god))
      (make-instance 'tms-tree
	:content node
	:descendants nil)
    (make-instance 'tms-tree
      :content node
      :descendants 
      (let ((result nil)
	    (just (car(gtre::tms-node-justifications node))))
	(push (construct-just-tree just b) result)))))

;;; ================================================================

(defmethod construct-just-tree (just-node (b jtms-browser))
  (make-instance 'just-tree
    :content just-node
    :descendants
      (let ((result nil)) 
	(dolist (tms (gtre::justification-antecedents just-node) result) 
	  (push (construct-tms-tree tms b) result)))))

;;; ==================================================================
;;;
;;; la modification sur la construction d'une justification d'un noeud
;;; est fait en rajoutant un niveau intermediaire.
;;; ce niveau est constitue de deux noeuds IN-LIST et OUT-LIST .
;;; Dans le noeud IN-LIST il y a les noeuds qui doivent etre IN,
;;; et Dans le noeud OUT-LIST il y a les noeuds qui doivent etre OUT .
;;; Si la liste out-list est vide on n'affiche pas le noeud OUT-LIST .
;;; Consequense de cette modification une modification sur le "label"
;;; dans la methode "defmethod label ((tree just-tree))" .
;;;
;;; ==================================================================

(defmethod construct-tree-in-list (just-node (b nmjtms-browser))
  (make-instance 'just-tree
    :content 'in-list
    :descendants 
    (let ((result nil))
      (dolist (tms (gtre::justification-in-list just-node) result) 
	(push (construct-tms-tree tms b) result)))))

(defmethod construct-tree-out-list (just-node (b nmjtms-browser))
   (make-instance 'just-tree
    :content 'out-list
    :descendants 
    (let ((result nil))
      (dolist (tms (gtre::justification-out-list just-node) result) 
	(push (construct-tms-tree tms b) result)))))

(defmethod construct-just-tree (just-node (b nmjtms-browser))
  (if (null (gtre::justification-out-list just-node))
	(make-instance 'just-tree
	  :content just-node
	  :descendants
	  (list (construct-tree-in-list just-node b)))
   (make-instance 'just-tree
    :content just-node
    :descendants
      (list (construct-tree-out-list just-node b)
	  (construct-tree-in-list just-node b)))))





;;; =======================================================================
;;; * END OF FILE *
;;; =======================================================================
