;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   input.cl
;;; Short Desc: Functions to read theorems from files
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Fabio Baj
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atp)



;;--------------------------------------------------------------------
;; read-input  : STRING ---> CONS(LIST-OF(FORMULA),LIST-OF(FORMULA))
;; Behavior    : reads an input file. Comments are skipped, command
;;             : lines are executed (flag settings, operator definitions)
;;             : the result is a cons of two lists of formulas: the axioms
;;             : and the set of support formulas.
;; Side Effects: Flags are set, infix/postfix operators lists are updated ..
;;             : eventually semantics is added to some symbols

(in-package :atp)

(defun read-input (th-name)
  (let ((filename (complete-filename (string th-name))))
    (with-open-file ( s filename)
      (let* ((line "")(ax nil)(sos nil))
	(loop
	  (setq line  (read-line s nil 'fine))
	  (cond ((eq line 'fine ) (close s) (return (cons ax sos))))
	  (execute-control-line  (remove '#\tab line))
	  (setq line (remove '#\   line))
	  (if (string-equal "list(axioms)." line) (setq ax  (read-formulas s)))
	  (if (string-equal "list(sos)."    line)(setq sos (read-formulas s))))))))


(defun read-formulas (stream)
  (let* 
      ((ax  (do* ((line "")(wb-line "")
		  (axioms  nil)
		  (fmla ""))
		((string-equal wb-line "end_of_list.") axioms)
	      (setq line (read-line  stream nil 'fine))
	      
	       (if ( equal line 'fine ) 
		  (progn (format t "*WARNING* \" end_of_list.\" missing~%")
			 (return axioms)))
			(setq wb-line  (without-blank line))
	      (cond ((not (or (string-equal "end_of_list."  wb-line)
			      (string-equal "" wb-line)
			      (equal (char wb-line 0) #\; )
			      (equal (char wb-line 0) #\% )))
		     (setq fmla (concatenate 'string  fmla (remove #\tab line)))
		     (cond ((terminates-with-dot wb-line)
			    (setq axioms 
			      (append axioms
				      (list  (string-right-trim '(#\. #\space) fmla))))
			    (setq fmla ""))))
		    ( t)))))
    ax))

(defun right-case(sym)
  (cond ((eq user::*current-case-mode* :case-insensitive-lower)
	 (intern (string-downcase sym) :atp))
	((eq user::*current-case-mode* :case-insensitive-upper)
	 (intern (string-upcase sym) :atp))
	(t sym)))
(defun execute-control-line (line)
  (if(and (not (string-equal line ""))
	  (not (equal (char line 0) #\; )))
      (let ((f-v (parse-control-line line )))
	(cond ((not (null f-v)) 
	       (let ((flag (right-case  (add-dollars (car f-v))))

		     (value (right-case (cadr f-v)))) 
		;; (format t "Ecco flag ~A~% ed ecco value ~A~%" flag value)
		 (cond ((member flag $flag-list$)
			(cond ((equal flag '$infix-functor$)
			       (setq $infix-functors$ 
				 (append(list  value) $infix-functors$)))
			      ((equal flag '$postfix-functor$)
			       (setq $postfix-functors$ 
				 (append (list  value) $postfix-functors$)))
			      ((equal flag '$evaluable$)
			       (attach-semantic (cdr f-v)))
			      ((typep value 'string)
			       (set flag (mk-sym value)))
			      ((equal flag '$strategy$)
			       (if (member value '(standard *sos* *man* ))
				    (set flag value)
				 (l-error(format nil  "~A is an unknown strategy" value))))
			      (t (set flag value))))
		       (t (l-error(format nil  "~A is an  unknown flag" (car f-v)))))))))))


;;------------------------------------------------------------------
;; attach-semantic:  ((SYMBOL DOM-1 ... DOM-n) FILE-NAME FUN-NAME)
;;                   ---> {nil}
;; side-effect    : Attaches to the symbol Symbol the function FUN-NAME
;;                : contained in file FILE-NAME. The symbol is added to the
;;                : evaluable-symbols list. The Prover now knows that the 
;;                : symbol has a semantic interpretation, and if the flag
;;                : $semantic-simplification$ is set, it will try evaluate it using this semantics.

(defun attach-semantic (spec)
  (let ((f-name (caar spec))
	(domain (cdar spec))
	(filename  (cadr spec))
	(lisp-f (caddr spec)))
    (setq $evaluable-symbols$ (cons f-name $evaluable-symbols$))
    (setf (get f-name 'domain) domain)
    (load-function lisp-f filename)
    (setf (get f-name 'attached-function) lisp-f)))
 
(defun load-function (funct source-cl)
    (with-open-file(sc
		    (atom-conc source-cl '.cl)
		    :direction :input)
      (let ((struct nil))
	(loop
	  (setq struct (read sc nil 'eof))
	  (if (equal struct 'eof) (return nil))
	  (if (and (equal (car struct ) 'defun)
		   (equal (cadr struct) funct)) (return (eval struct)))))))
	       

;;---------------------------------------------------------------
;; A control line is something like: set(<arg1>, ..., <argN>).
;; This function returns the list (<arg1> ... <argN>)

;; questa funzione andrebbe riscritta piu' specifica


(defun parse-control-line (line)
  (let ((f (atomic-f (make-token-list line))))
    (cond ((and (equal '#\. (cadr f))(equal '|set| (caar f)))
	   (cdar f)))))
			 

;;-----------------------------------------------------------------
;; Utility functions
;;
;; (add-dollars  'an-atom)                    ---> $an-atom$
;; (without-blank  " a string of chars")      ---> "astringofchars"
;; (terminates-with-dot "a line with a dot.") ---> t
;; (complete-filename "filename")             ---> "filename.th"


(defun add-dollars (name)
  (atom-conc '$ (atom-conc name '$)))

(defun without-blank (line)
  (remove '#\   (remove #\tab line)))

	  
(defun terminates-with-dot (line)
  (let ((line1 (string-right-trim '(#\space) line)))
    (cond ((eq 0 (length line1)) nil)
	  (t (equal (char  line1 (-  (length line1) 1))
		 '#\. )))))

(defun complete-filename (thname)
  (cond ((terminates-with ".th" thname)
	                        (concatenate 'string 
				(namestring   *theorems-dir*) thname))
	(t (concatenate 'string (namestring  *theorems-dir*)
		     (concatenate 'string (string thname) ".th")))))
  
  

 
