;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:RPG; Base:10;               -*-
;;;
;;; **************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; **************************************************************************
;;;
;;; Filename:   rpg-lib
;;; Short Desc: General Library stuff (from Repertory Grid)
;;; Version:    1.0b
;;; Status:     Beta testing
;;; Author:     ThE 
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;




;;; --------------------------------------------------------------------------
;;; Last Modified By: Thomas E. Rothenfluh
;;; Last Modified On: Thu Feb 20 01:18:10 1992
;;; Update Count    : 3
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================


;;; This file contains additional code which is useful beyond the module
;;; for which is was developed (Repertory Grid-Knowledge Acquisition)
 
;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :rpg)

(eval-when (compile load eval)
  (export '(init-triples triples 
	  randomize 
	  string-length string-counter char-repeat 
	  accept-alist-item 
	  print-alist-menu
	  accept-menu-item
	  print-selection-menu
	  list-to-stringlist string-to-list convert-to-integers)))


 
;;; --------------------------------------------------------------------------
;;; TRIPLES functions
;;; --------------------------------------------------------------------------
;;; DOCUMENTATION: (see in-code documentation)
;;; IMPLEMENTATION NOTE:
;;; The list with all possible combinations of ELEMENTS is implemented
;;; with information-hiding, thus preventing any interference/mixups.
;;; Only the function TRIPLES defined within the lexical scope of a variable
;;; (here: *triples [this syntax denotes semi-global, ie. "global in duration,
;;; but not in access]) have access to this variable.


(defun init-triples ()
  "Has to be called first to set up the pseudo-global variable and functions."
  (let ((*triples '()))
    (defun triples (&key init push pop pop-random length show)
      "This function is the triple-handler. It uses an internal list
      to keep track of all generated triples.
      As the keywords suggest, message-style invocation is possible.
       To generate all 3-permutation lists, use (triples :init '(a b c d)).
       To get a randomly selected element,  use (triples :pop-random t).
       (note that POP indicates destructive removal!)
      Other useful messages are:
       To get     the first       element,  use (triples :pop t).
       To get the number of remaining triples, use (triples :length t).
       To add an element at the beginning,  use (triples :push '(x y z))."
      (cond ((and init (listp init)) 	; Arguments to :init supplied
	      (setf *triples 	
		    (mapcon 
		      #'(lambda (x) 	      
			  (mapcon 
			    #'(lambda (y) 			  
				(mapcar 
				  #'(lambda (z) (list (car x) (car y) z))
				  (cdr y)))
			    (cdr x))) 	  
		      init))
		*triples)
	    (pop-random 		; Get and remove a random element
	     (let ((len (length *triples))
		   (selection nil)) 		
	       (cond ((< len 1) 
		      NIL) 		; the list is now empty
		     ((= len 1) 	; for efficiency
		      (pop *triples))
		     (t 		; else random selection
		      (setf selection 	; we have to return this...
			    (nth (1- (randomize 1 len)) *triples))
		      (setf *triples 	; ...and to remove it 
			    (delete selection
				    *triples :test #'equal)) ; destructive!
		      selection))))
	    (pop 			; Get FIRST and remove
	      (pop *triples))
	    ((and push 			; Add in front
		  (= (length push) 3))
	     (push push *triples))
	    (show 			; Have a look at it
	      *triples)
	    (length 			; How many are left?
	      (length *triples))) ) ))

(eval-when (eval load) (init-triples))

;;; Here is an example of how to use this 
;;; [evaluate all expressions sequentially]
;;; (init-triples) (triples :init '(a b c d e)) (triples :length t) 
;;; (triples :pop-random t)
;;; (triples :show t) (triples :push '(x y z)) (triples :show t) 
;;; (triples :pop t)

;;; --------------------------------------------------------------------------
;;; RANDOMIZE
;;; --------------------------------------------------------------------------

 
(defun RANDOMIZE (min max)
	  "Produces a random integer in the range between min and max (incl.)"
	  (+ (+ 0 min) 
	     #+GCLISP(random (1+ (- max min))) ; This IS Common Lisp
 					; here is the Allegro hack:
	     #-GCLISP(mod (random (1+ (- max min))) (1+ (- max min)))
	     ))

;;; --------------------------------------------------------------------------
;;; STRING-LENGTH, STRING-COUNTER, CHAR-REPEAT, STRING-PAD, STRING-TO-LIST
;;; CONVERT-TO-INTEGERS
;;; --------------------------------------------------------------------------
;;; Some functions to support pretty output of strings and type conversions

(defun string-length (string)
  "Get the length (in characters) of a symbol or string."
  (length (if (stringp string) string (format nil "~a" string))))

(defun string-counter (list fn)
  "Count the length of all symbols in list and apply fn to these values.
  For example, (string-counter '(sasa sasas sasasas) #'max)
  will return the number of characters of the longest symbol, i.e., 7."
  (apply fn (mapcar #'string-length list)))

(defun char-repeat (char n)
  "Repeat character n times and return it as a string."
  (make-string n :initial-element char))

(defun string-remove (bag str)
  "Remove all characters in bag from str."
  (let ((temp str))
    (mapcar #'(lambda (x)(setf temp (remove x temp)))
	    bag)
    temp))
;;; (string-remove '(#\Space #\Newline #\( #\) #\: #\") "asa(ss:asa(sa")

(defun string-cleanup (str &key (bag t))
  "Commonly used cleanups from string inputs.
   Without special bag, the returned value is symbolp (and thus settable)."
  (string-remove 
   (cond ((eq bag 'alphanum)
	  '(#\\ #\Space #\Tab #\Newline #\rubout #\page
	    #\backspace #\return #\linefeed #\|
	    #\! #\@ #\# #\$ #\% #\^ #\& #\( #\) 
	    #\{ #\} #\[ #\] #\: #\; #\" #\' #\~ #\`
	    #\< #\> #\, #\. #\? #\/))
	 ((every #'characterp (if (listp bag) bag (list bag)))
	  bag)
	 (bag
	  '(#\\ #\Space #\Tab #\Newline #\rubout #\page
	    #\backspace #\return #\linefeed
	    #\( #\) #\: #\" #\' #\` #\|)))
   str))
			
(defun string-pad (item 
		   &key (pad-char #\space)
			(before nil)
			(after nil))
  "Enhances a string with before (and/or after) pad-characters."
  (when (not (stringp item))
    (setf item (format nil "~a" item)))
  (when after
    (if (< after 0)(setf after 0))
    (setf item
      (concatenate 'string
	item (char-repeat pad-char after))))
  (when before
    (if (< before 0)(setf before 0))
    (setf item
      (concatenate 'string
	(char-repeat pad-char before) item)))  
  item)
;;; (print (string-pad 'hallo :before 3 :after 10))

(defun list-to-stringlist (list)
  "Converts a list to a list consisting of string elements."
  ;; IMPLEMENTATION NOTE: Does only work on top-level elements of list
  (let ((result nil))
    (if (listp list) nil (setf list (list list)))
    (dolist (el list (nreverse result))
	(push (format nil"~a" el) result))))


(defun string-to-list (string)
  "Converts a string to a list"
  (if (search "," string :test 'equalp)
      (display-error "Separate element names with blanks")
    (let ((len (length string))
	  result symbol (stop 0))
      (do ((start 0)(end len))
	  ((>= start end) (reverse result))
	(multiple-value-setq (symbol stop)
	  (read-from-string string nil 'done :start start))
	(push (intern symbol :dump) result)(setf start stop)
	) )))
;;;  (format t "~a" (string-to-list "hans eva heiri max"))


(defun convert-to-integers (list)
  "Convert a list (of symbols) to a list of integers,
   where same elements get same numbers."
  (mapcar #'(lambda (x)(position x list)) list))

;;; Example usage:
;;; (convert-to-integers '(a b c d d c b a))


;;; --------------------------------------------------------------------------
;;; Ascii-based INPUT and MENU functions
;;; --------------------------------------------------------------------------


(defun accept-alist-item (list &optional (out t) (in t))
  "Expects an a-list with numbers and labels. Returns selection."
  (when list
    (let* ((choices (print-alist-menu list))
	   (accept-num (first choices))
	   (accept-lab (second choices))
	   (choice-num (third choices))
	   (choice-lab (fourth choices))
	   (answer  (read in)))
      (format out "~%")
      (cond ((and (numberp answer)
		  (member answer accept-num)) 	; USER gives a number
	     (list answer (cadr (assoc answer choice-num))))
	    ((and (symbolp answer)
		  (member answer accept-lab)) 	; USER gives a label
	     (format out
	"~&You are supposed to type a NUMERIC VALUE from the displayed menu.~
	~%However, your input matches the printname of a possible choice.")
	     (if (y-or-n-p "~%>>>> Do you want to select item ~a? " answer)
		 (list (cadr (assoc answer choice-lab)) answer)
	       (accept-alist-item list)))
	    ((member answer '(stop quit help end bye)) :abort)
	    (t 
	     (format out
	     "~&Not one of the correct choices [1-~D], Please reenter: "
		     (length list))
	     (accept-alist-item list))))))

(defun print-alist-menu (in-list &optional (out t))
  "Print a menu from a-list: ( (num-1 label-1) (num-2 label-2) ..) ."
  (let ((len (length in-list))
	(rev-list nil)
	(firsts nil)
	(seconds nil)
	(result nil))
    (setf result
	  (dolist (el in-list 
		      (list (reverse firsts) 
			    (reverse seconds) 
			    in-list 
			    (reverse rev-list)))
	    (push (car el) firsts)
	    (push (cadr el) seconds)
	    (push (list (cadr el) (car el)) rev-list)
	    (format out "~&~3D. ~a" 
		    (car el) (cadr el))))
    (format out
	    "~%>>> ~R menuitem~:P to select from.~
	    Please enter number [1-~D]: " 
	    len len)
    result))


(defun accept-menu-item (list &optional (out t) (in t))
  "Print a menu with items from LIST to select one element."
  (when list
    (let ((choices (print-selection-menu list))
	  (answer  (read in)))
      (format out "~%")
      (cond ((and (numberp answer)
		  (> answer 0)
		  (<= answer (length choices)))
	     (cadr (assoc answer choices)))
	    ((member answer list) 
	     (format out
	"~&You are supposed to type the NUMERIC VALUE of the displayed menu.~
	~% However, your input matches the printname of a possible choice.")
	     (if (y-or-n-p "~%>>>> Do you want to select item ~a? " answer)
		 answer
	       (accept-menu-item list)))
	    ((member answer '(stop quit help end bye)) :abort)
	    (t 
	     (format out
		     "~&Not one of the correct choices [1-~D],~
		     Please reenter: "
		     (length list))
	     (accept-menu-item list))))))


(defun print-selection-menu (list &optional (out t))
  "Print a menu with numbers added in front of each element of the list"
  (let ((result nil)
	(len (length list)))
    (do ((i 1 (1+ i)) 
	 (el list (cdr el)))
	((> i len) (reverse result))
      (format out "~&~3D. ~a" i (car el))
      (push (list i (car el)) result))
    (format out 
	    "~&~%>>>> ~R menuitem~:P to select from. ~
	    Please enter number [1-~D]: " 
	    len len)
    result))

;;; (print-selection-menu '(a b c))


;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
