;;; -*- Mode: LISP; Package: STRIPS; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   strips.cl
;;; Short Desc: Basic STRIPS algorithm
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   24.1.92 DTA
;;; Author:     Hank Wan
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------
;;;; STRIPS planner

(in-package :strips)

(import '(pail-lib:op-list pail-lib:op-p pail-lib:*ops* pail-lib:add-list
	  pail-lib:del-list pail-lib:filter pail-lib:subgoals pail-lib:op
	  pail-lib:op-set pail-lib:use-ops pail-lib:not-same pail-lib:opname))

(defstruct (plan
	    (:print-function print-strips-plan))
  current-state
  pending-goals
  steps-taken-stack			; actual steps taken in reverse order
  bindings)

(defun print-strips-plan (plan &optional (stream t) ignore)
  (declare (ignore ignore))
  (format stream "~%PLAN:")
  (format stream "~% steps: ~a" (reverse
				 (plan-steps-taken-stack plan)))
  (format stream "~% state: ~a"
	  (plan-current-state plan))
  (format stream "~% goals: ~a" (plan-pending-goals plan))
  (format stream "~% bindings: ~a" (plan-bindings plan)))

(defun display-strips-plan (plan displ)
 (let ((h (- (gin::height displ) 10))
       (b (plan-bindings plan)))
  (gin::clear-display displ)
  (setf (gin:font displ) (cw:open-font :courier :non-italic 12 :weight :bold))
  (gin::write-display displ 
    "PLAN: " 0 h)
  (gin::write-display displ 
    (trim (format nil "~a" (reverse (plan-steps-taken-stack plan))) 400))
  (gin::write-display displ
    "STATE: " 0 (- h 100))
  (gin::write-display displ 
    (trim (format  nil "~a" (plan-current-state plan)) 400))
  (gin::write-display displ 
    "GOALS: " 0 (- h 200))
  (gin::write-display displ
        (format nil "~a" (mapcar (function 
	                    (lambda (x)
			      (cond ((op-p x)
				     (apply-sub (expand-lisp
						(opname x) b) b)
								  )
				    (t (apply-sub (expand-lisp x b) b)))))
				       (plan-pending-goals plan))))))

(defun display-backchain-proof (plan displ)
 (let ((h (- (gin::height displ) 10))
       (b (plan-bindings plan)))
  (gin::clear-display displ)
  (setf (gin:font displ) (cw:open-font :courier :non-italic 12 :weight :bold))
  (gin::write-display displ 
    "ACHIEVED: " 0 h)
  (gin::write-display displ 
    (trim (format nil "~a"
              (mapcar (function
	                (lambda (x) (format nil "<~a>" x)))
             (reverse (plan-steps-taken-stack plan))))
          400))
  (gin::write-display displ
    "WORKING MEMORY: " 0 (- h 100))
  (gin::write-display displ 
    (trim (format  nil "~a" (apply-sub (plan-current-state plan) b)) 400))
  (gin::write-display displ 
    "GOALS: " 0 (- h 200))
  (gin::write-display displ
        (format nil "~a" (mapcar (function 
	                    (lambda (x)
			      (cond ((op-p x)
				     (format nil "<~a>"(apply-sub (expand-lisp
						(car (add-list x)) b) b)))
				    (t (apply-sub (expand-lisp x b) b)))))
	       (plan-pending-goals plan))))
  (gin::write-display displ
    "BINDINGS: " 0 (- h 300))
  (gin::write-display displ
    (format nil "~a" b))))
    
(defun trim (string len)
  (if (< (length string) len) string
    (concatenate 'string (subseq string 0 len) "...")))


(defparameter *strips-planner*
    (planning:make-planner
     :next-states-fn 'achieve-one-goal
     :goal-p-fn 'complete-plan-p
     :plan-init-fn 'init-strips-plan
     :default-search-fn search:*depth-first-search*
     :graph-fn 'display-strips-plan
     :rank-fns '(norank)
     :default-rank-fn 'norank))

(defun norank (plan) -1)

(defun planning::fewgoals (plan) 
  (length (plan-pending-goals plan)))

(defun planning::near (plan)
  (let ((where (cadr (find-if #'(lambda (x)
				  (and (listp x) (eq (car x) 'dump::at)))
			      (apply-sub
			       (plan-pending-goals plan)
			       (plan-bindings plan))))))
    (if (numberp where) where 0)
    )
  )

(defun dump::lisp (x) (eval x))


(defun expand-lisp (sexp bind)
  (cond ((atom sexp) sexp)
	((eq (car sexp) 'dump::lisp)
	 (eval (apply-sub (cadr sexp) bind)))
	(t (loop for s in sexp collect (expand-lisp s bind)))))

;;;; Next state function for STRIPS

(defun achieve-one-goal (plan)
  "Returns a list of plans resulting from achieving the first pending goal"
  (setf plan (copy-plan plan))
  (let ((current-goal (pop (plan-pending-goals plan))))
    (cond ((op-p current-goal)		; operator ready to execute
	   (execute-op plan current-goal))
	  ((eq 'dump::lisp (car current-goal))
	   (if (eval (cadr (apply-sub current-goal (plan-bindings plan))))
	       (list plan)
	     nil))
	  (t (setf current-goal (expand-lisp current-goal (plan-bindings plan)))
	     (nconc
	      (establish-goal-in-current-state plan current-goal)
	      (establish-goal-with-new-step plan current-goal))))))

(defun achieve-one-goal-bc (plan)
  "Returns a list of plans resulting from achieving the first pending goal"
  (setf plan (copy-plan plan))
  (let ((current-goal (pop (plan-pending-goals plan))))
    (cond ((op-p current-goal)		; operator ready to execute
	   (execute-op-bc plan current-goal))
	  ((eq 'dump::lisp (car current-goal))
	   (if (eval (cadr (apply-sub current-goal (plan-bindings plan))))
	       (list plan)
	     nil))
	  (t (setf current-goal (expand-lisp current-goal (plan-bindings plan)))
	     (nconc
	      (establish-goal-in-current-state plan current-goal)
	      (establish-goal-with-new-step plan current-goal))))))

(defun execute-op (plan op)
  "Return, in a list, the plan after executing op"
  (let* ((bind (plan-bindings plan))
	 (new-plan (copy-plan plan))
	 (add-list (apply-sub (add-list op) bind))
	 (del-list (apply-sub (del-list op) bind))
	 (not-same (apply-sub (not-same op) bind)))
    (if (bad-equal not-same) nil
      (progn
	(push (apply-sub (opname op) bind)
	      (plan-steps-taken-stack new-plan))
	(setf (plan-current-state new-plan)
	  (append add-list
		  (remove-if #'(lambda (s) (find s del-list :test #'equal))
			     (plan-current-state new-plan))))
	(list new-plan)))))

(defun execute-op-bc (plan op)
  "Return, in a list, the plan after executing op"
  (let* ((bind (plan-bindings plan))
	 (new-plan (copy-plan plan))
	 (add-list (apply-sub (add-list op) bind))
	 (del-list (apply-sub (del-list op) bind))
	 (not-same (apply-sub (not-same op) bind)))
(format t "~%~%~a~%~a~%~a~%" (add-list op) bind add-list)
    (if (bad-equal not-same) nil
      (progn
	(push (apply-sub (car (add-list op)) bind)
	      (plan-steps-taken-stack new-plan))
	(setf (plan-current-state new-plan)
	  (append add-list
		  (remove-if #'(lambda (s) (find s del-list :test #'equal))
			     (plan-current-state new-plan))))
	(list new-plan)))))

(defun establish-goal-in-current-state (plan goal)
  "Return a list of possible plans from establishing goal in current state."
  (let ((bind (plan-bindings plan)))
					; try unify goal to each elt of state
    (mapcan #'(lambda (s)
		(let ((new-bind (unify s goal bind)))
		  (unless (eq 'unify::fail new-bind)
		    (let ((new-plan (copy-plan plan)))
;		      (setf (plan-bindings new-plan) (compose bind new-bind))
;		      (setf (plan-bindings new-plan) new-bind)
		      (setf (plan-bindings new-plan) (append new-bind bind))
		      (list new-plan)))))
	    (plan-current-state plan))))

(defun establish-goal-with-new-step (plan goal)
  "Return a list of possible plans from adding a step to achieve goal."
  (let ((bind (plan-bindings plan)))
    (unless (loop-detected plan goal)	; abort if looping
      (mapcan #'(lambda (op)
		  (setf op (dup-op op))
					; op is good if goal is in add-list:
		  (mapcan #'(lambda (a)
			      (let ((new-bind (unify a goal bind)))
				
				
				(unless (eq 'unify::fail new-bind)
				  (setf (plan-bindings plan) new-bind)
				  (add-new-step plan op))))
			  (add-list op)))
	      (op-list *ops*)))))

(defun bad-equal (not-same)
  (loop for ns in not-same thereis (equal (car ns) (cdr ns))))

(defparameter *process-filter* nil
  "If t, consider filters when adding operators; otherwise, treat
   filtering conditions as additional preconditions.

   STRIPS representation only has `preconditions'.  `Filters' is a useful
   operator representation formalism added to STRIPS representation.
   Filters specify conditions under which an opperator is appropriate.
   An operator should not be considered if filtering conditions cannot
   found by matching with current state.  This helps prune the search tree."
  )

(defun add-new-step (plan op)
  "Return, in a list, possible plans with op added"
  (if *process-filter*
      (mapcan #'(lambda (new-plan)
		  (push op (plan-pending-goals new-plan))
		  (setf (plan-pending-goals new-plan)
		    (append (subgoals op)
			    (plan-pending-goals new-plan)))
		  (list new-plan))
	      (apply-filter plan (filter op)))

					; preconditions otherwise
    (let ((new-plan (copy-plan plan)))
      (push op (plan-pending-goals new-plan))
      (setf (plan-pending-goals new-plan)
;	(append (and *process-filter* (filter op))	     ;)
	(append (filter op)				
		(subgoals op)
		(plan-pending-goals new-plan)))
      (list new-plan))))

(defun apply-filter (plan filters)
  "Return possible plans with filters satisfied by current state of plan"
  (if (null filters)
      (list plan)
    (mapcan #'(lambda (new-plan)
		(apply-filter new-plan (cdr filters)))
	    (establish-goal-in-current-state plan (car filters)))))

(defun loop-detected (plan goal)
  "Return t if loop detected."
  ; it maybe better here to use unify rather than #'equal.  Would that be
  ; correct?  If not, what would be better than equal but correct?
  (let ((goal-stack (remove-if #'op-p
			       (cdr (member-if #'op-p
					       (plan-pending-goals plan))))))
    (find (apply-sub goal (plan-bindings plan)) (apply-sub goal-stack (plan-bindings plan))
	  :test #'equal)))

(defun complete-plan-p (plan)
  (null (plan-pending-goals plan)))

(defun init-strips-plan (initial-state goals)
  (make-plan :current-state initial-state
	     :steps-taken-stack '(begin)
	     :pending-goals goals))

(defun init-bc-proof (initial-state goals)
  (make-plan :current-state initial-state
	     :steps-taken-stack nil
	     :pending-goals goals))

