;;; -*- Mode: LISP; Package: SNLP; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   structs.cl
;;; Short Desc: structure definitions for SNLP
;;; Version:    0.1
;;; Status:     Provisional
;;; Last Mod:   12.5.92 DTA
;;; Author:     Wan
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------
" (c) 1990, 1991 Copyright (c) University of Washington
  Written by Stephen Soderland, Tony Barrett and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-snlp@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Data Structures and variables for SNLPLAN
;;;
;;; Steven Soderland and Dan Weld
;;; Jan 1991

(in-package :snlp)

(export '(defstep reset-domain plan-steps))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SNLPLAN

(defstruct (snlp-plan (:constructor make-snlp-plan*)
                     (:print-function print-plan))
  steps                   ; list of steps
  links                   ; list of (ID1 condition ID2)
  unsafe                  ; list of (link clobber-ID clobber-bind)
  open                    ; list of (condition ID)   
  ordering                ; list of (ID1 ID2)
  bindings                ; hashtable of bind 
  high-step               ; integer number of highest step in plan
  rank                    ; used to cache rank value for best first search
  graphics				; used for graphical data
  )

(defstruct (mcs-plan (:constructor make-mcs-plan*)
                     (:print-function print-mcs))
  initial                 ; list of initial conditions
  steps                   ; list of steps
  goals                   ; list of sub-goals of current plan
  bindings                ; hashtable of bind
  high-step               ; integer number of highest step in plan
  open                    ; number of goals that don't unify with initial
  lazy                    ; used for lazy evaluation by mcstrips2
  rank                    ; used to cache evaluation of plan
  )

(defstruct snlp-step
  id                      ; integer step number
  action                  ; formula such as (puton ?X1 ?Y1)
  precond                 ; list of conditions such as (clear ?X1)
  add                     ; list of conditions asserted by step
  dele                    ; list of conditions denied by step
  )

(defstruct bind
  key            ; constant or id of variable
  value          ; constant it is bound to or nil
  syn            ; list of variables it is bound to
  not            ; hash table of var or constants constrain not to bind with
  )

;;; Variables are identified with ? such as ?X1
;;; (Adapted from Charniak 1987)
#+not-PAIL
(defstruct (variable (:print-function print-variable) (:predicate variable?))
  id)

;;; The Read-macro to build PCVAR structure when it sees a question mark.
#+not-PAIL
(set-macro-character #\?
  #'(lambda (stream char)
      (declare (ignore char))
      (make-variable :id (read stream t nil t)))
  t)


;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Testing

(defstruct (stat (:print-function print-stat))
  algo                                  ; tweak or strips
  date                                  ; when performed
  prob-num                              ; identifier
  num-init                              ; how many initial conditions
  num-goal
  plan-len                              ; how many steps
  reached-max?                          ; terminated because of nodes?
  complete?                             ; planner successful
  time                                  ; internal cpu time
  visited                               ; nodes-visted
  created                               ; calls to make-plan
  q-len                                 ; queue len at termination
  ave-branch                            ; average branching factor
  unify-count
  rank-unifies
  add-bindings
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global Variables  - general

(defvar *bindings* nil)      ; used in bind-variable
(defvar *templates* nil)     ; list of pairs of dummy step, bindings
(defvar *trace* 0)           ; 5 = list Queue
                             ; 4 = "Add Bind ..."
                             ; 3 = "Unifying ..."
                             ; 2 = "New Step? ..."
                             ; 1 = "* New Step ..."
                             ; 0 = "Plan at Current Node"
(defvar *nodes-visited* 0)
(defvar *unify-count* 0)
(defvar *compute-rank-unifies-count* 0)
(defvar *compute-rank-unifies* nil)
(defvar *computing-rank* nil)
(defvar *add-bind-count* 0)
(defvar *print-plan-fn* #'pprint)

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global Variables - mcstrips

(defvar *lazy* nil)
(defvar *mcs-plan* nil)
(defvar *mcs-limit* 400)     ; max number of plans created
(defvar *mcs-branch* 0)
(defvar *mcs-plans-created* 0)

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global Variables - SNLP

(defvar *snlp-limit* 400)     ; max number of plans created
(defvar *snlp-branch* 0)
(defvar *snlp-plans-created* 0)

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Globals for Testing

;;; Array of blocks
(defvar *blocks* 
  (make-array '45
              :initial-contents
              '(A B C D E F G H I J K L M N O P Q R S T U V W X Y Z 
                AA BB CC DD EE FF GG HH II JJ KK LL MM NN OO PP QQ RR SS)))
(defvar *stats* nil)      ; try to prseserve data in case of stack overflow
(defvar *old-stats* nil)  ; buffer for last run

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Interface functions

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This function is used to define plan steps of a domain theory.
(defun defstep (&key action precond add dele (equals nil))
  (push (list (make-snlp-step
	       :action action
	       :precond precond
	       :add add
	       :dele dele)
	      equals)
	*templates*))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Purge a previously defined domain theory.
(defun reset-domain ()
  (setf *templates* nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Utility functions

(defun make-snlp-plan (&rest args)
  (setq *snlp-plans-created* (+ 1 *snlp-plans-created*))
  (apply #'make-snlp-plan* args))

(defun make-mcs-plan (&rest args)
  (setq *mcs-plans-created* (+ 1 *mcs-plans-created*))
  (apply #'make-mcs-plan* args))

(defun print-variable (var stream depth)
  (declare (ignore depth))
  (if (listp (variable-id var))
      (format stream "?~s~s" (car (variable-id var)) (cadr (variable-id var)))
      (format stream "?~s" (variable-id var))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print functions for SNLP & SLP

(defun print-plan (s &optional (st t) ignore)
  (funcall *print-plan-fn* s st ignore))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print out statistics from single run
(defun print-stat (s &optional (st t) ignore)
  (declare (ignore ignore))
  (format st "~%~%~a (Init = ~2a ; Goals = ~2a) => ~a (~a steps)     CPU ~9a"
          (stat-algo s) (stat-num-init s) (stat-num-goal s)
          (if (stat-complete? s) "Win " "Lose")
          (stat-plan-len s) (stat-time s))
  (format st "~%     Nodes (V = ~4a; Q = ~4a; C = ~4a)             Branch ~10a"
          (stat-visited s) (stat-q-len s) (stat-created s)
          (stat-ave-branch s))
  (format st "~%     Working Unifies: ~25a       Bindings added: ~5a~%"
          (- (stat-unify-count s) (stat-rank-unifies s))
          (stat-add-bindings s)))

(defun print-mct0 (plan &optional (stream t) ignore)
  (declare (ignore plan stream ignore))
  )

;;;;;;;;;;;;;;;;
;;; This version does a toplogical sort and then prints out the steps
;;; in the order in which they should be executed
(defun print-snlp (plan &optional (stream *standard-output*) ignore)
  (declare (ignore ignore))
  (format stream "~%")
  (setf *bindings* (snlp-plan-bindings plan))
  (let ((steps (make-array (+ 1 (snlp-plan-high-step plan))))
        (order (top-sort (snlp-plan-ordering plan) (snlp-plan-high-step plan)))
        (goal nil))
    (dolist (step-n (snlp-plan-steps plan))
      (cond 
        ((eql (snlp-step-id step-n) '0)
         (format stream "~%Initial  : ~a~%" (snlp-step-add step-n)))
        ((eql (snlp-step-id step-n) :goal)
         (setf goal (snlp-step-precond step-n)))
        (t
         (setf (aref steps  (snlp-step-id step-n)) step-n))))
    (dotimes (i (snlp-plan-high-step plan))
      (let* ((sn (nth i order))
             (step (aref steps sn)))
        (format stream #-not-PAIL
		       "~%Step ~3a : ~15a Created ~2a"
		       #+not-PAIL
		       "~%Step ~3a : ~15a  from ~15a Created ~2a" 
                (+ 1 i)
                (mapcar #'bind-variable (snlp-step-action step))
		#+not-PAIL		; this bug is fixed in newer ver
                (bind-variable (make-variable 
                                :id (list 'Z sn)))
                sn)))
    (format stream "~%~%Goal     : ~a" goal)
    (format stream "~%~%Order    : ~a" (snlp-plan-ordering plan))
    (if (or (snlp-plan-unsafe plan)
            (snlp-plan-open plan))
        (format stream "~%Unsafe   : ~a ~%Open     : ~a"
                (mapcar #'bind-variable (snlp-plan-unsafe plan))
                (mapcar #'bind-variable (snlp-plan-open plan)))
        (format stream "~%Complete!"))))


;;; yields the plan itself, for use by a program.  Incomplete plan
;;; steps are not returned (i.e., the ones with variables still in
;;; them).
(defun plan-steps (plan)
  (setf *bindings* (snlp-plan-bindings plan))
  (let ((steps (make-array (+ 1 (snlp-plan-high-step plan))))
        (order (top-sort (snlp-plan-ordering plan) (snlp-plan-high-step plan)))
        (goal nil))
    (dolist (step-n (snlp-plan-steps plan))
      (cond 
        ((eql (snlp-step-id step-n) '0)
         (snlp-step-add step-n))
        ((eql (snlp-step-id step-n) :goal)
         (setf goal (snlp-step-precond step-n)))
        (t
         (setf (aref steps  (snlp-step-id step-n)) step-n)))) 
    (loop for i from 0 to (- (snlp-plan-high-step plan) 1)
	when (no-variables (mapcar #'bind-variable (snlp-step-action (aref
								 steps (nth i order)))))
	collect
	  (let* ((sn (nth i order))
		 (step (aref steps sn)))
	    (mapcar #'bind-variable (snlp-step-action step)))
      )
    
    ))
  


(defun no-variables (step) (loop for item in step never (variable? item)))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print function for Plan - all the ugly details
(defun print-snlp-verbose (plan &optional (stream *standard-output*))
  (cond (plan
         (print-snlp-nb plan)
         (print-bind (snlp-plan-bindings plan))
         (format stream "~%Links    : ~a ~%Ordering : ~a" 
                 (mapcar #'bind-variable (snlp-plan-links plan))
                 (snlp-plan-ordering plan)))
        (t 
         (format stream "No plan...")))
  (format stream "~%")
  )


;;;;;;;;;;;;;;;;
;;; Old, nonsorted version
(defun print-snlp-nb (plan &optional (stream *standard-output*))
  (format stream "~%")
  (setf *bindings* (snlp-plan-bindings plan))
  (dolist (step-n (snlp-plan-steps plan))
    (cond 
      ((eql (snlp-step-id step-n) '0)
       (format stream "~%Initial  : ~a" (snlp-step-add step-n)))
      ((eql (snlp-step-id step-n) :goal)
       (format stream "~%Goal     : ~a" (snlp-step-precond step-n)))
      (t
       (format stream #-not-PAIL
		      "~%Step ~3a : ~15a"
		      #+not-PAIL
		      "~%Step ~3a : ~15a  from ~a" (snlp-step-id step-n)
	       (mapcar #'bind-variable (snlp-step-action step-n))
	       #+not-PAIL
	       (bind-variable (make-variable 
			       :id (list 'Z (snlp-step-id step-n))))))))
  (format stream "~%Links    : ~a ~%Unsafe   : ~a ~%Open     : ~a ~%Ordering : ~a~%High-step : ~a"
	  (mapcar #'bind-variable (snlp-plan-links plan))
	  (mapcar #'bind-variable (snlp-plan-unsafe plan))
	  (mapcar #'bind-variable (snlp-plan-open plan))
	  (snlp-plan-ordering plan)
	  (snlp-plan-high-step plan))
  )

;;; Topological Sort   
;;; Returns correct order: first step at head
;;; Input: max is an integer
;;;    Ordering is a list of pairs (f l) where step number f must be before l
;;;    f, l <= max
;;; See Aho, Hopcoft, Ullman p70 for faster way
(defun top-sort (ordering max)
  (top-sort1 (copy-tree ordering) max))

;;; Topological Sort util  -   This code is DESTRUCTIVE!  Pass it a copy!
(defun top-sort1 (ordering max)
  (when (not (= 0 max))
    (let ((as (mapcar #'cadr ordering)))
      (do ((p ordering (cdr p)))
	  ((not (member (caar p) as))
	   (cons (caar p)
		 (top-sort1 (delete-if #'(lambda (x) (eql (car x) (caar p))) ordering)
			    (- max 1))))))))

