;;; -*- Mode: LISP; Package: SNLP; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   snlp.cl
;;; Short Desc: basic non-linear planner
;;; Version:    0.1
;;; Status:     Provisional
;;; Last Mod:   12.5.92 DTA
;;; Author:     Wan
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------
" (c) 1990, 1991 Copyright (c) University of Washington
  Written by Stephen Soderland, Tony Barrett and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-snlp@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Program   : "SNLPlan"
;;;  Author    : Stephen Soderland  
;;;              (modified & extended by Dan Weld and Tony Barrett)
;;;  Date      : Summer/Autumn 1990
;;;
;;;  Description:
;;;  
;;;    This program is a domain independent, conjuctive, non-linear planner
;;;    which claims to be both complete and sound.  The representation of
;;;    conditions and actions is similar to STRIPS, with each step having
;;;    a list of added conditions and a list of deleted conditions.  
;;;
;;;    A plan includes a list of steps and a list of links between steps. 
;;;    Links are in the form  (id1 condition1 id2) where id1 is a step that
;;;    establishes condition1, which is in turn a precondition of step id2.
;;;    The plan also has a list of open conditions (not yet acheived) and a 
;;;    list of unsafe links (possibly clobbered by another step).  The other 
;;;    components of a plan are ordering constraints and bindings of variables.
;;;
;;;    Each iteration of the main control level creates possible next plans 
;;;    which are added to a priority queue.  New plans are created by adding
;;;    constraints to resolve unsafe links.  If there are no unsafe links,
;;;    new plans are created by adding a new step or new link to achieve
;;;    an open condition.  The plan is completed when there are no more unsafe
;;;    links or open conditions.  
;;;  
;;;    SNLPlan is adapted from a non-linear planner by David McAllester.    
;;;    As the first phase of this project, I implemented Tweak, a non-linear
;;;    planner described by David Chapman in Artificial Intelligence 1987.
;;;    Code from my Tweak program was incorporated into SNLPlan.  The data
;;;    structures for variable bindings were later revised to enhance
;;;    performance.  Plan bindings involve a hash table that contains another
;;;    hash table.
;;;
;;;

(in-package :snlp)

(export '(plan rank rank1 rank2 rank3))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Outline of this file
;;;
;;; 1. Handy interface for using the program
;;; 2. Main control level of SNLPlan
;;; 3. Creating a list of one step refinements to a partial plan
;;; 4. Resolving an unsafe link
;;; 5. Adding a step
;;; 6. Handling links
;;; 7. Ranking partial plans
;;; 8. Statistics report routines
;;; 9. Print functions

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. Handy interface

;;; This returns two values: a plan and a stat record
;;; If this is the top-level call, then both are printed.
(defun plan (initial
	     goals
	     &optional
	     (rank-fun #'rank)
	     (search-fun #'bestf-search))
  (multiple-value-bind (plan done? time q-len av-branch)
      (plan* initial goals rank-fun search-fun)
    (values plan (make-stat :algo        "SNLPLAN"             
                            :date        (today)
                            :prob-num    1
                            :num-init    (length initial)       
                            :num-goal    (length goals)
                            :plan-len    (if plan (snlp-plan-high-step plan) 0)
                            :reached-max? (>= *nodes-visited* *snlp-limit*)
                            :complete?    done?
                            :time         time
                            :visited      *nodes-visited*     
                            :created      *snlp-plans-created*
                            :q-len        q-len
                            :ave-branch   (float av-branch)
                            :unify-count  *unify-count*
                            :rank-unifies *compute-rank-unifies*
                            :add-bindings *add-bind-count*)
            )))

;;; Quiet version - returns the details
(defun plan* (init goals rank-fun search-fun)
  (snlplan (make-snlp-plan
            :steps (list
                    (make-snlp-step :id '0
                               :add init)
                    (make-snlp-step :id :goal
                               :precond goals))  
            :open (mapcar #'(lambda (x) (cons x '(goal))) goals)
            :bindings (make-hash-table :test #'equal)
            :high-step 0)
           rank-fun
	   search-fun))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 2. New main control level of SNLPlan

;;; Returns 5 values: final plan, t or nil denoting success, run time
;;; length of q at termination, average branching factor
;;;
;;; Replaced by ACB 3/91 to factor out the search strategy.
(defun snlplan (initial-plan rank-fun search-fun)
  (reset-snlp-vars)                                ; clear globals for metering
  (setf *print-plan-fn* #'print-snlp)              ; Define plan print function
  (let* ((init-time (get-internal-run-time)))
    (multiple-value-bind (plan bfactor qlength)
	(funcall search-fun initial-plan
		#'plan-refinements #'plan-test rank-fun *snlp-limit*)
      (values plan                                 ; the plan itself
	      (and plan
		   (null (snlp-plan-unsafe plan))
		   (null (snlp-plan-open plan)))    ; plan successfull?
	      (- (get-internal-run-time) init-time) ; time
	      qlength                               ; final length of the queue
	      bfactor))))                           ; average branching factor

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  A simple best first search strategy.  Returns 3 values: the found state,
;;;  the average branching factor, and the number of generated but unexplored 
;;;  states.  The search will only generate up to LIMIT states.  
;;;  Calls snlp-search-trace for debugging purposes.
(defun bestf-search (initial-state daughters-fn goal-p rank-fn limit)
  (let ((branches nil)                         ; compute average branch factor
	(generation 0))
    (setf *user-stop* nil)
    (do* ((current-entry nil (caar search-queue))
          (current-state initial-state (cdr current-entry))
	  (ancesters nil (cdar search-queue))
	  (search-queue nil (cdr search-queue)))
         ((or *user-stop*
	      (null current-state)
	      (funcall goal-p current-state)
              (> 0 limit))
          (values current-state
                  (if (null branches) 0
                      (dnoz (apply #'+ branches) (length branches)))
                  (length search-queue)
		  ancesters))
      (incf *nodes-visited*)
      (search-trace current-state (car ancesters) search-queue rank-fn)
      (let ((children (funcall daughters-fn current-state)))
	(when (> *trace* 0.5) (format t "~%children: "))
        (setf limit (- limit (length children)))
        (setf search-queue
              (merge
                  'list search-queue
                  (sort (mapcar #'(lambda (x)
				    (when (> *trace* 0.5)
				      (format t "~d " (incf generation)))
				    (list* (cons (funcall rank-fn x) x)
					   generation ancesters))
                                children)
                        #'< :key #'caar)
                  #'< :key #'caar))
        (push (length children) branches)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Search trace function 
(defun search-trace (current-state gen search-queue &optional (rank-fn nil))
  (when (> *trace* 0.5)
    (format t "~%CURRENT PLAN (rank ~a): gen ~d"
	    (funcall rank-fn current-state)
	    gen)
    (print-snlp current-state))
  (if (> *trace* 0.5)
      (format t "~%QUEUE   -  Length ~a  " (length search-queue))
      (if (> *trace* 0) (format t " * ")))
  (if (and (> *trace* 4) search-queue)
      (dolist (q-plan search-queue)
	(setf q-plan (car q-plan))
	(format t "~%Rank ~a" (funcall rank-fn (caddr q-plan)))
	(print-snlp (caddr q-plan)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Goal test function for a search.
(defun plan-test (plan)
  (not (or (snlp-plan-unsafe plan)
           (snlp-plan-open plan))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Creating a list of one step refinements to a partial plan

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a list of one step refinements to the current plan.
;;;
;;;   Select one unsafe link and find all new plans that
;;; resolve it by separation, demotion, or promotion.  Separation is 
;;; adding bindings that prevent unifying of the condition with the
;;; added conditions of a possible clobbering step.  Demotion and
;;; promotion add ordering constraints so that a possibly clobbering
;;; step comes before the first step of the link or after the second
;;; step.
;;;   If there are no unsafe links, select an open condition to resolve
;;; by adding a new step or making a link to an existing step.  Adding 
;;; a new step or new link may cause a link to become unsafe (possibly 
;;; clobbered).
;;;
(defun plan-refinements (plan)
  (if (snlp-plan-unsafe plan)
      (multiple-value-bind (unsafe-link plan2)
          (remove-unsafe plan)
        (append (separate unsafe-link plan2)
                (demote unsafe-link plan2)
                (promote unsafe-link plan2)))
    (multiple-value-bind (open-cond plan2)
        (remove-open plan)
      (append (add-step open-cond plan2)
              (new-link open-cond plan2)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a new current plan with the first unsafe link removed, and
;;;  returns the removed link and the new plan.
(defun remove-unsafe (plan)
  (values (car (snlp-plan-unsafe plan))
          (make-snlp-plan
           :steps (snlp-plan-steps plan)
           :links (snlp-plan-links plan)
           :unsafe (cdr (snlp-plan-unsafe plan))
           :open (snlp-plan-open plan) 
           :ordering (snlp-plan-ordering plan)
           :bindings (snlp-plan-bindings plan)
           :high-step (snlp-plan-high-step plan))))


;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a new current plan with the first open condition removed, and
;;;  returns the removed open condition and the new plan.  If the
;;;  first open condition is a LISP evaluation expression, then we
;;;  should only try to form a new plan if the expression has all its
;;;  variables bound; otherwise, we should wait, and take another.  
(defun remove-open (plan)
  (let ((open-cond (car (last (snlp-plan-open plan))))) ; was car of the last
;   (print open-cond)
    (if (and (eq (caar open-cond) 'dump::lisp)
	     (stillvars (bind-variable (cadar open-cond) (snlp-plan-bindings plan))))
        (remove-open (make-snlp-plan
		      :steps (snlp-plan-steps plan)
		      :links (snlp-plan-links plan)
		      :unsafe (snlp-plan-unsafe plan)
		      :open (cons open-cond (remove open-cond (snlp-plan-open plan) :test #'equal))
		      :ordering (snlp-plan-ordering plan)
		      :bindings (snlp-plan-bindings plan)
		      :high-step (snlp-plan-high-step plan)))
      (values open-cond
	      (make-snlp-plan
	       :steps (snlp-plan-steps plan)
	       :links (snlp-plan-links plan)
	       :unsafe (snlp-plan-unsafe plan)
	       :open (remove open-cond (snlp-plan-open plan) :test #'equal)
	       :ordering (snlp-plan-ordering plan)
	       :bindings (snlp-plan-bindings plan)
	       :high-step (snlp-plan-high-step plan))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Resolving an unsafe link

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add constraints that prevent the bindings that
;;;  would clobber it.  The input is a link, the ID of the step that clobbers
;;;  it, and the bindings that clobber it.  The output is a list of new plans.
(defun separate (unsafe-link plan &optional (stream *standard-output*))
  (let ((new-plans nil)
	(neg-bind nil)
	(clobber-bind (third unsafe-link)))
    (dolist (bind clobber-bind)
      (if (eql (car bind) 'not)
	  (setf neg-bind (cdr bind))
	(setf neg-bind (list (list 'not bind))))
      (if (> *trace* 0.5) (format stream "~%   * Separate with ~a" neg-bind))
      (let ((new-bind (add-bind neg-bind
				(copy-bindings (snlp-plan-bindings plan)))))
	(when new-bind
	  (push (make-snlp-plan
		 :steps (snlp-plan-steps plan)
		 :links (snlp-plan-links plan)
		 :unsafe (snlp-plan-unsafe plan)
		 :open (snlp-plan-open plan)
		 :ordering (snlp-plan-ordering plan)
		 :bindings new-bind
		 :high-step (snlp-plan-high-step plan))
		new-plans))))
    new-plans))


;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add an ordering constraint so that the clobber
;;;  step comes before the establishing step.  The output is a list of new
;;;  plans.
(defun demote (unsafe-link plan &optional (stream *standard-output*))
  (let* ((clobber-id (cadr unsafe-link))
	 (id (caar unsafe-link))
	 (demotable (member clobber-id (possibly-prior id plan))))
    (if (> *trace* 1) (format stream "~%   Demote? step ~a" clobber-id))
    (if (> *trace* 0.5) 
	(if demotable (format stream "~%   * Demoting step ~a < ~a" clobber-id id)))
    (if demotable
	(list (make-snlp-plan :steps (snlp-plan-steps plan)
                             :links (snlp-plan-links plan)
                             :unsafe (snlp-plan-unsafe plan)
                             :open (snlp-plan-open plan)
                             :ordering (cons (list clobber-id id)
                                             (snlp-plan-ordering plan))
                             :bindings (snlp-plan-bindings plan)
                             :high-step (snlp-plan-high-step plan))))))
      
;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add an ordering constraint so that the clobber
;;;  step comes after the second step of the link.  The output is a list of new
;;;  plans.
(defun promote (unsafe-link plan &optional (stream *standard-output*))
  (let* ((clobber-id (cadr unsafe-link))
	 (link (car unsafe-link))
	 (id (third link))
	 (promotable (member id (possibly-prior clobber-id plan))))
    (if (> *trace* 1) (format stream "~%   Promote? step ~a" clobber-id))
    (if (> *trace* 0.5) 
	(if promotable (format stream "~%   * Promoting step ~a" clobber-id)))
    (if promotable
	(list (make-snlp-plan :steps (snlp-plan-steps plan)
                             :links (snlp-plan-links plan)
                             :unsafe (snlp-plan-unsafe plan)
                             :open (snlp-plan-open plan)
                             :ordering (cons (list id clobber-id)
                                             (snlp-plan-ordering plan))
                             :bindings (snlp-plan-bindings plan)
                             :high-step (snlp-plan-high-step plan))))))
      
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Adding a step

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;; Returns a list of pairs of new plans and step#.  The new plan
;;; establishes the condition by adding a new step, instantiated 
;;; from a step in *templates* with post-conditions that unify with
;;; the condition. 
;;; Modified by DSW 1/91 to reduce consing
(defun add-step (open-cond plan &optional (stream *standard-output*))
  (let ((new-plans nil)
	(new-plan nil)
	(condition (car open-cond))
	(id (cadr open-cond)))
    (dolist (templ *templates*) 
      (if (> *trace* 1) (format stream
				"~%New step? ~a" (snlp-step-action (car templ))))
      (let* ((new-step-num (+ (snlp-plan-high-step plan) 1))
             (step (instantiate-step (car templ) new-step-num))
             (templ-bind (add-bind 
			  (instantiate-bind (cadr templ) new-step-num)
			  (copy-bindings (snlp-plan-bindings plan)))))
	(when templ-bind
	  (dolist (add-cond (snlp-step-add step))
	    (let* ((new-bind (unify condition add-cond templ-bind)))
	      (cond
	       (new-bind
		(setf new-bind (car new-bind))
		(let ((new-bindings (add-bind new-bind
					      (copy-bindings templ-bind))))
		  (when new-bindings
		    (if (> *trace* 0.5) 
			(format
			 stream
			 "~%* New Step ~a  New bindings ~a  unify-count ~a"
			 (snlp-step-action step) new-bind *unify-count*))
		    (if (> *trace* 0) (format stream "S"))
		    
;		    (setf (snlp-step-precond step) (filter-preconds (snlp-step-precond step) new-bindings))
		    (setf (snlp-step-dele step) (expand-lisp (snlp-step-dele step) new-bindings))
		    (setf new-plan
		      (make-snlp-plan 
		       :steps (cons step (snlp-plan-steps plan))
		       :links (cons (list (snlp-step-id step) condition id)
				    (snlp-plan-links plan))
		       :ordering (if (eql id :goal)
				     (snlp-plan-ordering plan)
				   (cons (list (snlp-step-id step) id)
					 (snlp-plan-ordering plan)))
		       :open (nconc (new-step-open step)
				    (snlp-plan-open plan))
		       :bindings new-bindings
		       :high-step (snlp-step-id step)))
		    (setf (snlp-plan-unsafe new-plan)
		      (let ((l (test-link new-plan
					  (car (snlp-plan-links new-plan)))))
			(if l (append l (find-unsafe new-plan step))
			  (find-unsafe new-plan step))))
		    (setf new-plans (cons new-plan new-plans))))
		)))))))
  new-plans))


;;;;;;;;;;;;;;;;;;;
;;;  Checks to see if the lisp preconditions of this step are met;  if
;;;  not, then it does not treat it as a step.
(defun filter-preconds (precs binding)
  (loop for prec in precs
      when (or (not (eq 'dump::lisp (car prec)))
	       (stillvars (bind-variable prec binding))
	       (not (eval (bind-variable (cadr prec) binding))))
      collect (expand-lisp prec binding)))
		  

(defun stillvars (sexp)
  (cond ((atom sexp) (and (symbolp sexp)
			  (char= (schar (symbol-name sexp) 0) #\?)))
	(t (loop for item in sexp thereis (stillvars item)))))


;;;;;;;;;;;;;;;;;;;
;;;  Adds the preconditions of the new step as open conditions of the new plan.
(defun new-step-open (step)
  (let ((new-open nil))
    (dolist (precond (snlp-step-precond step))
      (setf new-open (cons 
			(list precond (snlp-step-id step))
			new-open)))
    new-open))


#| (defun new-step-open (step)
  (let ((new-open nil))
    (dolist (precond (snlp-step-precond step))
      (when (not (eq 'dump::lisp (car precond)))
	(setf new-open (cons 
			(list precond (snlp-step-id step))
			new-open))))
    new-open)) |#


(defun expand-lisp (sexp bind)
  (cond ((atom sexp) sexp)
	((eq (car sexp) 'dump::lisp) (eval (bind-variable (cadr sexp) bind)))
	(t (loop for s in sexp collect (expand-lisp s bind)))))

(defun expand-eval (sexp)
  (cond ((atom sexp) sexp)
	((eq (car sexp) 'dump::lisp) (eval (cadr sexp)))
	(t (loop for s in sexp collect (expand-eval s)))))
	 

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  After adding a new step each link of the new plan is 
;;;  tested to see if it is not clobbered.
(defun find-unsafe (plan step)
  (let ((new-unsafe nil))
    (dolist (link (snlp-plan-links plan) new-unsafe)
      (setf new-unsafe
	    (append (test-link plan link step)
		    new-unsafe)))
    new-unsafe))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6. Handling links

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tests whether a link is possibly clobbered by the delete conditions or
;;;  superceded by the add conditions of any steps possibly between the ID1
;;;  and ID2 of the link.  Returns nil if the link is safe, otherwise
;;;  returns a list of (link clobber-id clobber-bind)
(defun test-link (plan link &optional (st nil) (stream *standard-output*))
  (let ((new-unsafe nil)
	(bind2 (snlp-plan-bindings plan))
	(between-ids (possibly-between (car link) (third link) plan)))
    (if (> *trace* 1) 
        (format stream "~%Test link ~a with steps ~a" link between-ids))
    (dolist (step (if st (list st) (snlp-plan-steps plan)))
      (cond
	((member (snlp-step-id step) between-ids)
	 (if (> *trace* 2) 
             (format stream "~% Test step ~a" (snlp-step-id step)))
	 (dolist (dele-cond (snlp-step-dele step) new-unsafe)
	   (let ((clobber-bind (unify dele-cond (second link) bind2)))
	     (cond (clobber-bind
		    (setf clobber-bind (car clobber-bind))		   
		    (setf new-unsafe
			  (cons (list link (snlp-step-id step) clobber-bind)
				new-unsafe))

		    (if (> *trace* 0.5) 
			(format stream "~%   * New unsafe ~a   unify-count ~a"
				(list link (snlp-step-id step) clobber-bind)
				*unify-count*))
		    ))))
	 (dolist (add-cond (snlp-step-add step) new-unsafe)
	   (let ((clobber-bind (unify add-cond (second link) bind2)))
	     (cond (clobber-bind
		    (setf clobber-bind (car clobber-bind))		   
		    (setf new-unsafe
			  (cons (list link (snlp-step-id step) clobber-bind)
				new-unsafe))

		    (if (> *trace* 0.5) 
			(format stream "~%   * New unsafe ~a   unify-count ~a"
				(list link (snlp-step-id step) clobber-bind)
				*unify-count*))
		    )))))))
    new-unsafe))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns a list of step# that are possibly prior to the
;;; step being acheived, but not before the establishing step.
(defun possibly-between (estab-id id plan)
  (let ((not-after (list estab-id '0))
	(poss-between nil))
    (do* ((queue (list estab-id) (cdr queue))
	  (na-step estab-id (car queue)))
	((null queue) not-after)
      (setf not-after
	    (dolist (order (snlp-plan-ordering plan) not-after)
	      (cond ((eql na-step (cadr order))
		     (setf not-after (cons (car order)
					   not-after))
		     (setf queue (append queue
					 (list (car order)))))))))
    (dolist (prior-id (possibly-prior id plan))
      (if (not (member prior-id not-after))
	  (setf poss-between (cons prior-id poss-between))))
    poss-between))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns list of step-id's of steps possibly prior to a given
;;; step.  Possibly prior always includes the initial conditions.
;;; First build a list of steps constrained to be not prior by
;;; the ordering constraints.  Then add to possibly prior all
;;; steps that aren't in the not-prior list.
(defun possibly-prior (step-id plan)
  (if (not (eql step-id '0))
      (let ((not-prior (list step-id :goal))
	    (poss-prior (list '0)))
	(do* ((queue (list step-id) (cdr queue))
	      (np-step step-id (car queue)))
	     ((null queue) not-prior)
	  (setf not-prior
		(dolist (order (snlp-plan-ordering plan) not-prior)
		  (cond ((eql np-step (car order))
			 (unless (member (cadr order) not-prior)
                           (setf queue (append queue 
                                               (cdr order)))
                           (setf not-prior (append not-prior 
                                                   (cdr order)))))))))
	(dotimes (n (snlp-plan-high-step plan))
	  (if (not (member (+ n 1) not-prior))
	      (setf poss-prior (cons (+ n 1) poss-prior))))
	poss-prior)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates new plans to achieve an open condition.  Each possibly prior step
;;;  is tested to see if it has an add condition matching the open condition.
;;;  If so, a new plan is created with that link added.
(defun new-link (open-cond plan &optional (stream *standard-output*))
  (let ((new-plans nil)
	(new-plan nil)
	(new-link nil)
	(condition (car open-cond))
	(id (cadr open-cond))
	(prior-ids (possibly-prior (cadr open-cond) plan)))
    (if (> *trace* 2) (format stream "~%Possibly prior steps ~a" prior-ids))
    (dolist (step (snlp-plan-steps plan))
      (cond
       ((member (snlp-step-id step) prior-ids)
	(if (> *trace* 1) (format stream "~%New link?  step ~a" (snlp-step-id step)))
	(dolist (add-cond (snlp-step-add step))
	  (let ((new-bind (unify condition add-cond 
				 (snlp-plan-bindings plan))))
	    (cond
	     (new-bind
	      (setf new-bind (car new-bind))
	      (let ((new-bindings (add-bind new-bind
					    (copy-bindings
					     (snlp-plan-bindings plan)))))
		(if new-bindings
		    (progn
		      (setf new-link (list (snlp-step-id step) condition id))
		      (if (> *trace* 0.5) 
			  (format
			   stream "~% * New link ~a , bindings ~a    unify-count ~a"
			   new-link new-bind *unify-count*))
		      (if (> *trace* 0) (format stream "L"))
		      (setf new-plan
			(make-snlp-plan 
			 :steps (snlp-plan-steps plan)
			 :links (append (snlp-plan-links plan) (list new-link))
			 :ordering (if (or (eql id :goal) 
					   (eql (snlp-step-id step) '0))
				       (snlp-plan-ordering plan)
				     (cons (list (snlp-step-id step) id)
					   (snlp-plan-ordering plan)))
			 :open (snlp-plan-open plan) 
			 :bindings new-bindings
			 :high-step (snlp-plan-high-step plan)))		
		      (setf (snlp-plan-unsafe new-plan) 
			(append (test-link new-plan new-link) 
				(snlp-plan-unsafe new-plan)))
		      (setf new-plans (cons new-plan new-plans)))
		    
		  ))
	      )
	     )))
	(when (and (eq (snlp-step-id step) 0)
		   (eq (car condition) 'dump::lisp)
		   (not (stillvars (bind-variable condition (snlp::snlp-plan-bindings plan))))
		   (eval (bind-variable (cadr condition) (snlp::snlp-plan-bindings plan))))
	  (setf new-link (list (snlp-step-id step) condition id))
	  (if (> *trace* 0.5) 
	      (format
	       stream "~% * New link ~a , bindings ~a    unify-count ~a"
	       new-link new-bind *unify-count*))
	  (if (> *trace* 0) (format stream "L"))
	  (setf new-plan
	    (make-snlp-plan 
	     :steps (snlp-plan-steps plan)
	     :links (append (snlp-plan-links plan) (list new-link))
	     :ordering (if (or (eql id :goal) 
			       (eql (snlp-step-id step) '0))
			   (snlp-plan-ordering plan)
			 (cons (list (snlp-step-id step) id)
			       (snlp-plan-ordering plan)))
	     :open (snlp-plan-open plan) 
	     :bindings (snlp-plan-bindings plan)
	     :high-step (snlp-plan-high-step plan)))
	  (setf (snlp-plan-unsafe new-plan) 
	    (append (test-link new-plan new-link) 
		    (snlp-plan-unsafe new-plan)))
	  (setf new-plans (cons new-plan new-plans)))
	)))
    new-plans))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7. ranking partial plans

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;;  To find a plan with the least searching.  Ranking is based on the
;;;  number of unsafe plus the number of goals to be resolved plus the
;;;  number of steps.  Including the number of steps prevents creating
;;;  long plans that add cycles of moves getting no closer to the goal.
(defun rank (plan)
  (let ((num-steps (length (snlp-plan-steps plan)))
	(unsafe (length (snlp-plan-unsafe plan)))
	(open (length (snlp-plan-open plan))))
    (+ unsafe open num-steps)))

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;;  A slightly more agressive search, ignoring the number of steps
(defun rank1 (plan)
  (let ((unsafe (length (snlp-plan-unsafe plan)))
	(open (length (snlp-plan-open plan))))
    (+ unsafe open)))

(defun planning::myrank (plan)
  (loop for state
	in (snlp-plan-open plan)
	when (numberp (cadr state))
	sum (cadr state)))



(defun planning::snlpmiss (plan)
  (let* ((current (if (snlp-plan-open plan)
		      (expand-lisp (caar (snlp-plan-open plan)) (snlp-plan-bindings plan))
		    '(0 0 0 0 0 0 0 0)))
	 (mn (second current))
	 (ms (fourth current))
	 (cn (sixth current))
	 (cs (eighth current))
	 (goal (car (loop for step in (snlp-plan-steps plan)
			thereis (when (eql (snlp-step-id step) '0)
				  (snlp-step-add step)))))
	 (mng (second goal))
	 (msg (fourth goal))
	 (cng (sixth goal))
	 (csg (eighth goal)))
    
    (if (null (snlp-plan-open plan)) 0
      (if (or (> mn cn 0) (> ms cs 0)) 1000
	(+ (abs (- mn mng)) (abs (- ms msg)) (abs (- cn cng)) (abs (- cs csg))))))
  )

(defun planning::smiss (plan)
  (let* ((curr (find-if #'(lambda (item) (and (listp item) (eq 'dump::mn (car item))))
			 (strips::plan-pending-goals plan)))
	 (current (if curr (strips::expand-lisp
			    curr (strips::plan-bindings plan))
		    '(0 3 0 0 0 3 0 0)))
	 (mn (second current))
	 (ms (fourth current))
	 (cn (sixth current))
	 (cs (eighth current))
	 (goal (car (strips::plan-current-state plan)))
	 (mng (second goal))
	 (msg (fourth goal))
	 (cng (sixth goal))
	 (csg (eighth goal)))
    (if (null curr) -1
      (if (or (> mn cn 0) (> ms cs 0)) 1000
	(+ (abs (- mn mng)) (abs (- ms msg)) (abs (- cn cng)) (abs (- cs csg)))))))



(defun planning::hanoi (plan)
  
  (- 100 (loop for open in (bind-variable (snlp-plan-open plan) (snlp-plan-bindings plan))
      when (listp (car open)) count (member 'dump::aaa (car open)))))

(defun planning::shanoi (plan)
  (loop for step in (cdr (reverse (strips::plan-steps-taken-stack plan))) sum 
	(if (<= (+ (length (symbol-name (caddr step))) (length (symbol-name (cadr step)))) 6)
	    (- (length (symbol-name (caddr step))) (length (symbol-name (cadr step))))
	  0)))
	    

;;;;;;;;;;;;;;;;;;;;;;;;
;;; To find the plan with the fewest steps.  If a complete plan is found
;;; give it a lower rank than an incomplete plan with one fewer steps.
(defun rank2 (plan)
  (let ((num-steps (length (snlp-plan-steps plan)))
	(unsafe (length (snlp-plan-unsafe plan)))
	(open (length (snlp-plan-open plan))))
    (cond
      ((= 0 (+ unsafe open))
       (- num-steps 2))
      (t
       num-steps))))

(defun rank3 (plan)
  (let ((num-links (length (snlp-plan-links plan)))
	(unsafe (length (snlp-plan-unsafe plan)))
	(open (length (snlp-plan-open plan))))
    (+ unsafe open num-links)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  8. Statistics report routines

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Get today's date
(defun today ()
  (let ((d (multiple-value-list (get-decoded-time))))
    (format nil "~a/~a ~a at ~a:~a:~a"
            (nth 4 d) (nth 3 d) (nth 5 d) (nth 2 d) (nth 1 d) (nth 0 d))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Reset the global variables used for collecting statistics on the
;;;  planner.
(defun reset-snlp-vars ()
  (setf *nodes-visited* 0)
  (setf *unify-count* 0)
  (setf *compute-rank-unifies* 0)
  (setf *add-bind-count* 0)
  (setf *snlp-branch* 0)
  (setf *snlp-plans-created* 0)
  )

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  A divide routine that does not blow up on zero.
(defun dnoz (x y)
  (if (= y 0) (* 99999 99999 99999) (/ x y)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print out statistics from single run
(defun print-stat (s &optional (st *standard-output*) ignore)
  (declare (ignore ignore))
  (format st "~%~%~a (Init = ~2a ; Goals = ~2a) => ~a (~a steps)     CPU ~9a"
          (stat-algo s) (stat-num-init s) (stat-num-goal s)
          (if (stat-complete? s) "Win " "Lose")
          (stat-plan-len s) (stat-time s))
  (format st "~%     Nodes (V = ~4a; Q = ~4a; C = ~4a)             Branch ~10a"
          (stat-visited s) (stat-q-len s) (stat-created s)
          (stat-ave-branch s))
  (format st "~%     Working Unifies: ~25a       Bindings added: ~5a~%"
          (- (stat-unify-count s) (stat-rank-unifies s))
          (stat-add-bindings s)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 9. Print functions


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print value from hash table
(defun print-val (a b &optional (stream *standard-output*))
  (declare (ignore a))
  (format stream "~a " b))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print values in hash table
(defun print-tbl (tbl)
  (maphash #'print-val tbl))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print single binding for blocks world test data
(defun print-entry (id e &optional (stream *standard-output*))
  (declare (ignore id))
  (format stream "~%  ~5a has Value ~5a,  Synonyms ~a,  Constraints "
	  (bind-key e) (bind-value e) (bind-syn e))
  (print-tbl (bind-not e)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print list of bindings for blocks world test data
(defun print-bind (bind-tbl &optional (stream *standard-output*))
  (format stream "~%Bindings :")
  (maphash #'print-entry bind-tbl))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This suppresses printing of plan 
(defun no-print (plan)
  (declare (ignore plan)))


;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print function for Templates
(defun print-templates (&optional (templates *templates*)
				  (stream *standard-output*))
  (format stream "~&~%Templates:")
  (dolist (templ-n templates)
    (let ((action (snlp-step-action (car templ-n)))
	  (pre-cond (snlp-step-precond (car templ-n)))
	  (add (snlp-step-add (car templ-n)))
	  (dele (snlp-step-dele (car templ-n)))
	  (bind (cadr templ-n)))
      (format stream
	      "~&~a~%  Pre  : ~a~%  Add  : ~a~%  Dele : ~a~%  Bind : ~a~%"
	      action pre-cond add dele bind))))
    

