;;; -*- Mode: LISP; Package: SEARCH; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   search.cl
;;; Short Desc: Search routines for planning
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   24.1.92 DTA
;;; Author:     Hank Wan
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------
;;;; Best first search

(in-package :search)

(export '(*multiple-value-goal* *multiple-values*))

(import '(gin:format-display))

(defparameter *best-first-search* 'bestf-search)
(defparameter *user-stop* nil)
(defvar *multiple-value-goal* nil)
(defvar *multiple-values* nil)

(defstruct (search-tree
	    (:print-function print-search-tree))
  node
  parent
  (children nil)
  rank
  n-traverse				; num order in which tree was traversed
  x y					; for graphical interface
  )

; Originally from SNLP, U. of Washington.  See copyright notice elsewhere.
; Modified by hank, 8/91

;;  A simple best first search strategy.  The search will only generate
;;  up to LIMIT states.
;;  Calls search-trace for debugging purposes.
(defun bestf-search (initial-state daughters-fn goal-p rank-fn limit)
  (let ((branches nil)			; compute average branch factor
	(n-traverse 0)
	(planning-tree (make-search-tree :node initial-state :rank 0)))
    (setf *user-stop* nil)
    (do* ((current-entry planning-tree (car search-queue))
          (current-state initial-state (when current-entry
					 (search-tree-node current-entry)))
	  (search-queue nil (cdr search-queue)))
	((or *user-stop*
	     (null current-state)
	     (if planning::*multiple-value-goal*
		 (not (or (funcall goal-p current-state) (funcall daughters-fn current-state)))
	       (when (funcall goal-p current-state)
		 (setf (search-tree-n-traverse current-entry) (incf n-traverse))))
	     (> 0 limit))
	 (values current-state
		 planning-tree
					;		 (if (null branches) 0
					;		   (dnoz (apply #'+ branches) (length branches)))
					;		 (length search-queue)
		 ))
      (let ((children (funcall daughters-fn current-state)))
	(setf (search-tree-n-traverse current-entry) (incf n-traverse))
        (setf limit (- limit (length children)))
	(mapl #'(lambda (children)
		  (let ((child (car children)))
		    (setf (car children)
		      (make-search-tree :node child
					:parent current-entry
					:rank (funcall rank-fn child)))))
	      children)
	(setf (search-tree-children current-entry) children)
	(search-trace current-state search-queue current-entry)
        (setf search-queue (merge 'list search-queue
				  (sort (copy-tree children) ; destructive sort
					#'< :key #'search-tree-rank)
				  #'< :key #'search-tree-rank))
        (push (length children) branches)
	(if (and planning::*multiple-value-goal* (funcall goal-p current-state))
	    (push (unify::apply-sub planning::*multiple-value-goal*
				    (strips::plan-bindings current-state))
		  planning::*multiple-values*) 
	  )))))

;;;; Depth first search

(defparameter *depth-first-search* 'depth-first-search)

; H. Wan, 8/91: modeled after BFS in SNLP

(defun depth-first-search (initial-state children-fn goal-p rank-fn limit)
  (declare (ignore rank-fn))
  (let ((n-traverse 0)
	(planning-tree (make-search-tree :node initial-state)))
    (setf *user-stop* nil)
    (do* ((current-entry planning-tree (car search-queue))
          (current-state initial-state (when current-entry
					 (search-tree-node current-entry)))
	  (search-queue nil (cdr search-queue)))
	((or *user-stop*
	     (null current-state)
	     (if planning::*multiple-value-goal*
		 (not (or (funcall goal-p current-state) (funcall children-fn current-state)))
	       (when (funcall goal-p current-state)
		 (setf (search-tree-n-traverse current-entry) (incf n-traverse))))
	     (> 0 limit))
	 (values current-state
		 planning-tree
		 ))
      (let ((children (funcall children-fn current-state)))
	(setf (search-tree-n-traverse current-entry) (incf n-traverse))
	(setf limit (- limit (length children)))
	(mapl #'(lambda (children)
		  (setf (car children)
		    (make-search-tree :node (car children)
				      :parent current-entry)))
	      children)
	(setf (search-tree-children current-entry) children)
	(search-trace current-state search-queue current-entry)
	(setf search-queue (append children search-queue))
	(if (and planning::*multiple-value-goal* (funcall goal-p current-state))
	    (push (unify::apply-sub planning::*multiple-value-goal*
				    (strips::plan-bindings current-state))
		  planning::*multiple-values*) 
	  )
	))))

;;;; Breadth first search

(defparameter *breadth-first-search* 'breadth-first-search)

; H. Wan, 8/91: modeled after BFS in SNLP

(defun breadth-first-search (initial-state children-fn goal-p rank-fn limit)
  (declare (ignore rank-fn))
  (let ((n-traverse 0)
	(planning-tree (make-search-tree :node initial-state)))
    (setf *user-stop* nil)
    (do* ((current-entry planning-tree (car search-queue))
          (current-state initial-state (when current-entry
					 (search-tree-node current-entry)))
	  (search-queue nil (cdr search-queue)))
	((or *user-stop*
	     (null current-state)
	     (if planning::*multiple-value-goal*
		 (not (or (funcall goal-p current-state) (funcall children-fn current-state)))
	       (when (funcall goal-p current-state)
		 (setf (search-tree-n-traverse current-entry) (incf n-traverse))))
	     (> 0 limit))
	 (values current-state
		 planning-tree
		 ))
      (let ((children (funcall children-fn current-state)))
	(setf (search-tree-n-traverse current-entry) (incf n-traverse))
	(setf limit (- limit (length children)))
	(mapl #'(lambda (children)
		  (setf (car children)
		    (make-search-tree :node (car children)
				      :parent current-entry)))
	      children)
	(setf (search-tree-children current-entry) children)
	(search-trace current-state search-queue current-entry)
	(setf search-queue (append search-queue children))
	(if (and planning::*multiple-value-goal* (funcall goal-p current-state))
	    (push (unify::apply-sub planning::*multiple-value-goal*
				    (strips::plan-bindings current-state))
		  planning::*multiple-values*) 
	  )))))

;;;; Search Trace

(defun search-trace (current queue planning-tree)
  (if planning::*verbose* (format-display planning::*verbose-disp* "~%node ~s: ~d nodes"
	  (search-tree-n-traverse planning-tree)
	  (length (search-tree-children planning-tree)))))

(defun big (plan)
  (let ((goals
	 (mapcar (function (lambda (x)
			     (cond ((op-p x)
				    (apply-sub (opname x) (strips::plan-bindings plan))
				    )
				   (t (apply-sub x (strips::plan-bindings plan))))))
		 (strips::plan-pending-goals plan))))
    (setf *save* goals)
    (if (not (= (length goals) (length (remove-duplicates goals :test #'same-or-back))))
	1000
      (length (strips::plan-pending-goals plan)))))


(defun same-or-back (a b) (or (equal a b) (and (equal (car a) 'dump::from)
					       (equal (car b) 'dump::from)
					       (equal (cadr a) (nth 3 b))
					       (equal (nth 3 a) (cadr b)))))

;;;; Search Tree Routines

(defun print-search-tree (planning-tree stream ignore)
  (declare (ignore ignore))
  (format stream "<T~s"
	  (search-tree-n-traverse planning-tree))
  (when (search-tree-rank planning-tree)
    (format stream "(rank=~d)" (search-tree-rank planning-tree)))
  (format stream ">"))

(defun branching-factor (planning-tree)
  (multiple-value-bind (branches nodes)
      (n-tree-branches-and-nodes planning-tree)
    (if (zerop branches) 0
      (dnoz branches nodes))))

(defun n-tree-branches-and-nodes (planning-tree)
  (cond ((not (search-tree-visited planning-tree))
	 (values 0 0))
	(t (let ((branches (length (search-tree-children planning-tree)))
		 (nodes 1))
	     (mapc #'(lambda (child)
		       (multiple-value-bind (b n)
			   (n-tree-branches-and-nodes child)
			 (incf branches b)
			 (incf nodes n)))
		   (search-tree-children planning-tree))
	     (values branches nodes)))))

(defun quick-print-tree (planning-tree &optional (indent 0))
  (format t "~%~vtT~d" (* 2 indent) (search-tree-n-traverse planning-tree))
  (let ((new-indent (1+ indent)))
    (dolist (e (search-tree-children planning-tree))
      (quick-print-tree e new-indent))))

;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
