;;; -*- Mode: LISP; Package: PLANNING; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   bworld.cl
;;; Short Desc: Graphics for manipulating the blocks world
;;; Version:    0.1
;;; Status:     Provisional
;;; Last Mod:   14.1.91 DTA
;;; Author:     Winston, Gambardella, Allemang
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;




;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------




(in-package :planning)

(export '(hand brick wedge ball support-table put-on block-display *blocksize*))

(defparameter *blocksize* 20)
(defparameter *handwait* 0.5)

;;;;--------------------------------------
;;; evaluate it first !!!!

(defun remove-specializers (parameter-list)
  (mapcar #'(lambda (element)
	      (if (listp element)
		  (first element)
		element))
	  parameter-list))
;;;;--------------------------------------


(defclass animate-display  (display)
	  ((start-state :initform nil :accessor start-state)))

(defmethod start-state (a) nil)

(defclass block-display (animate-display)
	  ((hand :initarg :hand :initform  nil :accessor hand)
	   (hand-home :initarg :hand-home :initform '(10 100) :accessor hand-home)
	   (init-plan :initform nil :accessor init-plan)
	   ))


(defclass basic-block ()
  ((name :initarg :name  :accessor block-name)
   (width :initarg :width :accessor block-width)
   (height :initarg :height :accessor block-height)
   (position :initarg :position :accessor block-position)
   (supported-by :initform nil :accessor block-supported-by)
   (display :initform nil :accessor display :initarg :display)))

	  

(defclass movable-block (basic-block)
  ((width :initform *blocksize* :initarg :width :accessor block-width)
   (height :initform *blocksize* :initarg :height :accessor block-height)))


(defclass load-bearing-block (basic-block)
  ((support-for :accessor block-support-for :initform nil)))


(defclass brick (movable-block load-bearing-block)())


(defclass wedge (movable-block)())


(defclass ball (movable-block)())


(defclass support-table (load-bearing-block)
	  ((width :initform (* 7 *blocksize*) :initarg :width :accessor block-width)
	   (height :initform (* 2 *blocksize*) :initarg :height :accessor block-height)))

(defclass hand ()
  ((name :initarg :name :accessor hand-name)
   (position :initarg :position :accessor hand-position )
   (grasping :initform nil :accessor hand-grasping)
   (display :initform nil :accessor display :initarg :display)))






;;;===========================================================================
;;; methods
;;;===========================================================================

(defmethod block-support-for ((object basic-block))
  nil)

(defmethod put-on ((object movable-block)(support basic-block))
  (if (get-space object support)
      (and (grasp object)
	   (move-block object support)
	   (ungrasp object))
    (format t "~&Sorry, there is no room for ~a on ~a."
	    (block-name object)
	    (block-name support))))

(defmethod put-on (obj1 obj2) (documentation-print
			       (format nil
				       "~&Sorry, you cannot stack ~a on ~b,"
				       (block-name obj1)
				       (block-name obj2))))

(defmethod get-space ((object movable-block)(support basic-block))
  (or (find-space object support)
      (make-space object support)))


;;; the hand may be grasping the correct object
;;; the top of the object must be clear


(defmethod grasp ((object movable-block))
  (unless (eq (hand-grasping (hand (display object))) object)
    (when (block-support-for object)(clear-top object))
    (when (hand-grasping (hand (display object)))
      (get-rid-of  (hand-grasping (hand (display object)))))
    (format t "~&Move hand to pick up ~a at location ~a." 
	    (block-name object)
	    (top-location object))
    (setf (hand-position (hand (display object)))(top-location object))
    (format t "~&Grasp ~a." (block-name object))
    (setf (hand-grasping (hand (display object))) object))
  t)

(defmethod ungrasp ((object movable-block))
  (when (block-supported-by object)
    (format t "~&Ungrasp ~a." (block-name object))
    (setf (hand-grasping (hand (display object))) nil)
    (setf (hand-position (hand (display object))) (hand-home (display object)))
    t))


#| (defmethod get-rid-of ((object movable-block))
  (put-on object support-table)) |#


;;; In the original blocks world graphics, a bit of planning was done
;;; by the robot itself (i.e., the graphics).   If one tries to place
;;; an object on another, but there is no room, then this function is
;;; called.   It presumes that there is a table running around for
;;; getting rid of things.  Our planners are supposed to be able to
;;; deal with things like that.  Intermediate plans from TWEAK have
;;; not  yet sorted out their ordering properties, so they sometimes
;;; make such a request.  This reports that.

(defmethod get-rid-of ((object movable-block))
  (documentation-print (format nil "Unable to carry out plan request
because block ~a was in the way" object)))


(defmethod make-space ((object movable-block)(support basic-block))
  (dolist (obstruction (block-support-for support))
    (get-rid-of obstruction)
    (let ((space (find-space object support)))
      (when space (return space)))))


(defmethod clear-top ((support load-bearing-block))
  (dolist (obstacle (block-support-for support) t)
    (get-rid-of obstacle)))


(defmethod move-block ((object movable-block)(support basic-block))
  (remove-support object)
  (let ((newspace (get-space object support)))
    (format t "~&Move ~a to top of ~a at location ~a."
	    (block-name object)
	    (block-name support)
	    newspace)
    (setf (block-position object) newspace)
    (setf (hand-position (hand (display object)))(top-location object)))
  (add-support object support)
  t)


(defmethod remove-support ((object movable-block))
  (let ((support (block-supported-by object)))
    (when support
      (setf (block-support-for support)
	(remove object (block-support-for support)))
      (setf (block-supported-by object) nil)
      t)))


(defmethod add-support ((object movable-block)(support basic-block))
  t)


(defmethod add-support ((object movable-block)(support load-bearing-block))
  (push object (block-support-for support))
  (setf (block-supported-by object) support)
  t)
  


;;;===========================================================================
;;; space calculus
;;;===========================================================================

(defun find-space (object support)
  (dotimes (offset (+ 1 (- (block-width support)
			   (block-width object))))
    (unless (intersections-p object offset
			     (first (block-position support))
			     (block-support-for support))
      (return (list (+ offset (first (block-position support)))
		    (+ (second (block-position support))
		       (block-height support)))))))

(defun intersections-p (object offset base obstacles)
  (dolist (obstacle obstacles)
    (let* ((ls-proposed (+ offset base))
	   (rs-proposed (+ ls-proposed (block-width object)))
	   (ls-obstacle (first (block-position obstacle)))
	   (rs-obstacle (+ ls-obstacle (block-width obstacle))))
      (unless (or (>= ls-proposed rs-obstacle)
		  (<= rs-proposed ls-obstacle))
	(return t)))))


(defun top-location (object)
  (list (+ (first (block-position object))
	   (/ (block-width object) 2))
	(+ (second (block-position object))
	   (block-height object))))


;;;===========================================================================
;;; graphics interface
;;;===========================================================================
    

(defun crea-e-dimensiona (wstream)
  (reshape wstream (get-region *root-window*
			       :init-region
			       (window-stream-region wstream))))

(defun left-ko (wstream)
  (modify-window-stream-method wstream :frame-left-button-down
			       :after 'pre-flush))


(defun pre-flush (wstream &rest ignore)
  (flush wstream))




(defun start-window ()
  (initialize-common-windows)
  (setf *my-win* (make-window-stream :left 470 :bottom 590
                                     :height 200 :width 300
                                     :title "blocks"))
  (left-ko *my-win*)
  (modify-window-stream-method *my-win* :activate :after 'crea-e-dimensiona)
  (setf (window-stream-operation *my-win*) 8)
  (activate *my-win*))


(defun mystart ()
  (setf mydisp (make-instance 'display  :left 470 :bottom 590 :height 200 :width 300 :title "blocks"))
  )

;;;===========================================================================
;;; draw method
;;;===========================================================================

(defmethod draw ((object ball))
  (let ((start-x  (first (block-position object)))
	(start-y  (second (block-position object)))
	(radius (/  (block-width object) 2)))
  (draw-circle (display object) 
		  (+ start-x radius) 
		  (+ start-y radius)
		  radius
		  :brush-width 2 :operation boole-xor)
  (show object)))

(defmethod draw ((object load-bearing-block))
  (draw-rectangle  (display object) 
		      (+ 2 (first (block-position object)))
		      (-  (second (block-position object)) 2)
		      (- (block-width object) 2)
		      (- (block-height object) 2)
		      :brush-width 2 :operation boole-xor)
  (show object))

(defmethod draw ((object wedge))
  (let ((start-x  (first (block-position object)))
	(start-y  (second (block-position object)))
	(up (top-location object)))
    (draw-triangle (display object) 
		      start-x start-y
		      (first up)(second up)
		      (+ start-x  (block-width object)) start-y
		      :brush-width 2 :operation boole-xor)
    (show object)))
	
(defmethod show ((object basic-block))
  (write-display (display object)
		 (format nil "~a" (block-name object))
		 (+ (- (floor (font-string-width (font (display object))
						 (format nil "~a" (block-name object))
						 ) 2 ))
		    (floor (block-width object) 2)
		    (first (block-position object)))
		 (+ (- (floor (font-character-height (font (display object))
						     ) 2))
		    (floor (block-height object) 2)
		    (second (block-position object)))
		 :operation boole-xor))

(defmethod draw ((object hand))
  (draw-rectangle (display object) 
	       (first (hand-position object))
	       (second (hand-position object))
	       3 3
	       :brush-width 1 :operation boole-xor))



(defmethod initialize-instance :after ((instance basic-block) &rest ignore)
	   (draw instance))



(defmethod (setf block-position) :before (new-position (object basic-block))
	   (draw object))



(defmethod (setf block-position) :after (new-position (object basic-block))
	   (draw object))



(defmethod (setf hand-position) :before (new-position (object hand))
	   (draw object))

(defmethod (setf hand-position) :after (new-position (object hand))
	   (draw object)
	   #| (read-char) |#
	   (mp:process-wait-with-timeout "Blocks world" *handwait* #'(lambda nil nil)))


;;;===========================================================================
;;; 2nd version
;;;===========================================================================

(defmethod move-block ((object movable-block)(support basic-block))
;  (remove-support object)
  (let ((newspace (get-space object support)))
    (format t "~&Move ~a to top of ~a at location ~a."
	    (block-name object)
	    (block-name support)
	    newspace)
    (setf (block-position object) newspace)
    (setf (hand-position (hand (display object)))(top-location object)))
; (add-support object support)
  t)


(defmethod move-block :before ((object movable-block) ignored-parameter)
  (let ((support (block-supported-by object)))
    (when support
      (format t "~%Removing support relations between ~a and ~a."
	      (block-name object)(block-name support))
      (setf (block-support-for support)
	(remove object (block-support-for support)))
      (setf (block-supported-by object) nil)
      t)))

(defmethod move-block :after ((object movable-block)
			      (support load-bearing-block))
  (format t "~%Adding support relations between ~a and ~a."
	  (block-name object)(block-name support))
  (setf (block-support-for support)(cons object (block-support-for support)))
  (setf (block-supported-by object) support))

	       


;;;===========================================================================
;;; 3rd version
;;;===========================================================================

(defmethod move-block ((object movable-block)(support basic-block))
; (remove-support object)
  (let ((newspace (get-space object support)))
    (format t "~&Move ~a to top of ~a at location ~a."
	    (block-name object)
	    (block-name support)
	    newspace)
    (setf (block-position object) newspace))
; (setf (hand-position (hand (display object)))(top-location object))
; (add-support object support)
  t)


(defmethod (setf block-position) :after (new-position (object basic-block))
	   (draw object)
	   (setf (hand-position (hand (display object)))(top-location object)))


;;;===========================================================================
;;; Answering questions
;;;===========================================================================

(defclass node ()
  ((parent :initform nil :accessor node-parent)
   (children :initform nil :accessor node-children )
   (action :initform nil :accessor node-action)))

  
(defvar *current-node* (make-instance 'node))

(defmethod attach-parent ((child node)(parent node))
  (setf (node-parent child) parent)
  (setf (node-children parent)
    (append (node-children parent)(list child))))

(defmethod attach-action ((node node) action)
  (setf (node-action node) action))


(defmethod get-rid-of :around ((object movable-block))
  (let* ((parent *current-node*)
	 (*current-node* (make-instance 'node))
	 (primary-method-value (call-next-method)))
    (when primary-method-value
;      (attach-parent *current-node* parent)
      #| (attach-action *current-node* (list 'get-rid-of object)) |#)
    primary-method-value))


(defun remove-specializers (parameter-list)
  (mapcar #'(lambda (element)
	      (if (listp element)
		  (first element)
		element))
	  parameter-list))

(defmacro define-history-method (name parameters &rest body)
  `(defmethod ,name :around ,parameters
       (let* ((parent *current-node*)
	      (*current-node* (make-instance 'node))
	      (primary-method-value (call-next-method)))
	 (when primary-method-value
;	   (attach-parent *current-node* parent)
	   #| (attach-action *current-node* 
			  (list ',name
				,@(remove-specializers parameters))) |#)
	 primary-method-value)))


(define-history-method put-on ((object movable-block)
			       (support basic-block)))

(define-history-method get-rid-of ((object movable-block)))

(define-history-method make-space ((object movable-block)
				   (support basic-block)))
 
(define-history-method clear-top ((support load-bearing-block)))
 
(define-history-method move-block ((object movable-block)
				   (support basic-block)))

(define-history-method grasp ((object movable-block)))
		
(define-history-method ungrasp ((object movable-block)))

(defun show-simple-tree (node &optional (indentation 0))
  (format t "~&~vt~a"
	  indentation
	  (or (node-action node) 'top-of-tree))
  (dolist (node (node-children node))
    (show-simple-tree node (+ 2 indentation))))

(defmethod print-object ((x basic-block) stream)
  (format stream "~a" (block-name x)))

;;; Examples
;;;-------------------------------


#| (setf dd (make-instance 'block-display
                       :title "Blocks"
		       :width 300
		       :height 150
		       :borders 1
		       :left 200
		       :bottom 680
                       ))
 |#

#| (setq hand (make-instance 'hand :name 'hand :position '(0 6) :display dd)) |#

#| (setf (hand dd) hand) |#

#| (setf tab (make-instance 'support-table :name 'support-table :width 200 :height 20 
				:position '(0 0) :display dd)) |#

#| (setf a (make-instance 'brick :name 'a :width 20 :height 20 :position '(0 20) :display dd)) |#
#| (setf b (make-instance 'brick :name 'b :width 20 :height 20 :position '(40 20) :display dd )) |#

#| (setf c (make-instance 'brick :name 'c :width 40 :height 40 :position '(80 20) :display dd)) |#






;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
