;;; -*- Mode: LISP; Package: NP; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   np-graphics.cl
;;; Short Desc: graphics for neural networks
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   4.2.92 - TW
;;; Author:     Thomas Wehrle
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :np)


(defclass net-graphics ()
  ((net :accessor net)
   (a-disp :accessor a-disp :initform nil)
   (aa-disp :accessor aa-disp :initform nil)
   (w-disp :accessor w-disp :initform nil)
   (ww-disp :accessor ww-disp :initform nil)
   (net-display :accessor net-display :initform nil)
   (weight-display :accessor weight-display :initform nil)
   (nbr-of-layers :accessor nbr-of-layers)
   (nbr-of-nodes :accessor nbr-of-nodes)
   (matrix-len :accessor matrix-len)
   (w-max-button :accessor w-max-button :initform nil)
   (border :accessor border)
   (w-box-width :accessor w-box-width)
   (w-max :accessor w-max)
   (w-disp-width :accessor w-disp-width)
   (w-disp-height :accessor w-disp-height)
   (w-positions :accessor w-positions)
   (w-tabs :accessor w-tabs)
   (a-box-width :accessor a-box-width)
   (a-max :accessor a-max)
   (a-disp-width :accessor a-disp-width)
   (a-disp-height :accessor a-disp-height)
   (a-positions :accessor a-positions)))


(defmethod init ((graphics net-graphics) (net feed-forward-net-class)
		 &key
		 (max-activity 1)
		 (max-weight 1)
		 (border 30)
		 (x-gab 53)
		 (y-gab 118)
		 (gab 20)
		 (w-box-width 14)
		 (a-box-width 20))
  (let* (a-box-offset max-nodes-in-layer
	 (w-box-offset (- (/ w-box-width 2))))
    (setf (nbr-of-layers graphics) (length (get-node-seq-names net))
	  (nbr-of-nodes graphics) (mapcar 
				   (function (lambda (node-seq-name)
					       (length (get-node-seq net node-seq-name))))
				   (reverse (get-node-seq-names net)))
	  max-nodes-in-layer (apply #'max (nbr-of-nodes graphics))
	  a-box-width (min (truncate  (- (a-disp-width graphics) (* 2 border)) (* max-nodes-in-layer 2))
			   (truncate  (- (a-disp-height graphics) (* 2 border)) (* (nbr-of-layers graphics) 4)))
	  x-gab (+ a-box-width
		   (truncate (- (a-disp-width graphics) (* a-box-width max-nodes-in-layer) border)
			     (1+ max-nodes-in-layer)))
	  y-gab (+ a-box-width
		   (truncate (- (a-disp-height graphics) (* a-box-width (nbr-of-layers graphics)) border)
			     (nbr-of-layers graphics)))
			 
	  a-box-offset (/ a-box-width 2)
	  (net graphics) net)
    (setf *junk* graphics)
    (setf (matrix-len graphics) (apply #'+ (nbr-of-nodes graphics)))
    (setf (w-disp-width graphics) (+ (* 2 border)
				     (* (- (matrix-len graphics) 1)
					gab))
	  (w-disp-height graphics) (+ (* 2 border)
				      (* (- (matrix-len graphics) 1)
					 gab))
	  (a-box-width graphics) a-box-width
	  (w-box-width graphics) w-box-width
	  (a-max graphics) max-activity
	  (w-max graphics) max-weight
	  (border graphics) border

	  (a-disp-height graphics) (+ (* 2 border)
				      (* (- (nbr-of-layers graphics) 1)
					 y-gab))

	  )
	  (setf (a-positions graphics) (let ((nbr-of-layer 0)
					     (seq-names (reverse (get-node-seq-names net))))
					 (mapcar
					  #'(lambda (nbr-per-layer)
					      (incf nbr-of-layer)
					      (do ((count 1 (1+ count))
						   (nodes (get-node-seq net (nth (1- nbr-of-layer) seq-names)) (rest nodes))
						   (pos-list nil)
						   (layer-offset (/ (* (- (apply #'max (nbr-of-nodes graphics))
									  nbr-per-layer)
								       x-gab) 2)))
						  ((eq count (1+ nbr-per-layer)) (reverse pos-list))
						(setf pos-list
						  (cons
						   (list
						    (+ (round border 2) (* (1- count) x-gab) a-box-offset layer-offset)
						    (+ (round border 2) (* (1- nbr-of-layer) y-gab) a-box-offset)
						    (first nodes))
						   pos-list))))
					  (nbr-of-nodes graphics))))
	  (setf (w-positions graphics) (make-array (list (matrix-len graphics) 
							 (matrix-len graphics))
						   :adjustable nil
						   :initial-element nil))
	  (setf (w-tabs graphics) (make-array (list (matrix-len graphics) 3)
					      :adjustable nil
					      :initial-element nil))
	  (let ((all-nodes (apply #'append
				  (mapcar 
				   (function (lambda (node-seq-name)
					       (get-node-seq net node-seq-name)))
				   (reverse (get-node-seq-names net))))))
	    (dotimes (i (matrix-len graphics))
	      (let ((x-node (nth i all-nodes)))
		(dolist (connection (in-connections x-node))
		  (let ((j (position (from-node connection) all-nodes)))
		    (setf (aref (w-positions graphics)
				i
				j)
		      (list
		       (+ border (* j gab) w-box-offset)
		       (+ border (* i gab) w-box-offset)
		       connection)))))))
	  (dotimes (i (matrix-len graphics))
	    (setf (aref (w-tabs graphics) i 0)
	      (+ border (* i gab) w-box-offset))
	    (setf (aref (w-tabs graphics) i 1)
	      (+ border (* i gab) w-box-offset w-box-width))
	    (setf (aref (w-tabs graphics) i 2) nil))
	  (let ((pos 0))
	    (setf (aref (w-tabs graphics) pos 2)
	      (+ border (* pos gab)))
	    (dolist (nbr (butlast (nbr-of-nodes graphics)))
	      (incf pos nbr)
	      (setf (aref (w-tabs graphics) pos 2)
		(+ border (* pos gab)))))))


(defun node-info (node)
  (documentation-print (format nil "Activation is ~a" (activation node))))


(defun toggle-node (node)
  (setf (activation node) (- 1.0 (activation node)))
  (setf (output node) (compute-output node)))


(defun connection-info (connection)
  (documentation-print (format nil "Weight is ~a" (weight connection))))


(defmethod open-activation-display ((graphics net-graphics)
				    &key
				    (arrows-p nil)
				    (title "Topology")
				    (left 100)
				    (bottom 100)
				    (width (a-disp-width graphics))
				    (height (a-disp-height graphics))
				    (parent *root-window*))
  (setf (aa-disp graphics)
    (make-display :title title
		  :parent parent
		  :width (+ width cwex::*static-scroll-bar-width*)
		  :height (+ height cwex::*static-pann-bar-height*)
		  :left left
		  :bottom bottom))
  (setf (a-disp graphics)
    (make-display :title nil
		  :parent (aa-disp graphics)
		  :width width
		  :height height
		  :inner-width (max width (a-disp-width graphics))
		  :inner-height (max height (a-disp-height graphics))
		  :left cwex::*static-scroll-bar-width*
		  :bottom cwex::*static-pann-bar-height*))
  (cwex::make-static-pann-bar (window (a-disp graphics)))
  (cwex::make-static-scroll-bar (window (a-disp graphics)))

  (let ((width (a-box-width graphics)))
    (dolist (layers (a-positions graphics))
      (dolist (def layers)
	(let* ((x (round (first def)))
	       (y (round (second def)))
	       (n (third def))
	       (r (make-active-region (a-disp graphics) :left x :bottom y :width width :height width :active t)))
	  (add-active-region-method r :left-button-down :after (function (lambda (a b c)
									   (declare (ignore a b c))
									   (node-info n))))
	  (add-active-region-method r :middle-button-down :after (function (lambda (a b c)
									   (declare (ignore a b c))
									   (toggle-node n)
									   (update-net *demo-net* )
									   (show-activities *graphics* :arrows-p *connection-p*))))))))
  (show-activities graphics :arrows-p arrows-p))


(defmethod open-weight-display ((graphics net-graphics)
				&key
				(title "Weights")
				(left 100)
				(bottom 100)
				(width (w-disp-width graphics))
				(height (w-disp-height graphics))
				(parent *root-window*))
  (setf (ww-disp graphics)
    (make-display :title title
		  :parent parent
		  :width (+ width cwex::*static-scroll-bar-width*)
		  :height (+ height cwex::*static-pann-bar-height*)
		  :left left
		  :bottom bottom))
  (setf (w-disp graphics)
    (make-display :title nil
		  :parent (ww-disp graphics)
		  :width width
		  :height height
		  :inner-width (max width (w-disp-width graphics))
		  :inner-height (max height (w-disp-height graphics))
		  :left cwex::*static-scroll-bar-width*
		  :bottom cwex::*static-pann-bar-height*))
  (cwex::make-static-pann-bar (window (w-disp graphics)))
  (cwex::make-static-scroll-bar (window (w-disp graphics)))

  (let ((width (w-box-width graphics)))
    (dotimes (i (matrix-len graphics))
      (dotimes (j (matrix-len graphics))
	(let ((def (aref (w-positions graphics) i j)))
	  (when def
	    (let* ((x (first def))
		   (y (second def))
		   (c (third def))
		   (r (make-active-region (w-disp graphics) :left x :bottom y :width width :height width :active t)))
	      (add-active-region-method r :left-button-down :after (function (lambda (x y z)
									       (declare (ignore x y z))
									       (connection-info c))))))))))
  (show-weights graphics))


(defmethod close-activation-display ((graphics net-graphics))
  (close-display (aa-disp graphics)))


(defmethod close-weight-display ((graphics net-graphics))
  (close-display (ww-disp graphics)))


(defun find-definition (node graphics)
  (dolist (layers (a-positions graphics))
    (let ((def (dolist (def layers)
		 (when (equal (third def) node) (return def)))))
      (when def (return def)))))


(defmethod show-activities ((graphics net-graphics)
			    &key (arrows-p nil))
  (let* ((box-width (a-box-width graphics))
	 (box-middle (/ box-width 2))
	 (max (a-max graphics))
	 (disp (a-disp graphics)))
    (clear-display disp)
    (dolist (layers (a-positions graphics))
      (dolist (def layers)
	(let* ((x (first def))
	       (y (second def))
	       (activity (activation (third def)))
	       (width (* (sqrt (/ activity max)) box-width))
	       (overflow nil))
	  (when (> width box-width)
	    (setf width box-width)
	    (setf overflow t))
	  (draw-rectangle disp x y box-width box-width)
	  (draw-filled-rectangle disp x y width width)
	  (when overflow
	    (draw-circle disp
			 (+ x box-middle)
			 (+ y box-middle)
			 3
			 :color white)))))
    (when arrows-p
      (dolist (layers (a-positions graphics))
	(dolist (def layers)
	  (when (in-connections (third def))
	    (let* ((x2 (+ (first def) box-middle))
		   (y2 (second def)))
	      (dolist (in-node (get-from-nodes (third def)))
		(let* ((definition (find-definition in-node graphics))
		       (x1 (+ (first definition) box-middle))
		       (y1 (+ (second definition) box-width)))
		  (draw-line disp x1 y1 x2 y2 :arrow t))))))))
    (show-activities-net-specifics graphics (net graphics))))


(defmethod show-activities-net-specifics ((graphics net-graphics) (net feed-forward-net-class)))


(defmethod show-activities-net-specifics ((graphics net-graphics) (net bp-net-class))
  (let ((width (+ (a-box-width graphics) 4)))
    (dolist (seq-name (rest (get-node-seq-names net)))
      (let ((def (find-definition (first (get-node-seq net seq-name)) graphics)))
	(draw-rectangle (a-disp graphics) (- (first def) 2) (- (second def) 2) width width)))))


(defmethod show-weights ((graphics net-graphics))
  (let* ((box-width (w-box-width graphics))
	 (box-middle (/ box-width 2))
	 (max (w-max graphics))
	 (origin (- (border graphics) box-width 3))
	 (origin+1 (1+ origin))
	 (origin-1 (1- origin))
	 (origin-3 (- origin 3))
	 (disp (w-disp graphics)))
    (clear-display disp)
    
    
    
    (dotimes (i (matrix-len graphics))
      (dotimes (j (matrix-len graphics))
	(let ((def (aref (w-positions graphics) i j)))
	  (when def
	    (let* ((value (weight (third def)))
		   (width (* (/ (abs value) max) box-width))
		   (overflow nil))
	      (when (> width box-width)
		(setf overflow t)
		(setf width box-width))
	      (if (minusp value)
		  (progn
		    (draw-rectangle disp (first def) (second def) width width)
		    (when overflow
		      (draw-circle disp
				   (+ (first def) box-middle)
				   (+ (second def) box-middle)
				   3)))
		(progn
		  (draw-filled-rectangle disp (first def) (second def) width width)
		  (when overflow
		    (draw-circle disp
				 (+ (first def) box-middle)
				 (+ (second def) box-middle)
				 3
				 :color white)))))))))
    (draw-line disp origin origin
	       (- (inner-width disp) origin) origin)
    (draw-line disp origin origin
	       origin (- (inner-width disp) origin))
    (dotimes (i (matrix-len graphics))
      (draw-line disp
		 (aref (w-tabs graphics) i 0) origin+1
		 (aref (w-tabs graphics) i 1) origin+1)
      (draw-line disp
		 origin+1 (aref (w-tabs graphics) i 0)
		 origin+1 (aref (w-tabs graphics) i 1))
      (when (aref (w-tabs graphics) i 2)
	(draw-line disp
		   (aref (w-tabs graphics) i 2) origin-1
		   (aref (w-tabs graphics) i 2) origin-3)
	(draw-line disp
		   origin-1 (aref (w-tabs graphics) i 2)
		   origin-3 (aref (w-tabs graphics) i 2))))))


;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
