;;; -*- Mode: LISP; Package: np; Syntax: Common-lisp;      -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   np-def.cl
;;; Short Desc: Very primitives for neural network simulations
;;; Version:    1.0
;;; Status:     Experimental (July 1990)
;;; Last Mod:   27.1.92 - TW
;;; Author:     Thomas Wehrle
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;;
;;;


(in-package :np)

(export '())


; The NODE Class ***********************

(defclass node-class ()
  ((input          :initform 0           
		   :initarg :input
		   :accessor input
		   :type number)
   (activation     :initform 0           
		   :initarg :activation 
		   :accessor activation 
		   :type number)
   (output         :initform 0 
		   :initarg :output
		   :accessor output
		   :type number)
   (in-connections :initform nil 
		   :initarg :in-connections
		   :accessor in-connections
		   :type list))
  (:documentation "Foundation of all nodes"))


(defclass named-node-class ()
  ((name           :initform "anonymous" 
		   :initarg :name 
		   :accessor name 
		   :type string))
  (:documentation "Only for mixin"))


; If there aren't any incomming connections
; the input is set to 0 too. I expect input-
; nodes not to be updated if you do not want
; to refresh the input layer every cycle.

(defmethod compute-input ((node node-class))
  (let ((sum 0))
    (dolist (connection (in-connections node) sum)
      (incf sum 
	    (* (weight connection)
	       (output (from-node connection)))))))


(defmethod compute-activation ((node node-class))
  (let* ((node-input (input node))
	 (constrained-node-input (cond ((> (abs node-input) 15.0) 
					(* (signum node-input) 15.0))
				       ((< (abs node-input) 1.0e-37)
					(* (signum node-input) 1.0e-37))
				       (t node-input))))
    (/ 1.0 (+ 1.0 (exp (- constrained-node-input))))))


(defmethod compute-activation-derivative ((node node-class))
  (let ((current-activation (activation node)))
    (+ 0.1 (* current-activation (- 1.0 current-activation)))))    

		;+ 0.1 improves learning (cf. Fahlman 88) gutkn


(defmethod compute-output ((node node-class))
  (activation node))


(defmethod touch ((node node-class))
  (setf (input node) (compute-input node)))


(defmethod update ((node node-class))
  (setf (activation node) (compute-activation node))
  (setf (output node) (compute-output node)))

; the setf function for the activation slot should
; be modified in a way, that the output slot
; is automatically updated.


; all nodes that have a connection to node
(defmethod get-from-nodes ((node node-class))  
  (mapcar (function (lambda (a-connection)
		      (from-node a-connection)))
	  (in-connections node)))


; The CONNECTION class *****************

(defun random-weight (lower upper)
  (+ (random (* (- upper lower) 1.0)) lower))


(defclass connection-class ()
  ((weight    :initform (random-weight -1 1) 
	      :initarg :weight
	      :accessor weight
	      :type real)
   (from-node :initform nil
	      :initarg :from-node
	      :accessor from-node
	      :type node-class))
  (:documentation "Foundation of all connections"))


(defmethod set-random-weight ((connection connection-class) lower upper)
  (setf (weight connection) (random-weight lower upper)))


; Networks *****************************

(defclass net-class ()
  ((node-seqs   :initform nil
		:initarg :node-seqs
		:accessor node-seqs
		:type list)
   (graphics    :initform nil
		:accessor graphics
		:type list))
  (:documentation "Foundation of all nets"))


(defclass feed-forward-net-class (net-class)
  ()
  (::documentation "Foundation of all feed forward nets"))


(defmacro traverse-node-seq (node-seq node-var &rest body)
  `(dolist (,node-var ,node-seq) ,@body))


(defmacro traverse-all-nodes (net node &rest body)
  (let ((layer (gensym))
        (seq (gensym)))
    `(dolist (,layer (get-node-seq-names ,net))
       (let ((,seq (get-node-seq ,net ,layer)))
         (traverse-node-seq ,seq ,node ,@body)))))


(defmacro traverse-all-weights (net conn &rest body)
  (let ((layer (gensym))
        (seq (gensym))
        (node (gensym)))
    `(dolist (,layer (get-node-seq-names ,net))
       (let ((,seq (get-node-seq ,net ,layer)))
         (traverse-node-seq ,seq ,node 
                            (dolist (,conn (in-connections ,node)) ,@body))))))


(defun make-net (net-sym a-net-class &optional (node-seq-name-list nil))
  (set net-sym (make-instance		; top-down / output-to-input
		   a-net-class 
		 :node-seqs (do ((l1 node-seq-name-list (cdr l1))
				 (l2 nil))
				((null l1) l2)
			      (setf l2 (cons (car l1) (cons nil l2)))))))


(defmethod print-net ((net net-class) 
		      &key (number-of-nodes-per-line 2)
		      (with-connections nil)
		      (number-of-connections-per-line 1))
  (labels ((printlist (l &aux (col 0))
	     (cond (with-connections 
		    (dolist (n l)
		      (format t "    * ~A <~A>~%           (" n (activation n))
		      (setf col 0)
		      (dolist (m (in-connections n))
			(setf col (+ col 1))
                        (when (> col number-of-connections-per-line)
			  (format t "~%           ") 
			  (setf col 1))
			(format t "(~A ~A) " (weight m) (from-node m)))
		      (format t ")~%")))
	
		   (t (format t "   (")
		      (dolist (n l)
			(setf col (+ col 1))
			(when (> col number-of-nodes-per-line) 
			  (format t "~%    ") 
			  (setf col 1))
			(format t "~A " n))
		      (format t ")~%~%"))))

 	   (printassoc (pl)
	     (when pl 
	       (format t "~A~%" (car pl))
	       (if (null (cdr pl))
		   (format t "<no entry>")
		 (printlist (cadr pl)))
	       (printassoc (cddr pl)))))

    (printassoc (node-seqs net))))


(defmethod save-net ((net net-class) 
		     &key 
		     (filename "netdump.dat"))
  (with-open-file (output (add-path filename *nppath*) :direction :output)
    (traverse-all-weights net conn 
			  (print (weight conn) output))))


(defmethod load-net ((net net-class) 
		     &key
		     (filename "netdump.dat"))
  (with-open-file (input (add-path filename *nppath*) :direction :input)
    (traverse-all-weights net conn 
			  (setf (weight conn) 
			    (read input)))))


(defmethod init-node-seq ((net net-class)
			  node-seq-name &optional (node-seq nil))
  (setf (getf (node-seqs net) node-seq-name) node-seq))


(defmethod get-node-seq-names ((net net-class))
  (do ((assoc-list (node-seqs net) (cddr assoc-list))
       (result-list nil))
      ((null assoc-list) result-list)
    (setf result-list (cons (car assoc-list) result-list))))


(defmethod get-node-seq ((net net-class) node-seq-name)
  (getf (node-seqs net) node-seq-name))


(defmethod push-node-in-node-seq ((net net-class) node-seq-name node)
  (if (not (member node-seq-name (get-node-seq-names net)))
      (error "No such node-seq-name initialized (~A)" node-seq-name)
    (if (member node (get-node-seq net node-seq-name))
	(error "Node (~A) is already in sequence (~A)" node node-seq-name)
      (push node (getf (node-seqs net) node-seq-name)))))


(defmethod push-nodes-in-node-seq ((net net-class) node-seq-name node-seq)
  (dolist (node (reverse node-seq))
    (push-node-in-node-seq net node-seq-name node)))


(defmacro concatenate-node-seqs (&rest node-seqs)
  `(concatenate 'list ,@node-seqs))


(defmacro make-node (net a-node-class node-seq-name-list &rest args)
  (let ((node (gensym))
	(node-seq-name (gensym)))
     `(prog1
	 (setf ,node (make-instance ,a-node-class ,@args))
       (dolist (,node-seq-name ,node-seq-name-list)
	 (push-node-in-node-seq ,net ,node-seq-name ,node)))))


(defmacro make-connection (a-connection-class 
			   a-from-node a-to-node &rest args)
  (let ((connection (gensym)))
    `(prog1
	 (setf ,connection (make-instance ,a-connection-class ,@args))
       (setf (from-node ,connection) ,a-from-node)
       (setf (in-connections ,a-to-node)
	 (append (in-connections ,a-to-node) (list ,connection))))))


;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
