;;; -*- Mode: LISP; Package: ID3; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   id3-dialog.cl
;;; Short Desc: dialog handling for id3
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.4.91 - TW
;;; Author:     Thomas Wehrle
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :id3)



;;; ==========================================================================
;;; GLOBAL VARIABLE DECLARATIONS
;;; ==========================================================================

(defvar *pause* nil)
(defvar *quit* nil)
(defvar *verbose-disp* nil)

(defclass id3-display (display)
	  ((windows :accessor windows
		   :initarg :windows
		   :initform nil)
	   (browse :accessor browse
		   :initarg :browse
		   :initform nil)
	   (help-button :accessor help-button
			:initarg :help-button
			:initform nil)))

(defmethod close-display :before ((disp id3-display))
  (setf *quit* t)
  (close-display (technical-window (help-button disp)))
  (close-display (general-window (help-button disp)))
  (loop for d in (windows disp) do (close-display d))
  (loop for b in (browse disp) do (close-display b))
  (close-table-edit 'bla :close-all t)
)

(format t "about to make editor fcn")

(make-editor-fcn
 table-edit
 :obj-var tab
 :default-menu-entries t
 :menu
 `(("Cluster Values >>" ,(options (make-instance 'menu
				    :items `(("2 Clusters"
					      ,(function (lambda ()
							   (id3:cluster tab :clusters 2)
							   (refresh tab)))
					      "Make 2 groups named low and high")
					     ("3 Clusters" 
					      ,(function (lambda ()
							   (id3:cluster tab :clusters 3)
							   (refresh tab))) 
					      "Make 3 groups named low, medium and high")
					     ("n Clusters" 
					      ,(function (lambda ()
							   (id3:cluster tab)
							   (refresh tab)))
					      "Make n automatically named groups")
					     ("n named Clusters" 
					      ,(function (lambda ()
							   (id3:cluster tab :namedp t)
							   (refresh tab))) 
					      "Make n groups with user defined names")
					     ("(User function)"
					      ,(function (lambda ()
							   (id3::apply-user-function tab)
							   (refresh tab)))
					      "Apply user function to values of attribute"))))
			"Cluster numeric values (Menu)")
   ("Expand Wildcards" ,(function (lambda ()
				    (setf tab (id3:expand-wildcards tab))
				    (refresh tab)))
		       "Expand wildcard character '*'")
   ("Remove Duplicates" ,(function (lambda ()
				     (when (> (number-of-rows tab)
					      (number-of-rows (id3:reduce-data tab)))
				       (refresh tab))))
			"Remove duplicate examples")))


;;; specializing label on descision trees for browser
(defmethod label ((tree decision-tree))
  (if (or
       (> (length (descendants tree)) 1)
       (and (= (length (descendants tree)) 1)
	    (null (descendants (first (descendants tree))))))
      (symbol-name (content tree))
    (format nil "[~a]" (content tree))))


#| (defmethod find-sub-tree2 ((tree decision-tree))
  (let ((possible-values (mapcar (function (lambda (x)
					     (symbol-name (content x))))
				 (descendants tree))))
    (let ((value (ask  (format nil "What is the value of ~a ?~
                                  ~&   Possible values: ~a~
		                  ~&   ==> " (content tree) possible-values)
		       :title "ID3:Please answer")))
      (when (member value possible-values :test #'equal)
	(do* ((desc-list (descendants tree) (rest desc-list))
	      (sub-tree (first desc-list) (first desc-list)))
	    ((equal value (symbol-name (content sub-tree))) 
	     (first (descendants sub-tree)))))))) |#

(defmethod find-sub-tree2 ((tree decision-tree))
  (let ((possible-values (mapcar (function (lambda (x)
					     (format nil "~a" (content x))))
				 (descendants tree))))
    (let ((value (ask  (format nil "What is the value of ~a ?~
                                  ~&   Possible values: ~a~
		                  ~&   ==> " (content tree) possible-values)
		       :title "ID3:Please answer")))
      (when (member value possible-values :test #'equal)
	(do* ((desc-list (descendants tree) (rest desc-list))
	      (sub-tree (first desc-list) (first desc-list)))
	    ((equal value (format nil "~a" (content sub-tree))) 
	     (first (descendants sub-tree))))))))


(defmethod query2 (disp (tree decision-tree))
  (if (<= (length (descendants tree)) 1)
      (format-display disp "~%From what I know I expect ~a to be ~a."
		      (content tree)
		      (content (first (descendants tree))))
    (query2 disp (find-sub-tree2 tree))))


(defmethod query2 (disp something)
  (when (null something) 
    (format-display disp "~%No knowledge about this - sorry.")))


(defun make-title (nr)
  (format nil "A ~:R Demo on ID3" nr))

(defun start-id3-dialog (&optional (from-button nil))
  (setf *verbose-disp* (make-instance 'scroll-display
                                     :title "ID3:Verbose Comments"
				     :width (round (width *root-window*) 3)
				     :height (round (height *root-window*) 3)
				     :borders 1
				     :active nil
				     :left (round (width *root-window*) 2)
				     :bottom (round (height *root-window*) 3)))
  (if *verbose*
      (progn (activate-display *verbose-disp*)
	     (setf *verbose* *verbose-disp*))
    (deactivate-display *verbose-disp*))
  (let* ((disp (make-instance 'id3-display :title "ID3:Main Window"
			     :width 300
			     :height 150
			     :borders 1
			     :left (+ (width pail-lib::*main-window*) (left pail-lib::*main-window*))
			     :bottom (bottom pail-lib::*main-window*)))
	 (menu-button (make-instance 'pop-up-button
			:label "Demos"
			:width 100))
	 (exit-button (make-instance 'push-button :label "Exit" :width 70))
	 (tool-button (make-instance 'push-button :label "Tool" :width 100))
	 (help-button (setf (help-button disp)
			(make-instance 'help-button
			:technical (add-path "id3-desc.tec" *id3path*)
			:general (add-path "id3-desc.gen" *id3path*)
			:subject "ID3")))
	 (menu (make-instance 'menu
		 :items (let ((mlist nil))
			   (dotimes (i *number-of-demos* (nreverse mlist))
			     (setf mlist 
			       (cons (list (format nil "Demo ~a" (1+ i)) 
					   `(lambda () 
					      (progn (push (demo ,(1+ i) ,menu-button ,exit-button)
							   (windows ,disp))
						     (car (windows ,disp))))
					   (make-title (1+ i)))
				     mlist))))
       		 :query "DEMOS")))

    (setf (menu menu-button) menu)

    (copy-mask *pail-logo* 0 0 disp
	       (- (width disp) (cw:bitmap-width *pail-logo*) 15)
	       (- (height disp) (cw:bitmap-height *pail-logo*) 15))
  
    (setf (font disp) (cw:open-font :courier :italic 20 :weight :bold))
    (write-display disp
		   "Welcome to ..."
		   15 (- (height disp) 15 (cw:font-ascent (font disp))))
    (setf (font disp) (cw:open-font :courier :italic 45 :weight :bold))
    (write-display disp
		   "ID3"
		   (round (width disp) 3)
		   (- (height disp) 40 (cw:font-ascent (font disp))))
    (setf (font disp) *default-font*)
 
    (set-button exit-button disp 
		:left (- (width disp) (width exit-button) 15)
		:bottom 15
		:action `(lambda nil
			   (close-display *verbose-disp*)
			   (close-display ,disp)
			   (if ,from-button (reset-button ,from-button))))

    (set-button help-button disp :left 15 :bottom 15)
  

    (set-button menu-button disp 
		:left (round (- (width disp) (width menu-button)) 2)
		:bottom (+ 20 (height help-button)))
    (set-button tool-button disp 
		:left (round (- (width disp) (width tool-button)) 2)
		:bottom 15
		:action `(lambda ()
			   (push (tool ,tool-button ,exit-button)
				 (windows ,disp))
		))
    disp))


(defun demo (nr menu-button other-exit-button)
  (disable-button menu-button)
;  (disable-button other-exit-button)
  (let* ((dispwidth (min 500 (round (width *root-window*) 2)))
	 (dispheight (min 260 (round (height *root-window*) 2)))
	 (*default-font* (findfont (width *root-window*) 1140 13))
	 (disp (make-instance 'id3-display :title (concatenate 'string "ID3:" (make-title nr))
			      :width dispwidth
			      :height dispheight
			      :borders 1
			      :left (- (width *root-window*) dispwidth 15)
			      :bottom (- (height *root-window*) dispheight 20)))
	 (exit-button (make-instance 'push-button :label "Exit" :width (round (width disp) 5)))
	 (help-button (setf (help-button disp)
			(make-instance 'help-button
			  :width (round (width disp) 5)
			  :subject (concatenate 'string "ID3:" (make-title nr))
			  :technical (add-path
				      "demo-desc.tec"
				      *id3path*)
			  :general (add-path
				    (concatenate 'string
				      "demo" (write-to-string nr) "-desc.gen")
				    *id3path*) )))
	 (start-button (make-instance 'push-button :label "Start" :width (round (width disp) 5)))
	 (cont-button (make-instance 'push-button :label "Continue" :width (round (width disp) 5)))
	 (verb (make-instance 'radio-button :label "Verbose Mode"
			      :status *verbose*
			      :action 
			      '(lambda ()
				(if *verbose*
				    (progn
				      (setf *verbose* nil)
				      (deactivate-display *verbose-disp*))
				  (progn
				    (setf *verbose* *verbose-disp*)
				    (activate-display *verbose-disp*))))))
	 (clash (make-instance 'radio-button :label "Accept Clashes"
			       :action 
			       '(lambda () 
				 (setf *accept-clashes* (not *accept-clashes*)))))
	 (prec1 (make-instance 'radio-button :label "Single precision"
			       :action
			       '(lambda () 
				 (setf *precision* 'single-float))))
	 (prec2 (make-instance 'radio-button :label "Double precision"
			       :status t
			       :action 
			       '(lambda () 
				 (setf *precision* 'double-float)))))

    (setf *precision* 'double-float)	; make shure that default is set
    (setf *accept-clashes* nil)

    (setf (font disp) (cw:open-font :courier :italic 20 :weight :bold))
    (write-display disp 
		   (make-title nr)
		   15 (- (height disp) 15 (cw:font-ascent (font disp))))
    (setf (font disp) *default-font*)


    (copy-mask *pail-logo* 0 0 disp 
	       (- (width disp) (cw:bitmap-width *pail-logo*) 15)
	       (- (height disp) (cw:bitmap-height *pail-logo*) 15))

    (set-button exit-button disp 
		:left (- (width disp) (width exit-button) 15) 
		:bottom 15
		:action `(lambda ()
			   (setf *quit* t)
			   (enable-button ,menu-button)
;			   (enable-button ,other-exit-button)
			   (close-display ,disp)))

    (set-button help-button disp :left 15 :bottom 15)
    

    (set-button start-button disp 
		:left (round (+ 15
				(width start-button)
				(round (- (width disp)
					  30
					  (* 4 (width start-button)))
				       3)))
		:bottom 15
		:action
		#'(lambda ()
		    (disable-button exit-button)
		    (dialog-handler nr disp start-button cont-button exit-button)))
		
    (set-button cont-button disp 
		:left (round (+ 15
				(* 2 (width start-button))
				(* 2 (round (- (width disp) 
					       30 
					       (* 4 (width start-button)))
					    3))))
		:bottom 15
		:action 
		#'(lambda ()
		    (disable-button exit-button)
		    (setf *pause* (not *pause*))))
    (disable-button cont-button)

    (set-button prec1 disp :left (round (width disp) 10) :bottom 130)
    (set-button prec2 disp :left (round (width disp) 10) :bottom 110)
    (set-exclusive prec1 prec2)
		    
    (set-button verb disp :left (round (width disp) 10) :bottom 80)
    (set-button clash disp :left (round (width disp) 10)  :bottom 60)
    disp))
  



(defun auto-handler (nr parent start-button cont-button exit-button)
  (let ((text-disp (make-instance 'scroll-display
                                  :title "ID3:Comments"
				  :font (findfont (width *root-window*) 1140 10)
				  :parent parent
				  :width (round (width parent) 3)
				  :height (round (* 2 (height parent)) 3)
				  :borders 1
				  :left (round (width parent) 2)
				  :bottom (round (height parent) 4)))
	(table-disp (make-instance 'scroll-display
		                  :title "ID3:Examples"
				  :font (findfont (width *root-window*) 1140 13)
				  :width (- (round (width *root-window*) 2) 10)
				  :height (round (height *root-window*) 2)
				  :borders 1
				  :left 10
				  :bottom (round (height *root-window*)  2)))
	(rule-disp (make-instance 'scroll-display
                                  :title "ID3:Rules"
				  :font (findfont (width *root-window*) 1140 13)
				  :width (- (round (width *root-window*) 2) 10)
				  :height (round (height *root-window*) 2)
				  :borders 1
				  :left 10
				  :bottom 0)))
    (labels ((clean-up ()
;	       (close-display text-disp)
;	       (close-display rule-disp)
;	       (close-display table-disp)
	       (setf table-disp nil)
	       (close-display parent)
	       (unless *quit*
		 (enable-button exit-button)
		 (reset-button cont-button)
		 (disable-button cont-button)
		 (reset-button start-button))))
      (defun wait ()
	(unless *quit*
	  (enable-button cont-button)
	  (reset-button cont-button)
	  (enable-button exit-button)
	  (format-display text-disp "~&~%[Press <Continue> Button]")
	  (sleep 1)
	  (mp:process-wait "wait-on-click" (function (lambda () (not pail::*pause-demo*))))
	  (if pail::*stop-demo* (setq *quit* t))
	  (software-push cont-button)
					;(do () ((or *quit* *pause*)))
	  (if *quit* (clean-up)
	    (setf *pause* nil)))
	(not *quit*))
      (push rule-disp (windows parent))
      (push text-disp (windows parent))
      (push table-disp (windows parent))
      (setf *pause* nil)
      (setf *quit* nil)
      (let ((fun-sym (find-symbol (symbol-name (read-from-string (format nil "dialog-~a" nr))) :id3)))
	(if fun-sym
	    (funcall fun-sym text-disp table-disp rule-disp)
	  (format t "Demo not implemented")))
      (wait)
      (clean-up))))

(defun dialog-handler (nr parent start-button cont-button exit-button)
  (let* ((*default-font* (findfont (width *root-window*) 1140 13))
	 (text-disp (make-instance 'scroll-display
		      :title "ID3:Comments"
		      :font (findfont (width *root-window*) 1140 10)
		      :parent parent
		      :width (round (width parent) 2)
		      :height (round (height parent) 2)
		      :borders 1
		      :left (- (round (width parent) 2) 10)
		      :bottom 50))
	 (table-disp nil
		     #| (make-instance 'scroll-display
		                  :title "ID3:Examples"
				  :font *default-font*
				  :width (- (round (width *root-window*) 2) 10)
				  :height (round (height *root-window*) 2)
				  :borders 1
				  :left 10
				  :bottom (round (height *root-window*)  2)) |#)
	 (rule-disp (make-instance 'scroll-display
		      :title "ID3:Rules"
		      :font *default-font*
		      :width (- (round (width *root-window*) 2) 10)
		      :height (- (round (height *root-window*) 2) 20)
		      :borders 1
		      :left 10
		      :bottom 0)))
    (labels ((clean-up ()
;	       (close-display text-disp)
;	       (close-display rule-disp)
					;	       (close-display table-disp)
	       (setf table-disp nil)
	       (unless *quit*
		 (enable-button exit-button)
		 (reset-button cont-button)
		 (disable-button cont-button)
		 (reset-button start-button))))
      (defun wait ()
	(unless *quit*
	  (enable-button cont-button)
	  (reset-button cont-button)
	  (enable-button exit-button)
	  (format-display text-disp "~&~%[Press <Continue> Button]")
	  (mp:process-wait "wait-on-click" (function (lambda () (or *quit* *pause*))))
					;(do () ((or *quit* *pause*)))
	  (if *quit* (clean-up)
	    (setf *pause* nil)))
	(not *quit*))
      (push rule-disp (windows parent))
      (push text-disp (windows parent))

      (setf *pause* nil)
      (setf *quit* nil)
      (let ((fun-sym (find-symbol (symbol-name (read-from-string (format nil "dialog-~a" nr))) :id3)))
	(if fun-sym
	    (funcall fun-sym text-disp table-disp rule-disp)
	  (format t "Demo not implemented")))
      (wait)
      (clean-up))))




(defun tool (tool-button exit-button)
;  (disable-button exit-button)

  (let* ((*value-button-border-p* t)
	 (dispwidth 800)
	 (dispheight 260)
	 (disp (make-instance 'id3-display :title "ID3:Tool"
			      :width (min (width *root-window*) dispwidth)
			      :height (min (height *root-window*) dispheight)
			      :borders 1
			      :left (round (- (width *root-window*) (min (width *root-window*)
									 dispwidth)) 2)
			      :bottom (round (- (height *root-window*) (min (height *root-window*)
									    dispheight)) 2)))
	 (*default-font* (findfont (width disp) dispwidth 13))
	 (text-disp (make-instance 'scroll-display
		      :title "ID3:Comments"
		      :font (findfont (width disp) 800 10)
		      :parent disp
		      :width (round (* 3 (width disp)) 8)
		      :height (round (*  2 (height disp)) 3)
		      :borders 1
		      :left (+ 10 (round (width disp) 5) (round (width disp) 4))
		      :bottom (round (height disp) 5)))
	 (rule-disp nil)
	 (attribute nil)
	 (local-exit-button (make-instance 'push-button :label "Exit"
					   :width (round (width disp) 8)))
	 (help-button (setf (help-button disp)
			(make-instance 'help-button
				    :technical (add-path "id3-desc.tec" *id3path*)
			:general (add-path "id3-desc.gen" *id3path*)
			:subject "ID3" :width (round (width disp) 8)
				     )))
	 (classify-button (make-instance 'push-button :label "Classify"
					 :width (round (width disp) 8)))
	 (table-button (make-instance 'push-button :label "New Table"
				      :width (round (width disp) 8)))
	 (rules-button (make-instance 'push-button :label "Get rules"
				      :width (round (width disp) 8)))
	 (query-button (make-instance 'push-button :label "Query"
				      :width (round (width disp) 8)
				      :technical (add-path
					 "tool-desc.asc"
					 *id3path*)
				      :general (add-path
					 "tool-desc.asc"
					 *id3path*)))
	 (attr-button nil)
	 (verb (make-instance 'radio-button :label "Verbose Mode"
			      :status *verbose*
			      :action 
			      #'(lambda ()
				  (if *verbose*
				      (progn
					(setf *verbose* nil)
					(deactivate-display *verbose-disp*))
				    (progn
				      (setf *verbose* *verbose-disp*)
				      (activate-display *verbose-disp*))))))
	 (clash (make-instance 'radio-button :label "Accept Clashes"
			       :action 
			       '(lambda () 
				 (setf *accept-clashes* (not *accept-clashes*)))))
	 (prec1 (make-instance 'radio-button :label "Single precision"
			       :action
			       #'(lambda () 
				   (setf *precision* 'single-float))))
	 (prec2 (make-instance 'radio-button :label "Double precision"
			       :status t
			       :action
			       #'(lambda () 
				   (setf *precision* 'double-float))))
	
 (output-button (make-instance 'pool-button
			  :target-class 'decision-tree
			  :label "ID3 Tree output:"
			  :width (round (width disp) 5)
			  :pool *pail-pool*))
	 (input-button (make-instance 'pool-button
			 :target-class 'table
			 :label "ID3 Table input:"
			 :width (round (width disp) 5)
			 :pool *pail-pool*
			 :after-get #'(lambda (item)
					(let ((*default-font* (findfont  (width *root-window*) 1140 16)))
					  (table-edit (start-object item) :left 0 :bottom 0)))))

	 (file-button (make-instance 'file-button
			:w-directory *id3path*))
)
    (push text-disp (windows disp))
    

    (setf *precision* 'double-float)	; make sure that default is set
    (setf *accept-clashes* nil)

    (setf (font disp) (cw:open-font :courier :italic 20 :weight :bold))
    (write-display disp 
		   "ID3 tool"
		   15 (- (height disp) 15 (cw:font-ascent (font disp))))
    (setf (font disp) *default-font*)

    (copy-mask *pail-logo* 0 0 disp 
	       (- (width disp) (cw:bitmap-width *pail-logo*) 15)
	       (- (height disp) (cw:bitmap-height *pail-logo*) 15))
    
    (set-button file-button disp
		:left (- (width disp) (width file-button) 7)
		:bottom (- (height disp) (height file-button)
			   (cw:bitmap-height *pail-logo*) 20))  
    (set-button local-exit-button disp 
		:left (- (width disp) (width local-exit-button) 15) 
		:bottom 15
		:action #'(lambda ()
			    (reset-button tool-button)
			    (enable-button exit-button)
			    (close-display disp)))
 
    (set-button help-button disp 
		:left 15
		:bottom 15
		)
    
    (set-button table-button disp 
		:left (round (+ 15
				(* 1 (width table-button))
				(* 1 (round (- (width disp)
					   30
					   (* 6 (width table-button)))
					5))))
		:bottom 15
		:action
		#'(lambda ()
		    (let ((*default-font* (findfont  (width *root-window*) 1140 16)))
		      (disable-button local-exit-button)

		      (setf (start-object (button-value input-button))
			(table-edit (start-object (button-value input-button)) :left 0 :bottom 0))
		    
		      (reset-button table-button)
		      (enable-button local-exit-button))))

    (set-button classify-button disp 
		:left (round (+ 15
				(* 2 (width classify-button))
				(* 2 (round (- (width disp)
					   30
					   (* 6 (width classify-button)))
					5))))
		:bottom 15
		:action
		#'(lambda ()
		    (disable-button local-exit-button)
		    (unless (start-object (button-value input-button)) (display-error "No table specified!"))
		    (unless attribute (display-error "No target attribute specified!"))
		    (let ((ok (and (start-object (button-value input-button)) 
				   attribute 
			           (member attribute (attributes (start-object (button-value input-button))) :test #'equal))))
		      (when (and (start-object (button-value input-button)) attribute (not ok))
			(display-error "Target attribute does not exist!"))
		      (when ok
			(setf (start-object (button-value output-button)) (classify attribute (start-object (button-value input-button))))
			(format-display text-disp "~%Generated a decision tree.")
			(format-display text-disp "~&(see <Decision tree> window)")
			(multiple-value-bind (clashes examples)
			    (clash-p (start-object (button-value input-button)) attribute)
			  (format-display text-disp (format nil "~&Processed ~a unique example(s)." examples))
			  (when (plusp clashes)
			    (format-display text-disp (format nil "~&There were ~a clash(es) for this target" clashes))
			    (when *accept-clashes*
			      (format-display text-disp "~&Clashes were handled"))))
			(dolist (b (browse disp))
			  (setf (title (display b)) "ID3:Old Decision tree"))
			(push  (make-instance 'browser
					     :starting-tree (start-object (button-value output-button))
					     :title "ID3:Decision tree"
					     :left 100
					     :bottom 0
					     :font (findfont  (width *root-window*) 1140 16))
			       (browse disp))
			
			(setf (left (first (browse disp)))
			  (max 0 (- (width *root-window*) (width (first (browse disp)))))))
		      (reset-button classify-button)
		      (enable-button local-exit-button))))

    (set-button query-button disp 
		:left (round (+ 15
				(* 3 (width classify-button))
				(* 3 (round (- (width disp)
					   30
					   (* 6 (width classify-button)))
					5))))
		:bottom 15
		:action
		#'(lambda ()
		    (disable-button local-exit-button)
		    (unless (start-object (button-value output-button))
		      (display-error "No decision tree generated (Classify)"))
		    (when (start-object (button-value output-button))
		      (funcall (function query2) text-disp (start-object (button-value output-button))))
		    (reset-button query-button)
		    (enable-button local-exit-button)))

    (set-button rules-button disp 
		:left (round (+ 15
				(* 4 (width rules-button))
				(* 4 (round (- (width disp) 
					   30 
					   (* 6 (width rules-button)))
					5))))
		:bottom 15
		:action 
		#'(lambda ()
		    (disable-button local-exit-button)
		    (unless (start-object (button-value output-button)) (display-error "No decision tree generated (Classify)"))
		    (when (start-object (button-value output-button))
		      (when rule-disp (close-display rule-disp))
		      (push (setf rule-disp
			(make-instance 'scroll-display
				       :title "ID3:Rules"
				       :font (findfont  (width *root-window*) 1140 16)
				       :width 480
				       :height 200
				       :borders 1
				       :left 10
				       :bottom 143))
			    (windows disp))
		      (let ((*readable* t))
			(format-display rule-disp "~a"
					(change-type (start-object (button-value output-button))
						      (clos:class-prototype (find-class 'rule-set))))))
		    (reset-button rules-button)
		    (enable-button local-exit-button)))

    (set-button prec1 disp :left (round (width disp) 4) :bottom 230)
    (set-button prec2 disp :left (round (width disp) 4) :bottom 210)
    (set-exclusive prec1 prec2)
		    
    (set-button verb disp :left (round (width disp) 4) :bottom 180)
    (set-button clash disp :left (round (width disp) 4)  :bottom 160)

    (setf attr-button (make-instance 'value-button 
			:numeric nil 
			:label "Target Attribute:" 
			:width (round (width disp) 5)
			:menu
			(make-instance 'menu :items '(("No table defined - sorry" nil)))
			:before-action
			#'(lambda ()
			   (setf (menu attr-button) (make-instance 'menu
						      :items
						      (if (start-object (button-value input-button))
							  (mapcar (function (lambda (attr)
									      (list (all-symbol-names attr)
										    (all-symbol-names attr))))
								  (attributes (start-object (button-value input-button))))
							'(("No table defined - sorry" nil))))))
			:action
			#'(lambda ()
			    (setf attribute
			      (intern (read-from-string (button-value attr-button) nil) :dump)))))
    (set-button attr-button disp :left (round (width disp) 4) :bottom 120)
  
   (set-button output-button disp :left (round (width disp) 4) :bottom (- 115 (height  attr-button)))
   (set-button input-button disp :left (round (width disp) 4) :bottom (- 115 (* 2 (height  attr-button))))

    (format-display text-disp "~&Welcome to the ID3 tool!")
    (format-display text-disp "~%First you have to specify a table.")
    disp))



(defun auto-demo (nr  )
  (let ((disp (make-instance 'display :title (concatenate 'string "ID3:" (make-title nr))
			    :width 800
			    :height 200
			    :borders 1
			    :left 115
			    :bottom 610))
	(exit-button (make-instance 'push-button :label "Exit" :width 180))
	(help-button (make-instance 'help-button
		       :width 180
		       :subject (concatenate 'string "ID3:" (make-title nr))
		       :technical (add-path
					"demo-desc.tec"
					*id3path*)
		       :general (add-path
					(concatenate 'string
					  "demo" (write-to-string nr) "-desc.gen")
					*id3path*) ))
	(start-button (make-instance 'push-button :label "Start" :width 180))
	(cont-button (make-instance 'push-button :label "Continue" :width 180))
	(verb (make-instance 'radio-button :label "Verbose Mode"
			     :status *verbose*
			     :action 
			     '(lambda ()
			       (if *verbose*
				   (progn
				     (setf *verbose* nil)
				     (deactivate-display *verbose-disp*))
				 (progn
				   (setf *verbose* *verbose-disp*)
				   (activate-display *verbose-disp*))))))
	(clash (make-instance 'radio-button :label "Accept Clashes"
			      :action 
			      '(lambda () 
				(setf *accept-clashes* (not *accept-clashes*)))))
	(prec1 (make-instance 'radio-button :label "Single precision"
			      :action
			      '(lambda () 
				(setf *precision* 'single-float))))
	(prec2 (make-instance 'radio-button :label "Double precision"
			      :status t
			      :action 
			      '(lambda () 
				(setf *precision* 'double-float)))))

    (setf *precision* 'double-float)	; make shure that default is set
    (setf *accept-clashes* nil)

    (setf (font disp) (cw:open-font :courier :italic 20 :weight :bold))
    (write-display disp 
		   (make-title nr)
		   15 (- (height disp) 15 (cw:font-ascent (font disp))))
    (setf (font disp) *default-font*)


    (copy-mask *pail-logo* 0 0 disp 
	       (- (width disp) (cw:bitmap-width *pail-logo*) 15)
	       (- (height disp) (cw:bitmap-height *pail-logo*) 15))

    (set-button exit-button disp 
		:left (- (width disp) (width exit-button) 15) 
		:bottom 15
		:action `(lambda ()
			   (setf *quit* t)
			   (close-display ,disp)))

    (set-button help-button disp :left 15 :bottom 15)
    

    (set-button start-button disp 
		:left (round (+ 15
				(width start-button)
				(round (- (width disp)
				      30
				      (* 4 (width start-button)))
				   3)))
		:bottom 15
		:action
		#'(lambda ()
		    (disable-button exit-button)
		    (auto-handler nr disp start-button cont-button exit-button)))
		
    (set-button cont-button disp 
		:left (round (+ 15
				(* 2 (width start-button))
				(* 2 (round (- (width disp) 
					   30 
					   (* 4 (width start-button)))
					3))))
		:bottom 15
		:action 
		#'(lambda ()
		    (disable-button exit-button)
		    (setf *pause* (not *pause*))))
    (disable-button cont-button)

    (set-button prec1 disp :left (round (width disp) 4) :bottom 130)
    (set-button prec2 disp :left (round (width disp) 4) :bottom 110)
    (set-exclusive prec1 prec2)
		    
    (set-button verb disp :left (round (width disp) 4) :bottom 80)
    (set-button clash disp :left (round (width disp) 4)  :bottom 60)

    (setf pail::*stop-demo* nil)
    (software-push start-button)
    (software-push exit-button)
    ))



;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
