;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:GA; Base:10; -*-
;;;
;;; ******************************
;;; *  PORTABLE AI LAB - UNI ZH  *
;;; ******************************
;;;
;;; Filename:   ga.cl
;;; Short Desc: A simple Genetic Algorithm for Parameter Optimization
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   May 1991
;;; Author:     Nick Almassy
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; Files required: ga.cl		(this-file)
;;;		    ga-functions.cl	(a set of functions)
;;;                 ga-dialog.cl	(the window Interface)
;;;		    acl-gin		(the window-functions)
;;; --------------------------------------------------------------------------

(in-package :ga)

(export '(*acc*				flip
	  *allele-type*			format-genotype
	  *gray-coding*			generation
	  *max-allele*			genotype
	  *mutation-events*		individual
	  *mutation-probability*	initialize-population
	  *normalize-fitness-p*		maximum-fitness
	  *population-size*		minimum-fitness
	  *scale-fitness-p*		mutations
	  *trace* *mutations-p*		offline-performance
	  average-fitness		online-performance
	  cross-over			parameter-set
	  population
	  evaluate			populations
	  evolve			random-list
	  firstn			rnd-element
	  fitness			size
	  fitness-value			total-fitness
	  total-number
	  ))

(defparameter *trace* nil)
(setf *gin-p* t)

(defun find-best (l &optional (max (car l)))
  (if l
      (if (> (fitness-value (car l)) (fitness-value max))
	  (find-best (cdr l) (car l))
	(find-best (cdr l) max))
    max))

(defun flip (p)				; flip a biased coin - returns T with probability p
  (< (random 1.0) p))

(defun firstn (n l)			; returns the first n elements
  (if (> n 0)
      (cons (car l) (firstn (- n 1) (cdr l)))
    nil))

(defun rnd-element (l but-not-this)	; returns a random element of e list
  (if l
      (car (nthcdr (random (length l)) (set-difference l (list but-not-this))))))

;;; ==========================================================================
;;; Normal distributed random numbers, mu=0 sigma=1
;;; after ACM algorithm 167 / Box-Muller Method
;;; ==========================================================================

(defvar rndcalcflag t)
(defvar rndx1 0.0)
(defvar rndx2 0.0)

(defun noise ()
  (if rndcalcflag
      (let ((tt (* 6.2831853072 (random 1.0))))
	(setf rndx1 (sqrt (* -2.0 (log (random 1.0)))))
	(setf rndx2 (* rndx1 (sin tt)))
	(setf rndcalcflag nil)
	(* rndx1 (cos tt)))
    (progn
      (setf rndcalcflag t)
      rndx2)))


;;; ==========================================================================
;;; Functions to convert gray-coded bit-lists to integer and vice-versa
;;; ==========================================================================

(defun gray (l &optional (last nil))
  (if l
      (cons (not (equal (car l) last)) (gray (cdr l) (car l)))))


(defun degray (l &optional (last nil))
  (if l
      (if (car l)
	  (cons (not last) (degray (cdr l) (not last)))
	(cons last (degray (cdr l) last)))))


;;; ==========================================================================
;;; Constants of the GA
;;; ==========================================================================

(defparameter *population-size*       30)	; must be even!
(defparameter *crossover-probability* 0.6)
(defparameter *mutation-probability*  0.00433)
(defparameter *mutations-p*	      t)
(defparameter *inversion-probability* 0.0)      ; unused
(defparameter *generation-gap*        1.0)      ; unused
(defparameter *mutation-events*       0)
(defparameter *normalize-fitness-p*   t)
(defparameter *scale-fitness-p*	      t)	; for fitness-scaling
(defparameter *fitness-multiple*      2)
(defparameter *scale-a*               1.0)
(defparameter *scale-b*               0.0)
(defparameter *allele-type*	     '(nil t))	; the possible-values of the genotype
(defparameter *acc*		      0.0)	; this is a hack.
(defparameter *generations-to-evolve* 50)
(defparameter *current-generation*    nil)
(defparameter *proceed*		      t)	; if set to nil the evolving stops.

;;; ==========================================================================
;;; Constants of the parameter encoding scheme
;;; ==========================================================================

(defvar *fitness-function* nil)
(defvar *parameter* nil)
(defvar *bits-per-parameter* 8)
(defvar *precision* 0)
(defvar *max-allele* 0)
(defvar *lower-bound* 0)
(defvar *upper-bound* 1)
(defvar *itervall-length* 0)
(defvar *gray-coding* nil)
(defun degray1 (l) l)			; use this definition if you don't want gray coding to be used.
;;;(defun degray1 (l) (degray l))	; use this otherwise.

;;; ==========================================================================
;;; Classes
;;; ==========================================================================

(defclass ga-function ()
	  ((nparameter		:type integer :initform 1
				:initarg :nparameter
				:accessor nparameter)
	   (bits-per-parameter	:type integer :initform 8
				:initarg :bits-per-parameter
				:accessor bits-per-parameter)
	   (fitness-function	:initarg :fitness-function
				:accessor fitness-function)
	   (max-f		:type float :initform 1.0 :initarg :max-f :accessor max-f)
	   (min-f		:type float :initform 0.0 :initarg :min-f :accessor min-f)
	   (lower-bound		:type float :initform 0.0
				:initarg :lower-bound :accessor lower-bound)
	   (upper-bound		:type float :initform 1.0
				:initarg :upper-bound :accessor upper-bound)
	   (max-allele		:type integer :accessor max-allele)
	   (itervall-length	:type float :accessor itervall-length)
	   (precision		:type integer :accessor precision)
	   (gray-coded		:initform nil :initarg :gray-coded :accessor gray-coded))
  (:documentation "definition of a function to be optimized by the GA"))

(defmethod initialize-instance :after ((gf ga-function) &rest ignore)
  (setf (max-allele gf) (* (nparameter gf) (bits-per-parameter gf)))
  (setf (itervall-length gf) (- (upper-bound gf) (lower-bound gf)))
  (setf (precision gf) (expt 2 (bits-per-parameter gf))))

(defparameter *current-function* nil)

;;; ==========================================================================

(defclass individual ()
	  ((fitness :type float
		    :accessor fitness-value)
	   (genotype :type list
		     :initarg :genotype
		     :accessor genotype))
  (:documentation "A generic parameter-optimization genotype."))


(defclass population ()
	  ((minimum-fitness :type float :initform 0
			    :accessor minimum-fitness)
	   (average-fitness :type float :initform 0
			    :accessor average-fitness)
	   (maximum-fitness :type float :initform 0
			    :accessor maximum-fitness)
	   (total-fitness :type float :initform 0
			  :accessor total-fitness)
	   (size :type integer
		 :initarg :size
		 :accessor size)
	   (population :type list
		       :initarg :population
		       :accessor population))
  (:documentation "a set of indiviuals at time t."))


(defclass generation ()
	  ((mutation-events :type integer :initform 0
			    :initarg :mutations
			    :accessor mutations)
	   (total-number :type integer :initform 0
			 :initarg :total-number
			 :accessor total-number)
	   (online :type list :initform nil
		   :initarg :online-performance
		   :accessor online-performance) ; the convergence (best individuals)
	   (offline :initform nil
		    :initarg :offline-performance ; the evolving average since the start of the ga
		    :accessor offline-performance) ; (list of fitness values)
					   
	   (populations :type list :initform nil
			:initarg :populations
			:accessor populations))
  (:documentation "An evolving set of populations"))


;;; ==========================================================================
;;; Initialization
;;; ==========================================================================


(defmethod initialize-instance ((p population) &key
				(population (initialize-population *population-size* 'individual)))
  (setf (population p) population)
  (setf (size p) (length population))
  (setf (minimum-fitness p) 0.0)
  (setf (average-fitness p) 0.0)
  (setf (maximum-fitness p) 0.0)
  (setf (total-fitness p) 0.0)
  (evaluate p))

(defmethod initialize-instance ((i individual) &key
				(genotype (random-list *max-allele* *allele-type*)))
  (setf (genotype i) genotype))


;;; **********        Misc        **********

(defun random-list (length type)
  (if (> length 0)
      (cons (rnd-element type nil) (random-list (1- length) type))
    nil))

(defun initialize-population (population-size population-type)
  (let ((p nil))
    (dotimes (i population-size p)
      (push (make-instance population-type) p))))
#| (if (> population-size 0)
      (cons (make-instance population-type)
	    (initialize-population (1- population-size) population-type))
      nil) |#

(defun find-worst (l min &optional (the-worst (car l)))
  (if l
      (if (< (fitness-value (car l)) min)
	  (find-worst (cdr l) (fitness-value (car l)) (car l))
	(find-worst (cdr l) min the-worst))
    the-worst))


(defmethod best-individual ((p population))
  (find-best (population p)))

(defmethod best-individual ((g generation))
  (find-best (mapcar #'best-individual (populations g))))

(defmethod average-fitness ((g generation))
  (mapcar #'average-fitness (populations g)))

(defmethod maximum-fitness ((g generation))
  (mapcar #'maximum-fitness (populations g)))

;;; ==========================================================================
;;; Output functions
;;; ==========================================================================

(defun format-genotype (l)
  (if l
      (if (car l)
	  (concatenate 'string "1" (format-genotype (cdr l)))
	(concatenate 'string "0" (format-genotype (cdr l))))
    nil))

;(defmethod print-obj ((i individual) stream)
;  (format stream "#<Individual [~4,2F] ~A >"
;	  (fitness-value i)
;	  (format-genotype (genotype i))))

(defmethod print-object ((i individual) stream)
  (format stream "#<individual f=~4,2F>"
          (fitness-value i)))

(defmethod print-object ((p population) stream)
  (format stream "#<population of ~A Individuals, avg.f=~4,3F>"
	  (length (population p))
	  (average-fitness p)))

(defmethod print-object ((g generation) stream)
  (format stream "~%~A generations (~A mutations), average fitness developed: ~%~{~4,6F, ~}best individual is"
	  (total-number g)
	  (mutations g)
	  (mapcar #'average-fitness (populations g)))
  (print-object (best-individual g) stream))


;;; ==========================================================================
;;; Evaluation
;;; ==========================================================================

(defmethod evaluate ((i individual))
  (setq *acc* (+ *acc* (setf (fitness-value i) (evaluate (genotype i))))) i)

(defmethod evaluate :before ((p population))
  (setq *acc* 0))

(defmethod evaluate ((p population))
  (mapcar #'evaluate (population p)) p)

(defmethod evaluate :after ((p population))
  (progn (setf (total-fitness p) *acc*)
	 (setf (maximum-fitness p) (fitness-value (find-best (population p))))
	 (setf (average-fitness p) (/ *acc* (size p)))
	 (setf (minimum-fitness p) (fitness-value (find-worst (population p) (maximum-fitness p))))
         (if *scale-fitness-p* (scale-fitness p))
	 (if *normalize-fitness-p* (normalize-fitness p))
	 (display-message (format nil "finished evaluating~%   ~a" p))))

(defmethod evaluate (chromosome)
  (apply *fitness-function* (list (decode chromosome))))

(defun decode (l)
      (if l
	  (cons (extract (degray1 (firstn *bits-per-parameter* l))) ; degray1 if defined in ga-dialog.cl
		(decode (nthcdr *bits-per-parameter* l))) ; this funny solution kepps the performance acceptable
	nil))

(defun extract (l &optional (n 0) (i 0)) ; convert list to parameter
  (if l
      (extract (cdr l) (+ n (* (if (car l) 1 0) (expt 2 i))) (+ i 1))
    (+ *lower-bound* (* *itervall-length* (/ (+ n (random 1.0)) *precision*)))))


(defmethod scale-fitness :before ((p population))
  (display-message (format nil "scaling fitness of~%   ~a" p))
  (progn (setq *acc* 0)
	 (if (> (minimum-fitness p) (- (* *fitness-multiple* (average-fitness p)) (maximum-fitness p))) ; non-negative-test
	     (let ((delta (- (maximum-fitness p) (average-fitness p)))) ; normal scale
	       (if (eql delta 0)
		   (progn (setf *scale-a* 1) (setf *scale-b* 0))
		 (progn (setf *scale-a* (/ (* (- *fitness-multiple* 1.0) (average-fitness p)) delta))
			(setf *scale-b* (/ (* (- (maximum-fitness p) (* *fitness-multiple* (average-fitness p))) (average-fitness p)) delta)))))
	   (let ((delta (- (average-fitness p) (minimum-fitness p)))) ; scale as much as possible
	     (if (eql delta 0)
		 (progn (setf *scale-a* 1) (setf *scale-b* 0))
	       (progn (setf *scale-a* (/ (average-fitness p) delta))
		      (setf *scale-b* (- (/ (* (minimum-fitness p) (average-fitness p)) delta)))))))))

(defmethod scale-fitness ((p population))
  (mapcar #'scale-fitness (population p)))

(defmethod scale-fitness ((i individual))
  (incf *acc* (setf (fitness-value i) (+ (* (fitness-value i) *scale-a*) *scale-b*))))

(defmethod normalize-fitness ((p population))
  (mapcar #'normalize-fitness (population p)))

(defmethod normalize-fitness ((i individual))
  (setf (fitness-value i) (* 100 (/ (fitness-value i) *acc*))))


;;; ==========================================================================
;;; the selection process
;;; ==========================================================================

(defmethod select-parent :before ((p population))
  (setq *acc* (random (if *normalize-fitness-p* 100 (maximum-fitness p)))))

(defmethod select-parent ((p population))
  (select-parent-from-list (population p) 0))

(defun select-parent-from-list (list-of-individuals accu)
  (if list-of-individuals
      (if (< (+ (fitness-value (car list-of-individuals)) accu) *acc*)
	  (select-parent-from-list (cdr list-of-individuals)
				   (+ accu (fitness-value (car list-of-individuals))))
	(car list-of-individuals))
    nil))

(defmethod breed ((p population) n population-type)
  (if (> n 0)
      (nconc (cross-over (select-parent p) (select-parent p) population-type)
	     (breed p (- n 1) population-type))
    nil))

(defmethod cross-over ((mate1 individual) (mate2 individual) population-type)
  (if (flip *crossover-probability*)
      (let ((crossing-site (1+ (random (1- *max-allele*)))))
	(list (mutate (make-instance population-type
		:genotype (nconc (firstn crossing-site (genotype mate1))
				 (copy-list (nthcdr crossing-site (genotype mate2))))))
	      (mutate (make-instance population-type
		:genotype (nconc (firstn crossing-site (genotype mate2))
				 (copy-list (nthcdr crossing-site (genotype mate1))))))))
    (list (mutate (make-instance population-type
		:genotype (copy-list (genotype mate1))))
	  (mutate (make-instance population-type
		:genotype (copy-list (genotype mate2)))))))

(defmethod mutate ((i individual))
  (if *mutations-p*
      (dotimes (position (length (genotype i)) i)
	(if (flip *mutation-probability*)
	    (progn (setf *mutation-events* (1+ *mutation-events*))
		   (replace (genotype i) (list (rnd-element *allele-type* (car (nthcdr position (genotype i)))))
			    :start1 position :end1 (1+ position)
			    :start2 0 :end2 1))))
    i))



(defmethod make-generations ((p population) n population-type)
  (if *gin-p* (update-monitor n *mutation-events* p))
  (when (> n 0)
    (let ((base (make-instance 'population
		  :population (breed p (/ *population-size* 2) population-type))))
      (cons (list base
		  (best-individual base)
		  (average-fitness base))
	    (if *proceed* (make-generations base (- n 1) population-type)
	      (progn (display-message (format nil "evaluation interrupted at generation ~a" n))
		     nil))))))

(defmethod evolve :before ((base population) &key
			   (generations 1)
			   (population-type 'individual))
  (declare (ignore generations population-type))
  (setf *mutation-events* 0))

(defmethod evolve ((base population) &key
		   (generations 1)
		   (population-type 'individual))
  (let ((result (make-generations base generations population-type)))
    (make-instance 'generation :total-number generations
			       :populations (mapcar #'car result)
			       :offline-performance (mapcar #'cadr result)
			       :online-performance (mapcar #'caddr result)
			       :mutations *mutation-events*)))


(defmethod evolve ((base generation) &key
		   (generations 1)
		   (population-type 'individual))
  (evolve (car (last (populations base)))
	  :generations generations :population-type population-type))


;;; to test the code without interface:
#|
(setf *current-function* (car *known-functions*))
(defun setup-g (from-this-object)
  (setf *fitness-function* (fitness-function from-this-object))
  (setf *parameter* (nparameter from-this-object))
  (setf *bits-per-parameter* (bits-per-parameter from-this-object))
  (setf *lower-bound* (lower-bound from-this-object))
  (setf *upper-bound* (upper-bound from-this-object))
  (setf *precision* (expt 2 (bits-per-parameter from-this-object)))
  (setf *max-allele* (* *bits-per-parameter* *parameter*))
  (setf *itervall-length* (- *upper-bound* *lower-bound*))
  (setf *gray-coding* (gray-coded from-this-object)))
(setup-g *current-function*)
(defvar *verbose-mode*  nil)
(defun display-message (&rest ignore))
(setf *current-generation* (make-instance 'population
				 :size *population-size*))
(setf *current-generation* (evolve *current-generation* :generations 10))
 |#

;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================
