;;; -*- Mode: LISP; Package: EBG; Syntax:Common-Lisp; Base:10; -*-
;;; ************************************************************************
;;; PORTABLE AI LAB - UNI ZH
;;; ************************************************************************
;;;
;;; Filename:   ebg-dialog
;;; Short Desc: An implementation of Mitchell's EBG-Method
;;; Version:    1.0
;;; Status:     Final Beta
;;; Author:     Stefan Keller
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;




;;; ------------------------------------------------------------------------
;;; Change History:
;;; 28. 6.91  SK  Adding pool buttons DTA
;;; 22.11.91  SK  New Startup, Tools, Demos
;;; 29.01.92  SK  More clickable look and feel
;;; ========================================================================


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :ebg)

(defclass ebg-display (display)
	  ((help-button :accessor help-button
			:initarg :help-button
			:initform nil)
	   (eblist :accessor eblist
		   :initarg :eblist
		   :initform nil)))


(defmethod close-display :before ((disp ebg-display))
  (close-display (technical-window (help-button disp)))
  (close-display (general-window (help-button disp)))
  (loop for d in (eblist disp) do (close-display d)))


;;; ==========================================================================
;;; GLOBAL VARIABLE DECLARATIONS
;;; ==========================================================================


(defvar *verbose-disp*      nil              "Display for [verbose] comments")
(defvar *verbose*           nil              "Flag for verbose comments")
(defvar *exit*              nil              "Quit flag")
(defvar *pause*             nil              "Pause/Continue flag")


;;; ==========================================================================
;;; DIALOG
;;; ==========================================================================
 

(defun make-title (nr)
  (format nil "A ~:R Demo on EBG" nr))


(defun start-ebg-dialog (&optional (from-button nil))
  (setf *verbose-disp* (make-instance 'scroll-display
				     :title "EBG:Verbose Output"
				     :left (round (width *root-window*) 2)
				     :bottom (round (height *root-window*) 3)
				     :width (round (width *root-window*) 3)
				     :height (round (height *root-window*) 3)
				     :borders 1
				     :active nil))
  (if *verbose*
      (progn (activate-display *verbose-disp*)
	     (setf *verbose* *verbose-disp*))
    (deactivate-display *verbose-disp*))
  
  (let* ((disp (make-instance 'ebg-display :title "EBG:Main Window"
			     :left (+ (width pail-lib::*main-window*) (left pail-lib::*main-window*))
			     :bottom (bottom pail-lib::*main-window*) 
			     :width 300 :height 200
			     :borders 1))
         (menu-button (make-instance 'pop-up-button
			:label "Demos"
			:width 100))
	 (exit-button (make-instance 'push-button :label "Exit" :width 70))
	 (tool-button (make-instance 'push-button :label "Tool" :width 100))
	 (help-button (setf (help-button disp)
			(make-instance 'help-button
			:technical (add-path "ebg-desc.tec" *ebg-path*)
			:general   (add-path "ebg-desc.gen" *ebg-path*)
			:subject   "EBG")))
	 (menu (make-instance 'menu
		 :items (let ((mlist nil))
			   (dotimes (i *number-of-demos* (nreverse mlist))
			     (setf mlist 
			       (cons (list (format nil "Demo ~a" (1+ i)) 
					   `(lambda () 
					      (demo-control ,(1+ i) 
							    ,menu-button 
							    ,exit-button))
					   (make-title (1+ i)))
				     mlist))))
       		 :query "Demos and Tool")))

    (setf (menu menu-button) menu)

    (copy-mask *pail-logo* 0 0 disp
	       (- (width disp) (width *pail-logo*) 15)
	       (- (height disp) (height *pail-logo*) 15))
  
    (setf (font disp) (open-font :courier :italic 20 :weight :bold))
    (write-display disp 
                   "Welcome to..." 
                   15 (- (height disp) 15 20)) 
    (setf (font disp) (open-font :courier :italic 25 :weight :bold))
    (write-display disp "Explanation"     15 (- (height disp) 15 50))
    (write-display disp "Based"           15 (- (height disp) 15 75))
    (write-display disp "Generalization"  15 (- (height disp) 15 100))
    (setf (font disp) *default-font*)

    (set-button exit-button disp 
		:left (- (width disp) (width exit-button) 15)
		:bottom 15
		:action `(lambda nil
			   ;(close-display *verbose-disp*)
			   (close-display ,disp)
			   (if ,from-button (reset-button ,from-button))))

    (set-button help-button disp :left 15 :bottom 15)
  
    (set-button menu-button disp 
		:left (/ (- (width disp) (width menu-button)) 2)
		:bottom (+ 20 (height help-button)))
    (set-button tool-button disp 
		:left (/ (- (width disp) (width menu-button)) 2)
		:bottom 15
		:action `(lambda ()
			   (push (start-ebg-tool ,tool-button)
				 (eblist ,disp))))
    disp))


;;; ==========================================================================
;;; TOOL
;;; ==========================================================================

(defun start-ebg-tool (&optional (from-button nil))
  (setf *default-push-button-size-p* t)
  (setf *default-push-button-width* 140)
  
  (setf *verbose-display* (make-instance 'scroll-display
				     :title "EBG:Verbose Output"
				     :left (round (width *root-window*) 2)
				     :bottom (round (height *root-window*) 3)
				     :width (round (width *root-window*) 2)
				     :height (round (height *root-window*) 2)
				     :borders 1
				     :active nil))
  (setf *verbose-stream* *verbose-disp*)
  (let* ((ebg-left 
	     (round (width *root-window*) 3)
	    )
	 (ebg-bottom-offset 
	     (- (round (* 2 (height *root-window*)) 3) 25)
	   )
	 (verb (make-instance 'radio-button :label "Verbose Mode"
			      :status *verbose*
			      :action 
			      #'(lambda ()
				  (if *verbose*
				      (progn
					(setf *verbose* nil)
					(deactivate-display *verbose-disp*))
				    (progn
				      (setf *verbose* *verbose-disp*)
				      (activate-display *verbose-disp*))))))
	 
	 (ebg-display (make-instance 'ebg-display :title "A tool for running EBG"
                                    :left ebg-left :bottom ebg-bottom-offset 
                                    :width (round (width *root-window*) 3)
				    :height (round (height *root-window*) 3)
                                    :font *italic-font*))
         (help-button  (setf (help-button ebg-display)
			 (make-instance 'help-button 
			 :technical (add-path "tool-desc.tec" *ebg-path*)
			 :general   (add-path "tool-desc.gen" *ebg-path*)
			 :subject "EBG"
			 :width (round (width ebg-display) 5))))
         (exit-button  (make-instance 'push-button 
			 :label "Exit" :width (round (width ebg-display) 5)))
	 (run-button   (make-instance 'push-button
			 :label "Generalize" :width (round (* 2 (width ebg-display)) 5)))
	 (input-button  (make-instance 'pool-button
			  :name "Explanation input:"
			  :pool pail::*pail-pool*
			  :value ""
			  :width (round (width ebg-display) 3)
			  :border t
			  :target-class 'ebg-tree
			  :show-function `(lambda (item)
					    (push (make-instance 'ebg-browser
						    :starting-tree item)
						  (eblist ,ebg-display)))))
	 (output-button  (make-instance 'pool-button
			   :name "Rule output:"
			   :pool pail::*pail-pool*
			   :value ""
			   :width (round (width ebg-display) 3)
			   :target-class 'rule
			   :border t
			   :show-function #'(lambda (item)
					      (documentation-print
					       (print-rule item nil)))
			   ))
	 (file-button (make-instance 'file-button
			:w-directory *ebg-path*))
         )
    
;    (activate-display *verbose-display*)
    
    (set-button help-button ebg-display :left 5 :bottom 5)
    (set-button exit-button ebg-display
                :left (- (width ebg-display) (width help-button) 5) 
                :bottom 5
                :action `(lambda nil
;                           (close-display *verbose-display*)
                           (close-display ,ebg-display)
                           (reset-button ,from-button)))
    (set-button run-button ebg-display
                :left (round (- (width ebg-display) (width run-button)) 2)
                :bottom 5
                :action `(lambda nil
			   (setf (button-value ,output-button)
			     (make-instance 'pool-item
			       :start-object (ebg-m (start-object (button-value ,input-button)) nil)
			       ))
			   (reset-button ,run-button)))
    
;    (draw-line ebg-display 0 31 (width ebg-display) 31)
    
    (set-button input-button ebg-display
		:left (- (width ebg-display) (width input-button) 5)
		:bottom (round (height ebg-display) 4))

    (set-button output-button ebg-display
		:left (- (width ebg-display) (width input-button) 5)
		:bottom (+ (bottom input-button) (height input-button) 5))

    (set-button verb ebg-display :left (round (width ebg-display) 6)
		:bottom (+ (bottom output-button) (height input-button) 5))
    (setf (font ebg-display) (open-font :courier :italic 20 :weight :bold))
    (write-display ebg-display "Explanation" 10 (- (height ebg-display)
							      (font-character-height (font ebg-display))))
    (write-display ebg-display "Based"  10 (- (height ebg-display)
							      (*  2 (font-character-height (font ebg-display)))))
    (write-display ebg-display "Generalization"  10 (- (height ebg-display)
							      (*  3 (font-character-height (font ebg-display)))))
    (copy-mask *pail-logo* 0 0 ebg-display 
               (- (width ebg-display) 5 (width *pail-logo*))
               (- (height ebg-display) 5 (height *pail-logo*)))
    (set-button file-button ebg-display
		:left (- (width ebg-display) (width file-button) 7)
		:bottom (- (height ebg-display) (height file-button) (cw:bitmap-height *pail-logo*) 20))
    ebg-display
    ))



;;; ==========================================================================
;;; DEMO CONTROL
;;; ==========================================================================


(defun demo-control (nr menu-button main-exit-button)
  "Called by start-ebg-dialog. It's a plain window waiting for user clicking
   on Start button."

  (disable-button menu-button) 
  (disable-button main-exit-button)
  (let* ((*default-font* (findfont (width *root-window*) 1140 13))
	 (dispwidth (min 500 (round (width *root-window*) 2)))
	 (dispheight (min 260 (round (height *root-window*) 2)))
	 (disp (make-instance 'display :title (concatenate 'string "EBG:" (make-title nr))
			      :width dispwidth
			      :height dispheight
			      :left (- (width *root-window*) dispwidth 15)
			      :bottom (- (height *root-window*) dispheight 20)
			      :borders 1))
	 (exit-button (make-instance 'push-button :label "Exit"
				     :width (round (width disp) 5)))
	 (help-button (make-instance 'help-button
			:width (round (width disp) 5)
			:subject (concatenate 'string "EBG:" (make-title nr))
			:technical (add-path "demo-desc.tec" *ebg-path*)
			:general (add-path 
				  (concatenate 'string
				    "demo" (write-to-string nr) "-desc.gen")
				  *ebg-path*) ))
	 (start-button (make-instance 'push-button  :label "Start"    :width (round (width disp) 5)))
	 (cont-button  (make-instance 'push-button  :label "Continue" :width (round (width disp) 5)))
	 )

    (setf (font disp) (open-font :courier :italic 20 :weight :bold))
    (write-display disp 
		   (make-title nr)
		   15 (- (height disp) 15 (font-ascent (font disp))))
    (setf (font disp) *default-font*)

    (copy-mask *pail-logo* 0 0 disp 
	       (- (width disp) (width *pail-logo*) 15)
	       (- (height disp) (height *pail-logo*) 15))

    (set-button exit-button disp 
		:left (- (width disp) (width exit-button) 15) 
		:bottom 15
		:action `(lambda ()
			   (setf *exit* t)
			   (enable-button ,menu-button)
			   (enable-button ,main-exit-button)
			   (close-display ,disp)))

    (set-button help-button disp :left 15 :bottom 15)
    
    (set-button start-button disp 
		:left (round (+ 15
				(width start-button)
				(round (- (width disp)
				      30
				      (* 4 (width start-button)))
				   3)))
		:bottom 15
		:action
		#'(lambda ()
					;(disable-button exit-button)
		    (demo-dialog-control nr disp 
					 start-button cont-button exit-button)))
    
    (set-button cont-button disp 
		:left (round (+ 15
				(* 2 (width start-button))
				(* 2 (round (- (width disp) 
					   30 
					   (* 4 (width start-button)))
					3))))
		:bottom 15
		:action 
		#'(lambda ()
		    (disable-button exit-button)
		    (setf *pause* (not *pause*))))
    (disable-button cont-button)
    ))



(defun demo-dialog-control (nr parent start-button cont-button exit-button)
  "Called by demo-control. Sets up the in/out windows common for all demos and 
   typically used all over this module. 
   Immediately handles control over to demo-dialog-x."

  (let* ((*default-font* (findfont (width *root-window*) 1140 10))
	(guide-disp (make-instance 'scroll-display
		      :title "EBG:Guide & Comments"
		      :font *default-font*
		      :parent parent
		      :left (round (width parent) 24)
		      :bottom 50
		      :width (- (width parent) (round (width parent) 12))
		      :height (round (height parent) 2)
		      :borders 1))
	(rule-disp (make-instance 'scroll-display
		     :title "EBG:Input Rules"
		     :left 10   :bottom (round (height *root-window*) 2)
		     :width (- (round (width *root-window*) 2) 10)
		     :height (- (round (height *root-window*) 2) 20)
		     :font *default-font*
		     :borders 1))
	(trace-disp (make-instance 'scroll-display
		      :title "EBG:Trace Output"
		      :font *default-font*
		      :left 10   :bottom 5
		      :width (round (* 1 (width *root-window*)) 3)
		      :height (round (height *root-window*) 2) 
		      :borders 1))
	)
    (labels ((clean-up-all ()
	       (clean-up rule-disp trace-disp guide-disp))
	     (reset-all ()
	       (unless *exit* (enable-button exit-button)
		       (reset-button cont-button)
		       (disable-button cont-button)
		       (reset-button start-button)))
	     )

      ;; Prepare buttons
      (disable-button cont-button)
      (setf *pause* nil)
      (setf *exit* nil)
      

      (let ((fun-sym (find-symbol (write-to-string
				   (read-from-string (format nil "demo-dialog-~a" nr))) :ebg)))
	(if fun-sym
	    (funcall fun-sym 
		     guide-disp trace-disp rule-disp
		     cont-button exit-button)
	  (format t "~%Demo not implemented!")))
      
      ;; The last wait, then quit graciously to EBG:Demo
      (clean-up-all)
      (reset-all)
      )))


;;; ==========================================================================
;;; AUX FN'S
;;; ==========================================================================


(defun clicked-on-continue (cont-button exit-button disp)
  (unless *exit*
    (enable-button cont-button)
    (reset-button cont-button)
    (enable-button exit-button)
    (format-display disp "~&~%[Press <Continue> Button]")
    (mp:process-wait "wait-on-click" (function (lambda () (or *exit* *pause*))))
    (if *exit* 
	(clean-up disp)   ;user clicked in exit-button
      (setf *pause* nil))
    (not *exit*)))


(defun clean-up (&rest disp-list)
  (mapcar #'close-display disp-list))



;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================
