;;; -*- Mode: LISP; Package: cky; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   cky-grammar-gui.cl
;;; Short Desc: general user interface for the grammar routines
;;;            
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   18.2.92 - FB
;;; Author(s):  Fabio Baj
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;; Rod Johnson wrote a character oriented version of the interface
;;; Fabio Baj  wrote  the graphic user interface
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :cky)

;;GUI variables-------------------------------
(defvar *gram-disp*) (defvar *gram-msg-disp*)    (defvar *gram-subdisp*)        
(defvar *new-entry)     (defvar *rhslist* )    
(defvar *delete-rule*)     (defvar *find-rule*)
(defvar *load-grammar*)    (defvar *save-grammar*)
(defvar *exit-grammar*)    (defvar *help-grammar*)
(defvar *convert-grammar*)(defvar *view-grammar*)
(defvar *options-grammar*)
(defvar *gram-disp-width*)(defvar *gram-disp-height*)
(defvar *gram-disp-left*)(defvar *gram-disp-bottom*)
(defvar *gram-subdisp-width*)(defvar *gram-subdisp-height*)
(defvar *gram-msg-disp-width*)(defvar *gram-msg-disp-height*)
(defvar *gram-subdisp-left*)(defvar *gram-subdisp-bottom*)
(defvar *grammar-filename* nil)
;;=============   CKY variables =========================
(defvar *gfile*)
(defvar *grammar*) (defvar *syn-features* nil) (defvar lookahead) 



(defun init-grammar-tool()
  (setq gin::*default-display-border* 1)
  (if (null *grammar*) (setq *grammar-filename* nil))
  (setf *gram-disp-width*  (scalew 580) ) (setf *gram-disp-height* (scaleh 400))
  (setf *gram-disp-left*    (scalew 400) )  (setf *gram-disp-bottom*   (scaleh 100))
  (setf *gram-subdisp-width*  (scalew 380))
  (setf *gram-msg-disp-width*  *gram-disp-width* )  (setf *gram-msg-disp-height* (scaleh 50))
  (setf *gram-subdisp-height* (- *gram-disp-height* (+  36 *gram-msg-disp-height*)))
  (setf *gram-subdisp-left*   (- *gram-disp-width*  *gram-subdisp-width*) )
  (setf *gram-subdisp-bottom* (+  18 *gram-msg-disp-height* ))
  (setf *left-of-buttons* (scalew 20)) (setf *size-of-buttons* (scalew 160))
  (setf *new-entry* (make-instance 'push-button :font (my-findfont 13)
		      :width *size-of-buttons* :label "New Rule"))
  (setf *delete-rule* (make-instance 'push-button :font (my-findfont 13)
			:width *size-of-buttons* :label "Delete Rule"))
  (setf *find-rule* (make-instance 'push-button :font (my-findfont 13)
		      :width *size-of-buttons* :label "Expand Cat"))
  (setf *load-grammar* (make-instance 'push-button :font (my-findfont 13)
			 :width *size-of-buttons* :label "Load Grammar"))
  (setf *save-grammar* (make-instance 'push-button :font (my-findfont 13)
			 :width *size-of-buttons* :label "Save Grammar"))
  (setf *exit-grammar* (make-instance 'push-button :font (my-findfont 13)
			 :width *size-of-buttons* :label "OK"))
  (setf *convert-grammar* (make-instance 'push-button :font (my-findfont 13)
			    :width *size-of-buttons* :label "Convert Grammar"))
  (setf *view-grammar* (make-instance 'push-button :font (my-findfont 13)
			    :width *size-of-buttons* :label "View Grammar"))
  (setf *help-grammar* 
    (make-instance 'help-button :label "Help"
		   :width *size-of-buttons*   :font (my-findfont 13)
		   :subject "Chart Parsing"
		   :technical
		   (add-path "grammar-help.tec" (add-subdir *pail-path* "cky"))
		   :general 
		   (add-path "grammar-help.gen" (add-subdir *pail-path* "cky")))))
  
  
(defun grammar-tool()
  (let ((done nil))
    (init-grammar-tool)
    (setf *gram-disp* (make-instance 'display :font (my-findfont 13)
			:title "CKY: Grammar Tool" :width *gram-disp-width* :height  *gram-disp-height* 
				     :left *gram-disp-left* :bottom *gram-disp-bottom*))
    (push *gram-disp* *open-windows*)
    (setf *gram-subdisp* (make-instance  'scroll-display  :parent *gram-disp* 
					:title (concatenate 'string
						 "Grammar: " (or *grammar-filename*  ""))
					:width *gram-subdisp-width* :height  *gram-subdisp-height* 
					:left *gram-subdisp-left*   :font (my-findfont 10)
					:bottom  *gram-subdisp-bottom*))
    (setf *gram-msg-disp* (make-instance  'scroll-display :parent *gram-disp* :font (my-findfont 10)
					 :title "CKY: Messages " :width *gram-msg-disp-width* 
					 :height  *gram-msg-disp-height* 
					 :left 0
					 :bottom 0 ))
 (reset-b-f 9 (height *gram-subdisp*))
 (set-button *help-grammar* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f))
 (set-button *exit-grammar* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f)
	     :action #'(lambda nil (setf done t) (close-display *gram-disp*)))
 (set-button *view-grammar* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f)
	     :action '(lambda nil
		       (clear-scroll *gram-subdisp*)
		       (protect-display *gram-disp* t)
		       (reset-button *view-grammar*)
		       (viewsyn)
		       (protect-display *gram-disp* nil)))
 (set-button *convert-grammar* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f)
	     :action '(lambda nil
		       (clear-scroll *gram-subdisp*)
		       (protect-display *gram-disp* t)
		       (setq *grammar* (convert))
		       (reset-button *convert-grammar*)
		       (viewsyn)
		       (protect-display *gram-disp* nil)))
 
 (set-button *load-grammar* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f)
	     :action '(lambda nil
		       (clear-scroll *gram-subdisp*)
		       (protect-display *gram-disp* t)
		       (in-package :cky)
		       (loadsyn)
		       (reset-button *load-grammar*)
		       (protect-display *gram-disp* nil)
		       (viewsyn)))
 (set-button *save-grammar* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f)
	     :action '(lambda nil
		       (clear-scroll *gram-subdisp*)
		       (protect-display *gram-disp* t)
		       (savesyn)
		       (reset-button *save-grammar*)
		       (protect-display *gram-disp* nil)))
 (set-button *find-rule* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f)
	     :action '(lambda nil
		       (protect-display *gram-disp* t)
		       (format-display *gram-subdisp* "~A"
			(findsyn 
			 (to-symbol (no-blanks
				 (ask "Category?  "
				   :left (scalew 200) :bottom (scaleh 400)
				  
				   :font (my-findfont 10))))))
		       (reset-button *find-rule*)
		       (protect-display *gram-disp* nil)))
 (set-button *delete-rule* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f)
	     :action '(lambda nil
		       (protect-display *gram-disp* t)
		       (junksyn(to-symbol (no-blanks (ask "Enter Category :  "
					     :left (scalew 200) :bottom (scaleh 400)
					    :font (my-findfont 10) ))))
		       (viewsyn)
		       (reset-button *delete-rule*)
		       (protect-display *gram-disp* nil)))
 (set-button *new-entry* *gram-disp*
	     :left *left-of-buttons*
	     :bottom (b-f)
	     :action '(lambda nil
		       (protect-display *gram-disp* t)
		       (makesyn)
		       (reset-button *new-entry*)
		       (protect-display *gram-disp* nil)))
  (if (boundp '*grammar*) (progn (clear-scroll *gram-subdisp*)(viewsyn)))
 (if (not *demo-in-execution*) (mp::process-wait "exitin" #'(lambda nil done)))
))

(defun loadsyn()
  (let  ((filename
	  (cond ((not *demo-in-execution*) (choose-file-dialog :directory  (or *grammar-filename*
									       pail-lib::*user-dir*
									       (add-path "grammar/" *cky-path*))
							       :button-string "Load"
							       :prompt "Load grammar file..."
							       :left 400 :bottom 200))
		((= *demo-in-execution* 1) "grammar/g0.cgr")
		((= *demo-in-execution* 2) "grammar/number-grammar.cgr"))))
    (when filename (load-grammar filename))))

(defun load-grammar (filename)
  (if (probe-file filename)
      (with-open-file (infile filename :direction :input) 
	(setq *gfile* infile)(setq *grammar-filename* filename)
	(setq *grammar*  nil)
	(format-display *gram-msg-disp* "Loading grammar from ~A" filename)
	(setf (title *gram-subdisp*)(concatenate 'string "Grammar: " filename))
	(do ((entry (read infile nil 'eof)(read infile nil 'eof)))
	    ((eq 'eof entry))
	  (if (not (is-a-g-rule entry))
	      (format-display *gram-subdisp* "BAD RULE: ~A" entry)
	    (setq *grammar* (newsyn entry *grammar*)))
	  (sortsyn *grammar*)))
    (file-does-not-exist filename :font (my-findfont 13))))


;;;===========================================================================
(defun is-a-g-rule (r)
  (cond ( (atom r) nil)
	( t (and (is-lhs (car r) )
		 (is-rhs (cdr r))))))
(defun is-lhs(s)(atom s))
(defun is-rhs(s)(or (null s)(is-seq s)))
(defun is-seq(s)
  (cond ((atom s) nil)
	((and (is-node (car s))(null (cdr s))) t)
	((and (is-node (car s))(is-seq (cdr s))) t)))
		
(defun is-node(s)(or (is-self s) (is-alts s)(is-star s )(is-opts s)))
(defun is-self(s)
  (cond ((atom s) nil)
	(t (and (eq 'self (car s))
		(atom  (cadr s))))))
(defun is-star(s)(cond ((atom s) nil)
		       (t (and (eq 'star (car s))
			       (is-rhs (cdr s))))))
(defun is-alts(s)(cond ((atom s) nil)
		       (t(and (eq 'alts (car s))
			     (all-rhs(cdr s))))))
(defun all-rhs (s)(and (mapcar #'is-rhs s)))
			      
(defun is-opts(s)(cond ((atom s) nil)
		       (t(and (eq 'opts (car s))
			     (is-rhs(cdr s))))))

;;=========================================================




(defun sortsyn(grammar)())
(defun newsyn (g-rule grammar)
  (cond (g-rule
	 (dest-insert g-rule
		 grammar
		 '(lambda (x y) (g-alphalessp (car x) (car y)))
		 t))))

(defun g-alphalessp (x y) 
  (string< (string x)(string y)))

  

;;=============================================================

(defun findsyn (cat)
  (let ((rules (mapcan
		#'(lambda (prod-rule)
		  (cond ((eq (car prod-rule) (intern cat :cky))
			 (list prod-rule))))
		*grammar*)))
    (cond (rules
	   (seerules rules ))
	  (t
	   (format nil  "No rules for ~A" cat)))))
	    

(defun map-conc (fn list)
  (cond ((null list) "")
	(t (concatenate 'string (funcall fn (car list))
			(map-conc fn (cdr list))))))

(defun viewsyn ()  
  (format-display *gram-subdisp* 
"-----------------------------
          Current Grammar:
-----------------------------")
  (format-display *gram-subdisp* (seerules *grammar*)))

(defun seerules (grammar)
  (cond ((null grammar) (format nil "~%"))
	(t (concatenate 'string 
	     (showrule (car grammar)) (format nil "~%")
	     (seerules (cdr grammar))))))

(defun showrule (g-rule)
 (format nil  " ~A -> ~A"
    (string (car g-rule))
    (cond ((cdr g-rule)
	   (cond ((eq (caadr g-rule) 'alts)
		  (concatenate 'string
		    (showrhs (cadadr g-rule))
		    (map-conc '(lambda (seq)
				(concatenate 'string
				  (format nil " | ")
				  (showrhs seq)))
			      (cddadr g-rule))))
		 (t (showrhs (cdr g-rule))))))))

(defun showrhs (rhs)
  (concatenate 'string
    (showcat (car rhs))
    (map-conc '(lambda (cat) 
		(concatenate 'string " " (showcat cat)))
	      (cdr rhs))))

(defun showcat (cat)
  (cond ((eq (car cat) 'self)
	 (concatenate 'string (format nil "~A" (cadr cat))
		      (showfeats (cddr cat))))
	((eq (car cat) 'star) (cond ((null (cddr cat)) 
				     (concatenate 'string
				       (showcat (cadr cat))
				       "*"))
				    (t (concatenate 'string
					 "("
					 (showrhs (cdr cat))
					 ")*"))))
	((eq (car cat) 'alts)  (concatenate 'string  
				 "("
				 (showrhs (cadr cat))
				 (map-conc '(lambda (node)
					     (concatenate 'string
					       " | "
					       (showrhs node)))
					   (cddr cat))
				 ")"))
	((eq (car cat) 'opts)  (concatenate 'string "("
					    (showrhs (cdr cat))
					    ")"))
	(t  (concatenate 'string  "??? " (format nil "~A" cat) " ???"))))

(defun showfeats (features)
  (cond ((and *syn-features* features)
	 (concatenate 'string 
	   " {"
	   (showeqn (car features))
	   (map-conc '(lambda (eqn) 
		       (concatenate 'string
			 ", "
			 (showeqn eqn)))
		     (cdr features))
	   "}"))
	(t "")))

(defun showeqn (eqn)
  (cond ((atom eqn)
	 (format nil "~A" eqn))
	(t (concatenate 'string 
	     (showpath (car eqn))
	     (cond 
	      ((cdr eqn) (concatenate 'string " = " (showpath (cdr eqn))))
	      ( t ""))))))
  

(defun showpath (path)
  (cond ((atom path)
	 (format nil "~A"  path))
	(t (concatenate 'string 
	     "<"
	     (cond ((eq (car path) 'up)
		    "up")
		   ((eq (car path) 'down)
		    "down")
		   (t (showpath (car path))))
	     (map-conc '(lambda (node)
			 (concatenate 'string " " (showpath node)))
		       (cdr path))
	     ">"))))


;-------------------------------------------------------------

(defun convert ()
  (mapcar '(lambda (g-rule)
		   (cons (car g-rule) (convertrhs (cdr g-rule))))
	  *grammar*))

(defun convertrhs (rhs)
  (cond ((null rhs) nil)
	(t (cond ((eq (caar rhs) 'self)
		  (cons (list 'self (cdar rhs)) (convertrhs (cdr rhs))))
		 ((eq (caar rhs) 'opts) (cons (cons 'opts (convertrhs (cdar rhs)))
					      (convertrhs (cdr rhs))))
		 ((eq (caar rhs) 'alts) 
		  (cons (cons 'alts (mapcar 'convertrhs (cdar rhs)))
			(convertrhs (cdr rhs))))
		 ((eq (caar rhs) 'star)
		  (cons (cons 'star (convertrhs (cdar rhs)))
			(convertrhs (cdr rhs))))
		 (t (cons (car rhs) (convertrhs (cdr rhs))))))))

;;;====================================================================
(defun makesyn ()  
  (let* ((cat (to-symbol(no-blanks (ask "Left hand category? ... "
			     :left (scalew 200) :bottom (scalew 400)
			      :font (my-findfont 10))))))
    (cond ((string= cat "") (print 'erre)))
      (clear-scroll *gram-subdisp*)
    (format-display *gram-subdisp* "Existing rules for ~a : " cat  )
    (setq cat (intern  cat :cky)) 
    (format-display *gram-subdisp* "~A" (findsyn cat))
    (setq *rhslist* (mapcar #'to-symbol
      (scan-string 
       (ask 
	(format-display 
	 nil 
	 "Enter right hand side now...   ~A -> " cat)
	 :left (scalew 200) :bottom (scalew 400)
			      :font (my-findfont 10)))))
    (let* ((lookahead (myratom))
	   (rhs (readrhs nil)))
      (setq *grammar* (newsyn (cons cat  rhs) *grammar*)))
    (format-display *gram-subdisp*  "Rules for ~A are now:" cat)
     (format-display *gram-subdisp* "~A"   (findsyn cat))))
	

(defun readcat ()
  (let ((cat lookahead))
       (setq lookahead (myratom ))
       (cond ((eq lookahead '\{)
	      (cons 'self (cons cat (readfeats nil ))))
	     (t (list 'self (intern cat :cky))))))


(defun readrhs (rhs  )
  (cond ((null lookahead)
	 (reverse rhs))
	(t 
	 (cond ((string= lookahead "|")
		(setq lookahead (myratom  ))
		(let* ((alt (readrhs nil  ))
		       (this (reverse rhs))
		       (tail (cond ((and (eq (caar alt) 'alts)
					 (null (cdr alt)))
				    (cons 'alts (cons this (cdar alt))))
				   (t (list 'alts this	alt)))))
		  (list tail)))
	       ((string= lookahead "*")
		(setq lookahead (myratom  ))
		(let ((this (car rhs)))
		  (cond ((eq (car this) 'opts)
			 (readrhs (cons (cons 'star (cdr this))
					(cdr rhs)) ))
			(t
			 (readrhs (cons (list 'star this)
					(cdr rhs))
				  )))))
	       ((string= lookahead "(")
		(setq lookahead (myratom  ))
		(let ((seq (readrhs nil  )))
		  (cond ((string= lookahead ")")
			 (setq lookahead (myratom  ))
			 (cond ((and (null (cdr seq))
				     (eq (caar seq) 'alts))
				(readrhs (cons (car seq) rhs)  ))
			       (t (readrhs (cons (cons 'opts seq) rhs)
					   ))))
			(t (badrule ") expected after" seq)))))
	       ((string= lookahead ")")
		(reverse rhs))
	       (t (readrhs (cons (readcat) rhs)  ))))))

(defun myratom ()
  (setq lookahead (pop *rhslist*)))



;;;==================================================



(defun junksyn (lhs)
  (let ((lhs1 (intern lhs :cky)))
    (setq *grammar*
      (mapcan #'(lambda (g-rule)
		  (cond 
		   ((or (not (eq (car g-rule) lhs1))
			(eq 
			 (cky-y-or-n-dialog
			  (format nil "~A~A~A"
				  "Delete [ "
				  (showrule g-rule)
				  "] ?? ")
			  
			 ) :no))
		    (list g-rule))))
	      *grammar*))
    (format-display *gram-subdisp* "Rules for ~A are now:" lhs1)
    (format-display *gram-subdisp* "~A~% " (findsyn lhs1))))


;;===================================================

(defun savesyn ()
  (when *grammar-filename*
    (let  ((filename (filename (choose-new-file-dialog 
				:directory (or *grammar-filename*
					       pail-lib::*user-dir*
					       (add-path "grammar/" *cky-path*))
				:prompt "Save grammar (*.cgr) ..."
				:button-string "Save"
				:left 400
				:bottom 200))))
      (if (not(string= filename ""))
	  (with-open-file (port filename :direction :output :if-exists :supersede)  
	    (format-display *gram-msg-disp*  "Saving grammar to ~A" filename)
	    (dumpsyn *grammar* port))
	(format-display *gram-msg-disp*  "Filename not given : nothing saved")))))

(defun dumpsyn (grammar port)
(format port "
; Grammar source file

; Rule format is:-

; <rule>	::= ( <lhs> . <rhs> )
; <lhs>		::= <symbol>
; <rhs>		::= <nil> | ( <seq> )
; <seq>		::= <node> | <node> <seq> 
; <node>	::= <self> | <alts> | <star> | <opts>
; <self>	::= ( self . <category> )
; <alts>	::= ( alts . <rhs> ... <rhs> )
; <star>	::= ( star . <rhs> )
; <opts>	::= ( opts . <rhs> )
" )
(format port"
; <category>	::= ( <symbol> . <features> )
; <features>	::= <nil> | <feature> ... <feature>
; <feature>	::= <symbol> | ( <function> . <value> )
; <function>	::= ( <variable> ) | ( <variable> <symbol> ... <symbol> )
; <variable>	::= up | down
; <value>	::= <function> | <symbol> | <number>

"  ) 
(mapc #'(lambda (entry)
	  (print entry port))
	grammar))
