;;; -*- Mode: LISP; Package: pail-lib; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   lexicon-tool.cl
;;; Short Desc: general user interface for the lexicon routines
;;;            
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   18.2.92 - FB
;;; Author(s):  Fabio Baj 
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;; Rod Johnson wrote a character oriented version of the interface
;;; Fabio Baj  wrote  the graphic user interface
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :pail-lib)

;;GUI variables-------------------------------
(defvar *lex-disp*)     (defvar *lex-subdisp*)        
(defvar *new-entry*)    
(defvar *delete-word*)     (defvar *find-word*)
   (defvar *save-lexicon*)
(defvar *options-lexicon*) (defvar **)
(defvar *lex-disp-width*)(defvar *lex-disp-height*)
(defvar *lex-disp-left*)(defvar *lex-disp-bottom*)
(defvar *lex-subdisp-width*)(defvar *lex-subdisp-height*)
(defvar *msg-subdisp-width*)(defvar *msg-subdisp-height*)
(defvar *lex-subdisp-left*)(defvar *lex-subdisp-bottom*)
(defvar *left-of-buttons*) (defvar *size-of-buttons*)
(defvar *lexicon-filename* nil)
;;   CKY variables-----------------------------
(defvar *lfile*)
(defvar *separators*  '(#\, #\*  #\  #\= #\~ #\< #\> #\& #\+ #\( #\) #\[ #\]   #\|  #\: ))
(defvar *recgn-separators*  '(#\* #\( #\) #\| ))

(defclass lexicon-tool (display)
  ((new-entry-button           :accessor new-entry-button
			       :initarg :new-entry-button
			       :initform nil)
   (delete-entry-button        :accessor delete-entry-button
			       :initarg :delete-entry-button
			       :initform nil)
   (find-entry-button          :accessor find-entry-button
			       :initarg :find-entry-button
			       :initform nil)
   (load-button                :accessor load-button
			       :initarg :load-button
			       :initform nil)
   (save-button                :accessor file-button
			       :initarg :file-button
			       :initform nil)
   (exit-button                :accessor exit-button
			       :initarg :exit-button
			       :initform nil)
   (help-button                :accessor help-button
			       :initarg :help-button
			       :initform nil)
   (content-needs-saving-p     :accessor content-needs-saving-p
			       :initarg :content-needs-saving-p
			       :initform nil)
   (message-window             :accessor message-window
			       :initarg :message-window
			       :initform nil)
   (content-window             :accessor content-window
			       :initarg :content-window
			       :initform nil)))

(defmethod initialize-instance ((lt lexicon-tool)
				&key (filename nil))
  (let ((width (scalew 160))
	(left (salew 20)))
    (call-next-method :width (scalew 500)
		      :left (scalew 400)
		      :height (scaleh 400)
		      :bottom (scaleh 100))
    (setq gin::*default-display-border* 1)
    (setf (message-window lt)
	  (make-instance 'scroll-display
			 :parent lt
			 :font (my-findfont 11)
			 :title "Messages:"
			 :width (width lt)
			 :height (scaleh 50)))
    (setf (content-window lt)
	  (make-instance 'scroll-display :parent lt
			 :font (my-findfont 11) 
			 :title (concatenate 'string
					     "Lexicon: "
					     (if filename (namestring filename)
					       ""))
			 :width (scalew 300)
			 :height (- (height lt) (+ 36 (height (message-window lt))))
			 :left (- (width lt)  (width (message-window lt)))
			 :bottom  (+ 18 (height (message-window lt)))))
    (setf new-entry-button
	  (make-instance 'push-button   
			 :font (my-findfont 13) 
			 :width width
			 :left left
			 :label "New Entry"))
    (setf delete-entry-button
	  (make-instance 'push-button  
			 :font (my-findfont 13) 
			 :width width
			 :left left
			 :label "Delete Word"))
    (setf find-entry-word
	  (make-instance 'push-button  :font (my-findfont 13) 
			 :width width
			 :left left
			 :label "Find Word"))
    (setf load-button
	  (make-instance 'push-button  
			 :font (my-findfont 13) 
			 :width width
			 :left left
			 :label "Load Lexicon"))
    (setf save-button
	  (make-instance 'push-button  
			 :font (my-findfont 13) 
			 :width width
			 :left left
			 :label "Save Lexicon"))
    (setf exit-button
	  (make-instance 'push-button  
			 :font (my-findfont 13) 
			 :width width
			 :left left
			 :label "OK"))
    (setf help-button   
	  (make-instance 'help-button 
			 :label "Help"
			 :width size-of-buttons
			 :font (my-findfont 13)
			 :subject "Lexicon Tool"
			 :technical (add-path "lexicon-help.tec" 
					      (add-subdir *pail-path* "help"))
			 :general   (add-path "lexicon-help.gen" 
					      (add-subdir *pail-path* "help"))))))

				
  
(defun init-lexicon-tool()
  (if (or (null *lexicon*)(equal *lexicon* '(nil))(not (boundp '*lexicon*)))
      (setq *lexicon-filename* nil))
  (setf *lex-disp-width*  (scalew 500))  (setf *lex-disp-height* (scaleh 400))
  (setf *lex-disp-left*  (scalew 400))   (setf *lex-disp-bottom* (scaleh 100))
  (setf *lex-subdisp-width*  (scalew 300))
  (setf *msg-subdisp-width*  *lex-disp-width* )  (setf *msg-subdisp-height* (scaleh 50))
  (setf *lex-subdisp-height* (- *lex-disp-height* (+ 36 *msg-subdisp-height*)))
  (setf *lex-subdisp-left*   (- *lex-disp-width*  *lex-subdisp-width*) )
  (setf *lex-subdisp-bottom* (+ 18 *msg-subdisp-height* ))
  (setf *left-of-buttons* (scalew  20)) (setf *size-of-buttons* (scalew 160))
  (setf *help-lexicon*   
    (make-instance 'help-button :label "Help"
		   :width *size-of-buttons*   :font (my-findfont 13)
		   :subject "Chart Parsing"
		   :technical
		   (add-path "lexicon-help.tec" (add-subdir *pail-path* "cky"))
		   :general 
		   (add-path "lexicon-help.gen" (add-subdir *pail-path* "cky")))))
  
	
(defun lexicon-tool()
  (let ((done nil))
    (init-lexicon-tool)
    (setf *lex-disp* (make-instance 'display  :title "CKY: Lexicon Tool" :width *lex-disp-width* :height  *lex-disp-height* 
				    :font (my-findfont 13)   :left *lex-disp-left* :bottom *lex-disp-bottom*))
    (push *lex-disp* *open-windows*)
    (setf *msg-subdisp* (make-instance  'scroll-display :parent *lex-disp*   :font (my-findfont 11) 
					:title "CKY: Messages " :width *msg-subdisp-width* 
					:height  *msg-subdisp-height* 
					:left 0
					:bottom 0 ))    
    (reset-b-f 7 (- (height *lex-disp*) (scalew 100)))
    (set-button *help-lexicon* *lex-disp* :left  *left-of-buttons*
		:bottom (b-f ))
    (set-button *exit-lexicon* *lex-disp* :left  *left-of-buttons*
		:bottom (b-f)
		:action #'(lambda nil (setf done t)(close-display *lex-disp*)))
    (set-button *load-lexicon* *lex-disp* :left  *left-of-buttons*
		:bottom (b-f) :action '(lambda nil 
					(protect-display  *lex-disp* t)
					(loadlex)(viewlex)(reset-button *load-lexicon*)
					(protect-display  *lex-disp* nil)))
    (set-button *save-lexicon* *lex-disp* :left  *left-of-buttons*
		:bottom (b-f ) :action'(lambda nil 
					(protect-display  *lex-disp* t)
					(savelex)(reset-button *save-lexicon*)
					(protect-display  *lex-disp* nil)))
    (set-button *find-word* *lex-disp* :left  *left-of-buttons*
		:bottom (b-f )
		:action '(lambda nil (protect-display  *lex-disp* t)
			  (lookup (to-symbol (ask "Lookup for word?  " :font *small-font*)))
			  (viewlex)
			  (reset-button  *find-word*)
			  (protect-display  *lex-disp* nil)))
    (set-button *delete-word* *lex-disp* :left  *left-of-buttons*
		:bottom (b-f ) :action '(lambda nil
					 (protect-display  *lex-disp* t)
					 (junklex 
					  (to-symbol(ask "Delete word? :  " :font *small-font*)))
					 (viewlex)(reset-button *delete-word*)
					 (protect-display  *lex-disp* nil)))
    (set-button *new-entry* *lex-disp* :left  *left-of-buttons*
		:bottom (b-f  )
		:action '(lambda nil 
			  (protect-display  *lex-disp* t)
			  (makelex   (to-symbol(ask "Wordform? : " :font *small-font*)))
			  (viewlex)(reset-button *new-entry*)
			  (protect-display  *lex-disp* nil)))
    (if (boundp '*lexicon*) (viewlex))
(if (not *demo-in-execution* ) (mp::process-wait "exit" #'(lambda nil done)))
))


(defun loadlex()
  (let  ((filename 
	  (cond ((not *demo-in-execution*)
		 (choose-file-dialog :directory  
				     (or *lexicon-filename* pail-lib::*user-dir*
					 (add-path "lexicon/" *cky-path* ))
				:prompt "Load lexicon file..."
				:button-string "Load"
				:left 400 :bottom 200))
		((=  *demo-in-execution* 1) "lexicon/l0.lex")
		((= *demo-in-execution* 2) "lexicon/number-lexicon.lex"))))
    (when filename (load-lexicon filename))))

(defun load-lexicon (filename)
  (if (probe-file filename)
      (with-open-file (infile filename :direction :input) 
	(setq *lfile* infile)
	(setq *lexicon-filename* filename)
	(setq *lexicon* (list nil))
	(format-display *msg-subdisp*  "Loading lexicon file ~A" filename)
	(setf (title *lex-subdisp*)(concatenate 'string "Lexicon: " filename))
	(do ((entry (read infile nil 'eof)(read infile nil 'eof)))
	    ((eq 'eof entry))
	  (newlex (aexplodec (car entry))  (list (intern (cadr entry) :cky)) *lexicon*))
	(sortlex *lexicon*))
    (file-does-not-exist filename)))

  


(defun savelex ()
  (when *lexicon-filename*
    (let  ((filename (choose-new-file-dialog 
		      :directory (or *lexicon-filename*
				     pail-lib::*user-dir*
				     (add-path "lexicon/" *atn-path*))
		      :prompt "Save lexicon (*.lex)"
		      :button-string "Save"
		      :left 400
		      :bottom 200)))
      (if (not (string= filename ""))
	  (with-open-file (port filename :direction :output :if-exists :supersede)  
	    (format-display *msg-subdisp*  "Saving lexicon to ~A" filename)
	    (setq *lexicon-filename* filename)
	    (setf (title *lex-subdisp*) (concatenate 'string
					  "Lexicon: " (or  *lexicon-filename* "")))
	    (format port "(in-package :atn)
(")
	    (mapc #'(lambda (entry)
		      (format  port "(~a  ~a ~a)~%"(car entry)(string (cadr entry) (caddr entry)))
		  (growlex *lexicon* nil))
	    (format port ")")
	    )
	(format-display *msg-subdisp*  "Filename not given : nothing saved")))))


; This inserts a new item in the lexical tree
; name is a list of chars
(defun newlex (name value lexicon)
  (if (null name)
      (car (rplaca lexicon (ucons value (car lexicon))))
      (let ((alist (assq (car name) (cdr lexicon))))
	   (if (null alist)
	       (let ((newnode (list (car name) nil)))
		    (rplacd lexicon
			    (dest-insert newnode
				    (cdr lexicon)
			     '(lambda (x y) (alphalessp (car x) (car y)))
				    nil))
		    (newlex (cdr name) value (cdr newnode)))
	       (newlex (cdr name) value (cdr alist))))))



(defun lookup (word)
  (let ((cky-answer (car (search-word (aexplodec word) *lexicon*)))
	(yes-no nil))
    (cond (cky-answer  cky-answer)
	  (t (cond ((eq :yes 
			(setf yes-no
			  (cky-y-or-n-dialog
			   (format nil "   HELP! '~A' is not in the lexicon: is it a spelling error?" word) )))
		    (let ((newword (to-symbol (ask "Please give the correct form ... " :font *small-font* :width (scalew 350)))))
		      (lookup newword)))
		   ((eq yes-no :canceled) nil)		
		   (t
		    (let ((value (newword word)))
		      (newlex (aexplodec word) value *lexicon*)
		       (lookup word))))))))


(defun search-word (word lexicon)
  (if (null word)
      lexicon
      (let ((alist (assq (car word) (cdr lexicon))))
	   (if (null alist)
	       nil
	       (search-word (cdr word) (cdr alist))))))



(defun newword (word)
  (let ((cat (to-symbol (ask  "Please give its category ... " :font *small-font* ))))
      (cons (intern cat :cky)
	    (intern (getfeatures (ask  ".., and any other attributes (<enter> for none) ... " :font *small-font* )) :cky))))
     
	     

(defun sortlex (lexicon)
  (cond (lexicon
	(setf (cdr lexicon) (sort (cdr lexicon) #'(lambda (x y)(char< (car x)(car y)))))
	 (mapcar '(lambda (alist)
			(sortlex (cdr alist)))
	       (cdr lexicon)))))

;; ATTENZIONE: qui veniva usata la funzione my-read
(defun getfeatures (features)
  (cond ((string= features "")  nil)	
	(t  (to-symbol (car(word-cky (coerce   features 'list)))))))

;;=============================================================

(defun viewlex ()
 (let ((wordlist (growlex *lexicon* nil)))
       (clear-scroll *lex-subdisp* )(seewords wordlist 0)
       (format-display *msg-subdisp* "~D entries~%"  (length wordlist))))
       



(defun growlex (lexicon word)
  (mapcan #'(lambda (tree)
	    (let* ((item (cons (car tree) word))
		   ;;		  (name (maknam (reverse item)))
		   (name (coerce (reverse item) 'string))
		   (temp (mapcar #'(lambda (value)
				   (cons name value))
				 (cadr tree))))
	      (append temp (growlex (cdr tree) item))))
	  (cdr lexicon)))



(defun seewords (lexicon count)
  (cond ((null lexicon)
	 (cond ((> count 0))))
	(t
	 (showword (car lexicon))
	
	 (seewords (cdr lexicon) (1+ count)))))

(defun showword (entry)
  (format-display *lex-subdisp*  "~A : ~A  ~A " (car entry)(cadr entry) (cddr entry)))

(defun showword-string (entry)
  (format nil  "~A : ~A  ~A " (car entry)(cadr entry) (cddr entry)))



(defun junklex (word)
  (cond ( (not (string= word ""))
	  (let ((value (search-word (aexplodec word) *lexicon*)))
	    (cond ((car value)
		   (rplaca value
			   (mapcan #'(lambda (item)
				       (cond ((eq 
					       (cky-y-or-n-dialog
						(format nil "~A~A~A"
							"Delete [ "
							(showword-string (cons word item))
							"] ?? ")  ) :no)
					      (list item))))
				   (car value)))
		
		   (mapc #'(lambda (item)
			     (showword (cons word item))  )
			 (car value))	)
		  (t (format-display *msg-subdisp*  "~a is not in the dictionary~%"  word)))))))
	
(defun makelex (word)
  (if(not (string= word "")) ( newlex (aexplodec word) (newword word) *lexicon*)))
;-------------------------------------------------------------

(defun cky-y-or-n-dialog (query &key (title "Please Click"))
  (let ((dw (make-instance 'display   :width (if (> (cw::font-string-width *small-font* query) 195)
						 (+ (cw::font-string-width *small-font* query) 20)
					       205)
			   :font (my-findfont 11)
			   :left (scalew 200)
			   :bottom (scaleh 400)
			   :height (scaleh  100)
			   :title title))
	(done nil)
	(yb (make-instance 'push-button  :font (my-findfont 13) :label "  Yes "))
	(nb (make-instance 'push-button  :font (my-findfont 13) :label "  No  ")))
					;     (push  dw *open-displays-list*)
    (write-display dw query (scalew 10)(scaleh  60))
    (set-button yb dw :left (scalew 10)
		:bottom (scaleh 10) :action (function (lambda (&rest cw-internals)
							(declare (ignore cw-internals))
							(setf done :yes))))
    (set-button nb dw :left (- (width dw)(width nb) (scalew 10)) :bottom (scaleh 10)
		:action (function (lambda (&rest cw-internals)
				    (declare (ignore cw-internals))
				    (setf done :no))))
    (mp:process-wait "Exit"   #'(lambda nil done))
    (cw:flush (window dw))
    done))

(defun to-symbol(string)(intern (string-upcase string) :cky))











