;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   atp.lex.cl
;;; Short Desc:  Lexical data structure manager
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Mike Lenz
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atn)

;;
;; M. Lenz
;; pAIL ATN module
;;
;; Lexical data structure manager
;; 19-8-91
;;

;;
;; Defined herein:
;;
;; .  read-word
;; .  lex-word
;; .  lex-string
;; .  lookup-word
;; .  lookup-string
;; .  clear-lexicon
;;

;;
;; Returns next word of *input* as a string OR a list,
;; ignoring punctuation and spaces.
;; (Assumes global *input-length* has also been set.)
;; Also returns next input-head position of the string
;; via *nextpos*.
;; Note 2 loops: the first swallows any whitespace to the left
;; of the next word; the 2nd gets the word.
;;

(defun read-word (startpos)
  (do ((pos startpos (+ pos 1)))
      ((or (= pos *input-length*)
	   (alphanumericp (elt *input* pos)))
       (if (= pos *input-length*)
	   (return-from read-word nil))
       (setq startpos pos))
      )
  (do ((pos startpos (+ pos 1)))
      ((or (= pos *input-length*)
	   (not (alphanumericp (elt *input* pos))))
       (setq *nextpos* pos)
       (subseq *input* startpos pos))
      ))


;;
;; *lextree* stores our lexical data structure: a letter tree.
;; Each <node> is of the form
;;   (<character> (<node>)* <value1> <value2> ... <valuen>)
;;
;; e.g. after
;;   (lex-string "dog" 33)
;;   (lex-string "dog" 44)
;;   (lex-string "do" 55)
;;   (lex-string "" 66)
;; *lextree* looks like:
;;
;; (((#\d ((#\o ((#\g nil 44 33)) 55)))) 66)
;;
;; (Note that <character> of the root node is nil, for attaching
;;  values to the string "".)
;;

(defvar *lextree* '(nil))

(defun clear-lexicon ()
  (setf (car *lextree*) nil)
  (setf (cdr *lextree*) nil))

;;
;; Associates value with charlist (a list of characters)
;; in the lexical tree.
;;

(defun lex-word (charlist value)
  (let ((curlist *lextree*)
	(entry nil))
    (dolist (achar charlist)
      (let ((entry (assoc achar (car curlist) :test #'char=)))
	(if entry
	    (setq curlist (cdr entry))
	  (let* ((newtree (copy-list '(nil))))
	    (push (append (list achar) newtree) (car curlist))
	    (setq curlist newtree)))))
    (push value (cdr curlist))))

;;
;; Returns the list (value1 value2 ... valuen) of values
;; associated with charlist in the lexical tree; or nil
;; if the word is not in the lexicon (i.e. no values have
;; been associated).
;;

(defun lookup-word (charlist)
  (let ((curlist *lextree*)
	(entry nil))
    (dolist (achar charlist)
      (if (null curlist)
	  (return-from lookup-word nil))
      (setq curlist (cdr (assoc achar (car curlist) :test #'char=))))
    (cdr curlist)))

;;
;; Like lex-word and lookup-word, respectively, but these
;; accept a string argument rather than a character list.
;;

(defun lex-string (str val)
  (lex-word (str-to-list str) val))

(defun lookup-string (str)
  (lookup-word (str-to-list str)))


;;
;; Utility routines
;;

(defun str-to-list (str)
  (coerce str 'list))

;;

(defun random-string ()
  (let ((len (+ (random 7) 2))
	(str nil)
	(char #\a))
    (dotimes (i len)
      (setq char (character (+ *minchar* (random 26))))
      (setq str (nconc str (list char))))
    (setq str (coerce str 'string))))
