;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; File         : semp.cl
;; Description  : Routines for the simplification
;; Author       : Fabio Baj                   
;; Created      : 27-Feb-1990                    
;; Last Change  : 4-Sept-1990                    
;;
;; (c) Copyright (1990), IDSIA Lugano, all rights reserved
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package :atp)

;;----------------------------------------------------------------
;; simp_simps   : LIST-OF(RULE-ID) ---> LIST-OF(RULE-ID)
;; Behavior     : Simplifies the set of the simplifiers starting from
;;              : a set ns of new simplifiers. The result is a set of
;;              : new simplifiers generated during this process.
;; Side effects : rules can be modified or deleted from the database


(defun simp_simps (ns)
  (let ((simps $simps$)(s1A nil)(s1_id nil)(new_res nil))
    (do ((s_id (pop simps)(pop simps)))
        ((null simps) ns)
      (setq $d-reduced$ nil)
      (setq s1A (reduce s_id ns))	
      (if $d-reduced$ (progn 
			     (setq $d-reduced$ nil)
			     (setq new_res  (reduce_a_new_rule (car s1A)  $simps$))
			     (if (eq 'fail new_res) (setq s1A 'fail)
			       (setq s1A (cons (car new_res)
					       (list s_id 'simp (append (cadddr s1A)
									(caddr new_res))))))))
      (cond ((not(eq 'fail s1A))
	     (setq s1_id (store_rule s1A))
	     (if (not (null s1_id))(setq simps $simps$))
	     (setq ns (delete s_id ns))
	     (cond ((and (not(eq 'deleted s1_id))
			 )
		    (push s1_id ns)) (t)))
	    (t))))))

;;-----------------------------------------------------
;; simplify     : LIST-OF(RULE-ID) ---> {NIL}
;; Behavior     : simplifies the set of all the rules starting from
;;              : the set ns of new simplifiers.
;;              : it is mutually recursive with the memory managment module
;; Side effects : rules can be modified or deleted from the database


(defun simplify (ns)
  (let ((nsimps (simp_simps ns))(r1A nil)(r1_id nil)(new_res nil))
    (do* ((rules $rules$) 
	  (r_id (pop rules) (pop rules)))
	((null r_id) nil)
      (setq $d-reduced$ nil) 
      (setq r1A (reduce r_id nsimps))	
      (if $d-reduced$ (progn (print'ECCOMI) (nl)
			(setq $d-reduced$ nil)
			(setq new_res  (reduce_a_new_rule (car r1A)  $simps$))
			(if (eq 'fail new_res) (setq r1A 'fail)
			  (setq r1A (cons (car new_res)
					  (list r_id 'simp (append (cadddr r1A)
								   (caddr new_res))))))))
      (cond ((not(eq 'fail r1A)) 
	     (setq r1_id (store_rule r1A))
	     (cond ((member r1_id $simps$)
		    (push r1_id nsimps)
		    (setq rules $rules$)
		    (setq nsimps (simp_simps nsimps)))
		   (t)))
	    (t)))))


;;--------------------------------------------------------
;; reduce:  RULE-ID X LIST-OF(RULE-ID) ---> CONS(RULE,
;;                                              (RULE-ID, SIMP, LIST-OF(RULE-ID)))
;;                                          U {fail}
;; brings a rule in minimal form with respect to a set of simplifiers
;; the identifier of the rule is used to get the rule-body, and to
;; prevent to simplify the rule using itself. the result is a new rule,
;; togheter  with is ancestors. most part of this work is done with
;; a call to reduce00. the algorithm implemented is the following:
;;
;; split the simplifiers into                 (split-simps)
;; D= domain rules, NP= p rules and n rules
;; reduce the rule with simplifiers in NP     (reduce0)
;; REPEAT
;;   reduce the rule with simplifiers in D    (d-reduce-loop)
;; UNTIL no more d-simplification is possible
;; reduce the rule with simplifiers in NP     (reduce0)
;;

(defun reduce (r_id l_of_id)
 (let (( res (reduce00 r_id (get_rule r_id) l_of_id)))
   (cond ((null (cdr res))
	  'fail)
	 (t (cons (car res)
		  (list r_id 'simp (reverse (cdr res))))))))

;;----------------------------------------------------------------------
;; reduce_a_new_rule :  RULE-ID X LIST-OF(RULE-ID) 
;;                      ---> CONS(RULE, (SIMP, LIST-OF(RULE-ID)))
;;                                          U {fail}
;; it is the same of reduce, for the particular case
;; of a rule generated but not yet added to the system.

(defun reduce_a_new_rule (r_body l_of_id)
 (let (( res (reduce00 'no_name r_body l_of_id )))
    (cond ((null (cdr res)) res)
	  (t (cons (car res)
		  (list  'simp (reverse (cdr res))))))))

(defun reduce00 (r_id r_body l_of_id)
  (let* ((splitted_simps (split-simps l_of_id))
	(d_ids (car splitted_simps))
	(np_ids (cdr splitted_simps))
	(r-body1_simpused (reduce0-NP r_id r_body np_ids nil))
	(r-body2_simpused nil))
    (if (null d_ids) r-body1_simpused
      (progn
	(setq r-body2_simpused
	  (d_reduce_loop r_id (car r-body1_simpused) d_ids))
	(if (null (cdr r-body2_simpused)) r-body1_simpused
	   (reduce0-NP r_id (car r-body2_simpused) 
		 np_ids (append  (cdr r-body2_simpused)
				 (cdr r-body1_simpused))))))))
	    

(defun d_reduce_loop (r_id r_body d_ids)
  (let ((s-used nil)
        (rbody1-s_used (reduce0-D r_id r_body d_ids nil)))
    (loop      
      (if  (not (reduced  rbody1-s_used))
	  (return (cons (car  rbody1-s_used)  s-used)))
      (setq s-used (append (cdr  rbody1-s_used) s-used))
      (setq rbody1-s_used (reduce0-D r_id (car  rbody1-s_used)  d_ids nil)))))			 
  
(defun collect-d-rules ()
  (remove-if-not #'(lambda (id)
		     (member 'd (rinfotype-classlist (gethash id $rules_info$))))
		 $simps$))  
  
      
(defun split-simps (l_of_id)
  (split-s1 l_of_id nil nil))
(defun split-s1 (l-id d np)
  (cond ((null l-id) (cons d np))
	((member 'd (rinfotype-classlist (gethash (car l-id)  $rules_info$)))
	 (split-s1 (cdr l-id) (cons (car l-id) d) np))
	(t (split-s1 (cdr l-id) d (cons (car l-id) np)))))
	 
      
;;-----------------------------------------------------------------
;; reduce0: RULE_ID x RULE x LIST_OF(RULE_ID)
;;                         x LIST_OF(RULE_ID) ---> (RULE . LIST_OF(RULE_ID))
;;
;; Behavior    :  reduces r_body using the simplifiers in l_of_ids
;;             :  returnes the simplified rule toghther wiht the
;;             :  ids of the simplifiers used
;; PROSSIMAMENTE DEVO DIVIDERLA in REDUCE0-D e REDUCE-0-NP
;; E FARE LO STESSO PER REDUCE-1

(defun reduce0-D (r_id r_body l_of_id simps_used)
  ;(princ '#\s)(force-output)
  (cond ((null l_of_id)(cons r_body simps_used))
	((eq r_id (car l_of_id)) (reduce0-D r_id r_body (cdr l_of_id) simps_used))
	(t (let ((rbody1 (reduce1-D r_body (get_rule (car l_of_id)))))
	     (cond ((equal rbody1 r_body)
		    (reduce0-D r_id r_body (cdr l_of_id) simps_used))
		   (t (reduce0-D r_id
			       rbody1
			       (cdr l_of_id)
			       (cons (car l_of_id) simps_used))))))))


(defun reduce0-NP (r_id r_body l_of_id simps_used)
  ;(princ '#\s)(force-output)
  (cond ((null l_of_id)(cons r_body simps_used))
	((eq r_id (car l_of_id)) (reduce0-NP r_id r_body (cdr l_of_id) simps_used))
	(t (let ((rbody1 (reduce1-NP r_body (get_rule (car l_of_id)))))
	     (cond ((equal rbody1 r_body)
		    (reduce0-NP r_id r_body (cdr l_of_id) simps_used))
		   (t (reduce0-NP r_id
			       rbody1
			       (cdr l_of_id)
			       (cons (car l_of_id) simps_used))))))))

;;-----------------------------------------------------------------
;; reduce1: RULE X RULE ---> RULE
;;
;; Behavior    :  reduces rule using simp: if no reduction is possible
;;             :  rule itself is returned
;; Side effects:  if a d-rewriting has succesfully occurred, the flag
;;             :  $d-reduced$ is set to t. in this way the simplifier
;;             :  knows that the simplification process has to be 
;;             :  reiterated.
;; NOTE        :  rem-0col should be called only if the rule has been
;;             :  simplified, probably in the body of n-red, p-red ..

(defun reduce1-NP (rule simp) 
  (rem-0col  (cond ((and (= 1 (length ( car simp)))
			 (=  (length (cdr simp)) 2))
		    (p_red (caar simp) rule))
		   ((= 1 (length (cdr simp)))
		    (n_red (car simp) rule)))))

(defun reduce1-D (rule simp) 
   (cond
    ((and (listp (caar simp))(eq '-> (caaar simp)))
     (d-red (cadr(caar simp))(caddr(caar simp)) rule))
    ((and $extended-rewriting$ (listp (caar simp))(eq '= (caaar simp)))
     (caar simp) (extended-d-red (cadr(caar simp))(caddr(caar simp)) rule))))
    


;;======================================================================
;; reduction with n-rules
;; n-red        : N-TERM X RULE ---> RULE
;; Behavior     : returns the result of removing from rule n-terms
;;              : containing instances of ntx  
;; Side effects : none

(defun n_red (ntx rule)
  (b-nred1 ntx (length ntx) (car rule)(cdr rule) nil))

(defun b-nred1 (ntx l st bl tmbl) 
  (cond (( null bl) (cons st (reverse tmbl)))
	((<  (logcount (car bl)) l)
	 (cons st (append (reverse tmbl ) bl)))
	(t (let ((mnum 0)( s (n_match ntx (get_nt st (car bl)))))
	     (cond  ((not (eq 'fail s))
		     (setq mnum (num-of (apply_sub ntx s) st))
 		     (b-nred1 ntx
			      l 
			      st 
			      (remove-if #'(lambda (n)(is-present mnum n))(cdr bl))
			      (remove-if #'(lambda (n)(is-present mnum n))(cons (car bl) tmbl))))
		    (t (b-nred1 ntx l st (cdr bl) (cons (car bl) tmbl)))))))))
	
(defun is-present (np ng)
    (eq (logand  np ng ) np)))

(defun get_nt (st n)
  (map-bool (reverse st) n nil))

(defun map-bool (l n tmp)
	    (cond ((=  n 0) tmp)
		  ( (= 1 (rem n 2)) 
		    (map-bool (cdr l)(ash n -1)(cons (car l)tmp)))
		  (t (map-bool (cdr l)(ash n -1)tmp))))

(defun rem-bit ( n i ) 
  (+ (ash   (- n (rem n (ash 1 (1+ i)))) -1)
     (mod n (ash 1 i)))))

(defun rem-col (bm i)
  (mapcar #' (lambda (n)(rem-bit n i)) bm))


;;======================================================================
;; reduction with p-rules
;; p-red        : ATOMICF X RULE ---> RULE
;; Behavior     : returns the result of removing instances of at   from rule  
;; Side effects : none

(defun p_red (at rule)
  (b-p_red1 at (car rule) nil  (cdr rule) (1- (length (car rule)))))

(defun b-p_red1 (at st nst  bm i )
  (cond 
   ((null st) (cons (reverse nst) bm))
   ((not (eq 'fail (match at (car st))))
    (b-p_red1 at (cdr st) nst (rem-col bm i) (1- i)))
   (t 
    (b-p_red1 at (cdr st) (cons (car st) nst) bm (1- i)))))

(defun num-of (nterm st)
  (num1-of nterm st (1- (length st)) 0))
(defun num1-of (nt st n tmp)
  (cond ((< n 0) tmp)
	((member  (car st) nt :test #'equalatom)
	 (num1-of (del (car st) nt) (cdr st) (1- n) (+ tmp (ash 1 n))))
	(t (num1-of  nt (cdr st) (1- n) tmp ))))

(defun reduced (x)
  (not(null (cdr x))))


;;======================================================================
;; REDUCTION WITH ORIENTABLE EQUATIONS (ss > tt) 
;; d-red    : TERM X TERM X RULE ---> RULE
;; Behavior : substitutes in rule  instances of ss with the
;;          : corresponding instances of tt
;; Side effects : if a match succedes the flag $d-reduced$
;;              : is set to t.

(defun d-red (ss tt rule)
 (cons (mapcar #'(lambda (bt) 
		   (d-red1 ss tt bt)) (car rule)) (cdr rule)))
(defun d-red1 (ss tt term)
  (let ((sigma nil))
    (cond ((not (eq 'fail (setq sigma (match ss term))))
	   (setq $d-reduced$ t)
	   (d-red1 ss tt (apply_sub tt sigma)))
	  ((atom term) term)
	  (t (cons (car term)
		   (mapcar #'(lambda (bt) 
			       (d-red1 ss tt bt)) (cdr term)))))))


;;======================================================================
;; REDUCTION WITH NON ORIENTABLE EQUATIONS WITH INSTANCES ORIENTABLE
;; d-red        : TERM X TERM X RULE ---> RULE
;; Behavior     : substitutes in rule  instances of ss with the
;;              : corresponding instances of tt
;; Side effects : if al match succedes the flag $d-reduced$
;;              : is set to t.


(defun extended-d-red (ss tt rule)
  (let ((rulest (ext-d-red ss tt rule)))
      (if (not  (equal rule rulest))
	  rulest
	(let ((rulets (ext-d-red tt ss rule)))
	  (if (not (equal rule  rulets))
	      rulets
	    rule)))))

(defun ext-d-red (ss tt rule)
   (cons (mapcar #'(lambda (bt) 
		   (extended-d-red1 ss tt bt)) (car rule)) (cdr rule)))

;; extd-d-red returns NIL if the instance  of the equation can not be oriented

(defun extended-d-red1 (ss tt term)
  (let ((sigma nil))
    (cond ((and (not (eq 'fail (setq sigma (match ss term))))
		(eq '> (lro (apply_sub ss sigma)(apply_sub tt sigma)))
	       (extended-d-red1 ss tt (apply_sub tt sigma))))
	  ((atom term) term)
	  (t (cons (car term)
		   (mapcar #'(lambda (bt) 
			       (extended-d-red1 ss tt bt)) (cdr term)))))))





    

 
