;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   semp.cl
;;; Short Desc: Routines for the simplification of clauses
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Fabio Baj
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atp)



;;----------------------------------------------------------------
;; simp-simps   : LIST-OF(RULE-ID) ---> LIST-OF(RULE-ID)
;; Behavior     : Simplifies the set of the simplifiers starting from
;;              : a set ns of new simplifiers. The result is a set of
;;              : new simplifiers generated during this process.
;; Side effects : rules can be modified or deleted from the database


(defun simp-simps (ns)
  (let ((simps $simps$)(s1A nil)(s1-id nil)(new-res nil))
    (do ((s-id (pop simps)(pop simps)))
        ((or (null s-id) $contr$) ns)
      (setq $d-re-duced$ nil)
      (setq s1A (re-duce s-id ns))	
      (if $d-re-duced$ (progn 
			(setq $d-re-duced$ nil)
			(setq new-res  (re-duce-a-new-rule (car s1A) (remove s-id $simps$)))
			(if (eq 'fail new-res) (setq s1A 'fail)
			  (setq s1A (cons (car new-res)
					  (list s-id 'simp (append (cadddr s1A)
								   (caddr new-res))))))))
      (cond ((not(eq 'fail s1A))
	     (setq s1-id (store (car  s1A)(cdr s1A)))
	     (if (not (null s1-id))(setq simps $simps$))
	     (setq ns (delete s-id ns))
	     (cond ((and (not(eq 'deleted s1-id))
			 )
		    (push s1-id ns)) (t)))
	    (t)))))

;;-----------------------------------------------------
;; simplify     : LIST-OF(CLAUSE-ID) ---> {NIL}
;; Behavior     : simplifies the set of all the rules starting from
;;              : the set ns of new simplifiers.
;;              : it is mutually recursive with the memory managment module
;; Side effects : clauses can be modified or deleted from the database


(defun simplify (ns)
  (let ((nsimps (simp-simps ns))(r1A nil)(r1-id nil)(new-res nil))
    (do* ((rules $rules$) 
	  (r-id (pop rules) (pop rules)))
	((or (null r-id) $contr$) nil)
      (setq $d-re-duced$ nil) 
      (setq r1A (re-duce r-id nsimps))	
     (if $d-re-duced$ (progn 
			(setq $d-re-duced$ nil)
			(setq new-res  (re-duce-a-new-rule (car r1A)  $simps$))
			(if (eq 'fail new-res) (setq r1A 'fail)
			  (setq r1A (cons (car new-res)
					  (list r-id 'simp (append (cadddr r1A)
								   (caddr new-res))))))))
      (cond ((not(eq 'fail r1A)) 
	     (setq r1-id (store (car r1A)(cdr r1A)))
	     (cond ((member r1-id $simps$)
		    (push r1-id nsimps)
		    (setq rules $rules$)
		    (setq nsimps (simp-simps nsimps)))
		   (t)))
	    (t)))))


(defun simple-simplify (ns)
  (let ((nsimps ns)(r1A nil)(r1-id nil)(new-res nil))
    (do* ((rules $rules$) 
	  (r-id (pop rules) (pop rules)))
	((or (null r-id) $contr$) nil)
      (setq $d-re-duced$ nil) 
      (setq r1A (re-duce r-id nsimps))	
     (if $d-re-duced$ (progn 
			(setq $d-re-duced$ nil)
			(setq new-res  (re-duce-a-new-rule (car r1A)  $simps$))
			(if (eq 'fail new-res) (setq r1A 'fail)
			  (setq r1A (cons (car new-res)
					  (list r-id 'simp (append (cadddr r1A)
								   (caddr new-res))))))))
      (cond ((not(eq 'fail r1A)) 
	     (setq r1-id (store (car r1A)(cdr r1A)))
	     (cond ((member r1-id $simps$)
		    (push r1-id nsimps)
		    (setq rules $rules$)
		    )
		   (t)))
	    (t)))))


;;--------------------------------------------------------
;; re-duce:  RULE-ID X LIST-OF(RULE-ID) ---> CONS(RULE,
;;                                              (RULE-ID, SIMP, LIST-OF(RULE-ID)))
;;                                          U {fail}
;; brings a rule in minimal form with respect to a set of simplifiers
;; the identifier of the rule is used to get the rule-body, and to
;; prevent to simplify the rule using itself. the result is a new rule,
;; togheter  with is ancestors. most part of this work is done with
;; a call to re-duce00. the algorithm implemented is the following:
;;
;; split the simplifiers into                 (split-simps)
;; D= domain rules, NP= p rules and n rules
;; re-duce the rule with simplifiers in NP     (re-duce0)
;; REPEAT
;;   re-duce the rule with simplifiers in D    (d-re-duce-loop)
;; UNTIL no more d-simplification is possible
;; re-duce the rule with simplifiers in NP     (re-duce0)
;;

(defun re-duce (cl-id l-of-id)
 (let (( res (re-duce00 cl-id (get-clause cl-id) l-of-id)))
      (cond ((null (cdr res))
	  'fail)
	 (t (cons (car res)
		  (list cl-id 'simp (reverse (cdr res))))))))

;;----------------------------------------------------------------------
;; re-duce-a-new-rule :  RULE-ID X LIST-OF(RULE-ID) 
;;                      ---> CONS(RULE, (SIMP, LIST-OF(RULE-ID)))
;;                                          U {fail}
;; it is the same of re-duce, for the particular case
;; of a rule generated but not yet added to the system.

(defun re-duce-a-new-rule (r-body l-of-id)
 (let (( res (re-duce00 'no-name r-body l-of-id )))
    (cond ((null (cdr res)) res)
	  (t (cons (car res)
		  (list  'simp (reverse (cdr res))))))))

(defmethod re-duce00 (cl-id null-clause l-of-id) '(nil))
(defmethod re-duce00 (cl-id (cl-body clause-class) l-of-id)
  (let* ((splitted-simps (split-simps l-of-id))
	(d-ids (car splitted-simps))
	(np-ids (cdr splitted-simps))
	(cl-body1-simpused (re-duce0 cl-id cl-body np-ids nil))
	(cl-body2-simpused nil))
    (if (null d-ids) cl-body1-simpused
      (progn
	(setq cl-body2-simpused
	  (d-re-duce-loop cl-id (car cl-body1-simpused) d-ids))
	(if (null (cdr cl-body2-simpused)) cl-body1-simpused
	   (re-duce0 cl-id (car cl-body2-simpused) 
		 np-ids (append  (cdr cl-body2-simpused)
				 (cdr cl-body1-simpused))))))))
	    
(defmethod d-re-duce-loop (cl-id null-clause d-ids)
 '(nil))
(defmethod d-re-duce-loop (cl-id (cl-body clause-class) d-ids)
  (let ((s-used nil)
        (clbody1-s-used (re-duce0 cl-id cl-body d-ids nil)))
    (loop      
      (if  (not (re-duced  clbody1-s-used))
	  (return (cons (car  clbody1-s-used)  s-used)))
      (setq s-used (append (cdr  clbody1-s-used) s-used))
      (setq clbody1-s-used (re-duce0 cl-id (car  clbody1-s-used)  d-ids nil)))))			 
  
(defun collect-d-rules ()
  (remove-if-not #'(lambda (id)
		     (member 'd (classlist (get id 'info))))
		 $simps$))  
  
      
(defun split-simps (l-of-id)
  (split-s1 l-of-id nil nil))
(defun split-s1 (l-id d np)
  (cond ((null l-id) (cons d np))
	((member 'd (classlist (get (car l-id)  'info)))
	 (split-s1 (cdr l-id) (cons (car l-id) d) np))
	(t (split-s1 (cdr l-id) d (cons (car l-id) np)))))
	 
      
;;-----------------------------------------------------------------
;; re-duce0: RULE-ID x RULE x LIST-OF(RULE-ID)
;;                         x LIST-OF(RULE-ID) ---> (RULE . LIST-OF(RULE-ID))
;;
;; Behavior    :  re-duces r-body using the simplifiers in l-of-ids
;;             :  returnes the simplified rule toghther with the
;;             :  ids of the simplifiers used
;; PROSSIMAMENTE DEVO DIVIDERLA in RE-DUCE0-D e RE-DUCE-0-NP
;; E FARE LO STESSO PER RE-DUCE-1



(defmethod re-duce0 (cl-id null-clause l-of-id simps-used)
  (cons nil simps-used))
(defmethod re-duce0 (cl-id (cl-body clause-class) l-of-id simps-used)
   (cond ((null l-of-id)(cons cl-body simps-used))
	((eq cl-id (car l-of-id)) (re-duce0 cl-id cl-body (cdr l-of-id) simps-used))
	(t (let ((clbody1 (re-duce1 cl-body (get-clause (car l-of-id)))))
	     (cond ((clause-equal clbody1 cl-body)
		     (re-duce0 cl-id cl-body (cdr l-of-id) simps-used))
		   (t (re-duce0 cl-id
			       clbody1
			       (cdr l-of-id)
			       (cons (car l-of-id) simps-used))))))))




;;-----------------------------------------------------------------
;; re-duce1:       CLAUSE x CLAUSE  --->  CLAUSE
;;
;; Behavior    :  re-duces clause using simp: if no reduction is possible
;;             :  rule itself is returned
;; Side effects:  if a d-rewriting has succesfully occurred, the flag
;;             :  $d-re-duced$ is set to t. in this way the simplifier
;;             :  knows that the simplification process has to be 
;;             :  reiterated.

(defmethod copy-obj ((clause clause-class))
  (change-class 
   (make-clause (positive-atoms clause)
		(negative-atoms clause))
   (class-of clause)))
	      

;;(defmethod re-duce1 ((clause-to-re-duce clause-class) (simp clause-class))
;  (let ((susbstitutions-newcl (re-duce1answer clause-to-re-duce simp)))
;    
;    (if (not (eq 'fail susbstitutions-newcl))
;	(simp-combine-answers (cdr susbstitutions-newcl) clause-to-re-duce 
;			      simp  (car susbstitutions-newcl))
;    clause-to-re-duce)))
	
    
;(defmethod re-duce1 (null-clause anything) nil)

(defmethod re-duce1 (null-clause anything) nil)
(defmethod re-duce1 ((clause-to-re-duce clause-class) (simp clause-class))
  (let ((clause  (copy-obj clause-to-re-duce)))
    
    (cond 
;     ((not (eq 'fail (gen-subsumes simp clause))) (print 'SUBSUMPTION) nil)	
     ((and (null (positive-atoms simp))	(n-subsumes (negative-atoms simp) clause)) nil)
     ((and (null (negative-atoms simp)) (p-subsumes (positive-atoms simp) clause)) nil)
     
     ((null (positive-atoms simp)) (n-simp clause (negative-atoms simp)))
     ((is-positive-atom simp)
      (if (and (listp (car (positive-atoms simp)))(eq '-> (caar (positive-atoms simp))))
	  (d-red (cadar (positive-atoms simp))
		 (caddar(positive-atoms simp))
		 clause)
	(p-red (car (positive-atoms simp)) clause)))
     (t clause))))
  
	
;;=================================================================
;; reduction with n-rules
;; 

(defmethod n-simp ((clause clause-class) nterm)
  (let ((todo nterm)
	(literal nil)
	(sigma nil)(sigma1 nil)
	(s-literal nil))
    (loop
      (if (null (setq literal (pop todo))) (return clause))
      (if (not (eq 'fail 
		   (setq sigma
		     (first-match literal (to-ground (positive-atoms clause))))))
	  (progn
	    (setq sigma (car sigma))
	   (setq s-literal (apply-sub literal sigma))
	    (if (not 
		 (eq 'fail (setq sigma1 (n-match (apply-sub (remove literal nterm :test 'equalatom) sigma)
						 (negative-atoms  clause)))))
		
		  (return  (remake-clause 
			    (remove s-literal (positive-atoms clause) :test 'equalatom) 
			    (negative-atoms  clause)
			    clause))))))))
				  
	
(defmethod n-red (literal (clause clause-class))
  (remake-clause  
   (negative-atoms  clause)
   (pn-red1 literal (positive-atoms clause))
   clause))



;;======================================================================
;;
;; Subsumption
;;

(defmethod n-subsumes (nterm (clause clause-class))
  (subsumes nterm (negative-atoms  clause)))
(defmethod p-subsumes (pterm (clause  clause-class))
  (subsumes pterm (positive-atoms clause)))
(defun subsumes (x-literal-list literal-list)
  (not (eq 'fail (n-match x-literal-list literal-list))))

(defmethod gen-subsumes ((clausex clause-class)(clause clause-class))
  (n-match (make-standard-clause clausex)
	   (make-standard-clause clause)))

(defmethod make-standard-clause ((clause clause-class))
  (append (mapcar #'(lambda (x)(list '~ x)) (negative-atoms  clause))
	  (positive-atoms clause)))
		 


;;======================================================================
;; reduction with p-rules
;; p-red        : ATOMICF X RULE ---> RULE U {fail}
;; Behavior     : returns the result of removing instances of "literal"   
;;              : from "clause"


(defmethod p-red (literal (clause clause-class))
  (remake-clause
   (positive-atoms clause)
   (remove-if-match literal (negative-atoms clause))
   clause))

	
(defun remove-if-match (literal literal-list)
  (remove-if  #'(lambda (x) (not (eq 'fail (match literal x))))
	      literal-list))


(defun re-duced (x)
  (not(null (cdr x))))


;;======================================================================
;; REDUCTION WITH ORIENTABLE EQUATIONS (ss > tt) 
;; d-red    : TERM X TERM X CLAUSE ---> CLAUSE
;; Behavior : substitutes in rule  instances of ss with the
;;          : corresponding instances of tt
;; Side effects : if a match succedes the flag $d-re-duced$
;;              : is set to t.



(defmethod d-red (ss tt (clause clause-class))
  (remake-clause
   (mapcar #'(lambda (at) 
	       (d-red1 ss tt at)) (positive-atoms clause))
   (mapcar #'(lambda (at) 
	       (d-red1 ss tt at)) (negative-atoms  clause))
   clause))

(defun d-red1 (ss tt term)
  (let ((sigma nil))
    (cond ((not (eq 'fail (setq sigma (match ss term))))
	   (setq $d-re-duced$ t)
	   (d-red1 ss tt (apply-sub tt sigma)))
	  ((atom term) term)
	  (t (cons (car term)
		   (mapcar #'(lambda (bt) 
			       (d-red1 ss tt bt)) (cdr term)))))))




;;======================================================================
;; REDUCTION WITH NON ORIENTABLE EQUATIONS WITH INSTANCES ORIENTABLE
;; d-red        : TERM X TERM X RULE ---> RULE
;; Behavior     : substitutes in rule  instances of ss with the
;;              : corresponding instances of tt
;; Side effects : if al match succedes the flag $d-re-duced$
;;              : is set to t.


(defun extended-d-red (ss tt rule)
  (let ((rulest (ext-d-red ss tt rule)))
      (if (not  (equal rule rulest))
	  rulest
	(let ((rulets (ext-d-red tt ss rule)))
	  (if (not (equal rule  rulets))
	      rulets
	    rule)))))

(defun ext-d-red (ss tt rule)
   (cons (mapcar #'(lambda (bt) 
		   (extended-d-red1 ss tt bt)) (car rule)) (cdr rule)))

;; extd-d-red returns NIL if the instance  of the equation can not be oriented

(defun extended-d-red1 (ss tt term)
  (let ((sigma nil))
    (cond ((and (not (eq 'fail (setq sigma (match ss term))))
		(eq '> (lro (apply-sub ss sigma)(apply-sub tt sigma)))
	       (extended-d-red1 ss tt (apply-sub tt sigma))))
	  ((atom term) term)
	  (t (cons (car term)
		   (mapcar #'(lambda (bt) 
			       (extended-d-red1 ss tt bt)) (cdr term)))))))





    
 
