;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   folscan.cl
;;; Short Desc: A Lexical analizer for First Order languages
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Fabio Baj
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atp)

;;---------------------------------------------------
;; mkt          : LIST-OF(CHAR) --> LIST-OF(TOKEN)  
;; Behaviour    : Trasforms a string in a list of tokens 
;; Side effects : None
;; Example      : (mkt "all x p(x,y)=>q(y,x)") =     
;;                  '(all  x p  #\(  x  #\,  y  #\)  =>   q  #\(  y  #\,  x  #\) )
 
(defun mkt () 
   (delall #\space (scanner (coerce (read-line) 'list))))

(defun make-token-list (S) (delall #\space (scanner(delall #\newline (coerce S 'list)))))


(defun neck (ch-l)
   (and (eq (car ch-l) '#\:)
        (eq (cadr ch-l) '#\-)
        (cons '<= (cddr ch-l))))
(defun goes-to (ch-l)
   (and (eq (car ch-l) '#\-)
	(eq (cadr ch-l) '#\>)
	(cons '->  (cddr ch-l))))
(defun impl (ch-l)
   (and (eq (car ch-l) '#\=)
        (eq (cadr ch-l) '#\>)
        (cons '=> (cddr ch-l))))
(defun bi-impl (ch-l)
   (and (eq (car ch-l)   '#\<)
        (eq (cadr ch-l)  '#\=)
        (eq (caddr ch-l) '#\>)
        (cons  '<=> (cdddr ch-l))))
(defun lessp-eq (ch-l)
   (and (eq (car ch-l) '#\=)
        (eq (cadr ch-l) '#\<)
        (cons '=< (cddr ch-l))))
(defun greater-eq (ch-l)
   (and (eq (car ch-l) '#\>)
        (eq (cadr ch-l) '#\=)
        (cons '>= (cddr ch-l))))
(defun greater (ch-l)
   (and (eq (car ch-l) '#\>)
        (cons '> (cdr ch-l))))


(defun andsym (lch)
   (cond ((eq (car lch) '#\&) lch)))
(defun sp-sym (lch)
   (cond ((eq (car lch) '#\space) lch))) 
(defun orsym (lch)
   (cond ((eq  (car lch) '#\|) lch)))
(defun not-sym (lch)
   (cond ((eq  (car lch) '#\~) lch)))
	
(defun plus  (lch)
   (cond ((eq  (car lch) '#\+) (cons '+ (cdr  lch)))))
(defun minus (lch)
   (cond ((eq  (car lch) '#\-) (cons '- (cdr lch)))))




(defun comm-sym (lch)
  (cond ((eq (car lch) '#\,) lch)))
(defun sqlpsym (lch)
   (cond ((eq (car lch) '#\[ ) lch)))
(defun sqrpsym (lch)
   (cond ((eq (car lch) '#\] ) lch)))
(defun lpsym (lch)
   (cond ((eq (car lch) '#\( ) lch)))
(defun rpsym (lch)
   (cond ((eq (car lch) '#\) ) lch)))

(defun trim-head (l n)
   (cond  ((> n (length l)) nil)
          (t (trim-aux l nil n))))
(defun trim-aux (l x n)
   (cond  ((eq 0 n) (reverse x))
         (t (trim-aux (cdr l) (cons (car l) x) (- n 1)))))
(defun trim-tail (l n)
   (cond ((eq 0 n) l)
         (t (trim-tail (cdr l) (- n 1)))))

(defun recgn (x l)
    (let*  (( xl (coerce x 'list))
           ( n  (length xl) ))
           (cond ( (equal xl ( trim-head l n)) (trim-tail l n)))))

(defun scanner (lch)
    (scan-aux lch nil))
(defun scan-aux (lch aux)
    (cond ((null lch) (reverse aux))
          (t (let (( next-token (or (special-sym lch)
                                   (word lch)))) 
                  (scan-aux (cdr next-token) 
                            (cons (car next-token) aux))))))
(defun special-sym (lch)
  (or (impl     lch)
      (bi-impl  lch)
      (goes-to  lch)
      (neck     lch)
      (andsym   lch)
      (greater-eq lch)
      (greater lch)
      (lessp-eq lch)
      (comm-sym lch)
      (plus     lch)
      (minus    lch)
      (orsym    lch)
      (not-sym  lch)
      (sp-sym   lch)
      (lpsym    lch)
      (rpsym    lch)
      (sqlpsym  lch)
      (sqrpsym  lch)
      (and  (member (car lch)
		    '(#\, #\  #\. #\; #\:  #\~ #\= #\< #\& #\+ #\( #\) #\[ #\]  #\| #\-  ))
	    lch))) 
(defun word (lch)
   (trim-word lch nil))

(defun trim-word (lch  aux)
  (cond ((null lch)
	 (list (make-a-symbol aux)))
	((member (car lch) 
		 ;;; These are word separators
		 '(#\, #\  #\= #\~ #\< #\> #\& #\+ #\( #\) #\[ #\]   #\|  #\: ))
	 (cons (make-a-symbol aux)lch ))
	(t (trim-word (cdr lch) (cons (car lch) aux)))))

(defun is-a-variable-ch (l-of-ch)
     (if $prolog-mode$ (upper-case-p (car l-of-ch))
       (member (car l-of-ch)  '(#\x #\u #\y #\t #\z #\w #\v))))

;;  questa funsiona con insensitive upper
#|
 (defun make-a-symbol (list-of-chars)
      (let ((the-string  (coerce (reverse list-of-chars) 'string)))
         (cond (( is-a-number list-of-chars)
                   (read-from-string the-string))
          
         ((is-a-variable-ch (reverse list-of-chars))
                    (intern (concatenate 'string "$" the-string ):atp))
                   (t  (intern  the-string :atp)))))
|#
;;  questa funsiona con insensitive lower
 (defun make-a-symbol (list-of-chars)
   (let ((the-string  (coerce (reverse list-of-chars) 'string)))
   (if ( is-a-number list-of-chars)
	(read-from-string the-string)
    (intern  the-string :atp))))


(defun is-a-number (l-of-ch &optional (dot nil))
  (cond ((null l-of-ch) t)
	((and (eq #\. (car l-of-ch)) (not dot))
	 (is-a-number (cdr  l-of-ch) t))
	((digit-char-p (car l-of-ch))  
	 (is-a-number (cdr l-of-ch) dot))))
	
