LISP examples and code fragments from AI Expert's Expert's Toolbox,
April 1991, by Nick Bourbaki, entitled "An Initialization Protocol."

Listing 1

(defclass 2d-position (position) 
  ((x :accessor position-x :initarg :x) 
   (y :accessor position-y :initarg :y))) 


(defmethod print-object ((pos 2d-position) s)
  (format s "<~S,~S>" (position-x pos) (position-y pos)))


Listing 2

(defmethod position-rho ((pos 2d-position))
  (let ((x (position-x pos)) 
        (y (position-y pos)))
    (sqrt (+ (* x x) (* y y)))))


(defmethod position-theta ((pos 2d-position))
  (atan (position-y pos) (position-x pos)))


Listing 3

(defmethod (setf position-rho) (new-rho (pos 2d-position))
  (let* ((r (position-rho pos))
         (ratio (/ new-rho r)))
   (setf (position-x pos)
         (* ratio (position-x pos)))
   (setf (position-y pos)
         (* ratio (position-y pos)))
   new-rho))


(defmethod (setf position-theta) (new-theta (pos 2d-position))
  (let ((rho (position-rho pos)))
   (setf (position-x pos)
         (* rho (cos new-theta)))
   (setf (position-y pos)
         (* rho (sin new-theta)))
   new-theta)) 


Listing 4


> (setq xy (make-instance '2d-position :x 1.0 :y 1.0))
<1.0,1.0>
> (position-x xy)
1.0
> (setf (position-rho xy) 10.0)
10.0
> (position-x xy)
7.071067811865475
> (incf (position-theta xy) (position-theta xy))
1.5707963267948966
> (position-y xy)
10.0


Listing 5

(defmethod make-instance ((class standard-class) &rest initargs)
  (setq initargs (default-initargs class initargs))
  ...
  (let ((instance (apply #'allocate-instance class initargs)))
    (apply #'initialize-instance instance initargs)
    instance))


Listing 6

(defmethod initialize-instance
   ((instance standard-object) &rest initargs)
  (apply #'shared-initialize instance t initargs)))


Listing 7

(defmethod shared-initialize :after
           ((instance 2d-position) slot-names
            &key (x nil x-p) (y nil y-p)
                 (rho nil rho-p) (theta nil theta-p))
  (when (or (eq slot-names 't)
            (subsetp '(x y) slot-names))
    (cond ((and x-p y-p))
          ((and rho-p theta-p)
           (with-slots (x y) instance
             (setq x (* rho (cos theta)))
             (setq y (* rho (sin theta)))))
          ((and x-p rho-p)
           (with-slots (y) instance
             (setq y (sqrt (- (* rho rho) (* x x))))))
          ((and x-p theta-p)
           (with-slots (y) instance
             (setq y (* x (tan theta)))))
          ((and y-p rho-p)
           (with-slots (x) instance
             (setq x (sqrt (- (* rho rho) (* y y))))))
          ((and y-p theta-p)
           (with-slots (x) instance
             (setq x (/ y (tan theta)))))
          (t (error "Bad initargs (~S)" (list x y rho theta))))))


Listing 8

(defclass x-axis-position (2d-position) () (:default-initargs :y 0.0))


Listing 9

(defmethod print-object ((pos 2d-position) s)
  (format s "<~S,~S>" (position-rho pos) (position-theta pos)))


(defmethod update-instance-for-redefined-class :before
   ((pos 2d-position) added deleted plist &key)
  (declare (ignore added deleted))
  (let ((x (getf plist 'x))
        (y (getf plist 'y)))
    (setf (position-rho pos) (sqrt (+ (* x x) (* y y)))
          (position-theta pos) (atan y x))))

(defclass 2d-position (position)
    ((rho :accessor position-rho :initarg :rho)
     (theta :accessor position-theta :initarg :theta)))


Listing 10

(defclass x-y-position (position)
    ((x :accessor position-x :initarg :x)
     (y :accessor position-y :initarg :y )))
 
(defclass rho-theta-position (position)
    ((rho :accessor position-rho :initarg :rho)
     (theta :accessor position-theta :initarg :theta)))


Listing 11

(defmethod update-instance-for-different-class :before 
   ((old x-y-position) (new rho-theta-position) &key)
  (let ((x (slot-value old 'x))
        (y (slot-value old 'y)))
    (setf (slot-value new 'rho) (sqrt (+ (* x x) (* y y)))
          (slot-value new 'theta) (atan y x))))


Listing 12

> (setq p1 (make-instance 'x-y-position :x 1.0 :y 1.0))
<1.0,1.0>
> (change-class p1 'rho-theta-position)
<1.4142135623730952,0.7853981633974483>




