/*******************************************************************/
/*                                           GRANDMOTHER           */
/*   A simulation of a grandmother cell bin-sorting neural network */
/*                                                                 */
/*   Written by Maureen Caudill  in Lightspeed C on a Macintosh    */
/*                                                                 */
/*   (Note that Lightspeed C has a pre-defined constant for PI     */
/*    which is automatically included with the math package. )     */
/*                                                                 */
/*******************************************************************/

#include  <math.h>                    /* math library headers (for sin/cos)   */
#include  <stdio.h>                   /* standard i/o library headers         */
/*     (for printf and scanf) */

#define PI 3.141592653589
#define NUMBINS         5               /* number of grandmother cells   */
#define NUMATTS         2               /* number of pattern attributes  */
                                        /* or elements                   */

/*                        GLOBAL STORAGE                         */
/*  The weights of the grandmother cells are stored in a         */
/*  two-dimensional array called weights.  This array has one    */
/*  row for each grandmother cell, and the number of columns     */
/*  is determined by the number of weights on each cell.         */
/*                                                               */
/*  The resulting output of each grandmother cell is stored in   */
/*  the array called outputs.                                    */
/*                                                               */
/*  The input pattern to classify is stored in array pattern.    */
/*                                                               */
/*  Stopflag is used to exit the program when done.              */
/*                                                               */
/* Double precision floating point is used for easy compatibility*/
/* with the math functions  sin, cos, sqrt                       */
/*---------------------------------------------------------------*/
double  weights[NUMBINS][NUMATTS], outputs[NUMBINS];
double  pattern[NUMATTS];
int             stopflag;

/*---------------------------------------------------------------*/
initweights()
{
	/*  Initialize weights to correspond to "archetype" patterns.  */
	/*  For this example, the weights correspond to weight vectors */
	/*  at the angles of:  30!, 120!, 180!, 225!, 300! on the unit */
	/*  circle.                                                    */

	double  sin30, sin60, sin45;
	double  cos30, cos60, cos45;

	/*  set up constants */
	sin30   = sin(PI/6.0);
	sin60   = sin(PI/3.0);
	sin45   = sin(PI/4.0);
	cos45   = sin45;
	cos30   = sin60;
	cos60   = sin30;

	/*  now initialize the weights array   */
	weights[0][0] = cos30;        /*  30!  vector */
	weights[0][1] = sin30;
	weights[1][0] = -cos60;       /* 120! vector */
	weights[1][1] = sin60;
	weights[2][0] = -1;           /* 180! vector */
	weights[2][1] = 0;
	weights[3][0] = -cos45;       /* 225! vector */
	weights[3][1] = -sin45;
	weights[4][0] = cos60;        /* 300! vector */
	weights[4][1] = -sin60;
	return;
}

/*----------------------------------------------------------*/
getpattern()
{
	/* Have the user enter a pattern to classify.  Normalize it, and */
	/* set the stop flag if the pattern is 0,0                       */

	double  length, sumsq;
	int             i;

	printf("\n  Please enter the pattern you wish to classify.");
	printf("\n  The pattern should be in x,y coordinate format.");
	printf("\n  It will be normalized to a unit vector before classification.");
	printf("\n  For example, a 45! vector should be entered as 1 1");
	printf("\n  A pattern entry of 0,0 will stop the program.");
	printf("\n\n  The pattern to classify?     ");
	scanf("%lf %lf",&pattern[0],&pattern[1]);

	/* normalize the pattern entered */
	sumsq = 0.0;
	for (i=0;  i<NUMATTS;  i++)
		sumsq += pattern[i]*pattern[i];
	length = sqrt(sumsq);

	/* check for a 0,0 entry before completing normalization */
	if (length != 0.0)
	{
		/* not a 0,0 entry, so go ahead and normalize it */
		for (i=0; i<NUMATTS; i++)
			pattern[i] /= length;
	}
	else /* pattern was 0 0, so set stopflag */
		stopflag = 1;
	return;
}

/*----------------------------------------------------------*/
classify()
{
	/* Actually perform the classification of the pattern       */

	int             i,j;
	float   sum;

	for (i=0; i<NUMBINS; i++)
	{
		/* calculate weighted sum of input pattern for each cell */
		sum = 0;
		for (j=0; j<NUMATTS; j++)
			sum += pattern[j]*weights[i][j];
		/* save current weighted sum in array 'outputs'  */
		outputs[i] = sum;
	}
	return;
}

/*----------------------------------------------------------*/
displayvectors()
{
	/* Display the pattern to be classified, and the resulting outputs */
	/* and pattern classification.                                     */

	int             i;
	int             maxout;

	printf("\n\n  Pattern:      %06.3f    %06.3f",pattern[0],pattern[1]);
	printf("\n\n  Cell#:        WeightX    WeightY     Output");
	maxout  = 0;
	for (i=0; i<NUMBINS; i++)
	{
		printf(  "\n   %d            %06.3f     %06.3f      %06.3f",
		    i,weights[i][0],weights[i][1],outputs[i]);
		if (outputs[maxout] < outputs[i])
			maxout = i;
	}
	printf("\n\n  Pattern classification is  %d  \n\n", maxout);
	return;
}

/*-----------------------------------------------------------*/
/*  MAIN PROGRAM */

main ()
{
	stopflag = 0;
	initweights();
	getpattern();
	while (stopflag == 0)
	{
		classify();
		displayvectors();
		getpattern();
	}
	printf("\n\n  Stopping at user request.  Thank you. \n");
}
