/*********************************************************
 Backpropagation Network
 Written by Maureen Caudill 
 in Lightspeed C on a Macintosh
 
 This network will accept 5x7 input patterns
 and produce 8 bit output patterns.
 See the May, 1988 AI Expert, Neural Network Primer, Part 3 
 for a discussion of Backpropagation networks.
 
 *********************************************************/

/*********************************************************
 Structure of this program:
 
 Global data declarations
 
 Functions:
 do_forward_pass     propagate input activity forward thru network
 do_out_forward      do output layer, forward pass
 do_mid_forward      do middle layer, forward pass
 display_output      display output of network
 do_back_pass        propagate error activity backward thru network
 do_out_error        compute output layer errors
 do_mid_error        compute middle layer errors
 adjust_out_wts      adjust output layer weights
 adjust_mid_wts      adjust middle layer weights
 check_out_error     check to see if network knows all patterns yet
 initialize_net      do network initialization
 randomize_wts       randomize wts on middle & output layers
 read_data_file      read input/desired out patterns from data file
 display_mid_wts     output the weights on the middle layer neurodes
 display_out_wts     output the weights on the output layer neurodes
 main                main control program
*********************************************************/
/*********************************************************
       GLOBAL DATA & CONSTANT DECLARATIONS
*********************************************************/
#include <stdio.h>
#include <math.h>
/*#include <EventMgr.h>*/  /* included for Mac event traps only */
/*#include <MacTypes.h>*/  /* included for Mac event traps only */
#define T         1     
#define F         0
#define ERR      -1
#define MAXPATS  10   /* max number of patterns in data file */
#define IN_X_SIZE 5   /* number of neurodes/row of input layer */
#define IN_Y_SIZE 7   /* number of neurodes/col of input layer */
#define IN_SIZE  35   /* equals IN_X_SIZE*IN_Y_SIZE */
#define MID_SIZE  4   /* number of neurodes in middle layer */
#define OUT_SIZE  8   /* number of neurodes in output layer */
#define MARGIN  0.1   /* how near to 1,0 do we have to come to stop? */
#define BETA    0.3   /* beta learning constant */
#define ALPHA     0   /* momentum term constant */
double  mid_wts[MID_SIZE][IN_SIZE];  /* middle layer weights */
double  out_wts[OUT_SIZE][MID_SIZE]; /* output layer weights */
double  mid_out[MID_SIZE];   /* middle layer output */
double  out_out[OUT_SIZE];   /* output layer output */
double  mid_error[MID_SIZE]; /* middle layer errors */
double  out_error[OUT_SIZE]; /* output layer errors */
double  mid_wt_change[MID_SIZE][IN_SIZE];  /* storage for last wt change */
double  out_wt_change[OUT_SIZE][MID_SIZE]; /* storage for last wt change */
double  in_pats[MAXPATS][IN_SIZE];   /* input patterns */
double  out_pats[MAXPATS][OUT_SIZE]; /* desired output patterns */
double  tot_out_error[MAXPATS];  /* measure of whether net is done */
int iteration_count;  /* number of passes thru network so far */
int numpats;  /* number of patterns in data file */
int learned;  /* flag--if TRUE, network has learned all patterns */
FILE *fopen(), *outfile;
char *path = "backprop.out";
char *inpath = "alphabet";

/*********************************************************
  do_forward_pass(patt)
    control function for the forward pass through the network
*********************************************************/
do_forward_pass(patt)
int patt;   /* pattern number from input file */
{
	do_mid_forward(patt);   /* process forward pass, middle layer */
	do_out_forward();       /* process forward pass, output layer */
	display_output(patt);   /* display results of forward pass */
	return;
}

/*********************************************************
  do_mid_forward(patt)
   process the middle layer's forward pass
   The activation of middle layer's neurode is the weighted
   sum of the inputs from the input pattern, with sigmoid
   function applied to the inputs.
*********************************************************/

do_mid_forward(patt)
int patt; /* pattern number from input file */
{
	double sum;
	int neurode, i;

	for (neurode=0;neurode<MID_SIZE; neurode++)
	{
		sum = 0.0;
		for (i=0; i<IN_SIZE; i++)
		{/* compute weighted sum of input signals */
			sum += mid_wts[neurode][i]*in_pats[patt][i];
		}
		/* apply function f(x) = 1/(1+exp(-5x)) to weighted sum       */
		/* the factor of 5 is arbitrarily chosen to give a good range */
		/* from almost 0 to nearly 1                                  */
		sum = 1/(1+exp(-5*sum));
		mid_out[neurode] = sum;
	}
	return;
}

/**********************************************************
do_out_forward()
  process the forward pass through the output layer
  The activation of the output layer is the weighted sum of
  the inputs (outputs from middle layer), modified by the
  sigmoid function.
*********************************************************/
do_out_forward()
{
	double sum;
	int neurode, i;

	for (neurode=0; neurode<OUT_SIZE; neurode++)
	{
		sum = 0.0;
		for (i=0; i<MID_SIZE; i++)
		{   /* compute weighted sum of input signals from middle layer */
			sum += out_wts[neurode][i]*mid_out[i];
		}
		/* apply f(x) = 1/(1+exp(-5x)) to weighted input              */
		/* the factor of 5 is arbitrarily chosen to give a good range */
		/* from almost 0 to nearly 1                                  */
		sum = 1/(1+exp(-5*sum));
		out_out[neurode] = sum;
	}
	return;
}

/*******************************************************
display_output(patt)
Display the actual output vs. the desired output of the
network.
Once the training is complete, and the "learned" flag set
to TRUE, then display_output sends its output to both
the screen and to a text output file.
*******************************************************/
display_output(patt)
int patt;
{
	int i;

	fprintf(outfile,"\n Iteration # %d",iteration_count);
	fprintf(outfile,"\n Desired Output:  ");

	for (i=0; i<OUT_SIZE; i++)
	{
		fprintf(outfile,"%6.3f  ",out_pats[patt][i]);
	}
	fprintf(outfile,"\n Actual Output:   ");

	for (i=0; i<OUT_SIZE; i++)
	{
		fprintf(outfile,"%6.3f  ",out_out[i]);
	}
	fprintf(outfile,"\n");

	return;
}

/*********************************************************
do_back_pass(patt)
Process the backward propagation of error through network.
*********************************************************/
do_back_pass(patt)
int patt;
{
	do_out_error(patt);
	do_mid_error();
	adjust_out_wts();
	adjust_mid_wts(patt);
	return;
}

/********************************************************
do_out_error(patt)
Compute the error for the output layer neurodes.
This is simply Desired - Actual.
********************************************************/
do_out_error(patt)
int patt;
{
	int neurode;
	double error,tot_error;

	tot_error = 0.0;
	for (neurode=0; neurode<OUT_SIZE; neurode++)
	{
		out_error[neurode] = out_pats[patt][neurode] - out_out[neurode];
		/* while we're here, also compute magnitude of total error for
   output layer.  We use this to decide if we are done yet. */
		error = out_error[neurode];
		if (error <0)
		{
			tot_error -= error;
		}
		else
		{
			tot_error += error;
		}
	}
	tot_out_error[patt] = tot_error;
	/* used to check if network done */
	return;
}

/*********************************************************
do_mid_error()
Compute the error for the middle layer neurodes
This is based on the output errors computed above.
Note that the derivative of the sigmoid f(x) is
f'(x) = f(x)(1 - f(x))
Recall that f(x) is merely the output of the middle
layer neurode on the forward pass.

*********************************************************/
do_mid_error()
{
	double sum;
	int neurode, i;

	for (neurode=0; neurode<MID_SIZE; neurode++)
	{
		sum = 0.0;
		for (i=0; i<OUT_SIZE; i++)
		{
			sum += out_wts[i][neurode]*out_error[i];
		}
		/* apply the derivative of the sigmoid here */
		mid_error[neurode] = mid_out[neurode]*(1-mid_out[neurode])*sum;
	}
	return;
}

/*********************************************************
adjust_out_wts()
Adjust the weights of the output layer.  The error for
the output layer has been previously propagated back to
the middle layer.
Use the Delta Rule with momentum term to adjust the weights.
*********************************************************/
adjust_out_wts()
{
	int weight, neurode;
	double length;/* magnitude of input vector to neurode */
	double    learn,delta,alph;

	learn = BETA;
	alph  = ALPHA;
	length = 0.0;/* since all output neurodes see the same
                           input pattern, only have to compute this
	                           once!    */
	for (weight=0; weight<MID_SIZE; weight++)
	{
		length += mid_out[weight]*mid_out[weight];

		/* since we really want the square of the length, we don't
   bother taking the square root -- we'd only have to square
   it again */
	}
	if (length<=0.1) length = 0.1;

	for (neurode=0; neurode<OUT_SIZE; neurode++)
	{
		for (weight=0; weight<MID_SIZE; weight++)
		{
			delta =learn*out_error[neurode]*mid_out[weight]/length;
			out_wts[neurode][weight] += 
			    delta;
			out_wt_change[neurode][weight] = delta;

		}
	}
	return;
}

/********************************************************
adjust_mid_wts(patt)
Adjust the middle layer weights using the previously computed
errors.
We use the Generalized Delta Rule with momentum term
********************************************************/
adjust_mid_wts(patt)
int patt;
{
	int weight, neurode;
	double length;/* magnitude of input vector to neurode */
	double    learn,alph,delta;

	learn = BETA;
	alph  = ALPHA;
	length = 0.0;/* since all output neurodes see the same
                           input pattern, only have to compute this
	                           once!    */
	for (weight=0; weight<IN_SIZE; weight++)
	{
		length += in_pats[patt][weight]*in_pats[patt][weight];
		/* since we really want the square of the length, we don't
   bother taking the square root -- we'd only have to square
   it again */
	}
	if (length<=0.1) length = 0.1;

	for (neurode=0; neurode<MID_SIZE; neurode++)
	{
		for (weight=0; weight<IN_SIZE; weight++)
		{
			delta =learn*mid_error[neurode]*in_pats[patt][weight]/length;
			mid_wts[neurode][weight] +=
			    delta;
			mid_wt_change[neurode][weight] = delta;
		}
	}
	return;
}

/********************************************************
check_out_error()
Check to see if the error in the output layer is below
MARGIN*OUT_SIZE for all output patterns.  If so, then
assume the network has learned acceptably well.  This 
is simply an arbitrary measure of how well the network
has learned -- many other better standards are possible.
********************************************************/
check_out_error()
{
	int result, i, error;
	double standard, final_err;

	result = T;
	error   = F;
	final_err = 0.0;
	standard = 0.8;
	printf("\n Iteration # %d",iteration_count);
	for (i=0; i<numpats; i++)
	{
		printf("\n Total error pattern %d:  %8.3f",i,tot_out_error[i]);
		fprintf(outfile,
		    "\n Total error pattern %d:  %8.3f",i,tot_out_error[i]);
		final_err += tot_out_error[i];
		if (tot_out_error[i] >= standard)
			result = F;
		if (tot_out_error[i] >= 16.0)
			error = T;
	}
	if (error == T)
		result = ERR;
	printf("\n Total error this pass thru input file:  %6.3f",final_err);
	fprintf(outfile,"\n Total error this pass thru input file:  %6.3f",
	    final_err);
	return result;
}

/*******************************************************
initialize_net()
Do all the initialization stuff before beginning
*******************************************************/
initialize_net()
{
	int err_code;

	randomize_wts();
	err_code = read_data_file();
	iteration_count = 1;
	return err_code;
}

/******************************************************
randomize_wts()
Intialize the weights in the middle and output layers to
random values between -0.25..+0.25
******************************************************/
randomize_wts()
{
	int neurode,i;
	double value;

	printf("\n Please enter a random number seed (1..32767):  ");
	scanf("%d", &i);
	srand(i);
	for (neurode = 0; neurode<MID_SIZE; neurode++)
	{
		for(i=0; i<IN_SIZE; i++)
		{
			value = (((double) (rand())) /32767.0 ) - 0.5;
			mid_wts[neurode][i] = value/2;
			mid_wt_change[neurode][i] = 0.0;
		}
	}
	for (neurode=0; neurode<OUT_SIZE; neurode++)
	{
		for(i=0; i<MID_SIZE; i++)
		{
			value = (((double) (rand())) / 32767.0 ) - 0.5;
			out_wts[neurode][i] = value/2;
			out_wt_change[neurode][i] = 0.0;
		}
	}
	return;
}

/******************************************************
read_data_file()
Read in the input data file and store the patterns in
in_pats and out_pats.

The format for the data file is as follows:
line#   data expected
1      In-X-size,in-y-size,out-size
2      number of patterns in file
3      1st X row of 1st pattern
4      ...following rows of 1st pattern
       in-x+2y-out pattern
       +11st X row of 2nd pattern
       etc.

Each row of data is separated by commas or spaces.
The data is expected to be ascii text corresponding to
either a +1 or a 0.  

Sample input for a 1-pattern file (The comments to the
right may NOT be in the file unless more sophisticated
parsing of the input is done.):

5 7 8                   input is 5x7 grid, output is 8 bits
1                       one pattern in file
0 1 1 1 0               beginning of pattern for "O"
1 0 0 0 1
1 0 0 0 1
1 0 0 0 1
1 0 0 0 1
1 0 0 0 0
0 1 1 1 0
0 1 0 0 1 1 1 1         ASCII code for "O" -- 0100 1111

Clearly, this simple scheme can be expanded or enhanced
any way you like.

Returns -1 if any file error occurred, otherwise 0.
******************************************************/
read_data_file()
{
	FILE *infile;

	int xinsize,yinsize,youtsize;
	int patt, element, i, row, col;
	int vals_read;
	int val1,val2,val3,val4,val5,val6,val7,val8;

	printf("\n Opening and retrieving data from file.");

	infile = fopen(inpath, "r");
	if (infile == NULL)
	{
		printf("\n error in opening file!");
		return -1 ;
	}
	vals_read =fscanf(infile,"%d  %d  %d",&xinsize,&yinsize,&youtsize);
	if (vals_read != 3)
	{
		printf("\n Should read 3 items in line one; did read %d",vals_read);
		return -1;
	}
	vals_read=fscanf(infile,"%d",&numpats);
	if (vals_read !=1)
	{
		printf("\n Should read 1 item in line 2; did read &d",vals_read);
		return -1;
	}
	if (numpats > MAXPATS)
		numpats = MAXPATS;

	for (patt=0; patt<numpats; patt++)
	{
		element = 0;
		for (row = 0; row<yinsize; row++)
		{
			vals_read =fscanf(infile,"%d  %d  %d  %d  %d",
			    &val1, &val2, &val3, &val4, &val5);
			if (vals_read != 5)
			{
				printf ("\n failure in reading input!");
				return -1;
			}

			element=row*xinsize;
			in_pats[patt][element] = (double) val1; 
			element++;
			in_pats[patt][element] = (double) val2; 
			element++;
			in_pats[patt][element] = (double) val3; 
			element++;
			in_pats[patt][element] = (double) val4; 
			element++;
			in_pats[patt][element] = (double) val5; 
			element++;
		}
		for (i=0;i<IN_SIZE; i++)
		{
			if (in_pats[patt][i] >= 0.9)
				in_pats[patt][i] = 0.9;
			if (in_pats[patt][i] <= 0.1)
				in_pats[patt][i] = 0.1;
		}
		element = 0;
		vals_read = fscanf(infile,"%d  %d  %d  %d  %d  %d  %d  %d",
		    &val1, &val2, &val3, &val4, &val5, &val6, &val7, &val8);
		out_pats[patt][element] = (double) val1; 
		element++;
		out_pats[patt][element] = (double) val2; 
		element++;
		out_pats[patt][element] = (double) val3; 
		element++;
		out_pats[patt][element] = (double) val4; 
		element++;
		out_pats[patt][element] = (double) val5; 
		element++;
		out_pats[patt][element] = (double) val6; 
		element++;
		out_pats[patt][element] = (double) val7; 
		element++;
		out_pats[patt][element] = (double) val8; 
		element++;
	}

	printf("\n Closing the input file now. ");
	fclose(infile);
	return 0;
}

/*******************************************************
display_mid_wts()
Display the weights on the middle layer neurodes
*******************************************************/
display_mid_wts()
{
	int neurode, weight, row, col;

	fprintf(outfile,"\n Weights of Middle Layer neurodes:");
	for (neurode=0; neurode<MID_SIZE; neurode++)
	{
		fprintf(outfile,"\n  Mid Neurode # %d",neurode);
		for (row=0; row<IN_Y_SIZE; row++)
		{
			fprintf(outfile,"\n ");
			for (col=0; col<IN_X_SIZE; col++)
			{
				weight = IN_X_SIZE * row + col;
				fprintf(outfile," %8.3f ", mid_wts[neurode][weight]);
			}
		}
	}
	return;
}

/*******************************************************
display_out_wts()
Display the weights on the output layer neurodes
*******************************************************/
display_out_wts()
{
	int neurode, weight;

	fprintf(outfile,"\n Weights of Output Layer neurodes:");
	for (neurode=0; neurode<OUT_SIZE; neurode++)
	{
		fprintf(outfile,"\n  Out Neurode # %d \n",neurode);
		for (weight=0; weight<MID_SIZE; weight++)
		{
			fprintf(outfile," %8.3f ", out_wts[neurode][weight]);
		}
	}
	return;
}

/****************************************************
*****************************************************
                  MAIN PROGRAM
*****************************************************
*****************************************************/
main()
{
	int patt;
	int error;
	/* The next two definitions are Mac unique defined in Mac Toolbox */
	/*int mouse_pressed;*/    /* used to check for mouse press */
	/*EventRecord  *event;*/  /* used to check for mouse press */

	/* Set event mask to screen for either mouse pressed or released */
	/*mouse_pressed = BitOr (mouseDown, mouseUp);*/  /* Macintosh unique code */
	error = initialize_net();
	if (error!= 0)
	{
		printf("\n\n Program terminating with file error");
		return;
	}

	outfile = fopen(path,"w");         /* open an output text file */
	learned = F;
	while (learned == F)
	{
		for (patt=0; patt<numpats; patt++)
		{
			do_forward_pass(patt);
			do_back_pass(patt);
			iteration_count++;
			printf(" .");
		}
		learned = check_out_error();

		/* The following is Macintosh unique code to allow an easy bailout */
		/* It terminates the program if the user has pressed the mouse button */
		/* at any time since the last pass through the file   */
		/*if (GetNextEvent(mouse_pressed,event)==TRUE)
learned = T;*/
	}

	if (learned == ERR)
	{
		printf("\n\n Error # %d in learning after %d iterations",
		    learned,iteration_count);
		fprintf(outfile,"\n\n Error # %d in learning after %d iterations",
		    learned,iteration_count);
	}
	else
	{
		printf("\n\n  TRAINING COMPLETE after %d iterations", 
		    iteration_count);
		fprintf(outfile,
		    "\n\n  TRAINING COMPLETE after %d iterations", iteration_count);
	}

	printf("\n  Posting final weights to file...");
	display_mid_wts();
	display_out_wts();

	for (patt=0; patt<numpats; patt++)
	{
		do_forward_pass(patt);
	}
	fclose (outfile);
	return;
}
