
#|----------------------------------------------------------------------------
Artificial Intelligence, Second Edition
Elaine Rich and Kevin Knight
McGraw Hill, 1991

This code may be freely copied and used for educational or research purposes.
All software written by Kevin Knight.
Comments, bugs, improvements to knight@cs.cmu.edu
----------------------------------------------------------------------------|#

#|----------------------------------------------------------------------------
			PROPOSITIONAL RESOLUTION
			    "resolve.lisp"
----------------------------------------------------------------------------|#

;; --------------------------------------------------------------------------
;; Variable *AXIOMS* holds the list of axioms in clause form.  For example, 
;; the axioms below should be read:
;;
;; 		p
;;		~p \/ ~q \/ r
;;		~s \/ q
;;		~t \/ q
;;		t
;;
;; as in the example on page 150 of the text.

(defvar *axioms*) 

(setq *axioms*
  '((p)
    ((not p) (not q) r)
    ((not s) q)
    ((not t) q)
    (t)))

;; --------------------------------------------------------------------------
;; Function PROVE attempts to prove a statement using propositional resolution.
;; It adds the negation of the statement to the axioms and tries to find a 
;; contradiction.  It keeps a list of all axioms and inferred facts.  Every
;; time it finds a new fact, it pairs that fact with all other known facts
;; and adds the pairs to the end of pairs-to-resolve.  If a pair of facts
;; ever resolve to an empty clause (contradiction), then the program halts 
;; and returns the list of all known facts.  If pairs-to-resolve is exhausted,
;; then the program returns nil, as a sign that the proof could not be 
;; carried out.

(defun prove (statement)
  (let* ((new-axioms (cons (if (consp statement)
			       (cadr statement)
			       (list 'not statement))
			   *axioms*))
         (cf-axioms (mapcar #'convert-to-clause-form new-axioms))
         (pairs-to-resolve
	    (do ((c cf-axioms (cdr c))
		 (p nil))
	        ((null (cdr c)) p)
	      (setq p (append p (mapcar #'(lambda (x) 
		 			      (list (car c) x))
					(cdr c)))))))
     (do ((success nil))
         ((or (null pairs-to-resolve) success)
	  (nreverse success))
       (let* ((p (car pairs-to-resolve))
	      (resolvents (resolve (car p) (cadr p))))
	  (setq pairs-to-resolve (cdr pairs-to-resolve))
	  (dolist (r resolvents)
	     (setq pairs-to-resolve
		   (append pairs-to-resolve
			   (mapcar #'(lambda (x) (list x r))
				   cf-axioms)))
	     (setq cf-axioms (cons r cf-axioms)))
          (when (member nil cf-axioms)
	     (setq success cf-axioms))))))

;; Function RESOLVE returns all ways of resolving two clauses that remove
;; literals.

(defun resolve (x1 x2)
  (let ((r nil))
     (dolist (elt x1)
	(when (and (atom elt)
		   (member (list 'not elt) x2 :test #'equal))
	   (setq r (cons (append (remove elt x1)
				 (remove (list 'not elt) x2
					 :test #'equal))
			 r))))
     (dolist (elt x2)
	(when (and (atom elt)
		   (member (list 'not elt) x1 :test #'equal))
	   (setq r (cons (append (remove elt x2)
				 (remove (list 'not elt) x1
					 :test #'equal))
			 r))))
     r))

(defun convert-to-clause-form (x)
  (cond ((eq (car x) 'not) (list x))
	(t x)))


;; Example:
;;
;; (prove 'r)  ->
;;
;; ((T) ((NOT T) Q) ((NOT S) Q) ((NOT P) (NOT Q) R) (P) ((NOT R))
;; ((NOT P) (NOT Q)) ((NOT Q) R) ((NOT S) (NOT P) R)
;; ((NOT T) (NOT P) R) (Q) ((NOT Q)) ((NOT S) (NOT P))
;; ((NOT T) (NOT P)) ((NOT Q)) ((NOT S) R) ((NOT T) R)
;; ((NOT S) (NOT P)) ((NOT S) R) ((NOT T) (NOT P)) ((NOT T) R)
;; ((NOT P) R) (R) ((NOT P)) ((NOT P) R) NIL)

