;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX11:DIS$DISK:[GBB.V-120.LOCAL]UTILITIES.LISP *-*
;;;; *-* Edited-By: Cork *-*
;;;; *-* Last-Edit: Thursday, September 15, 1988  12:55:18 *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *              DEFSTRUCT FUNCTIONS, AND OTHER USEFUL CODE
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  01-08-86 File Created.  (Gallagher)
;;;     ...   Many changes too numerous to list here.  (Gallagher)
;;;  10-02-86 Moved the functions unit-type-p, unit-instance-p, and
;;;           get-unit-description from define_unit to this file.  (Gallagher)
;;;  10-13-86 Moved accessor and psuedo-type predicates to a new file,
;;;           accessors.lisp.  (Gallagher)
;;;  12-04-86 Added MAKE-PATHS and CHANGE-PATHS. (Johnson)
;;;  01-13-87 In BB-PARSE-NAME-AND-OPTIONS added the options :unnamed and
;;;           :constructor and deleted :hash-unit.  Added the functions
;;;           ANONYMOUS-UNIT-TYPE-P and ANONYMOUS-UNIT-INSTANCE-P.  (Gallagher)
;;;  01-14-87 Moved MAKE-PATHS, etc. from this file to space.  (Gallagher)
;;;  01-16-87 Added NCOPY-LIST and POP-SLICES.  (Gallagher)
;;;  02-06-87 Several changes because the name slot of a unit-instance is now a
;;;           slot in basic-unit.  (Gallagher)
;;;  02-10-87 Added DUMMY-UNIT-INSTANCE-P.  (JOHNSON)
;;;  02-10-87 Added ->.  (Gallagher)
;;;  02-22-87 Units now inherit the print-function from the included unit (if
;;;           any.  Added RESET-GBB and UNIT$NAME.  (Gallagher)
;;;  03-23-87 Rewrote event handling.  (Gallagher)
;;;  11-20-87 Changed the default from :no-subtypes to :plus-subtypes.  (Cork)
;;;  06-27-88 Add CHECK-OPTIONS.  (Gallagher)
;;;  08-12-88 Added unit-slot-names, unit-link-names, and all-unit-types.
;;;           (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'GBB :use '(lisp umass-extended-lisp))

(export '(unit-type-p
	  unit-instance-p
	  unit$name
          unit-slot-names
          unit-link-names
          all-unit-types
	  dummy-unit-instance-p
	  gbb-documentation
	  reset-gbb))

(use-package '(umass-extended-lisp))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))


;;; --------------------------------------------------------------------------
;;;   Support functions for DEFINE-UNIT
;;; --------------------------------------------------------------------------


(defun parse-name-and-options (name-and-options)
  
  "PARSE-NAME-AND-OPTIONS name-and-options

   Parse the defstruct name-and-options to handle GBB extensions.  Several
   more options are added, including :NAME-FUNCTION, :UNNAMED, and all the
   event keywords.  The argument to the :INCLUDE option can only be a unit
   -- not a defstruct.  Also, if no :CONC-NAME option is specified the
   :CONC-NAME will be ``NAME$''.

   This function returns four values:
     1. The new name-and-options list suitable for use by defstruct;
     2. A partially filled in unit-description;
     3. A flag indicating whether generated symbols should be exported;
     4. If #3 is true, a list of some of the symbols that should be
        exported (the unit name, the constructor name, and the predicate)."
  
  (unless (listp name-and-options)
    (setf name-and-options (list name-and-options)))
  (let* ((unit-name (first name-and-options))
	 (unit-name-string (string unit-name))
	 (external-conc-name (string-concatenate unit-name-string "$"))
	 (name-function nil)
	 (name-function-seen nil)
	 (unnamed nil)
	 (print-function nil)
	 (predicate (form-symbol unit-name-string "-P"))
	 (constructor (form-symbol "MAKE-" unit-name-string))
	 (constructor-seen nil)
	 (included-unit nil)
	 (unit-events nil)
	 (export nil)
	 description option-name args)

    (labels
      ;; These use OPTION-NAME and UNIT-NAME which are bound above.
      ((get-one-option-argument (args &optional default)
         ;; Check that the only zero or one arguments wer supplied.
         (cond ((null args) default)
               ((null-arg-p args) nil)
               ((/= (length args) 1)
                (error "Bad ~:@(~s~) option while defining ~s:~%~8@t(~s~{ ~s~})~@
                        The ~:@(~s~) option takes only zero or one arguments."
                       option-name unit-name option-name args option-name))
               (t (first args))))
       (get-one-symbol-argument (args &optional default)
         ;; Check that the argument is a symbol or omitted.
         (let ((the-arg (get-one-option-argument args default)))
           (error-unless (symbolp the-arg)
              "Bad ~:@(~s~) option while defining ~s:~%~8@t(~s~{ ~s~})~@
               The argument to ~:@(~s~) must be a symbol."
              option-name unit-name option-name args option-name)
           the-arg)))

      ;; For all the options if a symbol is given it is interpreted the
      ;; same as a one element list
      ;; (e.g., :CONC-NAME == (:CONC-NAME) == use the default conc-name).

      ;; This is a ridiculously long dolist...
      ;;  -- Yes, it is long but conceptually it is very simple.
      (dolist (option (cdr name-and-options))
        (cond ((symbolp option)
               (setf option-name option
                     args nil))
              (t (setf option-name (first option)
                       args (rest option))))
        (case option-name
          (:name-function
           (setf name-function (get-one-option-argument args))
           (setf name-function-seen t))
          (:unnamed
           (setf unnamed (get-one-option-argument args t)))
          (:print-function
           ;; Save the print function for inheritance via :include.
           (setf print-function (get-one-option-argument args)))
          (:conc-name
           (setf external-conc-name
                 (get-one-option-argument args external-conc-name)))
          (:include
           (setf included-unit (get-one-symbol-argument args))
           (error-unless (unit-type-p included-unit)
              "Bad :INCLUDE option while defining ~s.~@
               The argument to :INCLUDE must be a unit type.~%~s is not a unit."
              unit-name included-unit))
          (:export
           (setf export (or args t)))
          (:predicate
           (setf predicate (get-one-symbol-argument args predicate)))
          (:constructor
           (when constructor-seen
             (error "While defining ~s, more than one :CONSTRUCTOR option was seen."
                    unit-name))
           (setf constructor
                 (get-one-symbol-argument args constructor))
           (setf constructor-seen t))
          ((:copier :type :initial-offset :named)
           ;; Defstruct options that we don't support.
           (cerror "Ignore the ~*~:@(~s~) option."
                   "While defining ~s, the ~:@(~s~) option was seen.~@
                    This option is not supported for units."
                   unit-name option-name)
           (setf name-and-options (remove option name-and-options :test #'eq)))
          (otherwise
           (cond (;; Check for events
                  (event-type-name-p option-name)
                  (setf unit-events
                        (set-unit-events option-name
                                         unit-events
                                         (parse-events args))))
                 (t
                  (error "Unknown option given to DEFINE-UNIT: ~s~@
                         While defining ~s."
                         option-name unit-name)))))
        )  ;; End of ridiculously long DoList.

      (setf name-function (inherit-name-function name-function
                                                 name-function-seen
                                                 unnamed
                                                 included-unit
                                                 unit-name))

      (setf print-function (inherit-print-function print-function included-unit))

      ;; Make sure that the conc-name's are strings
      (setf external-conc-name (string external-conc-name))

      (setf description
            (make-unit-description
              :name			unit-name
              :package			(package-name *package*)
              :external-conc-name	external-conc-name
              :internal-conc-name	(string-concatenate "%%GBB-INTERNAL-" external-conc-name)
              :external-constructor	constructor
              :internal-constructor	(and constructor
					     (form-symbol "%%GBB-INTERNAL-CONSTRUCT-" unit-name))
              :included			included-unit
              :events			unit-events
              :name-function		name-function
              :print-function		print-function
              :name-and-options		name-and-options))

      (values `(,unit-name
                ,@(and predicate `((:predicate ,predicate)))
                (:print-function ,print-function)
                (:conc-name ,(intern (unit.internal-conc-name description)))
                (:constructor ,(and constructor (unit.internal-constructor description))))
              description
              export
              (cond ((null export) nil)
                    ((null constructor)
                     (list unit-name predicate))
                    (t (list unit-name predicate constructor)))))))

;;; -----------------------------------------------------------------------

(defun null-arg-p (args)
  "Returns true if ARGS is (NIL).  Used in processing defstruct
   style options."
  (equal args '(nil)))

;;; -----------------------------------------------------------------------

(defun inherit-name-function (name-function
			      name-function-seen
			      unnamed
			      included-unit
			      unit-name)
  ;; Name function rules:
  ;;  1. Can't specify both :unnamed and :name-function.
  ;;  2. If a :name-function option is given and an argument is supplied then
  ;;     use that argument.
  ;;  3. If the unit includes another unit and no :name-function option is
  ;;     given or it is given but no argument is supplied then use the name
  ;;     function from the included unit.
  ;;  4. Otherwise, use the default name function.

  (cond ((and unnamed name-function-seen)
	 (error "The incompatible options :NAME-FUNCTION and :UNNAMED~@
                 specified while defining the unit ~s."
		unit-name))
	(unnamed :unnamed)
	((and name-function-seen name-function)
	 name-function)
	(included-unit
	 (unit.name-function (get-unit-description included-unit)))
	(t 'default-unit-name-function)))

;;; -----------------------------------------------------------------------

(defun inherit-print-function (print-function included-unit)

  ;; Print function rules:
  ;;  1. If a :print-function option is given and an argument is supplied then
  ;;     use that argument.
  ;;  2. If the unit includes another unit and no :print-function option is
  ;;     given or it is given but no argument is supplied then use the print
  ;;     function from the included unit.
  ;;  3. Otherwise use the default print function.
  (cond (print-function print-function)
	(included-unit
	 (unit.print-function (get-unit-description included-unit)))
	(t 'default-unit-print-function)))

;;; -----------------------------------------------------------------------

(defun check-options (plist keywords)
  "This function checks that all the ``indicators'' in PLIST are legal.
   An indicator is legal if it is a member of KEYWORDS, which is a list
   of the known indicators.  This function returns a list of all the
   unknown indicators; Returns NIL if all the indicators are legal."
  (let ((result nil))
    (dolist-by-twos (key value plist)
      (unless (member key keywords :test #'eq)
        (push key result)))
    result))


;;; --------------------------------------------------------------------------
;;;   Predicates on Units Types and Unit Instances
;;; --------------------------------------------------------------------------

(defun unit-type-p (object)

  "UNIT-TYPE-P object

   This function returns true if OBJECT is a unit and false otherwise.
   This is not the same as a unit instance -- see UNIT-INSTANCE-P."

  (and (symbolp object)
       (gethash object *unit-description-hash-table*)))

;;; -----------------------------------------------------------------------

(defun unit-instance-p (object)

  "UNIT-INSTANCE-P object

   If OBJECT is an instance of a unit then this function returns the
   type of unit.  Otherwise it returns false."

  (and (typep object 'basic-unit)
       (unit-type-of object)))

;;; -----------------------------------------------------------------------

(defun equal-unit-instance-p (instance1 instance2)

  "EQUAL-UNIT-INSTANCE-P instance1 instance2

   Returns T if INSTANCE1 and INSTANCE2 are equal
   (i.e. if they are both unit instances of the same type, 
   have a name slot, and have the same value for the name slot.)
   Returns NIL otherwise."

  (let ((instance1-type (unit-instance-p instance1))
	(instance2-type (unit-instance-p instance2)))

    (and instance1-type instance2-type
	 (eq instance1-type instance2-type)
	 (not (anonymous-unit-type-p instance1-type))
	 (not (anonymous-unit-type-p instance2-type))
	 ;; if we've gotten this far, test the name slot values.
	 (equal (basic-unit.name instance1)
		(basic-unit.name instance2)))))

;;; -----------------------------------------------------------------------

(defun anonymous-unit-type-p (unit-type)

  "ANONYMOUS-UNIT-TYPE-P unit-type

   Returns true if instances of `Unit-Type' are not being named by GBB."

  (eq (unit.name-function (get-unit-description unit-type))
      :unnamed))

;;; -----------------------------------------------------------------------

(defun anonymous-unit-instance-p (unit-instance)

  "ANONYMOUS-UNIT-INSTANCE-P unit-instance

   Returns true if the `Unit-Instance' is an anonymous unit."

  (eq (unit.name-function (get-unit-description unit-instance))
      :unnamed))

;;; -----------------------------------------------------------------------

(defun dummy-unit-instance-exists-p (unit-instance)

  "DUMMY-UNIT-INSTANCE-EXISTS-P unit-instance

   Returns the corresponding dummy unit instance if it exists, NIL otherwise"

  (and (not (anonymous-unit-instance-p unit-instance))
       (gethash (basic-unit.name unit-instance)
		(unit-info.dummy-hash-table (get-unit-info unit-instance)))))

;;; -----------------------------------------------------------------------

(defun dummy-unit-instance-p (unit-instance)

  "DUMMY-UNIT-INSTANCE-P unit-instance

   Returns T if it is a dummy unit, NIL otherwise."

  (and (not (anonymous-unit-instance-p unit-instance))
       (eq unit-instance
	   (gethash (basic-unit.name unit-instance)
		    (unit-info.dummy-hash-table (get-unit-info unit-instance))))))


;;; --------------------------------------------------------------------------
;;;   Generic Unit Functions
;;; --------------------------------------------------------------------------

(defun unit$name (unit-instance)

  "UNIT$NAME unit-instance

   Returns the name of a unit instance or the symbol :anonymous
   if the unit is anonymous."

  (basic-unit.name unit-instance))

;;; -----------------------------------------------------------------------

(defun unit-slot-names (unit-type)

  "UNIT-SLOT-NAMES unit-type

   This function returns a list of the names of the slots for UNIT-TYPE."

  (slot-names (get-unit-description unit-type)))

;;; -----------------------------------------------------------------------

(defun unit-link-names (unit-type)

  "UNIT-LINK-NAMES unit-type

   This function returns a list of the names of the links for UNIT-TYPE."

  (link-names (get-unit-description unit-type)))

;;; -----------------------------------------------------------------------

(defun all-unit-types ()

  "ALL-UNIT-TYPES nil

   This function returns a list of all currently defined unit types."

  *all-units*)

;;; -----------------------------------------------------------------------

(defun gbb-documentation (symbol type)

  "GBB-DOCUMENTATION symbol type

   Returns the documentation associated with `symbol'.  If no
   documentation is found return nil.  Standard values of `type'
   are: SPACE, BLACKBOARD, INDEX-STRUCTURE, and UNIT."
  
  (let ((doc-type
          (assoc type
                 '(("SPACE" . get-space-documentation)
                   ("BLACKBOARD" . get-blackboard-documentation)
                   ("INDEX-STRUCTURE" . get-index-structure-documentation)
                   ("UNIT" . get-unit-documentation))
                 :test #'string=)))
    (error-unless doc-type
       "~s is not a GBB type.~@
        The GBB types are SPACE, BLACKBOARD, INDEX-STRUCTURE, and UNIT."
       type)
    (funcall (cdr doc-type) symbol)))

(defun get-space-documentation (symbol)
  (and (space-name-p symbol)
       (space.documentation (get-space symbol))))

(defun get-blackboard-documentation (symbol)
  (and (blackboard-name-p symbol)
       (bb.documentation (get-space symbol))))

(defun get-index-structure-documentation (symbol)
  (and (index-structure-name-p symbol)
       (index-structure.documentation (get-index-structure symbol))))

(defun get-unit-documentation (symbol)
  (and (unit-type-p symbol)
       (unit.documentation (get-unit-description symbol))))


;;; --------------------------------------------------------------------------
;;;   Unit Description Accessors, etc.
;;; --------------------------------------------------------------------------

(defun get-unit-description (unit &optional no-error-p)

  "GET-UNIT-DESCRIPTION unit

   Return the unit-description associated with UNIT.  UNIT may be a symbol
   or a unit instance.  If the optional argument, NO-ERROR-P, is true then
   no error will be signalled if a unit description can't be found.

   This function may be used as a place for for SETF if and only if the
   argument, UNIT, is a symbol."

  (cond ((typecase unit
           ;; Check for a unit instance first, because that is
           ;; probably what it will be.
           (basic-unit
            (gethash (unit-type-of unit) *unit-description-hash-table*))
           (unit-description
            unit)
           (symbol
            (gethash unit *unit-description-hash-table*))))
	(no-error-p nil)
	(t (error "~s is not a unit, a unit-instance, or a unit-description."
		  unit))))

;;; -----------------------------------------------------------------------

(defsetf get-unit-description (unit &optional no-error-p) (description)

  "Associate the argument, DESCRIPTION, with UNIT which must be a
   symbol."

  (declare (ignore no-error-p))
  `(progn
     (assert (symbolp ,unit) nil
	     "Can't set the unit description for ~s." ,unit)
     (setf (gethash ,unit *unit-description-hash-table*) ,description)))

;;; -----------------------------------------------------------------------

(defun get-included-unit-description (unit)

  "GET-INCLUDED-UNIT-DESCRIPTION unit

   If `Unit' includes another unit, then return the unit-description for
   the included unit, otherwise return nil."

   (let ((included-unit (unit.included (get-unit-description unit))))
     (and included-unit
	  (get-unit-description included-unit))))

;;; -----------------------------------------------------------------------

(defun get-unit-info (unit &optional no-error-p)

  "GET-UNIT-INFO unit

   Return the unit-information for UNIT.  UNIT may be a symbol or a unit
   instance.  If the optional argument, NO-ERROR-P, is true then no
   error will be signalled if a unit description can't be found.

   This function may be used as a place for for SETF if and only if the
   argument, UNIT, is a symbol."

  (cond ((typecase unit
           ;; Check for a unit instance first, because that is
           ;; probably what it will be.
           (basic-unit
            (gethash (unit-type-of unit) *unit-information-hash-table*))
           (unit-information
            unit)
           (unit-description
            (gethash (unit.name unit) *unit-information-hash-table*))
           (symbol
            (gethash unit *unit-information-hash-table*))))
	(no-error-p nil)
	(t (error "~s is not a unit, a unit-instance, or a unit-info."
		  unit))))

;;; -----------------------------------------------------------------------

(defsetf get-unit-info (unit &optional no-error-p) (info)

  "Associate the argument, INFO, with UNIT which must be a symbol."

  (declare (ignore no-error-p))
  `(progn
     (assert (symbolp ,unit) nil
	     "Can't set the unit info for ~s." ,unit)
     (setf (gethash ,unit *unit-information-hash-table*) ,info)))

;;; -----------------------------------------------------------------------

(defun unit-instantiable-p (description)

  "UNIT-INSTANTIABLE-P description

   Returns true if the unit type specified by `Description' can be
   instantiated (i.e., if it has a constructor defined for it.)"

  (unit.external-constructor description))

;;; -----------------------------------------------------------------------

(defun default-unit-print-function (object stream depth)

  "DEFAULT-UNIT-PRINT-FUNCTION object stream depth

   This function prints a unit instance."

  (declare (ignore depth))
  (cond (*print-escape*
	 (format stream "#<~s ~s ~x>"
		 (unit-type-of object)
		 (basic-unit.name object)
		 (%pointer object)))
	(t
	 (format stream "~a" (basic-unit.name object)))))

;;; -----------------------------------------------------------------------

(defun -> (object)

  "-> object

   This function tries to return a concise identification of object.
   If `Object' is a named unit-instance then it returns its name.
   Otherwise it simply returns the object itself.  It is used to
   concisely identify objects, especially in error messages."

  (cond ((not (unit-instance-p object)) object)
	((eq (basic-unit.name object) :anonymous)
	 object)
	(t (basic-unit.name object))))

;;; -----------------------------------------------------------------------

(defun unit-subtypep (unit1 unit2)

  "UNIT-SUBTYPEP unit1 unit2

   Returns true if UNIT1 is a subtype of UNIT2 (i.e., if UNIT1
   includes UNIT2 either directly or through inheritance).  Both UNIT1
   and UNIT2 must be acceptable to the function GET-UNIT-DESCRIPTION."

  (setf unit1 (get-unit-description unit1)
	unit2 (get-unit-description unit2))
  (let ((included-unit (unit.included unit1)))
    (cond ((eq unit1 unit2) t)
	  ((null included-unit) nil)
	  (t (unit-subtypep included-unit unit2)))))

;;; -----------------------------------------------------------------------

(defconstant SIMPLE-UNIT-TYPES-INCLUDE-SUBTYPES t

  "This constant governs the inheritance behavior of ``simple unit
   types.'' A simple unit type is a symbol (e.g., HYP).  If this
   constant, SIMPLE-UNIT-TYPES-INCLUDE-SUBTYPES, is true then a simple
   unit type will be considered as including its subtypes.  If this
   constant is false then a simple unit type will be considered to not
   include its subtypes.")

;;; -----------------------------------------------------------------------

(defun x-unit-type-of (obj &optional no-error-p)

  "X-UNIT-TYPE-OF obj

   Determines if OBJ is an ``extended unit type.'' An extended unit
   tyep is a simple unit type or a list of the form:

       (<unit-type> {:plus-subtypes | :no-subtypes}).

   This function returns two values: the simple unit type (a symbol),
   and whether or not the unit type should include its subtypes."
  
  (cond ((listp obj)
	 (cond ((and (= (length obj) 2)
		     (unit-type-p (first obj))
		     (member (second obj) '(:plus-subtypes :no-subtypes)
			     :test #'eq))
		(values (first obj) (eq (second obj) :plus-subtypes)))
	       (no-error-p nil)
               ((and (= (length obj) 2)
                     (unit-type-p (first obj)))
                (error "~s is not an extended unit type.~@
                        ~s is a unit type but ~s is not a valid subtype indicator."
                       obj (first obj) (second obj)))
	       (t (error "~s is not an extended unit type." obj))))
	((and (symbolp obj)
	      (gethash obj *unit-description-hash-table*))
	 (values obj simple-unit-types-include-subtypes))
	;; ((unit-description-p obj)
	;;  (values (unit.name obj) simple-unit-types-include-subtypes))
	;; ((basic-unit-p obj)
	;;  (values (unit-type-of obj) simple-unit-types-include-subtypes))
	(no-error-p nil)
	(t (error "~s is not an extended unit type." obj))))

;;; -----------------------------------------------------------------------

(defun x-unit-subtypep (unit1 unit2)

  "X-UNIT-SUBTYPEP unit1 unit2

   Returns true if UNIT1 is a subtype of UNIT2 (i.e., if UNIT1
   includes UNIT2 either directly or through inheritance).  UNIT1
   must be acceptable to the function GET-UNIT-DESCRIPTION, but
   UNIT2 may be an extended unit type."

  (let ((included-unit nil))
    (multiple-value-bind (unit2-sym include-subtypes?)
	(x-unit-type-of unit2)
      (setf unit1 (get-unit-description unit1)
	    unit2 (get-unit-description unit2-sym))
      (cond ((eq unit1 unit2) t)
	    ((not include-subtypes?) nil)
	    ((setf included-unit (unit.included unit1))
	     (unit-subtypep included-unit unit2))
	    (t nil)))))

;;; -----------------------------------------------------------------------

(defun find-ancestor (unit unit-types)

  "FIND-ANCESTOR unit unit-types

   Returns the element from UNIT-TYPES that is the closest ancestor of
   UNIT.  UNIT is an extended unit type and UNIT-TYPES is a list of
   extended unit types."

  (let* ((the-unit (first-if-list unit))
         (description (get-unit-description the-unit))
         (included (unit.included description)))
    (cond ((find the-unit unit-types :key #'first-if-list :test #'eq))
          (included
           (find-ancestor included unit-types))
          (t nil))))

;;; -----------------------------------------------------------------------

(defun unit-subtypes-included-p (unit-type)

  "UNIT-SUBTYPES-INCLUDED-P unit-type

   Returns true if UNIT-TYPE includes its subtypes."

  (multiple-value-bind (ignore include-subtypes?)
      (x-unit-type-of unit-type)
    (declare (ignore ignore))
    include-subtypes?))

;;; -----------------------------------------------------------------------

(defun expand-unit-type-list (unit-types)

  "EXPAND-UNIT-TYPE-LIST unit-types

   This function examines the argument, UNIT-TYPES, which may be a
   simple unit type, an extended unit type, or a list of simple or
   extended unit types.  It returns a list of simple unit types
   with subtypes of the units in UNIT-TYPES included when
   appropriate."

  (let ((result nil)
	unit-type include-subtypes?)

    ;; Check if UNIT-TYPES is a single simple or extended unit type.
    (multiple-value-setq (unit-type include-subtypes?)
      (x-unit-type-of unit-types t))

    (cond (unit-type
           ;; Single simple or extended unit type.
	   (cons unit-type
		 (if include-subtypes?
		     (unit.subtypes (get-unit-description unit-type))
		     nil)))
	  (t
           ;; List of simple or extended unit types.
	   (dolist (x-type unit-types)
             (multiple-value-setq (unit-type include-subtypes?)
               (x-unit-type-of x-type))
	     (when include-subtypes?
               (setf result
                     (union result (unit.subtypes (get-unit-description unit-type))
                            :test #'eq)))
	     ;; Include this unit type whether or not its subtypes are included.
	     (pushnew unit-type result :test #'eq))
	   result))))

;;; -----------------------------------------------------------------------

(defun simple-unit-type-list (unit-types)

  "EXPAND-UNIT-TYPE-LIST unit-types

   This function examines the argument, UNIT-TYPES, which may be a
   simple unit type, an extended unit type, or a list of simple or
   extended unit types.  It returns a list of simple unit types
   without any subtypes of the units in UNIT-TYPES regardless of
   whether subtypes were reuested."

  (let ((result nil)
	unit-type)

    ;; Check if UNIT-TYPES is a single simple or extended unit type.
    (setq unit-type (x-unit-type-of unit-types t))

    (cond (unit-type
           ;; Single simple or extended unit type.
	   (list unit-type))
	  (t
           ;; List of simple or extended unit types.
	   (dolist (x-type unit-types)
             (setq unit-type (x-unit-type-of x-type))
	     (pushnew unit-type result :test #'eq))
	   result))))

;;; -----------------------------------------------------------------------

(defun reset-gbb ()

  "RESET-GBB nil

   Resets GBB.  This tries to destroy as much of the saved state
   of GBB as it can.  It is not infallible, it still leaves some
   information around."

  ;; Clobber the compiler transforms.
  (dolist (unit *all-units*)
    (let ((description (get-unit-description unit)))
      (delete-transform (unit.external-constructor description) :gbb)))

  ;; Clobber user definitions.
  (clrhash *blackboard-hash-table*)
  (clrhash *space-hash-table*)
  (clrhash *unit-description-hash-table*)
  (clrhash *unit-information-hash-table*)
  (clrhash *link-description-hash-table*)
  (clrhash *index-structure-hash-table*)

  ;; Reset other variables to their startup value.
  (setq *all-units* nil)
  (setq *blackboard-database* nil)
  (setq *space-instance-list* nil)
  (setq %%run-events%% t)

  ;; Probably don't want to reset these.  For example, the graphics
  ;; uses these and reseting them breaks it.
  ;(setq *instantiate-bb-db-hook-functions* nil)
  ;(setq *clear-space-hook-functions* nil)
  ;(setq *insert-unit-hook-functions* nil)
  ;(setq *move-unit-hook-functions* nil)
  ;(setq *delete-unit-hook-functions* nil)
  )


;;; These add-xxx functions add the data (e.g., slots) from one unit
;;; to the data in another.  I use nconc rather than append to preserve
;;; the order of the entries.  NEW-DATA can be either a list or a
;;; unit-description.  D1 must be a unit-description.

(defun add-unit (unit-description included-unit)
  (let ((included-description
	  (if (unit-description-p included-unit)
	      included-unit
	      (get-unit-description included-unit))))
    (add-slots unit-description included-description)
    (add-links unit-description included-description)
    (add-d-indexes unit-description included-description)
    (add-p-indexes unit-description included-description)
    (add-paths unit-description included-description)))

(defun add-slots (d1 new-data)
  "Add the slots from new-data to the ones in d1."
  (when (unit-description-p new-data)
    (setf new-data (copy-list (unit.slot-list new-data))))
  (setf (unit.slot-list d1) (nconc (unit.slot-list d1) new-data)))

(defun add-d-indexes (d1 new-data)
  "Add some indexes to D1.  NEW-DATA may be a list of indexes or
   another unit description."
  (when (unit-description-p new-data)
    (setf new-data (copy-list (unit.d-index-list new-data))))
  (setf (unit.d-index-list d1)
	(nconc (unit.d-index-list d1) new-data)))

(defun add-p-indexes (d1 new-data)
  "Add some path indexes to D1.  NEW-DATA may be a list of indexes or
   another unit description."
  (when (unit-description-p new-data)
    (setf new-data (copy-list (unit.p-index-list new-data))))
  (setf (unit.p-index-list d1)
	(nconc (unit.p-index-list d1) new-data)))

(defun add-paths (d1 new-data)
  "Add the paths in d1 to the ones in d2."
  (when (unit-description-p new-data)
    (setf new-data (copy-list (unit.paths new-data))))
  (setf (unit.paths d1)
	(nconc (unit.paths d1) new-data)))

(defun add-links (d1 new-data)
  "Add the links in d1 to the ones in d2."
  (when (unit-description-p new-data)
    (setf new-data (copy-list (unit.link-list new-data))))
  (setf (unit.link-list d1)
	(nconc (unit.link-list d1) new-data)))

(defun add-unit-data (description slots indexes links path-indexes paths)
  "Add SLOTS, INDEXES, LINKS, etc. to DESCRIPTION.  SLOTS are
   copied, The others (INDEXES, LINKS, etc.) are not."
  (add-slots
    description
    ;; Make each slot spec a list to simplify further processing.
    ;; This also copies the list so I can nconc onto it later.
    (mapcar #'assure-list slots))
  (add-d-indexes description indexes)
  (add-links description links)
  (add-p-indexes description path-indexes)
  (add-paths description paths)
  (if (unit.included description)
      (add-paths description (get-included-unit-description description))))


;;; --------------------------------------------------------------------------
;;;   Event Inheritance
;;; --------------------------------------------------------------------------


(defun inherit-events (description)
  "If DESCRIPTION includes another unit then copy up to this unit
   all the events from the included unit."
  (setf description (get-unit-description description))
  (when (unit.included description)
    (dolist (event *event-types*)
      (set-unit-events event description (inherit-event event description)))))


(defun inherit-event (event-type this-desc)

  "Returns a list of EVENT-RECORDs with events suitably inherited.
   `Event-type' is an event type name (e.g., :creation-events).
   `This-desc' can be a unit-description or a unit-type.

   Event inheritance follows these rules:
     o By default, all events are inherited from the included unit
       to this unit. However, this mechanism can be overridden
       as follows:
     o If this unit specifies (:creation-events :no-default-inheritance),
       then events are not inherited from the included unit (except for events
       in the included unit specified as :always-inherit.)
     o If the included unit specifies (Event-X :always-inherit), then Event-X
       is inherited, even when this unit specifies :no-default-inheritance.
     o If this unit specifies (Event-X :never-inherit), then Event-X is not
       inherited from the included unit. It is an error for the included unit
       to specify (Event-X :always-inherit) and this unit to specify
       (Event-X :never-inherit).
     o Both the event name and its inheritance specification are inherited.
       Thus if unit4 includes unit3 includes unit2, and
       unit2 specifies (Event-X :always-inherit), unit3 and unit4 will inherit
       (Event-X :always-inherit), not just Event-X. However, :no-default-inheritance
       is not inherited to included units.
   See the GBB Reference Manual for more information."

  (setf this-desc (get-unit-description this-desc))
  (let* ((inc-desc (get-included-unit-description this-desc))
	 ;; We need the event records here, not just the names:
         (inc-events (get-unit-event-records event-type inc-desc))
         (these-events (get-unit-event-records event-type this-desc))
	 ;; The events that will be inherited:
	 (inherited-events nil))

    (cond ((member-if #'no-default-inheritance-p these-events)
	   (setf inherited-events
		 (mapc-condcons
		   #'(lambda (inc-event)
		       (check-for-inheritance-conflict
			 inc-event these-events inc-desc this-desc)
		       (and (always-inherit-p inc-event)
			    inc-event))
		   inc-events)))
          
          (t ;; inherit all slots, unless it's :never-inherit, or
             ;; it's the keyword :no-default-inheritance.
             ;; Signal an error if it's also :always-inherit.
	   (setf inherited-events
		 (mapc-condcons
		   #'(lambda (inc-event)
		       (check-for-inheritance-conflict
			 inc-event these-events inc-desc this-desc)
		       (unless (or (never-inherit-this-event inc-event these-events)
				   (never-inherit-p inc-event)
				   (no-default-inheritance-p inc-event))
			 inc-event))
		   inc-events))))

    ;; Remove duplicate events.
    (nfilter-duplicate-events inherited-events these-events)))


;;; The `events' stored in the various xxx-events slots of slots
;;; and links are simply lists of symbols.  The events in the events
;;; slot of the unit-description are structured objects.  The slot
;;; events don't need to be any more sophisticated than symbols
;;; because they are not inherited.

(defun mixin-included-events (event-keyword inc-events description)

  "Return a list of events to be stored in the slot or link object.
   INC-EVENTS are the events that were stored with the link/slot object
   of the included unit.  These are symbols.  EVENT-KEYWORD and
   DESCRIPTION allow us to get the set of new events to be added, as
   well as those events to take out (because they were specified as
   :NEVER-INHERIT."

  (let ((new-events (get-unit-event-records event-keyword description))
	(mixed-in-events nil))
    (when (member-if #'no-default-inheritance-p new-events)
      (setf inc-events nil))
    (dolist (new-event new-events)
      ;; Either add the event to inc-events, or remove it from inc-events.
      ;; When adding, maintain ordering of the events (inherited first).
      (cond ((no-default-inheritance-p new-event)
	     nil)
	    ((never-inherit-p new-event)
	     (setf inc-events (remove (event.name new-event) inc-events
				      :test #'eq)))
	    (t
	     (unless (member (event.name new-event) inc-events :test #'eq)
	       (push (event.name new-event) mixed-in-events)))))
    (append inc-events mixed-in-events)))
 
(defun check-for-inheritance-conflict (inc-event new-events inc-desc new-desc)

  "CHECK-FOR-INHERITANCE-CONFLICT inc-event new-events inc-desc new-desc
   Signal an error if the included unit's event is :always-inherit, and the
   same event is :never-inherit in the new-unit's event list"

  (let ((inherited-event (find (event.name inc-event) new-events
			       :key #'event.name :test #'eq)))
    
    (error-when (and inherited-event
		     (always-inherit-p inc-event)
		     (never-inherit-p inherited-event))
      "~s specifies :never-inherit for slot ~s, but~@
       ~s includes ~s, which specifies :always-inherit for ~s."
      (unit.name new-desc) (event.name inc-event) (unit.name new-desc)
      (unit.name inc-desc) (event.name inc-event))))

(defun nfilter-duplicate-events (some-events more-events)

  "NFILTER-DUPLICATE-EVENTS some-events more-events

   Merge `some-events' with `more-events'.  `Some-events' and
   `more-events' are lists of event-records (i.e., structures).  If
   there are any duplicates, the first event (from `some-events') takes
   precedence.

   Caution: The first argument is destructively modified."

  (cond ((null some-events)
	 more-events)
	((not (intersectp some-events more-events
			  :key #'event.name :test #'eq))
	 (nconc some-events more-events))
	(t (nconc some-events
		  (remove-if #'(lambda (e)
				 (member (event.name e) some-events
					 :key #'event.name :test #'eq))
			     more-events)))))

(defun filter-duplicate-event-names (some-symbols more-symbols)

  "FILTER-DUPLICATE-EVENT-NAMES some-symbols more-symbols

   Merges `some-symbols' with `more-symbols'.  This function is
   similar to UNION except that any symbol in some-symbols will
   preceed all the symbols in more-symbols and the order of elements
   in each list is maintained (except in cases of duplicate elements)."

  (setf some-symbols (assure-list some-symbols))
  (append some-symbols (set-difference more-symbols some-symbols :test #'eq)))


(defun no-default-inheritance-p (event)
  (eq (event.flag event) :no-default-inheritance))

(defun always-inherit-p (event)
  (eq (event.flag event) :always-inherit))

(defun never-inherit-p (event)
  (eq (event.flag event) :never-inherit))

(defun never-inherit-this-event (event event-list)
  "Returns true if `event' is found in `event-list' and it is
   its flag is :never-inherit in `event-list'."
  (let ((ev (find (event.name event) event-list :key #'event.name :test #'eq)))
    (and ev (never-inherit-p ev))))
	 

(defun pushnew-event (event event-list)
  (cond ((listp event) ;; then it's :always-inherit- remove any default occurrance.
         (setf event-list (cons event (remove-if #'(lambda (ele)
                                                     (or (equal ele (first event))
                                                         (equal ele event)))
                                                 event-list))))
         (t ;; pushnew unless it's already there as :always-inherit
           (pushnew event event-list :test #'(lambda (event event-in-list)
                                               (or (eq event event-in-list)
                                                   (and (listp event-in-list)
                                                        (eq event (first event-in-list)))))))))



;;; --------------------------------------------------------------------------
;;;   Meta-Access functions for UNIT-DESCRIPTION
;;; --------------------------------------------------------------------------

(defun find-slot (slot description)
  "FIND-SLOT slot description
   Returns the slot spec for SLOT.  SLOT may be defined directly in the
   unit or inherited from an included unit.  
   DESCRIPTION may be a unit type or a unit description."
  (setf description (get-unit-description description))
  (or (find slot (unit.slots description) :key #'slot.name :test #'string=)
      (and (unit.included description)
	   (find-slot slot (unit.included description)))))

(defun find-link (link description)
  "FIND-LINK link description
   Returns the link spec for LINK.  LINK may be defined directly in
   the unit or inherited from an included unit.
   DESCRIPTION may be a unit type or a unit description."
  (setf description (get-unit-description description))
  (or (find link (unit.links description) :key #'link.name :test #'string=)
      (and (unit.included description)
	   (find-link link (unit.included description)))))

(defun find-d-index (index description)
  "FIND-D-INDEX index description
   Returns the index spec for INDEX.  INDEX may be defined directly in
   the unit or inherited from an included unit.
   DESCRIPTION may be a unit type or a unit description."
  (setf description (get-unit-description description))
  (or (assoc index (unit.d-indexes description) :key #'unit-index.name :test #'string=)
      (and (unit.included description)
	   (find-d-index index (unit.included description)))))

(defun find-p-index (index description)
  "FIND-P-INDEX index description
   Returns the index spec for INDEX.  INDEX may be defined directly in
   the unit or inherited from an included unit.
   DESCRIPTION may be a unit type or a unit description."
  (setf description (get-unit-description description))
  ;; No need to look at the included units because the indexes are copied
  ;; up to this one.
  (find index (unit.p-indexes description)
	:key #'unit-index.name :test #'string=))
  

(defun slot-names (description)
  "SLOT-NAMES description
   Returns a list of the names (symbols) of the slots.  The argument may
   be a unit type or a unit description.  Be aware that this function
   conses a fresh list every time it is called."
  (setf description (get-unit-description description))
  (if (unit.slots description)
      (mapcar #'slot.name (unit.slots description))
      (delete-duplicates
        (nconc (mapcar #'car (unit.slot-list description))
               (and (unit.included description)
                    (slot-names (unit.included description))))
        :test #'string=)))

(defun link-names (description)
  "LINK-NAMES description
   Returns a list of the names (symbols) of the links.  The argument may
   be a unit type or a unit description.  Be aware that this function
   conses a fresh list every time it is called."
  (setf description (get-unit-description description))
  (if (unit.links description)
      (mapcar #'link.name (unit.links description))
      (delete-duplicates
        (nconc (mapcar #'car (unit.link-list description))
               (and (unit.included description)
                    (link-names (unit.included description))))
        :test #'string=)))

(defun included-unit-names (description)
  "INCLUDED-UNIT-NAMES description
   Returns a list of all the included units in a unit
   The argument may be a unit type or a unit description."
  (setf description (get-unit-description description))
  (nconc (list (unit.included description))
	 (and (unit.included description)
	      (included-unit-names (unit.included description)))))

(defun search-included-units (predicate description)
  "SEARCH-INCLUDED-UNITS predicate description
   Apply PREDICATE to DESCRIPTION and, recursively, its included
   units.  Return the first unit description that PREDICATE
   succeeds on; otherwise return nil."
  (setf description (get-unit-description description))
  (cond ((funcall predicate description)
         description)
        (t (let ((included-unit
                   (get-included-unit-description description)))
             (and included-unit
                  (search-included-units predicate included-unit))))))

(defun deletion-event-names (description)
  "DELETION-EVENT-NAMES description
   Returns a list of the names (symbols) of the deletion events.
   The argument may be a unit type or a unit description."
  (get-unit-events :deletion-events description))

(defun creation-event-names (description)
  "CREATION-EVENT-NAMES description
   Returns a list of the names (symbols) of the creation events.
   The argument may be a unit type or a unit description."
  (get-unit-events :creation-events description))


(defun get-unit-events (event-type description)

  "GET-UNIT-EVENTS event-type description
   Returns the set of event functions to be run for this type of event."

  (error-unless (event-type-name-p event-type)
     "~s is not the name of an event type.~@
      The event types available are ~s."
     event-type *event-types*)

  (setf description (get-unit-description description))
  (let ((events (find event-type (unit.events description)
		      :key #'event-type.type :test #'eq)))
    (and events
	 (event-type.functions events))))


(defun set-unit-events (event-type-name description events)

  "SET-UNIT-EVENTS event-type-name description events

   Sets the EVENT-TYPE-NAME for DESCRIPTION to be EVENTS.
   EVENT-TYPE-NAME is a keyword like :slot-access-events.  DESCRIPTION
   is a unit description or a list of events.  If it's a list then the
   new list is returned."

  ;; The feature of description being a list of events rather than a
  ;; unit description is used by PARSE-NAME-AND-OPTIONS because the
  ;; unit description hasn't been built yet.

  (error-unless (event-type-name-p event-type-name)
     "~s is not the name of an event type.~@
      The event types available are ~s."
     event-type-name *event-types*)

  (setf events (assure-list events))

  (let* ((list? (listp description))
	 (event-list (if list? description (unit.events description)))
	 (event-entry (find event-type-name event-list
			    :key #'event-type.type :test #'eq))
	 (found? (not (null event-entry))))

    (cond (found?
	   ;; If there was already an entry for this event type then
	   ;; change its event records.
	   (setf (event-type.records event-entry) events))
	  (t 
	   ;; If there was no entry then create one and add it to the list.
	   (setf event-entry (make-event-type :type event-type-name :records events))
	   (setf event-list (cons event-entry event-list))))

    ;; Update the function field.
    (update-event-type event-entry)

    ;; Figure out the return value.
    (cond
      ;; The argument was a list so return the updated list.
      (list? event-list)
      ;; The argument was a unit description so return
      ;; the updated unit description.  If the entry was 
      (t (setf (unit.events description) event-list)
	 description))))

(defun update-event-type (event-type)

  "Make sure that the FUNCTIONS slot of `event-type' accurately
   reflects the contents of the RECORDS slot."

  (setf (event-type.functions event-type)
	(mapc-condcons
	  #'(lambda (event)
	      (unless (or (no-default-inheritance-p event)
			  (never-inherit-p event))
		   (event.name event)))
	  (event-type.records event-type)))
  event-type)

(defun get-unit-event-records (event-type description)
  "Returns the event records for `event-type' from `description'."
  (let ((events (find event-type (unit.events description)
		      :key #'event-type.type :test #'eq)))
    (and events
	 (event-type.records events))))

(defun event-type-name-p (event-type)
  "EVENT-TYPE-NAME-P event-type
   Returns true if the argument, `Event-Type', is the name of
   an event type."
  (and (symbolp event-type)
       (member event-type *event-types* :test #'eq)))

(defun parse-events (event-specs)
  "The argument, `event-specs', is a list of event specifications
   like (:always-inherit foo) or :no-default-inheritance.  This
   function returns a list of event records."
  (mapcar #'parse-event event-specs))

(defun parse-event (event-spec)
  (cond ((keywordp event-spec)
	 (error-unless (eq :no-default-inheritance event-spec)
	    "Unknown event keyword: ~s" event-spec)
	 (make-event-record :flag :no-default-inheritance))
	((symbolp event-spec)
	 (make-event-record :name event-spec))
	((two-element-list-p event-spec)
	 (let ((name (first event-spec))
	       (flag (second event-spec)))
	   (error-unless (and (keywordp flag)
			      (member flag '(:always-inherit :never-inherit)))
	      "Unknown event keyword: ~s" event-spec)
	   (make-event-record :name name :flag flag)))
	(t (error "Unknown event keyword: ~s" event-spec))))

(defun build-event-call (event-fn &rest args)
  "Returns a form to call an event function.  EVENT-FN is the event
   function.  It may be a symbol or a lambda expression."
  #+SYMBOLICS (setf args (copy-list args))
  (cond ((or (symbolp event-fn) (functionp event-fn))
         ;; Must check for symbol for TI Explorer.
         `(,event-fn ,@args))
        ((atom event-fn)
         (error "The event function, ~s, is an atom~@
                 but doesn't seem to be a function." event-fn))
        ((eq (car event-fn) 'function)
         `(funcall ,event-fn ,@args))
        (t (error "I don't know what to do with this event function:~%~s."
                  event-fn))))


;;; --------------------------------------------------------------------------
;;;   Random useful code
;;; --------------------------------------------------------------------------


(defun ncopy-list (to-list from-list)
  "NCOPY-LIST to from
   Copies the list `From-List' to `To-List' by replacing the conses
   of `To-List'.  The two lists must be the same length.  Returns the
   new `To-List'."
  (mapl #'(lambda (to from)
	    (setf (first to) (first from)))
	to-list
	from-list)
  to-list)

(defun pop-slices (slices)
  "POP-SLICES slices
   `Slices' should be a list of lists.  Each of these sublists is
    poped and the altered `Slices' is returned."
  (mapl #'(lambda (list)
	    (pop (first list)))
	slices)
  slices)

(defun member-eq (item sequence)
  "MEMBER-EQ item sequence
   A function version of memq.  Used as a test for FIND, etc."
  (member item sequence :test #'eq))

(defun first-if-list (list-or-atom)

  "FIRST-IF-LIST list-or-atom

   Returns the first element of LIST-OR-ATOM if it's a list,
   returns LIST-OR-ATOM if it's an atom."

  (if (consp list-or-atom)
      (first list-or-atom)
      list-or-atom))


;;; These two functions save on consing when all you want to know
;;; is whether or not they intersect but don't care what the
;;; intersection is (similar to subsetp).

(defun intersectp (list1 list2 &key key (test #'eq))
  "INTERSECTP list1 list2 &key key test
   Returns true if list1 and list2 intersect."
  (dolist (e list1 nil)
    (when (if key
	      (member (funcall key e) list2 :key key :test test)
	      (member e list2 :test test))
      (return t))))

(defun overlapp (item list)
  "OVERLAPP item list
   Returns true if `item' overlaps with the elements in `list'.  `Item'
   may be an atom or a list.  `List' must be a list.  Uses EQ to test."
  (cond ((atom item)
	 (member item list :test #'eq))
	(t (dolist (i item nil)
	     (when (member i list :test #'eq)
	       (return t))))))


(defun set-equal (set1 set2)
  "Returns true if `set1' and `set2' contain the same elements.
   Uses EQL to compare elements."
  (and (subsetp set1 set2)
       (subsetp set2 set1)))


(defun assure-list (x)
  "ASSURE-LIST x
   Ensure that x is a list."
  (if (listp x) x (list x)))


(defun flatten (obj)
  "Return a list of all the atoms in `obj'."
  ;; Don't worry, this only gets called at compile time and
  ;; only on short lists.
  (cond ((null obj) nil)
	((atom obj) (list obj))
	(t (nconc (flatten (car obj))
		  (flatten (cdr obj))))))

;; This is a function because I need to be able to funcall it.

(defun make-keyword (name)
  "MAKE-KEYWORD name
   Interns NAME in the keyword package."
  (intern (string name) "KEYWORD"))

(defun keyword-first-elements (list)

  "KEYWORD-FIRST-ELEMENTS list

   The argument, LIST, should be a list of lists.  This function
   returns LIST with the first element of each sublist made into
   a keyword."

  (mapcar #'(lambda (x)
	      (if (listp x)
		  (cons (make-keyword (first x)) (rest x))
		  (list (make-keyword x))))
	  list))

(defun same-type-p (type-1 type-2)
  "SAME-TYPE-P type-1 type-2
   Returns true if TYPE-1 and TYPE-2 are equivalent."
  (or (eq type-1 type-2)
      (and (subtypep type-1 type-2) (subtypep type-2 type-1) t)))


;;; --------------------------------------------------------------------------
;;;   Code Used in Transforms
;;; --------------------------------------------------------------------------


(defun literalp (form)

  "LITERALP form
   Returns true if `Form' is a literal (i.e., it's self-evaluating
   or it's quoted."

  (or (numberp form)
      (keywordp form)
      (stringp form)
      (characterp form)
      (and (listp form) (eq (first form) 'quote))))

(defun all-literals-p (list)

  "ALL-LITERALS-P list
   Returns true if each element in `List' is a literal."

  (every #'literalp list))

(defun unquote (form &optional (noerror t))

  "UNQUOTE form
   Strips the quote from a form.  If `Form' is self-evaluating then this
   function just returns `Form'.  In the case that `Form' is'nt
   self-evaluating an error will be signaled if `Noerror' is nil;
   otherwise this function just returns nil."

  (cond ((and (listp form)
	      (= (list-length form) 2)
	      (eq (first form) 'quote))
	 (second form))
	((literalp form)
	 form)
	(noerror nil)
	(t (error "~s is not a literal." form))))

(defun quoted-symbol-p (form)

  "QUOTED-SYMBOL-P form
   Returns true if `Form' is a quoted symbol."

  (and (listp form)
       (= (list-length form) 2)
       (eq (first form) 'quote)
       (symbolp (second form))))

(defun keyword-or-quoted-keyword-p (form)

  "KEYWORD-OR-QUOTED-KEYWORD-P form
   Returns true if Form is a keyword or a quoted keyword."

  (or (keywordp form)
      (and (quoted-symbol-p form)
	   (keywordp (second form)))))


;;; --------------------------------------------------------------------------
;;;                                End of File
;;; --------------------------------------------------------------------------

