;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.DISTRIBUTION]SPACE.LISP *-*
;;;; *-* Edited-By: Gallagher *-*
;;;; *-* Last-Edit: Monday, July 24, 1989  16:57:30 *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 416) *-*
;;;; *-* Software: TI Common Lisp System 4.105 *-*
;;;; *-* Lisp: TI Common Lisp System 4.105 (0.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *           SPACE AND BLACKBOARD DEFINITION AND INSTANTIATION
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  01-08-86 File Created.  (Gallagher)
;;;  07-17-86 Space Hash Table and Default paths added. (Johnson)
;;;  08-25-86 Redo handling of unstructured spaces and unstructured unit
;;;           mappings.  (Gallagher)
;;;  10-10-86 Added DEFINE-SPACE and DEFINE-BLACKBOARD macro to allow GBB 
;;;           code to read better for single definitions.   (Cork)
;;;  10-30-86 Added SET-DATABASE-FILTER-FUNCTION (Johnson)
;;;  01-14-87 Moved MAKE-PATHS, etc. from utilities to this file.  (Gallagher)
;;;  01-30-87 Added FILL-OUT-REPLICATION-DESCRIPTION.  Rewrote
;;;           INSTANTIATE-BLACKBOARD-DATABASE,
;;;           INSTANTIATE-BLACKBOARD-DATABASE-1, etc. (Gallagher)
;;;  07-28-87 Changed GET-SPACE-INSTANCE-LIST-FROM-PATH.  It was returning more
;;;           space instances that it should have in the case where the path
;;;           ended in an index.  (Gallagher & Johnson)
;;;  10-26-87 Added UPDATE-SPACE-DIMENSION.  Also added coded to handle
;;;           extended unit inheritance.  (Gallagher)
;;;  01-29-88 Moved PP-BLACKBOARD-DATABASE into a new file, REPORT, which
;;;           has functions for describing the state of GBB.  (Gallagher)
;;;  09-01-88 Added ALL-PATHS.  (Gallagher)
;;;  07-24-89 Changed GET-SPACE-INSTANCE-LIST-FROM-PATH and GET-DBNODE-FROM-PATH
;;;           which were not finding some space instances in the case where
;;;           several trees in the database had the same root blackboard.
;;;           (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'GBB :USE '(lisp umass-extended-lisp))

(export '(define-spaces
          define-space
          define-blackboards
          define-blackboard
	  update-space-dimension
          instantiate-blackboard-database
	  clear-space
	  clear-blackboard-database
	  path-instantiated-p
          make-paths
          all-paths
          change-paths
	  get-space-instance-from-path-structure
	  get-path-structure-from-space-instance
	  get-space-name-from-path-structure
	  empty-space-p
	  space-occupied-p))

(use-package '(lisp umass-extended-lisp))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))


;;;; --------------------------------------------------------------------------
;;;;   Defining Spaces
;;;; --------------------------------------------------------------------------


(defun GET-SPACE (space)

  "GET-SPACE space

   Find the space associated with SPACE.  SPACE may be a symbol or a space.
   GET-SPACE may be used as a place for setf if SPACE is a symbol."

  (cond ((space-type-p space)
	 space)
        ((and (symbolp space)
	      (gethash space *space-hash-table*)))
        (t (error "~s is not a space." space))))


(defsetf GET-SPACE (space-name) (space)
  `(progn (unless (symbolp ,space-name)
            (error "Can't do a SETF on (GET-SPACE ~s)."
                   ,space-name))
          (setf (gethash ,space-name *space-hash-table*) ,space)
          ,space))


(defun SPACE-NAME-P (name)

  "SPACE-NAME-P name

   Returns true if name is a symbol which names a space."

  (and (symbolp name) (gethash name *space-hash-table*)))


(defmacro DEFINE-SPACES (spaces &rest args)

  "DEFINE-SPACES spaces [documentation] &KEY units dimensions

   Defines a set of spaces.  Units is a list of the the types of units that
   will be stored on the spaces.  Dimensions specifies the names and types 
   of the spaces' dimensionality."

  (let ((documentation (if (stringp (car args))
			   (pop args)
			   "No Documentation Supplied."))
	(dims nil))
    (with-keywords-bound ((units dimensions)
		    args
		    "~s is not a valid keyword for DEFINE-SPACES.")
      (setf dims (mapcar #'(lambda (dim)
			     (parse-space-dimension dim spaces))
			 dimensions))
      `(progn
	 (eval-when (load eval compile)
	   ,@(mapcar
	       #'(lambda (space-name)
		   `(progn
		      (setf (get-space ',space-name)
			    ',(make-space-type
				:name space-name
				:documentation documentation
				:units units
				:dimensions dims))
		      ,@(mapcar #'(lambda (d)
				    `(pushnew ',space-name
					      (get ',(dimension.name d) 'dimension-of)))
				dims)))
	       spaces)
	   ',spaces)))))


(defmacro DEFINE-SPACE (name &rest args)

  "DEFINE-SPACE space [documentation] &KEY units dimensions

   Defines a single space.  Units is a list of the the types of units that
   will be stored on the spaces.  Dimensions specifies the names and types 
   of the spaces' dimensionality."

  `(define-spaces ,(assure-list name) ,@args))


(defun PARSE-SPACE-DIMENSION (dimension space)
  "Parse a dimension specifier from DEFINE-SPACES and return
   a space-dimension."
  (declare (ignore space))
  (let* ((name (form-keyword (first dimension)))
	 (type (second dimension))
	 (arg1 (third dimension))
	 (options (cdddr dimension)))
    (create-dimension name type arg1 options)))


(defun create-dimension (name type arg1 options)
  (case type
    (:ordered
     (create-ordered-dimension name arg1 options))
    (:enumerated
     (create-enumerated-dimension name arg1 options))
    (otherwise
     (error "Bad dimension type: ~s.~@
             A dimension type must be one of :ORDERED or :ENUMERATED.~@
             The offending form is (~a ~s ~s~{ ~s~})."
	    type name type arg1 options))))


(defun create-ordered-dimension (name range options &optional odim)

  "Create or update an ordered dimension.  If ODIM is provided then
   it is changed.  Otherwise a new ordered dimension is created."

  (setf name (form-keyword name))

  (cond ((null odim)
	 (setf odim (make-ordered-dimension :name name :type :ordered)))
	((not (eq name (dimension.name odim)))
	 (error "Dimension supplied doesn't match dimension expected.~@
                 ~s is not a ~a dimension."
		odim name))
	((not (eq :ordered (dimension.type odim)))
	 (error "Dimension type mismatch.  ~s is an ~s dimension~@
                 but was expected to be an ordered dimension."
	      odim (dimension.type odim)))
	(t nil))

  (cond ((eq range :infinite)
	 (error "Infinite space dimensions are not implemented yet."))
	((not (rangep range))
	 (error "Bad dimension value for the ~a dimension.~@
                 ~s does not specify a range.~@
                 Ordered dimensions must be range of numbers (e.g., (0 100)).~@
                 The offending form is (~a :ORDERED ~s~{ ~s~})."
                 name range name range options))
	(options
	  (error "No options are available for ordered dimensions."))
	(t
	  (setf (odim.range odim) (cons (first range) (second range)))))

  odim)


(defun create-enumerated-dimension (name labelset options &optional edim)

  "Create or update an enumerated dimension.  If EDIM is provided then
   it is changed.  Otherwise a new ordered dimension is created."

  (setf name (form-keyword name))

  (cond ((null edim)
	 (setf edim (make-enumerated-dimension :name name :type :enumerated)))
	((not (eq name (dimension.name edim)))
	 (error "Dimension supplied doesn't match dimension expected.~@
                 ~s is not a ~a dimension."
		edim name))
	((not (eq :enumerated (dimension.type edim)))
	 (error "Dimension type mismatch.  ~s is an ~s dimension~@
                 but was expected to be an enumerated dimension."
	      edim (dimension.type edim)))
	(t nil))

  (error-unless (consp labelset)
     "Bad dimension value for the ~a dimension.~@
      ~s does not specify a list.~@
      Enumerated dimensions must be a list of labels.~@
      The offending form is (~a :ENUMERATED ~s~{ ~s~})."
     name labelset name labelset options)
  (with-keywords-bound (((test 'eq)) options
			"~s is not a valid keyword for the :ENUMERATED dimension type.")
    (setf (edim.labelset edim) labelset)
    (setf (edim.test edim) test))

  edim)


(defun equal-space-dimensions (d1 d2)
  "Returns true if the two dimensions are equal."
  (and (space-dimension-p d1)
       (space-dimension-p d2)
       (eq (dimension.name d1) (dimension.name d2))
       (eq (dimension.type d1) (dimension.type d2))
       (case (dimension.type d1)
         (:ordered
            (equal (odim.range d1) (odim.range d2)))
         (:enumerated
            (and (or (eq (edim.labelset d1) (edim.labelset d2))
                     (and (subsetp (edim.labelset d1) (edim.labelset d2))
                          (subsetp (edim.labelset d2) (edim.labelset d1))))
                 (eq (edim.test d1) (edim.test d2))))
         (otherwise
            nil))))


(defun UNSTRUCTURED-SPACE-P (space)
  (setf space (if (space-instance-p space)
		  (space-instance.space space)
		  (get-space space)))
  (null (space.dimensions space)))


(defun UPDATE-SPACE-DIMENSION (space dimension-name new-dimension-value)

  "UPDATE-SPACE-DIMENSION space dimension-name new-dimension-value

   Change the range or label set for a dimension for a space.
   DIMENSION-NAME is the name of a dimension of SPACE.
   NEW-DIMENSION-VALUE is the new range or label set."

  ;; Don't update the space dimensions until the end in case
  ;; an error is found.  This way the space will still be usable.

  (when (listp space)
    (dolist (s space)
      (update-space-dimension s dimension-name new-dimension-value))
    (return-from update-space-dimension nil))

  (setf space (get-space space))
  (let* ((dimensions (space.dimensions space))
	 (old-dimension (find dimension-name dimensions
			      :key #'dimension.name :test #'string=))
	 new-dimension)

    (error-unless old-dimension
       "~s is not a dimension in the space ~s." dimension-name (space.name space))
    (error-when (find space *space-instance-list*
		      :key #'space-instance.space :test #'eq)
       "The space, ~s, is instantiated in the current blackboard database."
       (space.name space) dimension-name)

    ;; Create the new dimension object.
    (ecase (dimension.type old-dimension)
      (:ordered
       (setf new-dimension (create-ordered-dimension
			     dimension-name new-dimension-value nil)))
      (:enumerated
       (setf new-dimension (create-enumerated-dimension
			     dimension-name new-dimension-value
			     nil))
       (setf (edim.test new-dimension) (edim.test old-dimension))))

    ;; Update any mappings that are affected.
    (setf (space.unit-mappings space)
	  (update-unit-mappings new-dimension
				(space.unit-mappings space)))

    ;; If we get to here then we can actually change the
    ;; dimension in the space.
    (setf (space.dimensions space)
	  (nsubstitute new-dimension old-dimension (space.dimensions space)
		       :test #'eq))

    nil))


;;;; --------------------------------------------------------------------------
;;;;   Defining Blackboards
;;;; --------------------------------------------------------------------------


(defun GET-BLACKBOARD (blackboard)

  "GET-BLACKBOARD blackboard

   Find the blackboard associated with BLACKBOARD.  BLACKBOARD may be a
   symbol or a blackboard.  GET-BLACKBOARD may be used as a place for setf
   if BLACKBOARD is a symbol."

  (cond ((blackboard-p blackboard)
	 blackboard)
        ((and (symbolp blackboard)
	      (gethash blackboard *blackboard-hash-table*)))
        (t (error "~s is not a blackboard." blackboard))))


(defsetf GET-BLACKBOARD (blackboard-name) (blackboard)
  `(progn (unless (symbolp ,blackboard-name)
            (error "Can't do a SETF on (GET-BLACKBOARD ~s)."
                   ,blackboard-name))
          (setf (gethash ,blackboard-name *blackboard-hash-table*) ,blackboard)
          ,blackboard))


(defun BLACKBOARD-NAME-P (name)

  "BLACKBOARD-NAME-P name

   Returns true if NAME is a symbol which names a blackboard."

  (and (symbolp name)
       (gethash name *blackboard-hash-table*)))


(defmacro DEFINE-BLACKBOARDS (blackboards components
			      &optional
			      (documentation "No Documentation Supplied."))

  "DEFINE-BLACKBOARDS blackboards components [documentation]

   Define blackboards.  BLACKBOARDS is a list of blackboards to be
   defined.  COMPONENTS is a list of spaces and/or blackboards from
   which the new blackboards are built. "


  (let ((component-list (assure-list components)))

    `(progn
       (error-when (some #'space-name-p ',blackboards)
	  "A space named ~s has been defined already.~@
           Blackboards and spaces must have different names.~@
           While defining blackboards ~s."
	  (find-if #'space-name-p ',blackboards) ',blackboards)
       (let ((component-bbs nil)
	     (component-spaces nil)
	     (extras nil))
	 (dolist (c ',component-list)
	   (cond ((space-name-p c)
		  (push (get-space c) component-spaces))
		 ((blackboard-name-p c)
		  (push (get-blackboard c) component-bbs))
		 (t (push c extras))))
	 (error-when extras
	    "~s ~[~;isn't a space or a blackboard.~:;aren't spaces or blackboards.~]"
	    extras (length extras))
	 (mapcar #'(lambda (bb-name)
		     (let ((bb (make-blackboard
				 :name bb-name
				 :documentation ',documentation
				 :spaces component-spaces
				 :blackboards component-bbs)))
                         (setf (get-blackboard bb-name) bb)))
		   ',blackboards))
       ',blackboards)))


(defmacro DEFINE-BLACKBOARD (name &rest args)

  "DEFINE-BLACKBOARD blackboard components [documentation]

   Define a single blackboard.  BLACKBOARD is the blackboard to be
   defined.  COMPONENTS is a list of spaces and/or blackboards from
   which the new blackboard is built. "

  `(define-blackboards ,(assure-list name) ,@args))


;;;; --------------------------------------------------------------------------
;;;;   Instantiating the Blackboard Database
;;;; --------------------------------------------------------------------------

;;; The argument to INSTANTIATE-BLACKBOARD-DATABASE is a list of
;;; `replication descriptions'.  Each replication description describes a
;;; blackboard hierarchy to be created.  It may be a symbol or a more
;;; complicated list.  The syntax of of each replication description is:
;;;
;;;     {name | (name [replication-count] [replication-description ...])}
;;;
;;; where name is the name of a blackboard or a space and replication-count
;;; is either a number or a range, which is a list of the inclusive lower
;;; bound and the exclusive upper bound.
;;;
;;; The function FILL-OUT-REPLICATION-DESCRIPTION puts the replication
;;; description into canonical form.  Canonical form is this:
;;;
;;;     (name (lower-bound upper-bound) [canonical-form ...])
;;;
;;; Differences to note between the replication description and the
;;; canonical form are:
;;;     1. The replication count must be a range.
;;;     2. The canonical form must be a list -- it can't be abbreviated to
;;;        a symbol.
;;; 
;;; FILL-OUT-REPLICATION-DESCRIPTION checks that the replication description
;;; is consistent with the blackboard definitions produced by DEFINE-SPACES
;;; and DEFINE-BLACKBOARDS.  If include-all-components is true it will add
;;; all the component blackboards that aren't explicitly mentioned in the
;;; replication description.

(defun INSTANTIATE-BLACKBOARD-DATABASE (&rest args)
  
  "INSTANTIATE-BLACKBOARD-DATABASE {descriptions}*
                                   &key (mode :ask) 
                                        (include-all-components t)

   Create the internal structures and storage for the blackboard database.
   Each element of DESCRIPTIONS (called a replication description)
   describes a blackboard hierarchy to be created.  It may be a symbol or
   a more complicated list.  The syntax of of each replication description
   is:

      {name | (name [replication-count] [replication-description ...])}

   MODE may be one of :overwrite, :append, or :ask. :Overwrite means the new
   database will clobber the old database. :Append means that the new database
   will be appended onto the previous one. :Ask means that the user will be
   prompted to choose between :overwrite and :append.

   INCLUDE-ALL-COMPONENTS may be one of T, nil or :ask.  If is nil then only
   that portion of the blackboard hierarchy that is explicitly mentioned in
   the DESCRIPTIONS is instantiated.  This is useful in conjunction with
   append mode.  If INCLUDE-ALL-COMPONENTS is T then all the components of a
   blackboard are instantiated.  :Ask means that the user will be prompted to
   choose between including all the components or not."

  (multiple-value-bind (descriptions mode include-all)
      (extract-bb-db-descriptions args)
    
    (flet ((filter-replication-description (d)
	     (fill-out-replication-description d include-all)))

      (setf descriptions (mapcar #'filter-replication-description descriptions))

      (cond ((null *blackboard-database*)
	     (overwrite-existing-blackboard descriptions))
	    ((eq mode :ask)
	     (case (choose-overwrite-or-append)
		(:overwrite (overwrite-existing-blackboard descriptions))
		(:append (append-to-existing-blackboard descriptions))))
	    ((eq mode :overwrite)
	     (overwrite-existing-blackboard descriptions))
	    ((eq mode :append)
	     (append-to-existing-blackboard descriptions))
	    (t (error ":MODE must be one of :ASK, :OVERWRITE, or :APPEND")))

      (dolist (fn *instantiate-bb-db-hook-functions*)
	(funcall fn)))))


(defun EXTRACT-BB-DB-DESCRIPTIONS (args)
  "Parse the arguments to instantiate-blackboard-description."
  ;; Use member instead of getf because `args' is not in plist form.
  (let* ((mode-present (member :mode args :test #'eq))
	 (include-present (member :include-all-components args :test #'eq))
	 description mode include-all)

    (setf description
	  (cond ((and mode-present include-present)
		 (if (or (and (= (length mode-present) 4)
			      (= (length include-present) 2))
			 (and (= (length include-present) 4)
			      (= (length mode-present) 2)))
		     (butlast args 4)
		     (error "Ill formed argument list: ~s" args)))
		((or mode-present include-present)
		 (if (or (= (length mode-present) 2)
			 (= (length include-present) 2))
		     (butlast args 2)
		     (error "Ill formed argument list: ~s" args)))
		(t args)))
    ;; Include-all-components defaults to t.
    (setf include-all
	  (cond ((not include-present) t)
		((eq (second include-present) :ask)
		 (choose-include-or-not))
		(t (second include-present))))
    ;; Mode defaults to :ask
    (setf mode
	  (cond ((not mode-present) :ask)
		(t (second mode-present))))

    (values description mode include-all)))


(defun OVERWRITE-EXISTING-BLACKBOARD (descriptions)
  "Sets up a new blackboard database and clobbers the previous one, if it exists."

  (clear-blackboard-database t)
  ;; This may not always be the right thing to do (but where
  ;; else would I do it).
  (clear-unit-hash-tables)
  (setf *blackboard-database*
        (mapcar #'instantiate-blackboard-database-1 descriptions))
  (build-bb-storage *blackboard-database*)
  (update-space-instance-variables *blackboard-database*))


(defun APPEND-TO-EXISTING-BLACKBOARD (descriptions)
  "Adds the new blackboard to the previously existing one."
  
  (let ((new-bb-nodes (mapcar #'instantiate-blackboard-database-1 descriptions)))
    (build-bb-storage new-bb-nodes)
    (update-space-instance-variables new-bb-nodes)
    (setf *blackboard-database* (nconc *blackboard-database* new-bb-nodes))))


(defun INSTANTIATE-BLACKBOARD-DATABASE-1 (description &optional parent)

  "INSTANTIATE-BLACKBOARD-DATABASE-1 description &optional parent

   Parses DESCRIPTION and returns a database-node.  PARENT is the
   parent node of the one to be built.  DESCRIPTION must be in
   canonical form."

  (let* ((name (first description))
	 (start (first (second description)))
	 (end (second (second description)))
	 (size (- end start))
	 (tail (cddr description))
	 type node vector)

    (setf type (cond ((blackboard-name-p name) (get-blackboard name))
		     ((space-name-p name) (get-space name))
		     ;; Error in fill-out-replication-description
		     (t (error "~s is not a blackboard or a space." name)))
	  vector (make-array size)
	  node (make-database-node :name name :parent parent :type type
				   :start start :end end :vector vector))
    (dotimes (instance size)
      (setf (svref vector instance)
	    (if (space-type-p type)
		(instantiate-space type node)
		(instantiate-blackboard-part tail node))))
    node))


(defun INSTANTIATE-BLACKBOARD-PART (component-descriptions parent)
  "Return a list of database-nodes -- one for each element of
   COMPONENT-DESCRIPTIONS.  COMPONENT-DESCRIPTIONS must be in
   canonical form."
  (mapcar #'(lambda (d)
	      (instantiate-blackboard-database-1 d parent))
	  component-descriptions))


(defun INSTANTIATE-SPACE (space parent)
  "INSTANTIATE-SPACE space
   Create a space instance."
  (setf space (get-space space))
  (make-space-instance :space space
                       :name (space.name space)
                       :parent parent))


(defun BUILD-BB-STORAGE (nodes)
  
  ;; A simple tree walk:
  (cond ((space-instance-p nodes)
         (build-space-storage nodes))
        ((database-node-p nodes)
         (map nil #'build-bb-storage (db-node.vector nodes)))
        (t (map nil #'build-bb-storage nodes))))


(defun BUILD-SPACE-STORAGE (instance)
  "Create the storage arrays, etc. for the space instance
   that has been previously created by instantiate space."

  (let* ((space (space-instance.space instance))
         (path (get-path-from-space-instance instance))
         (mappings (cdr (assoc path
                               *unit-mappings-for-space-paths*
                               :test #'equal)))
         units)
    (cond ((null mappings)
           ;; No mappings defined for this instance so use the
           ;; mappings for the space.
           (create-additional-unit-mappings space)
           (setf mappings (space.unit-mappings space)))
          (t
           ;; There are some mappings defined for this instance.
           ;; If any units that can be stored on this space are
           ;; not included, give them a empty mapping.
           (setf units (remove-if #'(lambda (u)
                                      (find u mappings :test #'member-eq
                                            :key #'unit-mapping.units))
                                  (expand-unit-type-list (space.units space))))
           (push (make-unstructured-unit-mapping units space)
                 mappings)))
    (setf (space-instance.data instance)
	  (mapcar #'build-space-instance-data-entry mappings))
    instance))


(defun BUILD-SPACE-INSTANCE-DATA-ENTRY (unit-mapping)
  "Build an alist entry with the local data for a space instance.
   This function returns an alist entry where the unit-mapping is the key."
  (setf (unit-mapping.in-use unit-mapping) t)
  (cond ((unstructured-unit-mapping-p unit-mapping)
	 (cons unit-mapping nil))
	(t
         (cons unit-mapping (build-space-data-array unit-mapping)))))

(defun BUILD-SPACE-DATA-ARRAY (unit-mapping)
  "Build an `index array table' (also called a `storage element').
   This table is an alist of indexes and corresponding arrays.
   For example:
     (((x y)  . <array for x and y>)
      ((time) . <array for time>))."
  (let* ((umi-list (unit-mapping.umi-list unit-mapping))
	 (index-array-table nil)
	 array dims umi)
    ;; Loop through the indexes and build the arrays
    (dolist (indexes (unit-mapping.indexes unit-mapping))
      (setf dims nil)
      (do ((p indexes (rest p))
	   (i 0 (1+ i)))
	  ((endp p)
	   (setf dims (nreverse dims)))
	(setf umi (find (car p) umi-list :key #'umi.name))
	(push (umi.array-size umi) dims))
      (setf array (make-array dims :initial-element nil))
      (push (cons indexes array) index-array-table))
    ;; Sort the elements so that the `coarsest' array is first.
    (sort index-array-table
          #'<
          :key #'(lambda (element)
                   (array-total-size (storage-element.array element))))
    ))


(defun CREATE-ADDITIONAL-UNIT-MAPPINGS (space)

  "Creates unit-mappings for units that can be stored on SPACE
   but which haven't had a unit-mapping explicitly defined."

  (flet ((add-unit-mapping (new-unit-mapping units space)
	   (setf (unit-mapping.units new-unit-mapping) units)
	   (push new-unit-mapping (space.unit-mappings space))))

    (let* ((space-units (space.units space))
	   (units-with-mappings
	     (mapcan #'(lambda (mapping)
			 (copy-list (unit-mapping.units mapping)))
		     (space.unit-mappings space)))
	   current-units ancestor descendants)
      
      (dolist (mapping (space.unit-mappings space))
	(setf descendants nil)
	;; The units that participate in this unit mapping:
	(setf current-units (copy-list (unit-mapping.units mapping)))
	(dolist (unit current-units)
          (setf ancestor (find-ancestor unit space-units))
          (when (unit-subtypes-included-p ancestor)
            (npush-list (descendants-without-mappings
                          unit
                          units-with-mappings)
                        descendants)))
        (setf descendants (delete-duplicates descendants))
	(when descendants
	  (add-unit-mapping (copy-unit-mapping mapping)
			    descendants
			    space)))
    
      (dolist (unit space-units)
	(unless (find (x-unit-type-of unit) (space.unit-mappings space)
		      :key #'unit-mapping.units :test #'member-eq)
	  (add-unit-mapping
	    (make-unstructured-unit-mapping nil space "Default unit-mapping.")
	    (descendants-without-mappings unit units-with-mappings)
	    space))))))


(defun DESCENDANTS-WITHOUT-MAPPINGS (ancestor units-with-mappings)

  "Returns a list of the subtypes of ANCESTOR which don't have a
   unit-mapping yet.  (The result will contain ANCESTOR unless it's
   in UNITS-WITH-MAPPINGS.) Whether a unit has a unit-mapping is
   determined by examining the list UNITS-WITH-MAPPINGS."

  (let ((result nil))
    (multiple-value-bind (ancestor-type include-subtypes?)
        (x-unit-type-of ancestor)
      (when include-subtypes?
        (let* ((description (get-unit-description ancestor-type))
               (subtypes (unit.subtypes description)))
          (dolist (unit subtypes)
            (unless (find-if #'(lambda (element)
                                 (and (not (unit-subtypep ancestor element))
                                      (unit-subtypep unit element)))
                             units-with-mappings)
              (push unit result)))))
      (unless (member ancestor-type units-with-mappings :test #'eq)
        (push ancestor-type result)))
    result))


(defun EMPTY-SPACE-P (path-structures)

  "EMPTY-SPACE-P path-structures

   This function returns T if there are no objects on the space (or
   spaces) represented by PATH-STRUCTURES.  It returns NIL if the
   space is occupied."

  (catch 'exit
    (map-space #'(lambda (object)
                   (declare (ignore object))
                   #-(and DEC VAX) (return-from empty-space-p nil)
                   #+(and DEC VAX) (throw 'exit nil))
               t
               path-structures)
    ;; If we get here then the space is empty.
    t))

(defun SPACE-OCCUPIED-P (path-structures)

  "SPACE-OCCUPIED-P path-structures

   Returns T if there are objects on the space(s), NIL otherwise."

  (not (empty-space-p path-structures)))


(defun clear-blackboard-database (&optional delete-p)

  "CLEAR-BLACKBOARD-DATABASE &optional delete-p

   Removes all units from the blackboard database.  If the optional
   argument, DELETE-P, is nil then the database itself remains intact.
   If DELETE-P is true then the blackboard database is destroyed."

  (setf *find-units-count* nil)
  (cond (delete-p
	 (dolist (si *space-instance-list*)
	   (dolist (fn *clear-space-hook-functions*)
	     (funcall fn si)))
	 (setq *space-instance-list* nil)
	 (setq *space-instance-table* nil)
	 (setq *blackboard-database* nil))
	(t
	 (mapc #'clear-space-instance *space-instance-list*)))
  (clear-unit-hash-tables)
  nil)


(defun clear-space (path-structure &optional delete-p)

  "CLEAR-SPACE path-structure

   Removes all units from the space(s) indicated by PATH-STRUCTURE.
   Note that the units will still exist and will remain on any other
   spaces that they are stored on."

  (error-when delete-p
    "You can't delete individual spaces yet.")
  (dolist-or-atom (path path-structure)
    (let ((si (get-space-instance-from-path-structure path)))
      (clear-space-instance si t))))


(defun clear-space-instance (space-instance &optional carefully)

  "CLEAR-SPACE-INSTANCE space-instance &optional carefully

   Removes all units from the space-instance.  If the optional argument,
   CAREFULLY, is true then all the units that were stored on the space
   instance will have their %%SPACE-INSTANCES%% slot updated to reflect
   the fact that they are no longer on this space instance.  If
   CAREFULLY is nil then the %%SPACE-INSTANCES%% slot is not updated.
   This is much faster, but should be used with caution.  It is intended
   to be used when those units will not be used again."

  (flet ((remove-space-instance-from-unit (unit)
	   (setf (basic-unit.%%space-instances%% unit)
		 (remove space-instance (basic-unit.%%space-instances%% unit)
			 :test #'eq))))

    (dolist (fn *clear-space-hook-functions*)
      (funcall fn space-instance))

    (let* ((space-instance-data (space-instance.data space-instance))
	   unit-mapping data)

      ;; Loop though the data alist
      (dolist (storage-entry space-instance-data)
	(setf unit-mapping (storage-entry.unit-mapping storage-entry)
	      data (storage-entry.data storage-entry))

	(cond 
	  ((unstructured-unit-mapping-p unit-mapping)
	   ;; There should be no duplicates in this list so there is
	   ;; no need to mark the units.
	   (when carefully
	     (mapc #'remove-space-instance-from-unit data))
	   (setf (storage-entry.data storage-entry) nil))

	  (t
	   (when carefully
	     (let* ((array (smallest-space-data-array data))
		    (region (entire-array-region array)))
	       ;; Apply the function to each unit.
	       (do-array-region (subscripts region)
		 (mapc #'remove-space-instance-from-unit
		       (apply #'aref array subscripts)))))
	   (dolist (storage-element data)
	     (let* ((array (storage-element.array storage-element))
		    (region (entire-array-region array)))
	       (do-array-region (subscripts region)
		 (setf (apply #'aref array subscripts) nil))))))))))


(defun delete-space-instance (path-structure &optional carefully)
  
  (dolist-or-atom (path path-structure)
    (let ((si (get-space-instance-from-path-structure path)))
      (delete-space-instance-1 si))))

(defun delete-space-instance-1 (si)

  (setf *space-instance-list* (delete si *space-instance-list* :test #'eq))
  (remhash si *space-instance-table*)
  (clear-space-instance si t)
  (delete-db-node si (space-instance.parent si)))

(defun delete-db-node (node &optional (parent (db-node.parent node)))

  (let* ((end (db-node.end parent))
         (start (db-node.start parent))
         (length (- end start))
         (vector (db-node.vector parent))
         )
    ;; There are several cases:
    ;;  - Only one instance.
    ;;      (e.g. (end - start) == 1 and 
    ;;      only one element in the vector element (vector [0]).
    ;;  - Delete instance from the end
    ;;  - Delete instance from the middle
    (cond ((and (= length 1)
                (or (atom (svref vector 0)) ;; space-instance
                    (= (length (svref vector 0)) 1)))
           (delete-db-node parent))
          ;; ...
          )
    ))



(defun FILL-OUT-REPLICATION-DESCRIPTION (description
					 &optional (include-all-components t))

  "FILL-OUT-REPLICATION-DESCRIPTION description
				    &optional (include-all-components t)

   This function looks at `Description' and returns an expanded description.
   If include-all-components is true then add any components that are declared
   by define-blackboards but not explicitly mentioned in `Description'.  If
   include-all-components is nil then this function ensures that the leaves
   of `Description' are space names.  In either case fill in any missing 
   replication counts."

  (let (start end type tail name)
    (cond ((symbolp description)
	   (setf start 0 end 1
		 type description
		 tail nil))
	  ((numberp (second description))
	   (setf start 0 end (second description)
		 type (first description)
		 tail (cddr description)))
	  ((rangep (second description))
	   (setf start (first (second description))
		 end (second (second description))
		 type (first description)
		 tail (cddr description)))
	  (t (setf start 0 end 1
		   type (car description)
		   tail (cdr description))))

    (cond ((blackboard-name-p type)
	   (setf type (get-blackboard type)
		 name (bb.name type)))
	  ((space-name-p type)
	   (setf type (get-space type)
		 name (space.name type)))
	  (t (error "~s does not name a space or a blackboard." type)))

    (cond ((and (not include-all-components) (null tail))
	   (unless (space-type-p type)
	     (error "A leaf in the replication description ... ~s is not a space name."
		    description))
	   `(,name (,start ,end)))
	  ((space-type-p type)
	   (unless (null tail)
	     (error "Extraneous information in the replication description: ~s."
		    tail))
	   `(,name (,start ,end)))
	  (t `(,name (,start ,end)
	       ,.(fill-out-replication-description-1 type tail include-all-components))))))


(defun FILL-OUT-REPLICATION-DESCRIPTION-1 (blackboard
					   component-descriptions
					   include-all-components)
  "Helper function for Fill-Out-Replication-Description."
  (mapcar #'(lambda (d)
	      (fill-out-replication-description d include-all-components))
	  (if include-all-components
	      (merge-blackboard-components blackboard component-descriptions)
	      component-descriptions)))

(defun MERGE-BLACKBOARD-COMPONENTS (blackboard component-descriptions)

  "MERGE-BLACKBOARD-COMPONENTS blackboard component-descriptions

   Add all the components of `Blackboard' (both blackboards and spaces)
   as defined by Define-Blackboards that aren't explicitly mentioned in
   `Component-Descriptions'.  Return the new component-description."

  (let ((bb-spaces (bb.spaces blackboard))
	(bb-blackboards (bb.blackboards blackboard))
	name)
    ;; Check that all the components in the description are children
    ;; of this blackboard.
    (dolist (description component-descriptions)
      (setf name (if (listp description) (car description) description))
      (cond ((not (or (space-name-p name) (blackboard-name-p name)))
	     (error "~s is not the name of a blackboard or a space (in ~s)."
		    name description))
	    ((not (or (find name bb-spaces :test #'eq :key #'space.name)
		      (find name bb-blackboards :test #'eq :key #'bb.name)))
	     (error "~s is not a component of the blackboard ~s."
		    name (bb.name blackboard)))
	    (t nil)))
    (nconc
      ;; Copy the components that are explicitly supplied.
      (copy-list component-descriptions)
      ;; Add the component spaces that aren't mentioned.
      (mapc-condcons #'(lambda (s)
			 (unless (find (space.name s) component-descriptions
				       :key #'first-if-list)
			   (space.name s)))
		     bb-spaces)
      ;; Add the component blackboards that aren't mentioned.
      (mapc-condcons #'(lambda (b)
			 (unless (find (bb.name b) component-descriptions
				       :key #'first-if-list)
			   (bb.name b)))
		     bb-blackboards))))



(defun CHOOSE-OVERWRITE-OR-APPEND ()
  "Returns :OVERWRITE  or :APPEND, depending upon user response.  Used
   by Instantiate-Blackboard-Database."
  #+LISPM
  (#+SYMBOLICS scl::fquery
   #-SYMBOLICS ticl::fquery
          '(:fresh-line t :beep t
	    :choices (((:overwrite "Overwrite.") #\O #\o)
		      ((:append    "Append.")    #\A #\a)))
	  "A blackboard database already exists.~@
           Do you want to overwrite the existing blackboard~@
           or append the new one onto it? ")
  #-LISPM
  (do ((answer nil))
      (answer answer)
    (format *query-io* "~&A blackboard database already exists.~@
                        You can overwrite the existing blackboard ~
                        or append the new one onto it.")
    (setf answer
	  (cond ((y-or-n-p "~&Do you want to overwrite the existing blackboard? ")
		 :overwrite)
		((y-or-n-p "~&Do you want to append the new one onto it? ")
		 :append)
		(t (format *query-io*
			   "~2&Please choose one of overwrite or append.~2&")))))

  )


(defun CHOOSE-INCLUDE-OR-NOT ()
  "Asks the user whether or not to include all the blackboard
   components.  Returns T or NIL depending on the user's response."
  (y-or-n-p "Do you want to include all the blackboard components? "))


;;;; --------------------------------------------------------------------------
;;;;   Building the Space Instance Hash Table
;;;; --------------------------------------------------------------------------


(defun UPDATE-SPACE-INSTANCE-VARIABLES (nodes)

  "UPDATE-SPACE-INSTANCE-VARIABLES nodes

   Initializes or updates *space-instance-table* and *space-instance-list*."

  (let ((space-instances (mapcan #'space-instance-list nodes)))
    (setf *space-instance-table* (build-space-instance-table space-instances))
    (setf *space-instance-list* (nconc space-instances *space-instance-list*))))


(defun SPACE-INSTANCE-LIST (node &optional (space-instances nil))

  "SPACE-INSTANCE-LIST node &optional (space-instances nil)

   Returns a list of space instances below NODE.  this is a freshly
   consed list.  The optional argument SPACE-INSTANCES is for internal
   use only."

  ;; The result is accumulated in the argument SPACE-INSTANCES.
  
  (cond ((typep node 'space-instance)  ;;bottom of tree
         (push node space-instances))
	
        ((typep node 'database-node)   ;;intermediate node
	 (dotimes (svindex (- (db-node.end node) (db-node.start node)))
	   (setf space-instances (space-instance-list
				   (svref (db-node.vector node) svindex)
				   space-instances))))
        
        ((listp node) ;; replicated descriptions 
	 (dolist (element node)
	   (setf space-instances (space-instance-list element space-instances))))
	
	;; If the blackboard database is screwed up.
        (t (error "Unplanned case in space-instance-list (see philip).")))
  
  ;; Return the path list for this node.
  space-instances)


(defconstant *BB-PATHNAME-ERROR-MSG*
	"Ambiguous specification in INSTANTIATE-BLACKBOARD-DATABASE.~@
         Some bb/space paths can select more than one space instance.~@
         For example, ~s does not uniquely specify a space.~@
         Check your blackboard and space definitions as well as your~@
         call to INSTANTIATE-BLACKBOARD-DATABASE.~%")

(defun BUILD-SPACE-INSTANCE-TABLE (space-instance-list)

  "BUILD-SPACE-INSTANCE-TABLE space-instance-list

   Returns a new space instance table built from *space-instance-table*,
   with keys equal to fully replicated path lists and values equal to
   the corresponding space instance.  We make a new copy of
   *space-instance-table* so that if a new path conflicts with an old
   one, we can bomb out without having corrupted the current
   *space-instance-table* (we hope)."

  (let ((new-space-table (if *space-instance-table*
			     (copy-equal-hash-table *space-instance-table*)
			     (make-hash-table :test 'equal))))
    (mapc #'(lambda (path space-instance)
	      (if (gethash path new-space-table)
		  (error *bb-pathname-error-msg* path)
		  (setf (gethash path new-space-table) space-instance)))
	  (mapcar #'get-path-from-space-instance space-instance-list)
	  space-instance-list)
  new-space-table))

(defun COPY-EQUAL-HASH-TABLE (old-table)
  (let ((new-table (make-hash-table :test 'equal)))
    (maphash #'(lambda (key value)
		 (setf (gethash key new-table) value))
	     old-table)
    new-table))


;;;; --------------------------------------------------------------------------
;;;;   Conversion between instances and paths.
;;;; --------------------------------------------------------------------------


(defun path-instantiated-p (paths)

  "PATH-INSTANTIATED-P paths

   This predicate returns true if each BB/space path in PATHS is a path
   to an instantiated space.  PATHS is a list of bb/space paths or a
   single bb/space path.  Note that these are bb/space paths not path
   structures!"

  (dolist-or-atom (path paths (listp (first paths)))
    (unless (get-space-instance-from-path path nil)
      ;; An uninstantiated path...  Return nil.
      (return-from path-instantiated-p nil)))

  ;; Return true if all the paths are instantiated.
  t)

(defun GET-SPACE-INSTANCE-FROM-PATH (path &optional (signal-error t))
  
  "GET-SPACE-INSTANCE-FROM-PATH path
   
   Given a path, returns the space instance, or signals an error
   if the space instance can't be found."
  
  (let* ((full-path (add-replications path nil signal-error))
	 (space-instance (gethash full-path *space-instance-table*)))
    (cond (space-instance
           (when (null (space-instance.path space-instance))
             (setf (space-instance.path space-instance) full-path))
	   space-instance)
	  ((null signal-error) nil)
	  (t 
	   (error "Illegal path: ~s.~@
                   Cannot find the corresponding space instance.~@
                   Use PP-BLACKBOARD-DATABASE to see the legal paths."
		  path)))))


(defun GET-PATH-FROM-SPACE-INSTANCE (space-instance)

  "GET-PATH-FROM-SPACE-INSTANCE space-instance

   Return a fully specified blackboard/space path that
   specifies SPACE-INSTANCE."

  (when (space-instance.path space-instance)
    (return-from get-path-from-space-instance
      (space-instance.path space-instance)))

  (let* ((node (space-instance.parent space-instance))
	 (path (list (space-instance.name space-instance)
                     (+ (db-node.start node)
                        (position space-instance (db-node.vector node) :test #'eq)))))
    (do ((parent (db-node.parent node) (db-node.parent node)))
	((null parent)
	 (setf (space-instance.path space-instance) path))
      (setf path
	    (list* (db-node.name parent)
		   (+ (db-node.start parent)
		      (position node (db-node.vector parent) :test #'member-eq))
		   path))
      (setf node parent))))


(defun GET-SPACE-NAME-FROM-PATH (path)

  "GET-SPACE-NAME-FROM-PATH path

   Return a the space name (a symbol) for this path."

  (let* ((l (length path))
         (tail (nthcdr (- l 2) path))
         (space (cond ((symbolp (second tail))
                       (second tail))
                      ((and (numberp (second tail))
                            (symbolp (first tail)))
                       (first tail))
                      (t nil))))
    (error-unless (and space (space-name-p space))
       "Ill-formed path: ~s~@
        Use PP-BLACKBOARD-DATABASE to see the legal paths."
       path)
    space))


(defun ADD-REPLICATIONS (path &optional (except-last-element nil)
                                        (signal-error t))
  
  "ADD-REPLICATIONS path &optional (except-last-element nil)

   Returns a path specification built from path, where all implicit
   replication counts have been explicitly included.  For example:

     (add-replications '(bb1 bb2 1 space3)) => '(bb1 0 bb2 1 space3 0)

   If except-last-element is true, then don't add a default replication
   count to the last path element (this is useful for saving blackboards).
   If signal-error is true then an error will be signaled if any of the
   blackboards or spaces are not defined."

  (labels
    ((number (tail)
       (cond ((null tail)
              (if except-last-element '() '(0)))
             ((numberp (first tail))
              (cons (first tail) (non-number (rest tail))))
             (t (cons 0 (non-number tail)))))
     (non-number (tail)
       (if (null tail)
           '()
           (let ((name (first tail)))
             (cond ((or (blackboard-name-p name)
                        (space-name-p name))
                    (cons name (number (rest tail))))
                   (signal-error
                    (error
                      "~s is not a blackboard or space name~%in the path ~s."
                      name path))
                   (t #-(and DEC VAX) (return-from add-replications nil)
                      #+(and DEC VAX) (throw 'exit nil)))))))

    #-(and DEC VAX)  (non-number path)
    #+(and DEC VAX)  (catch 'exit (non-number path))
    ))


;;; --------------------------------------------------------------------------
;;;   Path structure creation and manipulation code
;;; --------------------------------------------------------------------------

(defun GET-SPACE-NAME-FROM-PATH-STRUCTURE (path-structure)
  (space-instance.name
    (get-space-instance-from-path-structure path-structure)))

(defun GET-SPACE-INSTANCE-FROM-PATH-STRUCTURE (path-structure)

  ;; accept lists of length 1, otherwise signal an error.
  (when (listp path-structure)
    (if (> (length path-structure) 1)
	(error "Can't manipulate more than one path structure in this context.")
	(setf path-structure (first path-structure))))
  
  (or (path-structure.space-instance path-structure)
      (setf (path-structure.space-instance path-structure)
	    (get-space-instance-from-path
	      (path-structure.path-value path-structure)))))

(defun GET-SPACE-INSTANCES-FROM-PATH-STRUCTURES (path-structs)

  "GET-SPACE-INSTANCES-FROM-PATH-STRUCTURES path-structs
   Returns a list of space instances corresponding to the list of path structs."

  (mapcar #'get-space-instance-from-path-structure path-structs))

(defun GET-PATH-FROM-PATH-STRUCTURE (path-structure)

  ;; accept lists of length 1, otherwise signal an error.
  (when (listp path-structure)
    (if (> (length path-structure) 1)
	(error "Can't manipulate more than one path structure in this context.")
	(setf path-structure (first path-structure))))

  (or (path-structure.path-value path-structure)
      (setf (path-structure.path-value path-structure)
	    (get-path-from-space-instance
	      (path-structure.space-instance path-structure)))))

(defun GET-PATH-STRUCTURE-FROM-SPACE-INSTANCE (space-instance)
  (or (space-instance.path-structure space-instance)
      (setf (space-instance.path-structure space-instance)
	    (make-path-structure :space-instance space-instance))))

(defun GET-PATH-STRUCTURE-FROM-PATH (path)
  (get-path-structure-from-space-instance
    (get-space-instance-from-path path)))

(defun EQUAL-PATHS-P (path1 path2)
  "EQUAL-PATHS-P path1 path2
   Returns T if PATH1 and PATH2 are both path structures
   indicating the same space instance, NIL otherwise."

  (and (path-structure-p path1)
       (path-structure-p path2)
       (eq (get-space-instance-from-path-structure path1)
	   (get-space-instance-from-path-structure path2))))


(defun MAKE-PATHS (&key paths unit-instances)

  "MAKE-PATHS &key paths unit-instances

   This function returns a list of path structures.  A path structure can
   be derived from a blackboard/space path or a unit-instance.  The the
   PATHS argument may be a single blackboard/space path or a list of
   blackboard/space paths.  The blackboard/space paths can specify an
   entire path or a partial path.  The argument to UNIT-INSTANCES is a
   single unit instance or list of unit instances.

   Note that though the same path may be specified multiple times in the
   argument list, only one corresponding path structure will be returned."

  ;; path structures have slots for both the path list and the corresponding space
  ;; instance pointed to.  MAKE-PATHS does not necessarily provide values for both
  ;; of the slots, which may result in some extra computation in CHANGE-PATHS.

  (let ((path-structs nil))

    (when paths
      (dolist-or-atom (path paths (listp (first paths)))
	(dolist (space-instance (get-space-instance-list-from-path path))
	  (pushnew (get-path-structure-from-space-instance space-instance)
		   path-structs
		   :test #'equal-paths-p))))

    (dolist-or-atom (unit-instance unit-instances)
      (dolist (space-inst (basic-unit.%%space-instances%% unit-instance))
        (pushnew (get-path-structure-from-space-instance space-inst)
		 path-structs
		 :test #'equal-paths-p)))

    path-structs))


(defun ALL-PATHS ()

  "ALL-PATHS

   Returns the list of path structures for all instantiated spaces."

  (mapcar #'get-path-structure-from-space-instance *space-instance-list*))


;;; Warning:  DEFINE-UNIT depends on CHANGE-PATHS returning a
;;;           freshly consed list.  If CHANGE-PATHS is modified so
;;;           that it no longer conses a new list then the function
;;;           BUILD-PATH-GENERATION-FORMS must be changed as well.

(defun CHANGE-PATHS (path-structures &rest change-specs)

  "CHANGE-PATHS path-structures &rest change-specs

   PATH-STRUCTURES is a list of path structures, and each change-spec should
   evaluate to one of the following:

     {({:CHANGE-SUBPATH | :CHANGE-INDEX} old-value new-value) |
      (:CHANGE-RELATIVE ({:UP}* {path-element}*))}   

   In general, a list of new path structures is returned, where each path
   structure has been built from one of the passed path structures, modified
   by the application in order of each of the passed CHANGE-SPECS.

   When a change-spec specifies :CHANGE-SUBPATH, the OLD-VALUE within the
   path-structure will be replaced by NEW-VALUE.

   When a change-spec specifies :CHANGE-INDEX, the element following
   OLD-VALUE (presumably an index) is replaced by NEW-VALUE (also presumably
   an index.)
 
   When a change-spec specifies :CHANGE-RELATIVE, the path is traced up the
   tree from the space instance as many times as :UP is specified, then the
   specified PATH-ELEMENTS are appended to form the new path.

   A CHANGE-SPEC may specify NIL for NEW-VALUE, in which case the OLD-VALUE
   is simply deleted from the path.  It is an error for NEW-VALUE to be NIL
   or to not be present within each of the passed PATH-STRUCTURES."
  
  (let ((new-path-structs nil))
    (dolist-or-atom (path-struct path-structures)
      (let ((new-path-value (copy-list
                              (or (path-structure.path-value path-struct)
                                  (get-path-from-space-instance
                                    (path-structure.space-instance path-struct))))))
        (dolist (change-spec change-specs)
	  (let ((change-type (first change-spec)))
	    
	    (cond ((member change-type '(:CHANGE-SUBPATH :CHANGE-INDEX))
		   (let* ((index-p (eq change-type :CHANGE-INDEX))
			  (old-path-sublist (second change-spec))
			  (new-path-sublist (third change-spec)))
		     (setf new-path-value
			   (splice-path-list new-path-value
					     old-path-sublist
					     new-path-sublist
					     index-p))))
		  ((eq change-type :CHANGE-RELATIVE)
		   (let* ((upward-amount (count :UP (second change-spec)))
			  (path-tail (member-if-not #'(lambda (element)
							(eq element :UP))
						    (second change-spec)))
			  (path-head (subseq new-path-value
					     0
					     (- (length new-path-value)
						(* 2 upward-amount)))))
		     (setf new-path-value (nconc path-head path-tail))))
		  (t (error "Illegal change type ~s." change-type)))))
        (push (get-path-structure-from-path new-path-value) new-path-structs)))
    (nreverse new-path-structs)))


(defun SPLICE-PATH-LIST (old-list old-sublist new-sublist index-p)

  "SPLICE-PATH-LIST old-list old-sublist new-sublist index-p

   Returns a list with the section of OLD-LIST equal to OLD-SUBLIST replaced
   by NEW-SUBLIST.  If NEW-SUBLIST is nil, then the effect is to delete
   OLD-SUBLIST from OLD-LIST.  If INDEX-P is non-nil, then the element
   following OLD-SUBLIST is replaced with NEW-SUBLIST.  (In this case
   NEW-SUBLIST should be a replacement index value.) The returned list will
   share as much structure as possible with OLD-LIST, so watch out.  An
   error is signalled if OLD-SUBLIST is not a subsequence of OLD-LIST or is
   nil."
  
  (setf old-sublist (assure-list old-sublist)
        new-sublist (assure-list new-sublist)
        old-list (assure-list old-list))

  (when (null old-sublist)
    (error "SPLICE-LIST does not allow null old-sublists.~@
            You probably need APPEND instead."))
  
  (let* ((front-ptr old-list)
        (back-ptr old-list)
        (old-frag-ptr old-sublist))
    ;; find the old-sublist within the copy of old-list
    ;; move front and back pointer along the old list until an element
    ;; matching the car of old-sublist is found, then extend back-ptr
    (while (and old-frag-ptr front-ptr)
      (cond ((equal (first old-frag-ptr) (first back-ptr))
             (setf old-frag-ptr (cdr old-frag-ptr)
                   back-ptr    (cdr back-ptr)))
            (t
             (setf old-frag-ptr old-sublist)
             (if (eq front-ptr back-ptr)
                 (setf back-ptr (cdr front-ptr))
                 (setf front-ptr (cdr front-ptr)
                       back-ptr (cdr front-ptr))))))
    
    (cond ((and (null old-frag-ptr) index-p)
	   ;; we've found the fragment and need to replace the index
           (replace old-list new-sublist
		    :start1 (- (length old-list) (length back-ptr))))
          ((null old-frag-ptr)
           (if (and (eq front-ptr old-list)
                    (eq (first front-ptr) (first old-sublist)))
               (if (null new-sublist)
                   (setf old-list back-ptr)
                   (setf old-list new-sublist
                         (cdr (last new-sublist)) back-ptr))
               (if (null new-sublist)
                   (setf (cdr front-ptr) back-ptr)
                   (setf (cdr front-ptr) new-sublist
                         (cdr (last new-sublist)) back-ptr))))
          (t (error "The sublist ~s was not found in the list ~s."
		    old-sublist old-list)))
    old-list))


(defun GET-SPACE-INSTANCE-LIST-FROM-PATH (path)

  "GET-SPACE-INSTANCE-LIST-FROM-PATH path

   Returns a list of space instances below the node specified by
   PATH, which is a bb/space path."
  
  (let ((result nil))
    (dolist (root-node *blackboard-database*)
      (multiple-value-bind (node index)
          (get-dbnode-from-path path root-node)
        (when node
          ;; Node will be the dbnode that is the nearest parent to the path.
          ;; Index will be either nil or an index into the node's vector of
          ;; children.  
          (setf result
                (nconc result
                       (space-instance-list
                         (if index
                             (svref (db-node.vector node) (- index (db-node.start node)))
                             node)))))))
    (if (null result)
        (error "Bad blackboard/space path: ~s.~@
                This path does not name an instantiated space.~@
                Use the PP-BLACKBOARD DATABASE to see the legal paths."
               path)
        result)))


(defun GET-DBNODE-FROM-PATH (path root-node)
  
  "GET-DBNODE-FROM-PATH path root-node

   Find the nearest parent node to PATH in the tree rooted at ROOT-NODE.
   Returns 2 values, the database node, and the replication number for
   it, if one is supplied.  If there is no replication index, then the
   value for the index is nil.  If PATH can't be found in this tree then
   just return nil -- no error is signaled."

  ;; A little hack here to save a cons.  NODE-LIST will be a list of DB
  ;; nodes every iteration through the loop except for the first time
  ;; when it is a single DB node (namely the root node).
  (let* ((p path)
         (node-list root-node)
         name node index)
    (loop
      (setf name (pop p))
      (setf index (and (numberp (car p)) (pop p)))
      ;; Special case first iteration here.
      (unless (setf node (if (listp node-list)
                             (find name node-list :key #'db-node.name)
                             (and (eq name (db-node.name node-list)) node-list)))
        (return nil))
      (cond ((null p)
             ;; Final element in the path.  Return the node and index.
             (when (and index
                        (not (and (<= (db-node.start node) index)
                                  (< index (db-node.end node)))))
               (return nil))
             (return (values node index)))
            (t
             ;; More path to match.
             ;; If index is omitted use zero.
             (unless index (setf index 0))
             (unless (and (<= (db-node.start node) index)
                          (< index (db-node.end node)))
               (return nil))
             (setf node-list (svref (db-node.vector node)
                                    (- index (db-node.start node)))))))))


;;; ---------------------------------------------------------------------------
;;;                                 End of File
;;; ---------------------------------------------------------------------------
