;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:SIMPLE-SHELL; Base:10 -*-
;;;; *-* File: VAX11:DIS$DISK:[GBB.V-120.DISTRIBUTION.SIMPLE-SHELL]SIMPLE-SHELL.LISP *-*
;;;; *-* Last-Edit: Monday, January 29, 1990  13:26:12; Edited-By: Cork *-*
;;;; *-* Machine: Caliban (Explorer II, Microcode 429) *-*
;;;; *-* Software: TI Common Lisp System 6.9 *-*
;;;; *-* Lisp: TI Common Lisp System 6.9 *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                   SIMPLE AGENDA-BASED CONTROL SHELL
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Daniel Corkill
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988, 1989, 1990 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  08-25-86 File Created.  (Cork)
;;;  11-18-86 Testing Completed.  (Cork)
;;;  02-17-87 Converted to GBB V1.00.  (Cork)
;;;  11-04-87 Converted to GBB V1.10.  (Cork)
;;;  04-22-88 Repaired DEFINE-LEVELS to correctly handle multiple definitions.
;;;           Added missing DEFINE-LEVEL macro.  Added check for precondition
;;;           function if input-levels are specified in DEFINE-KS.
;;;           Added RESET-SIMPLE-SHELL.  (Cork)
;;;  06-15-88 Converted to single file.  Changed DEFINE-KS to use keyword rather
;;;           than positional arguments.  (Cork)
;;;  07-13-88 Modified the unit creation event handler to allow units that
;;;           include basic-unit to be placed on generic spaces (rather than
;;;           simple-shell triggering levels.  (Cork)
;;;  07-20-88 Transferred the generic queue-unit functions to QUEUE.LISP.
;;;           (Cork)
;;;  08-02-88 Made major V1.2 changes including buffering events until end of
;;;           KSI and in KSI triggering conditions.  (Cork)
;;;  06-21-89 Added :FILTER-AFTER predicates to handle cross-product KS
;;;           trigger errors.  (Cork)
;;;  07-21-89 Added *SIMPLE-SHELL-POST-KSI-EXECUTION-HOOK*.  (Gallagher)
;;;  08-01-89 Added *SIMPLE-SHELL-EXIT-HOOK*.  (Cork)
;;;  01-05-90 Repaired errors in unit-creation and unit-deletion triggering
;;;           mechanisms.  (Cork)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(provide "GBB SIMPLE-SHELL")

(in-package "SIMPLE-SHELL")

(export '(*current-ksi*
          *event-print-types*
          *executed-ksis*
          *legal-event-print-types*
          *minimum-ks-execution-rating*
          *scheduling-queue*
          *simple-shell-cycle-hook*
          *simple-shell-exit-hook*
          *simple-shell-post-ksi-execution-hook*
          add-unit-to-space-event-handler
          basic-unit
          basic-ks
          basic-ksi
          build-basic-unit-queue
          defevent-printer
          define-ks
          initial-basic-unit-queue-unit
          insert-on-basic-unit-queue
          insert-on-basic-unit-queue-end
          instantiate-simple-control-shell
          ks
          ksi
          ksis
          kss
          link-access-event-handler
          link-update-event-handler
          make-ks
          make-ksi
          map-basic-unit-queue
          next-basic-queue-unit
          on-basic-unit-queue-p
          previous-basic-queue-unit
          remove-from-basic-unit-queue
          remove-unit-from-space-event-handler
          reset-simple-shell
          simple-control-shell
          slot-access-event-handler
          slot-update-event-handler
          unit-creation-event-handler
          unit-deletion-event-handler
          unlink-event-handler))

#-TI
(require "GBB QUEUE-UNIT")
#+TI
;; Improve TI's REQUIRE error message ::
(unless (member "GBB QUEUE-UNIT" user::*modules* :TEST 'string=)
  (error "Module GBB QUEUE-UNIT is not loaded."))

(use-package '("LISP" "UMASS-EXTENDED-LISP" "GBB"))

(proclaim '(optimize (speed 3) (safety 1)))

;;; ---------------------------------------------------------------------------
;;;
;;; Type Declarations :::
;;;
;;; ---------------------------------------------------------------------------

(deftype RATING ()

  "RATING

This data type specifies the range of KSI ratings."

  `(integer ,0 ,most-positive-fixnum))

;;; ---------------------------------------------------------------------------
;;;
;;; Global Variables :::
;;;
;;; ---------------------------------------------------------------------------

(eval-when (compile load eval)

(defconstant *SIMPLE-SHELL-EVENT-TYPES*
             '(nil       ; We include nil for KSs without relevant event types.
               :unit-creation
               :unit-deletion
               :unit-addition-to-space
               :unit-removal-from-space
               :slot-update
               :link-update
               :unlink
               :slot-access
               :link-access
               :queue-quiescence)

  "*SIMPLE-SHELL-EVENT-TYPES*

Contains the list of legal simple-shell event triggers.")

;;; --------------------------------------------------------------------------

(defvar *CURRENT-KSI* '()

  "*CURRENT-KSI*

This control shell variable contains the currently executing ksi or nil
if there is no ksi running at the moment.")

;;; --------------------------------------------------------------------------

(defvar *EVENT-PRINT-TYPES* '()

  "*EVENT-PRINT-TYPES*

This control shell variable controls which event printers are active.  By
default this value is a list containing keywords for all defined event print 
names.")

;;; --------------------------------------------------------------------------

(defvar *LEGAL-EVENT-PRINT-TYPES* '()

  "*LEGAL-EVENT-PRINT-TYPES*

This control shell variable contains a list of all defined event print 
names (keywords).")

) ; End of Eval-When

;;; --------------------------------------------------------------------------

(defvar *SIMPLE-SHELL-CYCLE-HOOK* nil

  "*SIMPLE-SHELL-CYCLE-HOOK*

This control shell variable allows a function (of no arguments) to be invoked
at the end of each scheduling cycle in the simple control shell.  This
function should not cause any blackboard events -- they will be ignored.  By
default this variable is nil.")

;;; --------------------------------------------------------------------------

(defvar *SIMPLE-SHELL-EXIT-HOOK* nil

  "*SIMPLE-SHELL-EXIT-HOOK*

This control shell variable allows a function (of no arguments) to be invoked
immediately prior to the simple shell exiting due to a lack of executable
KSIs.  This function can cause blackboard events.  Unless this function
returns :STOP, the simple shell will continue to cycle (and perhaps
repeatedly invoke the *SIMPLE-SHELL-EXIT-HOOK* function).  By default this
variable is nil.")

;;; --------------------------------------------------------------------------

(defvar *SIMPLE-SHELL-POST-KSI-EXECUTION-HOOK* nil

  "*SIMPLE-SHELL-POST-KSI-EXECUTION-HOOK*

This control shell variable allows a function (of no arguments) to be invoked
immediately after the execution of a KSI by the simple control shell.  This
function can cause blackboard events.  By default this variable is nil.")

;;; --------------------------------------------------------------------------

(defvar *KS-DEFINITION-FORMS* `()

  "*KS-DEFINITION-FORMS*

This control shell variable holds calls to MAKE-KS between definition (via
DEFINE-KS) and KS unit creation (performed during 
INSTANTIATE-BLACKBOARD-DATABASE).")

;;; --------------------------------------------------------------------------

(defvar *KSS-SPACE-PATH* :uninitialized

  "*KSS-SPACE-PATH*

This control shell variable holds the path to the KSS space (avoiding repeated
calls to MAKE-PATHS.")

;;; --------------------------------------------------------------------------

(defvar *STIMULUS-EVENT-BUFFER* :uninitialized

  "*STIMULUS-EVENT-BUFFER*

This control shell variable holds all stimulus events created during a KS
execution unit the end of the cycle, when they are processed by 
FIND-AND-INVOKE-PRECONDITION-FUNCTIONS.")

;;; --------------------------------------------------------------------------

(defvar *LAST-BASIC-KSI-NUMBER* :UNINITIALIZED

  "*LAST-BASIC-KSI-NUMBER*

This control shell variable contains the instance counter for the last
generated BASIC-KSI unit.")

;;; --------------------------------------------------------------------------

(defvar *SCHEDULING-QUEUE* :UNINITIALIZED

  "*SCHEDULING-QUEUE*

This global variable points to the header for the KSI scheduling queue.")

;;; --------------------------------------------------------------------------

(defvar *EXECUTED-KSIS* :UNINITIALIZED

  "*EXECUTED-KSIS*

This global variable points to the header for the executed KSI queue.")

;;; --------------------------------------------------------------------------

(defparameter *MINIMUM-KS-EXECUTION-RATING* 1

  "*MINIMUM-KS-EXECUTION-RATING*

This global parameter contains the minimum rating a ksi must have in order
to be executed.")

;;; ---------------------------------------------------------------------------

(define-index-structure EVENT-TYPES
  :COMPOSITE-TYPE list
  :COMPOSITE-INDEX :none
  :ELEMENT-TYPE symbol
  :INDEXES ((event-type :label)))

;;; ---------------------------------------------------------------------------

(define-index-structure UNIT-TYPES
  :COMPOSITE-TYPE list
  :COMPOSITE-INDEX :none
  :ELEMENT-TYPE t
  :INDEXES ((unit-type :label)))

;;; ---------------------------------------------------------------------------

(define-index-structure SLOT/LINK-NAMES
  :COMPOSITE-TYPE list
  :COMPOSITE-INDEX :none
  :ELEMENT-TYPE t
  :INDEXES ((slot/link-name :label)))

;;; ---------------------------------------------------------------------------

(define-index-structure PATHS
  :COMPOSITE-TYPE list
  :COMPOSITE-INDEX :none
  :ELEMENT-TYPE t
  :INDEXES ((path :label)))

;;; ---------------------------------------------------------------------------

(define-space KSS
                  
  "KSS

This space contains the defined knowledge sources for the system."

  :UNITS (basic-ks)
  :DIMENSIONS
  ((event-type :ENUMERATED #.*simple-shell-event-types*)

   ;; Nil is included in the following for use in KSs where the dimension is
   ;; not applicable.
   (unit-type :ENUMERATED (nil))
   (slot/link-name :ENUMERATED (nil))
   (path :ENUMERATED (nil))))

;;; ---------------------------------------------------------------------------

(define-space KSIS

  "KSIS

This space contains the KSI records that are pending execution (the scheduling
queue) and those that have executed (the executed KSI history list)."

  :UNITS (queue-unit)
  :DIMENSIONS nil)

;;; --------------------------------------------------------------------------
;;;
;;; Basic-unit Unit Definition ::
;;;
;;; --------------------------------------------------------------------------

(define-unit (BASIC-UNIT
               :EXPORT
               (:CONSTRUCTOR nil))    ; This structure exists only for
                                      ; inclusion in user UNIT structure.
  
  "BASIC-UNIT (Stimulus Unit)"

  :LINKS
  ((stimulated-ksis (basic-ksi stimulus-units))))

;;; --------------------------------------------------------------------------
;;;
;;; Basic KS Unit Definition ::
;;;
;;; --------------------------------------------------------------------------

(defun GENERATE-BASIC-KS-NAME (ks)
  
  "GENERATE-BASIC-KS-NAME ks

This function returns the name string for a KS unit."
  
  (basic-ks$name ks))

;;; ---------------------------------------------------------------------------

(proclaim '(function basic-ks-printer (t t t) t))

(define-unit (BASIC-KS
               :EXPORT
               (:CONSTRUCTOR nil)  ; This structure exists only for
                                   ; inclusion in user KS structure.
               (:NAME-FUNCTION generate-basic-ks-name)
               (:PRINT-FUNCTION basic-ks-printer))

  "BASIC KS (Knowledge Source)

This unit type contains knowledge source information."
  
  :SLOTS
  ((ksi-creation-function 'make-ksi :PRIVATE t)
   (ks-function)
   (precondition-function)
   (trigger-conditions '() :TYPE list)
   (event-types nil :TYPE event-types :PRIVATE t)
   (unit-types nil :TYPE unit-types :PRIVATE t)
   (slot/link-names nil :TYPE slot/link-names :PRIVATE t)
   (paths nil :TYPE paths :PRIVATE t))

  :LINKS
  ((ksis (basic-ksi ks :singular)))

  :DIMENSIONAL-INDEXES
  ((event-type event-types)
   (unit-type unit-types)
   (slot/link-name slot/link-names)
   (path paths))

  :PATHS
  ((:paths `(kss))))

;;; --------------------------------------------------------------------------

(defun BASIC-KS-PRINTER (ks stream depth)

  "BASIC-KS-PRINTER ks stream depth

Print a KS unit."
  
  (declare (ignore depth))
  (cond
    ;; Prin1 ::
    (*print-escape* (format stream "#<~A>" (basic-ks$name ks)))
    ;; Princ ::
    (t (format stream "~A" (basic-ks$name ks)))))

;;; --------------------------------------------------------------------------
;;;
;;; Basic-unit Queue Unit Unit Definition ::
;;;
;;; --------------------------------------------------------------------------

(define-unit (BASIC-UNIT-QUEUE-UNIT :EXPORT :UNNAMED
                                    (:INCLUDE basic-unit))
  
  "BASIC-UNIT-QUEUE-UNIT

This unit type provides a primitive queue unit for BASIC-UNIT types. 
It is used as a special ``header'' or ``sentinel'' node and as an included
unit in other queue units."
  
  :LINKS
  ((next-basic-queue-unit
     :SINGULAR
     (basic-unit-queue-unit previous-basic-queue-unit :SINGULAR))
   (previous-basic-queue-unit
     :SINGULAR
     (basic-unit-queue-unit next-basic-queue-unit :SINGULAR)))

  :PATHS
  ((:PATH)))

;;; --------------------------------------------------------------------------
;;;
;;; Basic KSI Unit Definition ::
;;;
;;; --------------------------------------------------------------------------

(defun GENERATE-BASIC-KSI-NAME (ksi)
  
  "GENERATE-BASIC-KSI-NAME ksi

This function returns the name string for a basic KSI unit."
  
  (declare (ignore ksi))
  (format nil "ksi.~5,'0D" (incf *last-basic-ksi-number*)))

;;; ---------------------------------------------------------------------------

(proclaim '(function basic-ksi-printer (t t t) t))

(define-unit (BASIC-KSI
               :EXPORT
               (:INCLUDE queue-unit)
               (:CONSTRUCTOR nil)          ; This structure exists only for
                                           ; inclusion in user KSI structure.
               (:NAME-FUNCTION generate-basic-ksi-name)
               (:PRINT-FUNCTION basic-ksi-printer)
               (:CREATION-EVENTS ks-instantiation-event-printer)
               (:DELETION-EVENTS basic-ksi-deletion-event-handler))
  
  "BASIC-KSI (Knowledge Source Instance)

This unit type represents a knowledge source instantiation (or activation)
record on the scheduling queue."
  
  :SLOTS
  ((rating most-negative-fixnum :TYPE rating)
   (response-frame))

  :LINKS
  ((ks :SINGULAR (basic-ks ksis))
   (stimulus-units (basic-unit stimulated-ksis)))

  :PATHS
  ((:PATH '(ksis))))

;;; --------------------------------------------------------------------------

(defun BASIC-KSI-PRINTER (ksi stream depth)
  
  "BASIC-KS-PRINTER ksi stream depth

Print a KSI unit."
  
  (declare (ignore depth))
  (cond
    ;; Prin1 ::
    (*print-escape* (format stream "#<~A>" (basic-ksi$name ksi)))
    ;; Princ ::
    (t (format stream "~A" (basic-ksi$name ksi)))))

;;; --------------------------------------------------------------------------

(defun BASIC-KSI-DELETION-EVENT-HANDLER (ksi)

  "BASIC-KSI-DELETION-EVENT-HANDLER ksi

Removes a KSI from its queue if it is deleted."

  (remove-from-queue ksi))

;;; --------------------------------------------------------------------------
;;;
;;; Basic-unit Queue Management Functions :::
;;;
;;; --------------------------------------------------------------------------

(defmacro INITIAL-BASIC-UNIT-QUEUE-UNIT (queue-header)

  "INITIAL-BASIC-UNIT-QUEUE-UNIT queue-header

This macro returns the first unit in the queue headed by QUEUE-HEADER or
nil if the queue is empty.  The queue itself is not changed."

  `(basic-unit-queue-unit$next-basic-queue-unit ,queue-header))

;;; --------------------------------------------------------------------------

(defmacro BUILD-BASIC-UNIT-QUEUE (queue)

  "BUILD-BASIC-UNIT-QUEUE queue

This macro creates a doubly-linked list header queue-unit unit pointing to
itself as both next and previous KSI and assigns it to the symbol QUEUE."

  `(let ((queue-header (make-basic-unit-queue-unit)))
     (setf ,queue queue-header)
     (linkf (basic-unit-queue-unit$next-basic-queue-unit queue-header)
            queue-header)))

;;; --------------------------------------------------------------------------

(defun INSERT-ON-BASIC-UNIT-QUEUE (queue-header basic-unit-queue-unit key
                                   &OPTIONAL (test '>))

  "INSERT-ON-QUEUE queue-header basic-unit-queue-unit key [test]

This function inserts BASIC-UNIT-QUEUE-UNIT at the proper place in the
doubly-linked queue headed by QUEUE-HEADER based on the field selected by
KEY.  TEST is the comparison function used to determine where to insert the 
unit (e.g., > would yield a numerically decreasing queue)."

  (let ((pointer (initial-basic-unit-queue-unit queue-header))
        (test-value (funcall key basic-unit-queue-unit)))

    ;; Find the proper position ::
    (until (or (eq pointer queue-header)             ; We've reached the end.
               (funcall test test-value
                        (funcall key pointer)))                ; We're here!
       (setf pointer (basic-unit-queue-unit$next-basic-queue-unit pointer)))

    ;; Do the insertion (order is crucial) ::
    (let ((previous-basic-queue-unit
            (basic-unit-queue-unit$previous-basic-queue-unit pointer)))
      (unlinkf-all (basic-unit-queue-unit$previous-basic-queue-unit pointer))
      (linkf (basic-unit-queue-unit$previous-basic-queue-unit
               basic-unit-queue-unit)
             previous-basic-queue-unit))
    (linkf (basic-unit-queue-unit$next-basic-queue-unit
             basic-unit-queue-unit)
           pointer)))

;;; --------------------------------------------------------------------------

(defun INSERT-ON-BASIC-UNIT-QUEUE-END (queue-header basic-unit-queue-unit)
  
  "INSERT-ON-BASIC-UNIT-QUEUE-END queue-header basic-unit-queue-unit 

This function inserts BASIC-UNIT-QUEUE-UNIT at the end of the 
doubly-linked queue headed by QUEUE-HEADER."
  
  ;; Do the insertion (order is crucial) ::
  (let ((previous-basic-queue-unit
          (basic-unit-queue-unit$previous-basic-queue-unit queue-header)))
    (unlinkf-all (basic-unit-queue-unit$previous-basic-queue-unit
                   queue-header))
    (linkf (basic-unit-queue-unit$previous-basic-queue-unit
             basic-unit-queue-unit)
           previous-basic-queue-unit))
  (linkf (basic-unit-queue-unit$next-basic-queue-unit
           basic-unit-queue-unit)
         queue-header))

;;; --------------------------------------------------------------------------

(defun REMOVE-FROM-BASIC-UNIT-QUEUE (basic-unit-queue-unit)

  "REMOVE-FROM-BASIC-UNIT-QUEUE basic-unit-queue-unit 

This function removes BASIC-UNIT-QUEUE-UNIT from the doubly-linked queue
on which it resides."

  (let ((previous-basic-queue-unit
          (basic-unit-queue-unit$previous-basic-queue-unit
            basic-unit-queue-unit))
        (next-basic-queue-unit
          (basic-unit-queue-unit$next-basic-queue-unit
            basic-unit-queue-unit)))
    (unlinkf-all (basic-unit-queue-unit$next-basic-queue-unit
                   previous-basic-queue-unit))
    (unlinkf-all (basic-unit-queue-unit$previous-basic-queue-unit
                   next-basic-queue-unit))
    (linkf (basic-unit-queue-unit$next-basic-queue-unit
             previous-basic-queue-unit)
           next-basic-queue-unit)))

;;; --------------------------------------------------------------------------

(defun ON-BASIC-UNIT-QUEUE-P (basic-unit-queue-unit)

  "ON-BASIC-UNIT-QUEUE-P basic-unit-queue-unit 

This function returns a non-nil value if QUEUE-UNIT is not on a queue.
Otherwise, nil is returned."

  (basic-unit-queue-unit$previous-basic-queue-unit basic-unit-queue-unit))

;;; --------------------------------------------------------------------------

(defun MAP-BASIC-UNIT-QUEUE (function queue-header)

  "MAP-QUEUE function queue-header

This function applys FUNCTION to each unit in the basic-unit queue headed
by QUEUE-HEADER in queue order."

  (let ((pointer (initial-basic-unit-queue-unit queue-header)))
    (until (eq pointer queue-header)                 ; We've reached the end.
      (funcall function pointer)                     ; Apply the function.
      (setf pointer (basic-unit-queue-unit$next-basic-queue-unit pointer)))))

;;; --------------------------------------------------------------------------
;;;
;;; Default Event Printers :::
;;;
;;; --------------------------------------------------------------------------

(defmacro DEFEVENT-PRINTER (event-symbol args
                                 documentation-string
                                 &KEY
                                 (ignored-args nil)
                                 (predicate nil)
                                 (title-string nil)
                                 (print-spec nil))

  "DEFEVENT-PRINTER event-name arguments documentation-string
                    &KEY :IGNORED-ARGS :PREDICATE :TITLE-STRING :PRINT-SPEC

Defines a print format for event EVENT-NAME to the simple shell.  In order
to be printed, EVENT-NAME (a keyword) must be a member of the global symbol
*event-print-types*.  A list of the defined event-name keywords is contained
in the global symbol *legal-event-print-types*."

  (let ((keyword-event-name (intern (string event-symbol) "KEYWORD")))
    (check-type event-symbol symbol)
    (unless title-string
      (let* ((event-name (symbol-name event-symbol))
             (string-position (search "-EVENT" event-name
                                      :FROM-END t
                                      :TEST #'char=)))
        (unless string-position
          (error "Event-symbol ~S does not contain ``-EVENT''~@
                and no title-string is supplied."
                 event-symbol))
        (setf title-string (subseq event-name 0 string-position))))
    `(progn
       (eval-when (compile load eval)
         (pushnew ',keyword-event-name *event-print-types*)
         (pushnew ',keyword-event-name *legal-event-print-types*))
       (defun ,event-symbol ,args
         ,documentation-string
         ,@(when ignored-args
             `((declare (ignore ,@ignored-args))))
         ,@(when print-spec
             `((when (and
                       (member ',keyword-event-name *event-print-types*
                               :TEST #'eq)
                       ,@(when predicate
                           `(,predicate)))
                 (format t "~&~21,,,'-@<~A~1,1@T~>-->~1,1@T~@?"
                         ,title-string
                         ,(first print-spec)
                         ,@(rest print-spec)))))
         ,(first args)))))

;;; ---------------------------------------------------------------------------

(defevent-printer QUIESCENCE-EVENT-EVENT-PRINTER ()

  "QUIESCENCE-EVENT-EVENT-PRINTER nil

This print event is signalled whenever the scheduling queue contains no
executable KS Instances."

  :PRINT-SPEC ("Scheduling Queue"))

;;; ---------------------------------------------------------------------------

(defevent-printer INVOKED-PRECONDITION-EVENT-PRINTER (ks stimulus-unit-or-nil)

  "INVOKED-PRECONDITION-EVENT-PRINTER ks stimulus-unit-or-nil

This print event is signalled whenever a knowledge source precondition is
invoked."

  :PRINT-SPEC ("~A ~@[~A~]"
               (basic-ks$name ks)
               (unless (null stimulus-unit-or-nil)
                 (basic-unit$name stimulus-unit-or-nil))))

;;; ---------------------------------------------------------------------------

(defevent-printer KS-INSTANTIATION-EVENT-PRINTER (ksi)

   "KS-INSTANTIATION-EVENT-PRINTER ksi

This print event is signalled whenever a knowledge source instance (KSI)
is created."

  :PRINT-SPEC ("~A ~A ~A {~D}"
               (basic-ksi$name ksi)
               (basic-ks$name (basic-ksi$ks ksi))
               (basic-ksi$stimulus-units ksi)
               (basic-ksi$rating ksi)))

;;; ---------------------------------------------------------------------------

(defevent-printer KS-INVOCATION-EVENT-PRINTER (ksi)

   "KS-INVOCATION-EVENT-PRINTER

This event is signalled whenever a knowledge source instance (KSI)
is selected for execution from the scheduling queue."

  :PRINT-SPEC ("~A ~A ~A {~D}"
               (basic-ksi$name ksi)
               (basic-ks$name (basic-ksi$ks ksi))
               (basic-ksi$stimulus-units ksi)
               (basic-ksi$rating ksi))) 

;;; ---------------------------------------------------------------------------
;;;
;;; EVENT BUFFERING FUNCTIONS ---
;;;     (for inclusion in unit definition event handlers) :::
;;;
;;; ---------------------------------------------------------------------------

(defun UNIT-CREATION-EVENT-HANDLER (unit)

  "UNIT-CREATION-EVENT-HANDLER unit

This shell function handles unit creation events.  The event description is
saved on a list until the currently executing KS completes."

  (push `(handle-creation-event ,unit)
        *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------

(defun UNIT-DELETION-EVENT-HANDLER (unit)

  "UNIT-DELETION-EVENT-HANDLER unit

This shell function handles unit creation events.  The event description is
saved on a list until the currently executing KS completes."

  (push `(handle-deletion-event ,unit)
        *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------

(defun ADD-UNIT-TO-SPACE-EVENT-HANDLER (unit path)

  "ADD-UNIT-TO-SPACE-EVENT-HANDLER unit path

This shell function handles unit/space addition events.  The event description 
is saved on a list until the currently executing KS completes."

  (push `(handle-add-unit-to-space-event ,unit ,path)
        *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------

(defun REMOVE-UNIT-FROM-SPACE-EVENT-HANDLER (unit path)

  "REMOVE-UNIT-FROM-SPACE-EVENT-HANDLER unit path

This shell function handles unit/space removal events.  The event description 
is saved on a list until the currently executing KS completes."

  (push `(handle-remove-unit-from-space-event ,unit ,path)
        *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------

(defun SLOT-ACCESS-EVENT-HANDLER (unit slot current-value)

  "SLOT-ACCESS-EVENT-HANDLER unit slot current-value

This shell function handles unit slot access events.  The event description 
is saved on a list until the currently executing KS completes."

  (declare (ignore current-value))
  (push `(handle-slot-access-event ,unit ,slot)
        *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------

(defun LINK-ACCESS-EVENT-HANDLER (unit link current-links)

  "LINK-ACCESS-EVENT-HANDLER unit link current-links

This shell function handles unit link access events.  The event description 
is saved on a list until the currently executing KS completes."

  (declare (ignore current-links))
  (push `(handle-link-access-event ,unit ,link) *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------

(defun SLOT-UPDATE-EVENT-HANDLER (unit slot current-value previous-value)

  "SLOT-UPDATE-EVENT-HANDLER unit slot current-value previous-value

This shell function handles unit slot update events.  The event description 
is saved on a list until the currently executing KS completes."

  (declare (ignore current-value previous-value))
  (push `(handle-slot-update-event ,unit ,slot) *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------

(defun LINK-UPDATE-EVENT-HANDLER (unit link current-links added-links)

  "LINK-UPDATE-EVENT-HANDLER unit link current-links added-links

This shell function handles unit link update events.  The event description 
is saved on a list until the currently executing KS completes."

  (declare (ignore current-links added-links))
  (push `(handle-link-update-event ,unit ,link) *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------

(defun UNLINK-EVENT-HANDLER (unit link current-links removed-links)

  "UNLINK-EVENT-HANDLER unit link current-links removed-links

This shell function handles unit unlink events.  The event description 
is saved on a list until the currently executing KS completes."

  (declare (ignore current-links removed-links))
  (push `(handle-unlink-event ,unit ,link) *stimulus-event-buffer*))

;;; ---------------------------------------------------------------------------
;;;
;;; EVENT HANDLERS (called by FIND-AND-INVOKE-PRECONDITION-FUNCTIONS) ::
;;;
;;; ---------------------------------------------------------------------------

(defun UNIT-EVENT-KS-RETRIEVER (unit event)

  (find-units '(basic-ks :PLUS-SUBTYPES)
              *kss-space-path*
              `(:AND
                 (:PATTERN-OBJECT (:INDEX-TYPE   event-types
                                   :INDEX-OBJECT (,event))
                  :MATCH (:COUNT 1))
                 (:PATTERN-OBJECT (:INDEX-TYPE   unit-types
                                   :INDEX-OBJECT (,(type-of unit)))
                  :MATCH (:COUNT 1)))
              :FILTER-AFTER
              #'(lambda (ks)
                  (some #'(lambda (trigger-condition)
                            (and (eq (first trigger-condition) event)
                                 (some #'(lambda (specification)
                                           (member (type-of unit) specification))
                                       (rest trigger-condition))))
                        (basic-ks$trigger-conditions ks)))))

;;; ---------------------------------------------------------------------------

(defun HANDLE-CREATION-EVENT (unit)

  "HANDLE-CREATION-EVENT unit

Returns KSs that might be interested in UNIT's creation."

  (unit-event-ks-retriever unit :unit-creation))

;;; ---------------------------------------------------------------------------

(defun HANDLE-DELETION-EVENT (unit)

  "HANDLE-DELETION-EVENT unit

Returns KSs that might be interested in UNIT's deletion."

  (unit-event-ks-retriever unit :unit-deletion))

;;; ---------------------------------------------------------------------------

(defun UNIT/SPACE-EVENT-KS-RETRIEVER (unit path event)

  (find-units '(basic-ks :PLUS-SUBTYPES)
              *kss-space-path*
              `(:AND
                 (:PATTERN-OBJECT (:INDEX-TYPE   event-types
                                   :INDEX-OBJECT (,event))
                  :MATCH (:COUNT 1))
                 (:PATTERN-OBJECT (:INDEX-TYPE   unit-types
                                   :INDEX-OBJECT (,(type-of unit)))
                  :MATCH (:COUNT 1))
                 (:PATTERN-OBJECT (:INDEX-TYPE   paths
                                   :INDEX-OBJECT (,path))
                  :MATCH (:COUNT 1)))
              :FILTER-AFTER
              #'(lambda (ks)
                  (some #'(lambda (trigger-condition)
                            (and (eq (first trigger-condition) event)
                                 (some #'(lambda (specification)
                                           (and (member (type-of unit)
                                                        (first specification))
                                                (member path
                                                        (second specification))))
                                       (rest trigger-condition))))
                        (basic-ks$trigger-conditions ks)))))

;;; ---------------------------------------------------------------------------

(defun HANDLE-ADD-UNIT-TO-SPACE-EVENT (unit path)

  "HANDLE-ADD-UNIT-TO-SPACE-EVENT unit path

Returns KSs that might be interested in the addition of UNIT to PATH."

  (unit/space-event-ks-retriever unit path :unit-addition-to-space)) 

;;; ---------------------------------------------------------------------------

(defun HANDLE-REMOVE-UNIT-FROM-SPACE-EVENT (unit path)

  "HANDLE-REMOVE-UNIT-FROM-SPACE-EVENT unit path

Returns KSs that might be interested in the removal of UNIT from PATH."

  (unit/space-event-ks-retriever unit path :unit-removal-from-space)) 

;;; ---------------------------------------------------------------------------

(defun SLOT/LINK-EVENT-KS-RETRIEVER (unit link/slot event)

  (find-units '(basic-ks :PLUS-SUBTYPES)
              *kss-space-path*
              `(:AND
                 (:PATTERN-OBJECT (:INDEX-TYPE   event-types
                                   :INDEX-OBJECT (,event))
                  :MATCH (:COUNT 1))
                 (:PATTERN-OBJECT (:INDEX-TYPE   unit-types
                                   :INDEX-OBJECT (,(type-of unit)))
                  :MATCH (:COUNT 1))
                 (:PATTERN-OBJECT (:INDEX-TYPE   slot/link-names
                                   :INDEX-OBJECT (,link/slot))
                  :MATCH (:COUNT 1)))
              :FILTER-AFTER
              #'(lambda (ks)
                  (some #'(lambda (trigger-condition)
                            (and (eq (first trigger-condition) event)
                                 (some #'(lambda (specification)
                                           (and (member (type-of unit)
                                                        (first specification))
                                                (member link/slot
                                                        (second specification))))
                                       (rest trigger-condition))))
                        (basic-ks$trigger-conditions ks)))))


;;; ---------------------------------------------------------------------------

(defun HANDLE-SLOT-ACCESS-EVENT (unit slot)

  "HANDLE-SLOT-ACCESS-EVENT unit slot

Returns KSs that might be interested in the access of SLOT in UNIT."

  (slot/link-event-ks-retriever unit slot :slot-access))

;;; ---------------------------------------------------------------------------

(defun HANDLE-LINK-ACCESS-EVENT (unit link)

  "HANDLE-LINK-ACCESS-EVENT unit link

Returns KSs that might be interested in the access of LINK in UNIT."

  (slot/link-event-ks-retriever unit link :link-access)) 

;;; ---------------------------------------------------------------------------

(defun HANDLE-SLOT-UPDATE-EVENT (unit slot)

  "HANDLE-SLOT-UPDATE-EVENT unit slot

Returns KSs that might be interested in the access of SLOT in UNIT."

  (slot/link-event-ks-retriever unit slot :slot-update)) 

;;; ---------------------------------------------------------------------------

(defun HANDLE-LINK-UPDATE-EVENT (unit link)

  "HANDLE-LINK-UPDATE-EVENT unit link

Returns KSs that might be interested in the update of LINK in UNIT."

  (slot/link-event-ks-retriever unit link :link-update)) 

;;; ---------------------------------------------------------------------------

(defun HANDLE-UNLINK-EVENT (unit link)

  "HANDLE-UNLINK-EVENT unit link

Returns KSs that might be interested in the unlinking of LINK in UNIT."

  (slot/link-event-ks-retriever unit link :unlink)) 

;;; ---------------------------------------------------------------------------

(defun HANDLE-QUEUE-QUIESCENCE-EVENT ()

  "HANDLE-QUEUE-QUIESCENCE-EVENT

Returns KSs that might be interested in queue quiescence."

  (find-units '(basic-ks :PLUS-SUBTYPES)
              *kss-space-path*
              `(:AND
                 (:PATTERN-OBJECT (:INDEX-TYPE   event-types
                                   :INDEX-OBJECT (:queue-quiescence))
                  :MATCH (:COUNT 1)))))

;;; --------------------------------------------------------------------------
;;;
;;; Main Control Shell Functions/Macros :::
;;;
;;; --------------------------------------------------------------------------

(defmacro DEFINE-KS (name &KEY trigger-conditions
                          precondition-function ks-function)

  "DEFINE-KS name &KEY trigger-conditions
                       precondition-function ks-function

Notifies the control shell of the existence of KS NAME.  NAME must be a
symbol naming the KS.

The legal trigger conditions are:
  (:UNIT-CREATION unit-types)
  (:UNIT-DELETION unit-types)
  (:UNIT-ADDITION-TO-SPACE {(unit-types path-specificiations)}+)
  (:UNIT-REMOVAL-FROM-SPACE {(unit-types path-specificiations)}+)
  (:SLOT-ACCESS {(unit-types slot-names)}+)
  (:SLOT-UPDATE {(unit-types slot-names)}+)
  (:LINK-ACCESS {(unit-types link-names)}+)
  (:LINK-UPDATE {(unit-types link-names)}+)
  (:UNLINK {(unit-types link-names)}+)
  (:QUEUE-QUIESCENCE).

KS-FUNCTION and PRECONDITION-FUNCTION must be valid arguments for
FUNCALL.  (PRECONDITION-FUNCTION can be nil for the initial KS)."

  ;; Verify arguments ::
  (check-type name symbol)
  (when (null ks-function)
    (error "A KS-FUNCTION must be supplied to DEFINE-KS."))
;  (when (and (null precondition-function)
;             (not (null trigger-conditions)))
;    (error "A TRIGGER-CONDITIONS argument cannot be specified without ~
;            also supplying a PRECONDITION-FUNCTION argument."))

  (let ((event-types '())
        (unit-types '())
        (slot/link-names '())
        (paths '()))
    (labels

      ;; Helper to check the number of values given to trigger-condition
      ;; parameters ::
      ((value-check (value min max name &OPTIONAL (form value))
         (let ((length (length value)))
           (when (< length min)
             (error "Too few values were supplied to ~A.~
                     ~%   ~S ~:[are~;is~] required, ~S ~:[were~;was~] ~
                     supplied in:~
                     ~%   ~S."
                    name min (= 1 min) length length form))
           (when (and max (> length max))
             (error "Too many values were supplied to ~A.~
                     ~%   ~S ~:[are~;is~] required, ~S ~:[were~;was~] ~
                     supplied in:~
                     ~%   ~S."
                    name max (= 1 max) length length form))))

       ;; Helper to process individual trigger condition.  Since trigger
       ;; conditions containing bb/space-paths must be converted to
       ;; MAKE-PATH form, this function returns the (possibly modified)
       ;; trigger condition for all event types.
       (process-trigger-condition (trigger-condition)
         (let ((event-type (first trigger-condition)))
           (case event-type

             ;;
             ;; Unit Creation and Deletion Triggers ::
             ;;
             ((:unit-creation :unit-deletion)
              (value-check (rest trigger-condition) 1 1
                           ':trigger-condition trigger-condition)
              (pushnew event-type event-types :TEST 'eq)
              (mapc #'(lambda (unit-type)
                        (check-type unit-type symbol)
                        (pushnew unit-type unit-types :TEST 'eq))
                    (gbb::assure-list (second trigger-condition)))
              `',trigger-condition)             ; Return it unchanged.
             
             ;;
             ;; Unit Addition and Removal Triggers ::
             ;;
             ((:unit-addition-to-space :unit-removal-from-space)
              (value-check (rest trigger-condition) 1 nil
                           ':trigger-condition trigger-condition)
              (pushnew event-type event-types :TEST 'eq)
              (check-type (rest trigger-condition) cons)
              (let ((subconditions
                      (mapcar
                        #'(lambda (subcondition)
                            (value-check subcondition 2 2
                                         "unit-type/path specification")
                            (let ((unit-types-spec
                                    (gbb::assure-list (first subcondition)))
                                  (paths-spec (second subcondition)))
                              (dolist (unit-type unit-types-spec)
                                (check-type unit-type symbol))
                              (setf unit-types
                                  (union unit-types-spec unit-types
                                         :TEST 'eq))
                              (check-type paths-spec cons)
                              (setf paths (union paths-spec paths
                                                 :TEST 'equal))
                              `(list ',unit-types-spec
                                     (make-paths :PATHS ',paths-spec))))
                        (rest trigger-condition))))

                ;; Return modified trigger-condition ::
                `(list ',(first trigger-condition) ,@subconditions)))

             ;;
             ;; Slot and Link Access, Update, and Unlink Triggers ::
             ;;
             ((:slot-access :slot-update :link-access :link-update :unlink)
              (pushnew event-type event-types :TEST 'eq)
              (value-check (rest trigger-condition) 1 nil
                           ':trigger-condition trigger-condition)
              (dolist (subcondition (rest trigger-condition))
                (value-check subcondition 2 2
                             "unit-type/path specification")
                (let ((unit-types-spec
                        (gbb::assure-list (first subcondition)))
                      (slots/links-spec
                        (gbb::assure-list (second subcondition))))
                  (dolist (unit-type unit-types-spec)
                    (check-type unit-type symbol))
                  (dolist (slot/link-name slots/links-spec)
                    (check-type slot/link-name symbol))
                  (setf unit-types
                        (union unit-types-spec unit-types
                               :TEST 'eq))
                  (setf slot/link-names
                        (union slots/links-spec slot/link-names
                               :TEST 'eq))))
              `',trigger-condition)             ; Return it unchanged.
             
             ;;
             ;; Queue Quiescence Trigger ::
             ;;
             (:queue-quiescence
              (value-check (rest trigger-condition) 0 0
                           ':trigger-condition trigger-condition)
              (pushnew event-type event-types :TEST 'eq)
              `',trigger-condition)             ; Return it unchanged.
             
             (otherwise
              (error "Illegal trigger condition event name: ~S." event-type))))))

      (check-type trigger-conditions list)
      (setf trigger-conditions
            (mapcar #'process-trigger-condition trigger-conditions)))

  ;; Define the KS ::
  `(progn
     (pushnew-acons
       *ks-definition-forms*
       ',name
       '(make-ks :NAME ',name
                 :KS-FUNCTION ,ks-function
                 :PRECONDITION-FUNCTION ,precondition-function
                 :TRIGGER-CONDITIONS (list ,@trigger-conditions)
                 :EVENT-TYPES ',event-types
                 :UNIT-TYPES ',unit-types
                 :SLOT/LINK-NAMES ',slot/link-names
                 :PATHS (append ,@(mapcar #'(lambda (path)
                                              `(make-paths :PATHS '(,path)))
                                          paths)))
       :TEST #'eq)
     ',name)))

;;; --------------------------------------------------------------------------

(defun INSTANTIATE-SIMPLE-CONTROL-SHELL (&REST args)

  "INSTANTIATE-SIMPLE-CONTROL-SHELL {descriptions}* &KEY (mode :ask)

This function builds the control data structures for the simple control
shell.  It should be run in place of INSTANTIATE-BLACKBOARD-DATABASE to
create the internal structures and storage for the blackboard database.
Each element of DESCRIPTIONS (called a replication description)
describes a blackboard hierarchy to be created.  It may be a symbol or a
more complicated list.  The syntax of of each replication description
is:

  {name | (name [replication-count] [replication-description ...])}

MODE may be one of :overwrite, :append, or :ask.  :Overwrite means the
new database will clobber the old database.  :Append means that the new
database will be appended onto the previous one.  :Ask means that the
user will be prompted to choose between :overwrite and :append."

; Once DELETE-SPACE-INSTANCE is available in GBB we will use it here!
; (when (path-instantiated-p '(kss))
;    (delete-space-instance (make-paths :PATHS '(kss))))

  (apply #'instantiate-blackboard-database args)
  (setf *last-basic-ksi-number* 0)

  ;; Make sure KS and KSI units are defined.
  (unless (and (gbb::get-unit-description 'ks t)
               (gbb::get-unit-description 'ksi t))
    (error "One or both of the units KS and KSI are not defined."))

  ;; Instantiate the KSS blackboard and define the KSS ::  
  (update-space-dimension 'kss 'unit-type (cons nil (all-unit-types)))
  (update-space-dimension 'kss 'path (cons nil (all-paths)))
  (update-space-dimension 'kss 'slot/link-name
			  (adjoin nil (all-unit-slot-and-link-names)))
  (instantiate-blackboard-database 'kss 'ksis :MODE :append)
  (setf *kss-space-path* (make-paths :PATHS '(kss)))
  (dolist (ks-defining-form *ks-definition-forms*)
    (eval (rest ks-defining-form))))

;;; --------------------------------------------------------------------------

(defun all-unit-slot-and-link-names ()

  "ALL-UNIT-SLOT-AND-LINK-NAMES ()

   Returns a list of all the slot and link names for all currently defined
   units."

  (let ((result (list nil)))
    (dolist (unit-type (all-unit-types))
      (dolist (slot (unit-slot-names unit-type))
	(pushnew slot result :test #'eq))
      (dolist (link (unit-link-names unit-type))
	(pushnew link result :test #'eq)))
    result))

;;; --------------------------------------------------------------------------

(defun RESET-SIMPLE-SHELL ()

  "RESET-SIMPLE-SHELL nil

Resets the simple-shell (but not GBB).  This destroys all KS definitions from
the control shell and clears the blackboard database."

  (setf *ks-definition-forms* '())
;; Until the changes are available ::
  (clear-blackboard-database t)
; (delete-space-instance 'kss)
  )

;;; --------------------------------------------------------------------------

(defun SIMPLE-CONTROL-SHELL (initial-ks-name)

  "SIMPLE-CONTROL-SHELL initial-ks-name

This function invokes the control shell beginning with an instance of
INITIAL-KS.  INITIAL-KS is invoked with NIL as its stimulus-units.
Execution continues until either:
  1. the scheduling-queue is empty;
  2. no pending KSI on the scheduling queue is rated higher than
     *minimum-ks-execution-rating*;
  3. a knowledge-source returns the value :STOP."

  (build-queue *scheduling-queue*)
  (build-queue *executed-ksis*)
  (let* ((initial-ks-types
           (gbb::expand-unit-type-list 'basic-ks))
         (initial-ks
           ;; Note that we cannot have two KSs of the same name in
           ;; the simple shell due to the *KS-DEFINITION-FORMS* alist.
           (some #'(lambda (ks-type)
                     (find-unit-by-name initial-ks-name ks-type))
                 initial-ks-types)))
    (unless initial-ks
      (error "KS ~S was never defined." initial-ks-name))
    (format t "~2&~60,,,'=<~>")
    (insert-on-queue *scheduling-queue*
                     (funcall (basic-ks$ksi-creation-function initial-ks) 
                            :KS initial-ks
                            :RATING most-positive-fixnum)
                     #'basic-ksi$rating)
    (simple-shell-cycle-loop)))

;;; ---------------------------------------------------------------------------

(defun SIMPLE-SHELL-CYCLE-LOOP ()

  "SIMPLE-SHELL-CYCLE-LOOP nil

Implements the scheduling-execution cycle loop.  Execution continues until
either:
  1. the scheduling-queue is empty;
  2. no pending KSI on the scheduling queue is rated higher than
     *minimum-ks-execution-rating*;
  3. a knowledge-source returns the value :STOP."

  (let* ((ksi (initial-queue-unit *scheduling-queue*))
         (quiescence-handled? nil)
         (ksi-counter 0)
         (*stimulus-event-buffer* nil))

    (flet ((stop-message (msg &rest args)
             (format t "~&~60,,,'=<~>")
             (format t "~2&***** ~? *****~%" msg args)))

      (loop

        (cond
          ;; Quiescence condition ::
          ((or (eq ksi *scheduling-queue*)                   ; Queue is empty.
               (< (basic-ksi$rating ksi) *minimum-ks-execution-rating*))
           (cond ((and quiescence-handled?
                       (not (when *simple-shell-exit-hook*
                              (neq (funcall *simple-shell-exit-hook*) :STOP))))
                  (stop-message "The Scheduling Queue has no Executable KS Instances")
                  (return))
                 (t (setf quiescence-handled? t)
                    (quiescence-event-event-printer)
                    (find-and-invoke-precondition-functions
                      '(handle-queue-quiescence-event)))))
          (t

           (setf quiescence-handled? nil)
           (remove-from-queue ksi)
           (insert-on-queue-end *executed-ksis* ksi)
           (format t "~&~30,30,,'-<~> KSI ~A ~30,30,,'-<~>" (incf ksi-counter 1))
           (ks-invocation-event-printer ksi)
           (let* ((*current-ksi* ksi)
                  (ks (basic-ksi$ks ksi))
                  (ks-return-value (funcall (basic-ks$ks-function ks) ks ksi)))
             (when (eq ks-return-value :STOP)
               (stop-message "Explicit STOP Returned by ~A" ksi)
               (return :STOP)))
           (when (and *simple-shell-post-ksi-execution-hook*
                      (eq (funcall *simple-shell-post-ksi-execution-hook*) :STOP))
             (stop-message "Explicit STOP Returned by *SIMPLE-SHELL-POST-KSI-EXECUTION-HOOK* function")
             (return :STOP))))

        ;; Don't just clobber the event buffer in case any preconditions
        ;; generate events.
        (let ((recent-events (nreverse *stimulus-event-buffer*)))
          (setf *stimulus-event-buffer* nil)
          (dolist (stimulus-event recent-events)
            (find-and-invoke-precondition-functions stimulus-event)))
        (when (and *simple-shell-cycle-hook*
                   (eq (funcall *simple-shell-cycle-hook*) :STOP))
          (stop-message "Explicit STOP Returned by *SIMPLE-SHELL-CYCLE-HOOK* function")
          (return :STOP))
        (setf ksi (initial-queue-unit *scheduling-queue*))))))

;;; ---------------------------------------------------------------------------

(defun FIND-AND-INVOKE-PRECONDITION-FUNCTIONS (event)

       "FIND-AND-INVOKE-PRECONDITION-FUNCTIONS event

This function looks on the KSS space to determine which KSs might be
interested in the event and runs their preconditions.  KSs whose
preconditions return a positive rating are instantated and placed on the
scheduling queue."

  (let* ((unit-or-nil (second event))
         ;; APPLY is used here to avoid evaluating the args (the rest of the list).
         (interested-kss (apply (first event) (rest event))))
    (dolist (ks interested-kss)
      (multiple-value-bind (rating response-frame)
          (progn (invoked-precondition-event-printer ks unit-or-nil)
                 (funcall (basic-ks$precondition-function ks) ks unit-or-nil))
        (check-type rating rating)
        (when (plusp rating)
          (let ((ksi (funcall (basic-ks$ksi-creation-function ks)
                              :KS ks
                              :STIMULUS-UNITS unit-or-nil
                              :RESPONSE-FRAME response-frame
                              :RATING rating)))
            (insert-on-queue *scheduling-queue* ksi #'basic-ksi$rating)))))))

;;; **************************************************************************
;;;                                End of File
;;; **************************************************************************



