;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.LOCAL]REPORT.LISP *-*
;;;; *-* Edited-By: Gallagher *-*
;;;; *-* Last-Edit: Tuesday, December 6, 1988  17:11:37 *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 308) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (0.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *               FUNCTIONS TO EXAMINE THE STATE OF GBB
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  01-27-88 File Created.  (Gallagher)
;;;  01-27-88 Moved PP-BLACKBOARD-DATABASE, DESCRIBE-SPACE and
;;;           DESCRIBE-SPACE-INSTANCE into this file to collect all the
;;;           `describe' type of functions in one file.  (Gallagher)
;;;  02-19-88 Changed PP-BLACKBOARD-DATABASE do DESCRIBE-BLACKBOARD-DATABASE.
;;;           (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'GBB :USE '(lisp umass-extended-lisp))

(export '(describe-blackboard-database
          pp-blackboard-database            ; old name
          describe-space
          describe-space-instance))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))


;;;; --------------------------------------------------------------------------
;;;;   Printing the Blackboard Database Structure
;;;; --------------------------------------------------------------------------


(defun DESCRIBE-BLACKBOARD-DATABASE (&optional (verbose t))

  "DESCRIBE-BLACKBOARD-DATABASE &optional verbose

   Print a description of the blackboard database to *standard-output*.
   If VERBOSE is true (the default) then the number of units on each
   space instance is also shown.  If VERBOSE is false then a brief
   description of the structure of the blackboard database is printed.
   This format is the same as that of the replication descriptions given
   to INSTANTIATE-BLACKBOARD-DATABASE."

  (let ((length (length *blackboard-database*)))
    (case length
	  (0 (format t "~&The blackboard database is empty.~%"))
	  (1 (format t "~&There is one blackboard tree ~
                        in the blackboard database:"))
	  (t (format t "~&There are ~d blackboard trees ~
                        in the blackboard database:" length)))

    (mapc (if verbose #'ds-bb-db-long #'ds-bb-db-short)
          *blackboard-database*)
    (fresh-line)
    nil))

;; Keep this around for compatability.
(defun PP-BLACKBOARD-DATABASE (&optional verbose)

  "PP-BLACKBOARD-DATABASE &optional verbose

   See DESCRIBE-BLACKBOARD-DATABASE."

  (describe-blackboard-database verbose))

(defun DS-BB-DB-SHORT (node &optional (left-margin 0))
  (if (= (db-node.start node) 0)
      (format t "~%~vt(~s ~d"
	      left-margin (db-node.name node) (db-node.end node))
      (format t "~%~vt(~s (~d ~d)"
	      left-margin (db-node.name node)
	      (db-node.start node) (db-node.end node)))
  (when (blackboard-p (db-node.type node))
    (dolist (child (svref (db-node.vector node) 0))
      (ds-bb-db-short child (+ left-margin 4))))
  (format t ")"))


(defun DS-BB-DB-LONG (node &optional (depth 0))

  (let ((indent (1+ (* 3 depth))))

    (labels ((print-line (node indent &optional show-index? i vi)
               (format t "~%~v@t~s" indent (db-node.name node))
               (when show-index?
                 (format t " [~d]" (or i 0)))
               (when (space-type-p (db-node.type node))
                 (print-counts (svref (db-node.vector node)
                                      (or vi 0)))))
             (print-counts (space-instance)
               (let ((totals nil)
                     (total 0))
                 (map-space-instance #'(lambda (u)
                                         (incf (getf totals (unit-type-of u) 0)))
                                     t
                                     space-instance)
                 (cond ((null totals)
                        (format t " ~20,10t      Empty"))
                       (t
                        (dolist-by-twos (type count totals)
                          (incf total count))
                        (format t " ~20,10t~5d Units: (~{~*~d ~2:*~s~*~^, ~})"
                                total totals))))))
      (cond ((null node)
             ;; Base Case
             nil)
      
            ((listp node)
             ;; List of Nodes
             (mapc #'(lambda (node) (ds-bb-db-long node depth)) node))

            ((and (= (db-node.start node) 0)
                  (= (db-node.end node) 1))
             ;; No index necessary
             (print-line node indent nil nil)
             (unless (space-type-p (db-node.type node))
               (mapc #'(lambda (node)
                         (ds-bb-db-long node (1+ depth)))
                     (svref (db-node.vector node) 0))))
      
            (t
             ;; Index required
             (do ((i (db-node.start node) (1+ i))
                  (vi 0 (1+ vi)))
                 ((>= i (db-node.end node)))
               (print-line node indent t i vi)
               (unless (space-type-p (db-node.type node))
                 (mapc #'(lambda (node)
                           (ds-bb-db-long node (1+ depth)))
                       (svref (db-node.vector node) vi))))))

      nil)))


;;;; --------------------------------------------------------------------------
;;;;   Describe-Space
;;;; --------------------------------------------------------------------------


(defun DESCRIBE-SPACE (space)

  "DESCRIBE-SPACE space

   Print information about the space specified by SPACE, which must be
   symbol which names a space.  The output is printed on *standard-output*."

  (labels ((print-space-dimension (d)
             (ecase (dimension.type d)
               (:ordered
                (format t "~a :ORDERED ~d-~d"
                        (dimension.name d)
                        (%range-start (odim.range d))
                        (%range-end (odim.range d))))
               (:enumerated
                (format t "~a :ENUMERATED ~s"
                        (dimension.name d)
                        (edim.labelset d))))))

    (check-type space symbol)
    (cond ((space-name-p space)
           (setf space (get-space space)))
          ((blackboard-name-p space)
           (error "~s is the name of a blackboard not a space." space))
          (t (error "~s is not the name of a space." space)))

    (let ((dimensions (space.dimensions space)))
      (format t "~2&Space ~s." (space.name space))
      (format t "~%~4tUnits:~18t~s" (space.units space))
      (format t "~%~4tDimensions:~18t")
      (cond ((null dimensions)
             (princ "NIL"))
            (t
             (print-space-dimension (car (space.dimensions space)))
             (dolist (d (cdr (space.dimensions space)))
               (format t "~%~18t")
               (print-space-dimension d))))
      (terpri))))


;;;; --------------------------------------------------------------------------
;;;;   Describe-Space-Instance
;;;; --------------------------------------------------------------------------


(defun DESCRIBE-SPACE-INSTANCE (path-or-paths
                                &key (stream *standard-output*)
                                     (units t)
                                     (indexes t)
                                     (show-units nil)
                                     (show-totals t)
                                     (show-empty-buckets nil))

  "DESCRIBE-SPACE-INSTANCE path-or-paths &key (stream *standard-output*)
                                           (units t)
                                           (indexes t)
                                           (show-units nil)
                                           (show-totals t)
                                           (show-empty-buckets nil)

   Display information about the space instance(s) specified by the
   argument PATH-OR-PATHS, which is a path structure or list of path
   structures.  In particular, this function shows the mappings that are
   defined for the space instance and the distribution of units within the
   buckets.  Output will be directed to STREAM.

   The keyword arguments UNITS and INDEXES restrict the display to only
   those mappings which include the specified units and indexes.  However,
   the count of units in each bucket will still include all units in the
   bucket regardless of type.  If SHOW-UNITS is true then the list of
   units will be printed rather than the count."

  (unless (eq units t)
    (setf units (expand-unit-type-list units)))
  (unless (eq indexes t)
    (setf indexes (assure-list indexes)))
  (dolist-or-atom (p path-or-paths (not (or (space-instance-p path-or-paths)
                                            (path-structure-p path-or-paths)
                                            (symbolp (first path-or-paths)))))
    (describe-space-instance-1
      (etypecase p
        (space-instance p)
        (path-structure (get-space-instance-from-path-structure p))
        (list (get-space-instance-from-path p)))
      stream
      units
      indexes
      show-units
      show-totals
      show-empty-buckets)))

(defun describe-space-instance-1 (space-instance
                                  stream
                                  units
                                  indexes
                                  show-units
                                  show-totals
                                  show-empty-buckets)

  (labels

    ((describe-mapping (unit-mapping elements)
       (let ((umi-list (unit-mapping.umi-list unit-mapping))
             n-indexes tab-to)
         (format stream "~2%~2tUnits: ~s" (unit-mapping.units unit-mapping))
         (cond ((null umi-list)
                (format stream "~%~4tNo Indexes (unstructured)   ~3d"
                        (length elements)))
               (t 
                (when show-totals
                  (print-total-units (unit-mapping.units unit-mapping)))
                (setf n-indexes (mapc-max #'(lambda (e)
                                              (length (storage-element.indexes e)))
                                          elements))
                (setf tab-to (max 24 (+ 16 (* n-indexes 8))))
                (dolist (element elements)
                  (let ((array (storage-element.array element))
                        (array-indexes (storage-element.indexes element)))
                    (when (or (eq indexes t)
                              (overlapp indexes array-indexes))
                      (describe-array-mapping
                        array array-indexes umi-list tab-to))))))))

     (print-total-units (units)
       (let ((totals (mapcar #'(lambda (u) (cons u 0)) units))
             (comma nil))
         (map-space-instance #'(lambda (u)
                                 (incf (cdr (assoc (unit-type-of u) totals :test #'eq))))
                             units
                             space-instance
                             t)
         (format stream "~%~2t~d total units~:*~[~:;: ~]" (mapc-+ #'cdr totals))
         (dolist (total totals)
           (unless (zerop (cdr total))
             (format stream "~:[~;, ~]~d ~s" comma (cdr total) (car total))
             (setf comma t)))
         (terpri stream)))

     (describe-array-mapping (array array-indexes umi-list tab-to)
       (format stream "~%~4tIndexes: (~{~a~^ ~}), ~d Buckets."
               array-indexes (array-total-size array))
       (let ((empty-bucket-count 0)
             (bucket-size 0))
         (do-array-region (subscripts (entire-array-region array))
           (setf bucket-size (length (apply #'aref array subscripts)))
           (when (and (not show-empty-buckets)
                      (not (zerop bucket-size))
                      (not (zerop empty-bucket-count)))
             (format stream "~%~8t -~d-~vt    0" empty-bucket-count tab-to)
             (setf empty-bucket-count 0))
           (when (and (zerop bucket-size)
                      (not show-empty-buckets))
             (incf empty-bucket-count))
           (when (or (not (zerop bucket-size))
                     show-empty-buckets)
             (format stream "~%~8t[")
             (dolists ((i array-indexes) (s subscripts))
               (unless (eq i (first array-indexes))
                 (format stream ", "))
               (print-bucket-ranges-from-array-subscript
                 (find i umi-list :test #'eq :key #'umi.name)
                 s
                 stream))
             (if show-units
                 (format stream "]~vt~s" tab-to (apply #'aref array subscripts))
                 (format stream "]~vt~5d" tab-to bucket-size))))
         (when (and (not show-empty-buckets)
                    (zerop bucket-size)
                    (not (zerop empty-bucket-count)))
           (format stream "~%~8t -~d-~vt    0" empty-bucket-count tab-to)))))

    (let ((space (space-instance.space space-instance)))
      (format stream "~2&~s is an instance of ~s."
              (get-path-from-space-instance space-instance)
              (space.name space))
      (format stream "~2&Mappings:")
      (dolist (entry (space-instance.data space-instance))
        (let ((unit-mapping (storage-entry.unit-mapping entry)))
          (when (or (eq units t)
                    (overlapp units (unit-mapping.units unit-mapping)))
            (describe-mapping unit-mapping
                              (storage-entry.data entry)))))
      (terpri stream))))

(defun print-bucket-ranges-from-array-subscript (umi subscript
                                                 &optional (stream *standard-output*))
  (error-when (or (< subscript 0)
                  (>= subscript (umi.array-size umi)))
     "Subscript out of range.")
  (ecase (umi.type umi)
    (:subranges
     (let* ((bucket (find subscript (umi.buckets umi)
                          :test #'<
                          :key #'(lambda (bucket)
                                   (+ (bucket.offset bucket)
                                      (bucket.count bucket)))))
            (bucket-width (bucket.width bucket))
            (bucket-start (+ (bucket.start bucket)
                             (* bucket-width
                                (- subscript (bucket.offset bucket))))))
       (format stream "~d-~d" bucket-start
               (+ bucket-start bucket-width))))
    (:groups
     (let ((table (umi.buckets umi))
           (comma? nil))
       (princ #\( stream)
       (map-index-table #'(lambda (label label-subscript)
                            (when (= label-subscript subscript)
                              (if comma?
                                  (princ #\, stream)
                                  (setf comma? t))
                              (write label :stream stream
                                     :pretty nil :escape t)))
                        table)
       (princ #\) stream)))))


;;; --------------------------------------------------------------------------
;;;                                End of File
;;; --------------------------------------------------------------------------
