;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.DISTRIBUTION]LINKS.LISP *-*
;;;; *-* Edited-By: Gallagher *-*
;;;; *-* Last-Edit: Wednesday, March 29, 1989  16:23:26 *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 416) *-*
;;;; *-* Software: TI Common Lisp System 4.105 *-*
;;;; *-* Lisp: TI Common Lisp System 4.105 (0.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                      LINK FUNCTIONS AND MACROS
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights reserved.
;;;
;;; Development of this code was partially supported by:
;;;    NSF CER grant DCR-8500332;
;;;    Donations from Texas Instruments, Inc.;
;;;    ONR URI grant N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  10-24-86 File Created.  (Gallagher)
;;;           Moved code that handles link definition, access, etc. to a
;;;           separate file (namely, this one).
;;;  10-24-86 Added ADD-LINK, DELETE-LINK, and DELETE-ALL-LINKS -- function
;;;           versions of LINKF, UNLINKF, and UNLINKF-ALL.  (Gallagher)
;;;  10-30-86 Added link event inheritance.  (Johnson)
;;;  01-19-87 Added LINKF!.  (Gallagher)
;;;  06-19-87 Changed the order of arguments to the update event functions.
;;;           This was done to make update events consistent with 
;;;           initialization events.  (Gallagher)
;;;  04-28-88 Added private link capability.  (Gallagher)
;;;  11-14-88 Corrected event argument order in DO-LINK-UPDATE-EVENTS and
;;;           DO-LINK-UNLINK-EVENTS.  (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'GBB :use '(lisp umass-extended-lisp))

(export '(check-unit-links
	  linkf
	  linkf-list
	  linkf!
	  unlinkf
	  unlinkf-list
	  unlinkf-all
          get-unit-link
          add-unit-link
          add-unit-link!
          delete-unit-link
          delete-all-links))

(use-package '(lisp umass-extended-lisp))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))

;;; ---------------------------------------------------------------------------


(defun get-link-description (link-accessor &optional no-error-p)

  "GET-LINK-DESCRIPTION link-accessor &optional no-error-p

   ..."

  (cond ((gethash link-accessor *link-description-hash-table*))
	(no-error-p nil)
	(t (error "~s does not access a link." link-accessor))))

;;; -----------------------------------------------------------------------

(defsetf get-link-description (link-accessor &optional no-error-p) (link)

  "Associate LINK-ACCESSOR (the name of a link accessor function)
   with LINK (the description of the link)."

  (declare (ignore no-error-p))
  `(progn
     (setf (gethash ,link-accessor *link-description-hash-table*) ,link)))


;;;; --------------------------------------------------------------------------
;;;;   LINKF and Associated Functions
;;;; --------------------------------------------------------------------------

;;; LINKF
;;;
;;; No setf method is provided for links -- use LINKF instead.  LINKF is
;;; like SETF.  It is the form used to update a link slot in a unit.  In
;;; addition to adding the new outgoing link, it also adds the the
;;; corresponding backlink.
;;;
;;; Linkf works as follows.  The link-accessor argument (e.g., unit1$link1)
;;; has a link description (cf.  get-link-description) which describes the
;;; link including the slot name and unit name of the inverse link.  Each
;;; link can be singular or not.  If a link is singular then the value of
;;; the slot will just be the other unit (i.e., an atom).  If it is not a
;;; singular link then the slot will contain a list of the units it is
;;; linked to.
;;;
;;; With the information from the link description the appropriate calls to
;;; setf are generated.
;;;
;;; The argument LINK-ACCESSOR to this macro is the public (external)
;;; accessor.  The internal accessor function (which is capable of modifying
;;; the slot value) is in the accessor slot of the link object.


(defmacro linkf ((link-accessor this-unit) other-unit)

  "LINKF (link-accessor this-unit) other-unit

   Add a link between `This-Unit' and `Other-Unit'.  The link that is
   set is specified by `Link-Accessor'."

  (assert (get-link-description link-accessor t)
	  (link-accessor)
	  "~s does not access a link in ~s." link-accessor this-unit)
  (let* ((link (get-link-description link-accessor))
	 (i-link (get-inverse-link link))
	 (this-unit-temp (newsym "THIS-"))
	 (other-unit-temp (newsym "OTHER-")))

    `(let ((,this-unit-temp ,this-unit)
	   (,other-unit-temp ,other-unit))

       ;; If either link is singular then make sure it's empty.
       ,.(build-singular-link-empty-check-code link this-unit-temp)
       ,.(build-singular-link-empty-check-code i-link other-unit-temp)

       ;; Set the links.
       ,.(build-add-link-pointer-code link this-unit-temp other-unit-temp)
       ,.(build-add-link-pointer-code i-link other-unit-temp this-unit-temp)

       ;; Return the last argument like SETF.
       ,other-unit-temp)))

;;; -----------------------------------------------------------------------

;;; Linkf-list is exactly like linkf (from an implementation perspective)
;;; except that there is a dolist to iterate over each element in the list
;;; of units.  References to `other-unit-temp' in the body of linkf should
;;; be references to `item-temp' in the body of the dolist in linkf-list.

(defmacro linkf-list ((link-accessor this-unit) list-of-units)

  "LINKF-LIST (link-accessor this-unit) list-of-units

   Add a link between `This-Unit' and each of the elements of
   `List-Of-Units.'  The link that is set is specified by `Link-Accessor.'"

  (assert (get-link-description link-accessor)
	  (link-accessor)
	  "~s does not access a link in ~s." link-accessor this-unit)
  (let* ((link (get-link-description link-accessor))
	 (i-link (get-inverse-link link))
	 (item-temp (newsym "ITEM-"))
	 (this-unit-temp (newsym "THIS-"))
	 (list-of-units-temp (newsym "UNIT-LIST-")))

    `(let ((,this-unit-temp ,this-unit)
	   (,list-of-units-temp ,list-of-units))

       (dolist (,item-temp ,list-of-units-temp)

	 ;; If either link is singular then make sure it's empty.
	 ,.(build-singular-link-empty-check-code link this-unit-temp)
	 ,.(build-singular-link-empty-check-code i-link item-temp)

	 ;; Set the links.
	 ,.(build-add-link-pointer-code link this-unit-temp item-temp nil)
	 ,.(build-add-link-pointer-code i-link item-temp this-unit-temp))

       ;; Run the events for this unit.
       ,.(build-link-update-event-code link this-unit-temp list-of-units-temp)

       ;; Return the last argument like SETF.
       ,list-of-units-temp)))

;;; -----------------------------------------------------------------------

;;; Linkf! is exactly like linkf except that if either link is singular
;;; and is not empty then linkf! arranges to do an unlink first.

(defmacro linkf! ((link-accessor this-unit) other-unit)

  "LINKF! (link-accessor this-unit) other-unit

   Add a link between `This-Unit' and `Other-Unit.'  The link that is
   set is specified by `Link-Accessor.'  If either end of the link is
   singular and is linked to another unit then linkf will undo that
   link."

  (assert (get-link-description link-accessor)
	  (link-accessor)
	  "~s does not access a link in ~s." link-accessor this-unit)
  (let* ((link (get-link-description link-accessor))
	 (i-link (get-inverse-link link))
	 (this-unit-temp (newsym "THIS-"))
	 (other-unit-temp (newsym "OTHER-"))
	 (temp-unit (gensym)))

    `(let ((,this-unit-temp ,this-unit)
	   (,other-unit-temp ,other-unit)
	   (,temp-unit nil))

       ;; If either link is singular then make sure it's empty.
       ,.(build-delete-link-backpointer-code link i-link this-unit-temp temp-unit)
       ,.(build-delete-link-backpointer-code i-link link other-unit-temp temp-unit)

       ;; Set the links.
       ,.(build-add-link-pointer-code link this-unit-temp other-unit-temp)
       ,.(build-add-link-pointer-code i-link other-unit-temp this-unit-temp)

       ;; Return the last argument like SETF.
       ,other-unit-temp)))

;;; -----------------------------------------------------------------------

(defmacro unlinkf ((link-accessor this-unit) other-unit)

  "UNLINKF (link-accessor this-unit) other-unit

   Remove the link specified by Link-Accessor between This-Unit
   and Other-Unit."

  (assert (get-link-description link-accessor)
	  (link-accessor)
	  "~s does not access a link in ~s." link-accessor this-unit)
  (let* ((link (get-link-description link-accessor))
	 (i-link (get-inverse-link link))
	 (this-unit-temp (gensym))
	 (other-unit-temp (gensym)))
    `(let ((,this-unit-temp ,this-unit)
	   (,other-unit-temp ,other-unit))
       ,.(build-link-membership-check-code link i-link this-unit-temp other-unit-temp)
       ,.(build-delete-link-pointer-code link this-unit-temp other-unit-temp)
       ,.(build-delete-link-pointer-code i-link other-unit-temp this-unit-temp)
       nil)))

;;; -----------------------------------------------------------------------

(defmacro unlinkf-list ((link-accessor this-unit) other-units)

  "UNLINKF-LIST (link-accessor this-unit) other-units

   Remove the link specified by Link-Accessor between This-Unit
   and each element of Other-Units."

  (assert (get-link-description link-accessor)
	  (link-accessor)
	  "~s does not access a link in ~s." link-accessor this-unit)
  (let* ((link (get-link-description link-accessor))
	 (i-link (get-inverse-link link))
	 (item-temp (gensym))
	 (this-unit-temp (gensym))
	 (other-units-temp (gensym)))
    `(let ((,this-unit-temp ,this-unit)
	   (,other-units-temp ,other-units))
       (dolist (,item-temp ,other-units-temp)
	 ,.(build-link-membership-check-code link i-link this-unit-temp item-temp)
	 ,.(build-delete-link-pointer-code link this-unit-temp item-temp nil)
	 ,.(build-delete-link-pointer-code i-link item-temp this-unit-temp))
       ;; Run the events for this unit.
       ,.(build-link-unlink-event-code link this-unit-temp other-units-temp)
       nil)))

;;; -----------------------------------------------------------------------

(defmacro unlinkf-all ((link-accessor unit))

  "UNLINKF-ALL (link-accessor unit)

   Unlink Unit from all the units in the link specified by
   Link-Accessor."

  (assert (get-link-description link-accessor)
	  (link-accessor)
	  "~s does not access a link in ~s." link-accessor unit)
  (let* ((link (get-link-description link-accessor))
	 (i-link (get-inverse-link link))
	 (real-link-accessor (link.accessor link))
	 (item (newsym "ITEM-"))
	 (unit-temp (newsym "UNIT-"))
	 (the-links (newsym "LINKS-")))
    `(let* ((,unit-temp ,unit)
	    (,the-links (,real-link-accessor ,unit-temp)))
       (when ,the-links
	 ,@(if (link.singular link)
	       `(,.(build-link-membership-check-code-1 i-link the-links unit-temp)
		 ,.(build-delete-link-pointer-code i-link the-links unit-temp))
	       `((dolist (,item ,the-links)
		   ,.(build-link-membership-check-code-1 i-link item unit-temp)
		   ,.(build-delete-link-pointer-code i-link item unit-temp))))
         (setf (,real-link-accessor ,unit-temp) nil))
       ,.(build-link-unlink-event-code link unit-temp the-links)
       nil)))


;;;; --------------------------------------------------------------------------
;;;;   Helper functions for LINKF
;;;; --------------------------------------------------------------------------

(defun get-i-link-accessor (link)

  "GET-I-LINK-ACCESSOR link

   Get the internal accessor function for the inverse link of
   LINK.  The i-link-accessor slot (of LINK) gets filled the
   first time this function is called."

  (or (link.i-link-accessor link)
      (let* ((i-link (get-inverse-link link)))
	(inverse-link-match-or-error link i-link)
	;; Don't set the inverse's accessor too because it may have
	;; been from an included unit.
	;; (setf (link.i-link-accessor i-link) (link.accessor link))
	(setf (link.i-link-accessor link) (link.accessor i-link)))))

;;; -----------------------------------------------------------------------

(defun get-inverse-link (link &optional (unit (link.i-link-unit link)))

  "GET-INVERSE-LINK link &optional (unit (link-i-link-unit link))

   Get the inverse link for LINK.  UNIT is the name of the unit on
   whose property list to look for the inverse.  If an inverse is
   not found then included units of UNIT will be searched as well."

  (let* ((description (get-unit-description unit))
	 (included-unit (unit.included description))
	 i-link)
    (setf i-link (find (link.i-link-name link)
		       (unit.links description)
		       :key #'link.name
		       :test #'eq))
    (or i-link
	(and included-unit
	     (get-inverse-link link included-unit))
	(error "There is no inverse link for ~s."
	       link))))

;;; -----------------------------------------------------------------------

(defun get-link-from-link-name (link-or-name this-unit)

  "Returns a link object for Link-Name.  If Link-Or-Name is a link object
   already then this just checks that it is a link for This-Unit.  If
   Link-Or-Name is a symbol then it looks for such a link in This-Unit's
   unit description."

  (let ((description (get-unit-description this-unit)))
    (cond ((link-p link-or-name)
	   (error-unless (eq (link.unit link-or-name) (unit.name description))
	      "~s is not a link in ~s." link-or-name description)
	   link-or-name)
	  ((and (symbolp link-or-name)
		(find link-or-name (unit.links description)
		      :test #'string= :key #'link.name)))
	  (t (error "~s is not a link in ~s." link-or-name description)))))

;;; -----------------------------------------------------------------------
;;; These build-...-code functions return a list of forms to be spliced
;;; into the various link macros.  (The list is freshly consed so it can
;;; be spliced in destructively.)  The arguments LINK and I-LINK are link
;;; objects.  THIS-UNIT and OTHER-UNIT are symbols (usually gensyms) that
;;; will be bound at execution time to unit instances.
;;; -----------------------------------------------------------------------

(defun build-link-membership-check-code (link i-link this-unit other-unit)
  "Builds code to check that the unit instance to be deleted is really
   contained in the link slot."
  `(,.(build-link-membership-check-code-1 link this-unit other-unit)
    ,.(build-link-membership-check-code-1 i-link other-unit this-unit)))

;;; -----------------------------------------------------------------------

(defun build-link-membership-check-code-1 (link this-unit other-unit)
  (let ((accessor (link.accessor link)))
    (if (link.singular link)
	`((error-unless (eq ,other-unit (,accessor ,this-unit))
	     "~s is not in the ~s link of ~s.~@
              The contents of the link is ~s."
	     (-> ,other-unit) ',(link.name link) (-> ,this-unit)
	     (-> (,accessor ,this-unit))))
	`((error-unless (member ,other-unit (,accessor ,this-unit) :test #'eq)
	     "~s is not in the ~s link of ~s.~@
              The contents of the link is ~s."
	     (-> ,other-unit) ',(link.name link) (-> ,this-unit)
	     (-> (,accessor ,this-unit)))))))
	     
;;; -----------------------------------------------------------------------

(defun build-singular-link-empty-check-code (link this-unit)
  "Builds code to check that if a link is singular then it must be empty."
  (when (link.singular link)
    (let ((accessor (link.accessor link)))
      `((error-when (,accessor ,this-unit)
	   "The ~s link in ~s already contains ~s.~@
            Singular links can only be set if they are empty."
	   ',(link.name link) (-> ,this-unit) (-> (,accessor ,this-unit)))))))

;;; -----------------------------------------------------------------------

(defun build-add-link-pointer-code (link this-unit other-unit
                                    &optional (include-events t))
  "Build code to add a pointer from this-unit to other-unit."
  (let ((accessor (link.accessor link)))
    `(,(if (link.singular link)
	   `(setf (,accessor ,this-unit) ,other-unit)
	   `(pushnew ,other-unit (,accessor ,this-unit) :test #'eq))
      ,.(when include-events
          (build-link-update-event-code link this-unit other-unit)))))

;;; -----------------------------------------------------------------------

(defun build-delete-link-pointer-code (link this-unit other-unit
				       &optional (include-events t))
  "Build code to remove the pointer from this-unit to other-unit."
  (let ((accessor (link.accessor link)))
    `(,(if (link.singular link)
	   `(setf (,accessor ,this-unit) nil)
	   `(setf (,accessor ,this-unit)
		  (delete ,other-unit (,accessor ,this-unit) :test #'eq)))
      ,.(when include-events
	  (build-link-unlink-event-code link this-unit other-unit)))))

;;; -----------------------------------------------------------------------

;;; This function builds code to delete the backpointer for a link in the
;;; case where the outgoing pointer is singular.  To illustrate the
;;; problem, assume that Unit-A is linked to Unit-B and the link on
;;; Unit-A's side is singular.  Now if Unit-A is linked to Unit-C the
;;; pointer to Unit-B will be clobbered (because the link is singular).
;;; If nothing was done there would be a pointer from Unit-B to Unit-A but
;;; no corresponding pointer from Unit-A to Unit-B.

(defun build-delete-link-backpointer-code (link i-link this-unit temp)
  "Build code to delete the backpointer of a link if necessary."
  (let ((this-link-accessor (link.accessor link)))
    (when (link.singular link)
      `((when (setf ,temp (,this-link-accessor ,this-unit))
	  ,.(build-delete-link-pointer-code i-link temp this-unit))))))

;;; -----------------------------------------------------------------------

(defun build-link-update-event-code (link this-unit other-unit)
  "Build code to run the update events associated with link."
  (let ((arguments `(,this-unit
		     ',(link.name link)
		     (,(link.accessor link) ,this-unit)
		     ,other-unit)))
    `(,@(when (link.update-events link)
	 `((when %%run-events%%
	    ,.(mapcar #'(lambda (event)
                          (apply #'build-event-call event arguments))
		      (link.update-events link))))))))

;;; -----------------------------------------------------------------------

(defun build-link-unlink-event-code (link this-unit other-unit)
  "Build code to run the unlink events associated with link."
  (let ((arguments `(,this-unit
		     ',(link.name link)
		     (,(link.accessor link) ,this-unit)
		     ,other-unit)))
    `(,@(when (link.unlink-events link)
	 `((when %%run-events%%
	    ,.(mapcar #'(lambda (event)
			  (apply #'build-event-call event arguments))
		      (link.unlink-events link))))))))


;;;; --------------------------------------------------------------------------
;;;;   Generic Link Access Functions
;;;; --------------------------------------------------------------------------

(defun get-unit-link (unit-instance link-name)

  "GET-UNIT-LINK unit-instance link-name

   Return the value of LINK-NAME in UNIT-INSTANCE.  LINK-NAME should be a
   symbol which names a link in UNIT-INSTANCE.  This function will run any
   access-events defined for that link."

  (let* ((description (get-unit-description unit-instance))
         (link (find-link link-name description))
         internal-accessor)
    (error-unless link
       "~a is not the name of a link in ~s."
       link-name unit-instance)
    (setf internal-accessor (form-symbol-in-package
                              (unit.package description)
                              (unit.internal-conc-name description)
                              (link.name link)))
    (when %%run-events%%
      (dolist (event (link.access-events link))
        (funcall event unit-instance
                 link
                 (funcall internal-accessor unit-instance))))
    (funcall internal-accessor unit-instance)))

;;; -----------------------------------------------------------------------

(defun add-unit-link (link-name this-unit other-units)

  "ADD-UNIT-LINK link-name this-unit other-units

   Add a link between THIS-UNIT and OTHER-UNITS via LINK-NAME.  THIS-UNIT
   must be a unit instance.  OTHER-UNITS may be either a single unit
   instance or a list of unit instances.  LINK-NAME must be the name of
   a link in THIS-UNIT."

  (check-type this-unit basic-unit "a unit instance")
  (let* ((link (get-link-from-link-name link-name this-unit))
	 (i-link (get-inverse-link link)))

    (if (listp other-units)
        (progn
          (dolist (other-unit other-units)
            ;; If either link is a singular link then check that it is empty.
            (do-singular-link-empty-check link this-unit)
            (do-singular-link-empty-check i-link other-unit)
            ;; Set the links.
            (do-add-link-pointer link this-unit other-unit nil)
            (do-add-link-pointer i-link other-unit this-unit))
          (do-link-update-events link this-unit other-units))
        (progn
          ;; If either link is a singular link then check that it is empty.
          (do-singular-link-empty-check link this-unit)
          (do-singular-link-empty-check i-link other-units)
          ;; Set the links.
          (do-add-link-pointer link this-unit other-units)
          (do-add-link-pointer i-link other-units this-unit)))

    nil))

;;; -----------------------------------------------------------------------

(defun add-unit-link! (link-name this-unit other-unit)

  "ADD-UNIT-LINK! link-name this-unit other-unit

   Add a link from THIS-UNIT to OTHER-UNIT and back via LINK-NAME.
   THIS-UNIT and OTHER-UNIT should be unit instances.  LINK-NAME should be
   the name of a link in THIS-UNIT.  If either end of the link is singular
   and is linked to another unit then add-unit-link! will undo that link."

  (assert (unit-instance-p this-unit) (this-unit)
          "~s is not a unit instance." this-unit)
  (assert (unit-instance-p other-unit) (other-unit)
          "~s is not a unit instance." other-unit)
  (let* ((link (get-link-from-link-name link-name this-unit))
	 (i-link (get-inverse-link link)))

    ;; If either link is a singular link then delete its backpointer.
    (do-delete-link-backpointer link i-link this-unit)
    (do-delete-link-backpointer i-link link other-unit)

    ;; Set the links.
    (do-add-link-pointer link this-unit other-unit)
    (do-add-link-pointer i-link other-unit this-unit)

    nil))

;;; -----------------------------------------------------------------------

(defun delete-unit-link (link-name this-unit other-unit)

  "DELETE-UNIT-LINK link-name this-unit other-unit

   Remove the link from This-Unit to Other-Unit and back via Link-Name.
   This-Unit and Other-Unit are unit instances.  Link-Name is the name
   of a link in This-Unit."
  ;; LINK-NAME may also be a link object. 
  
  (assert (unit-instance-p this-unit) (this-unit)
          "~s is not a unit instance." this-unit)
  (assert (unit-instance-p other-unit) (other-unit)
          "~s is not a unit instance." other-unit)
  (let* ((link (get-link-from-link-name link-name this-unit))
	 (i-link (get-inverse-link link)))
    (do-link-membership-check link i-link this-unit other-unit)
    (do-delete-link-pointer link this-unit other-unit)
    (do-delete-link-pointer i-link other-unit this-unit)
    nil))

;;; -----------------------------------------------------------------------

(defun delete-all-links (link-name this-unit)

  "DELETE-ALL-LINKS link-name this-unit

   Remove all the links between This-Unit and any other units via
   Link-Name.  This-Unit is a unit instance.  Link-Name is the name
   of a link in This-Unit."
  ;; LINK-NAME may also be a link object. 
  
  (assert (unit-instance-p this-unit) (this-unit)
          "~s is not a unit instance." this-unit)
  (let* ((link (get-link-from-link-name link-name this-unit))
	 (i-link (get-inverse-link link))
	 (link-name (link.name link))
	 (other-units (get-structure-slot this-unit link-name)))
    (cond ((null other-units))
	  ((link.singular link)
	   (do-link-membership-check link i-link this-unit other-units)
	   (do-delete-link-pointer link this-unit other-units)
	   (do-delete-link-pointer i-link other-units this-unit))
	  (t (dolist (other-unit other-units)
	       (do-link-membership-check link i-link this-unit other-unit)
	       (do-delete-link-pointer link this-unit other-unit)
	       (do-delete-link-pointer i-link other-unit this-unit))))
    nil))


;;;; --------------------------------------------------------------------------
;;;;   Helper functions for ADD-LINK, Etc.
;;;; --------------------------------------------------------------------------

;;; These do-...  functions are exactly analagous to the build-...-code
;;; functions above except that they actually go ahead and do it rather
;;; than return a list of forms to be spliced into the various link macros.
;;; (I used the prefix `do-' so that there would be no confusion.)  The
;;; arguments Link and I-Link are link objects.  This-Unit and Other-Unit
;;; are unit instances.


(defun do-link-membership-check (link i-link this-unit other-unit)
  "Check that the unit instance to be deleted is really
   contained in the link slot."
  (do-link-membership-check-1 link this-unit other-unit)
  (do-link-membership-check-1 i-link other-unit this-unit))

;;; -----------------------------------------------------------------------

(defun do-link-membership-check-1 (link this-unit other-unit)
  (let ((link-name (link.name link)))
    (error-unless
      (if (link.singular link)
	  (eq other-unit (get-structure-slot this-unit link-name))
	  (member other-unit (get-structure-slot this-unit link-name) :test #'eq))
       "~s is not in the ~s link of ~s.~%The contents of the link is ~s."
       (-> other-unit) link-name (-> this-unit)
       (-> (get-structure-slot this-unit link-name)))))
	     
;;; -----------------------------------------------------------------------

(defun do-add-link-pointer (link this-unit other-unit
                            &optional (run-events t))
  "Add a pointer from this-unit to other-unit."
  (let ((link-name (link.name link)))
    (if (link.singular link)
	(setf (get-structure-slot this-unit link-name) other-unit)
	(pushnew other-unit (get-structure-slot this-unit link-name) :test #'eq))
    (when run-events
      (do-link-update-events link this-unit other-unit))))

;;; -----------------------------------------------------------------------

(defun do-delete-link-pointer (link this-unit other-unit)
  "Remove the pointer from this-unit to other-unit."
  (let ((link-name (link.name link)))
    (if (link.singular link)
	(setf (get-structure-slot this-unit link-name) nil)
	(setf (get-structure-slot this-unit link-name)
	      (delete other-unit (get-structure-slot this-unit link-name)
		      :test #'eq)))
    (do-link-unlink-events link this-unit other-unit)))

;;; -----------------------------------------------------------------------

;;; This function deletes the backpointer for a link in the case where the
;;; outgoing pointer is singular.  To illustrate the problem, assume that
;;; Unit-A is linked to Unit-B and the link on Unit-A's side is singular.
;;; Now if Unit-A is linked to Unit-C the pointer to Unit-B will be
;;; clobbered (because the link is singular).  If nothing was done there
;;; would be a pointer from Unit-B to Unit-A but no corresponding pointer
;;; from Unit-A to Unit-B.

(defun do-delete-link-backpointer (link i-link this-unit)
  "Delete the backpointer of a link if necessary."
  (let ((link-name (link.name link))
	temp)
    (when (link.singular link)
      (when (setf temp (get-structure-slot this-unit link-name))
	(do-delete-link-pointer i-link temp this-unit)))))

;;; -----------------------------------------------------------------------

(defun do-singular-link-empty-check (link this-unit)
  "Check that if a link is singular then it must be empty."
  (when (link.singular link)
    (let ((link-name (link.name link)))
      (error-when (get-structure-slot this-unit link-name)
         "The ~s link in ~s already contains ~s.~@
          Singular links can only be set if they are empty."
         link-name (-> this-unit) (-> (get-structure-slot this-unit link-name))))))
  
;;; -----------------------------------------------------------------------

(defun do-link-update-events (link this-unit other-unit)
  "Run the update events associated with Link."
  (when %%run-events%%
    (let ((link-name (link.name link)))
      (dolist (event (link.update-events link))
	(funcall event
		 this-unit
		 link-name
		 (get-structure-slot this-unit link-name)
		 other-unit)))))

;;; -----------------------------------------------------------------------

(defun do-link-unlink-events (link this-unit other-unit)
  "Run the unlink events associated with Link."
  (when %%run-events%%
    (let ((link-name (link.name link)))
      (dolist (event (link.unlink-events link))
	(funcall event
		 this-unit
		 link-name
		 (get-structure-slot this-unit link-name)
		 other-unit)))))

;;; -----------------------------------------------------------------------

;;; Something to watch out for: The link objects associated with the
;;; accessor function (via the *link-description-hash-table*) must be
;;; the same objects as (i.e., EQ to) the links in the links field of
;;; the unit-description.  This is because get-inverse-link finds the
;;; inverse link by looking at the links field of the other unit.  If
;;; they weren't EQ then changes to one link object wouldn't be seen by
;;; the other one.

(defun make-unit-links (description)
  
  "Put the information about each link into a structure and return a
   list of them all.  The links aren't checked for consistency here.  Use
   CHECK-UNIT-LINKS to do that.  Description is the unit-description of
   the unit."

  ;; The simple case of a link looks like this:
  ;;   (subgoals (goal supergoals) :access-events (event1 event2))
  
  (let* ((unit-name (unit.name description))
	 (internal-conc-name (unit.internal-conc-name description))
         (included-links nil))

    ;; When there is an included unit, copy the link structures, updating as
    ;; necessary.  We don't need to recur through all levels of inclusion
    ;; here because the `links' slot of the unit description contains a link
    ;; descriptor for every link in the unit -- including the included
    ;; links.
    (when (unit.included description)
      (setf included-links
            (mapcar #'(lambda (link) (copy-included-link link description))
                    (unit.links (get-included-unit-description description)))))

    (nconc
      included-links
      (mapcar
        #'(lambda (link)
	    ;; Parse a single link specification.
	    (let* ((name (car link))
		   (pointer (cdr link))
		   (singular (not (null (member :singular pointer))))
		   (reflexive (not (null (member :reflexive pointer))))
		   (dimensions (second (member :dimensions pointer)))
		   i-link-name i-link-unit i-link-singular)
	      
	      ;; move past :reflexive, :singular, and :dimensions 
	      (setf pointer (member-if-not
			      #'(lambda (x)
				  (member x '(:reflexive :singular)))
			      (cdr link)))
	      (setf pointer (if (eq (first pointer) :dimensions)
				(cddr pointer)
				pointer))
	      (if reflexive
		  (setf i-link-name name
			i-link-unit unit-name
			i-link-singular singular)
		  ;; Else inverse link information must be parsed as well
		  (let ((i-link (pop pointer)))
		    (setf i-link-name (second i-link)
			  i-link-unit (first i-link)
			  i-link-singular (eq (third i-link) :singular))))
	      
              (macrolet ((error! (&body args)
                           `(error "Syntax error in link ~a, unit ~s.~%~@?"
                                   name (unit.name description) ,@args)))
                ;; Now, pointer points to a list of link options.
                (when (oddp (length pointer))
                  (error! "Odd number of options and values:~%~8t~s" link))
                (when (check-options pointer *link-option-keywords*)
                  (error! "Unknown link option keyword(s): ~s."
                          (check-options pointer *link-option-keywords*))))

	      ;; Now that I've got all the necessary information in hand
	      ;; from the link specification, create the link.
	      (make-link :name name
			 :unit unit-name
                         :inherited-p nil
                         :private (getf pointer :private)
			 :singular singular
			 :i-link-name i-link-name
			 :i-link-unit i-link-unit
			 :i-link-singular i-link-singular
			 :dimensions (assure-list dimensions)
			 :accessor (form-symbol internal-conc-name name)
			 :initialization-events
			     (filter-duplicate-event-names
			       (getf pointer :initialization-events)
			       (get-unit-events :link-initialization-events description))
			 :access-events
			     (filter-duplicate-event-names
			       (getf pointer :access-events)
			       (get-unit-events :link-access-events description))
			 :update-events
			     (filter-duplicate-event-names
			       (getf pointer :update-events)
			       (get-unit-events :link-update-events description))
			 :unlink-events
			     (filter-duplicate-event-names
			       (getf pointer :unlink-events)
			       (get-unit-events :unlink-events description)))))
	(unit.link-list description)))))
      

(defun copy-included-link (link-struct description)

  "Copy the link description from an included unit."

  (let ((new-link (copy-link link-struct)))
    (setf (link.unit new-link) (unit.name description)
          (link.inherited-p new-link) t 
          (link.initialization-events new-link) (mixin-included-events
                                                  :link-initialization-events
                                                  (link.initialization-events new-link)
                                                  description)
          (link.access-events new-link) (mixin-included-events
                                          :link-access-events
                                          (link.access-events new-link)
                                          description)                     
          (link.update-events new-link) (mixin-included-events
                                          :link-update-events
                                          (link.update-events new-link)
                                          description)   
          (link.unlink-events new-link) (mixin-included-events
                                          :unlink-events
                                          (link.unlink-events new-link)
                                          description))
    new-link))

;;; -----------------------------------------------------------------------

(defun inverse-links-match-p (link1 link2)

  "INVERSE-LINKS-MATCH-P link1 link2

   Returns true if LINK1 and LINK2 are inverses of each other
   and all their attributes match."

  (and (link-p link1) (link-p link2)
       (eq (link.unit link1) (link.i-link-unit link2))
       (eq (link.name link1) (link.i-link-name link2))
       (eq (link.singular link1) (link.i-link-singular link2))
       (eq (link.i-link-unit link1) (link.unit link2))
       (eq (link.i-link-name link1) (link.name link2))
       (eq (link.i-link-singular link1) (link.singular link2))))

;;; -----------------------------------------------------------------------

(defun inverse-link-match-or-error (link1 link2)

  "INVERSE-LINK-MATCH-OR-ERROR link1 link2

   Returns true if LINK1 and LINK2 are inverses of each other and all
   their attributes match.  If they don't match then an error is
   signalled.  LINK1 must be a valid link.  The error message is based
   on LINK1."

  (cond ((not (link-p link1))
	 (error "These links don't match:~%~10t~s and~%~10t ~s."
		link1 link2))
	((null link2)
	 (error
	   "There is no inverse link for the ~s link of ~s.~
	    ~% ~s should have a link like (~s (~s ~s)),~
	    ~% but it doesn't.~% The links it does have are: ~s."
	   (link.name link1) (link.unit link1)
	   (link.i-link-unit link1) (link.i-link-name link1)
	   (link.unit link1) (link.name link1)
	   (unit.links
	     (get-unit-description (link.i-link-unit link1)))))
	 ((not (link-p link2))
	  (error "These links don't match:~%~10t~s and~%~10t ~s."
		link1 link2))
	 ((not (inverse-links-match-p link1 link2))
	  (error
	    "The ~s link of ~s does not match it's inverse link.~
	     ~% The link is         ~s.~
	     ~% The inverse link is ~s."
	    (link.name link1) (link.unit link1) link1 link2))
	 (t t)))

;;; -----------------------------------------------------------------------

(defun check-unit-links (&optional (error-p t)
			 (units *all-units*)
			 &aux (error-count 0))

  "CHECK-UNIT-LINKS &optional (error-p t) (units *all-units*)

   Check that the link definitions between the unit types given by UNITS
   are consistent.  If no inconsistencies are found this function simply
   returns nil.  If any inconsistencies are found and ERROR-P is true
   then an error is signaled.  If ERROR-P is false, a description of
   the problem is printed and the remaining unit types are checked.
   CHECK-UNIT-LINKS returns a count of the number of errors found."

  (declare (special *all-units*))

  (macrolet ((error-or-report (string &rest args)
	       `(cond (error-p
		       (error ,string ,@args))
		      (t (when (zerop error-count)
			   (format *error-output* "~2&~70,,,'-<-~>~%"))
			 (format *error-output* ,string ,@args)
			 (format *error-output* "~&~70,,,'-<-~>~%")
			 (incf error-count)))))

    (dolist-or-atom (unit units)
     (dolist (link (unit.links (get-unit-description unit)))
      (let* ((name (link.name link))
	     (inverse-link-unit (link.i-link-unit link))
	     (inverse-link-name (link.i-link-name link))
	     (inverse-links
	       (unit.links (get-unit-description inverse-link-unit)))
	     (i-link
	       (find-if #'(lambda (item) ;; Inverse-link-p
			    (and (eq name (link.i-link-name item))
				 (eq unit (link.i-link-unit item))))
			inverse-links))
	     i-description)
	(when (unit-type-p inverse-link-unit)
	  (setf i-description
                (get-unit-description inverse-link-unit)))

	(cond
	  
	  ;; Don't check included links.
	  ((or (link.inherited-p link)
	       (link.inherited-p link)))

	  ;; Links are fine.
	  ((inverse-links-match-p link i-link))
	  
	  ;; Inverse link's unit isn't even defined.
	  ((not (unit-type-p inverse-link-unit))
	   (error-or-report
	     "The ~a link of ~s refers to ~s which isn't a unit.~@
	      The link spec in ~s is ~a."
	     name unit inverse-link-unit unit link))
	  
	  ;; No inverse link is defined in the other unit.
	  ((null i-link)
	   (error-or-report
	     "There is no inverse link for the ~a link of ~s.~@
	      ~s should have a link like~%    (~a (~a ~a)),~@
	      but it doesn't.~%The links it does have are:~{~%    ~a~}."
	     name unit inverse-link-unit
	     inverse-link-name unit name inverse-links))
	  
	  ;; An inverse link was defined but it doesn't match.
	  (t
	   (error-or-report
	     "The ~a link of ~s does not match it's inverse link.~@
	      The link is         ~a.~@
	      The inverse link is ~a."
	     name unit link i-link)))

	))))

  error-count)

;;; -----------------------------------------------------------------------

(defun check-unit-links-if-possible (description)

  "CHECK-UNIT-LINKS-IF-POSSIBLE description

   Checks all the links for DESCRIPTION if all of the other units
   are defined.  An error is signaled if any inconsistency is found."

  (let ((unit-name (unit.name description))
	(links (unit.links description)))
    (when (every #'(lambda (link)
		     (let ((i-unit (link.i-link-unit link)))
		       (or (eq i-unit unit-name)
			   (get-unit-description i-unit t))))
		 links)
      (unless (zerop (check-unit-links nil unit-name))
	(cerror "Ignore the inconsistency and go on."
                "Some links for ~s were inconsistent (see above).~@
                 Note: If you have added or deleted any links from this unit~
               ~%      you can probably disregard this error."
                unit-name)))))

;;; -----------------------------------------------------------------------

;; Note that the links slot of the unit-description has a link for *all*
;; the links --- included links as well as links directly present in the
;; unit definition.

(defun build-link-access-forms (description &optional
				(external-conc-name
				  (unit.external-conc-name description)))

  "Builds code to define the link access forms."

  (flet ((event-forms (access-events link-name internal-name unit-v)
           (when access-events
             `((when %%run-events%%
                 ,.(mapcar #'(lambda (event)
                               `(,event unit ',link-name (,internal-name ,unit-v)))
                           access-events))))))

    (let ((internal-conc-name (unit.internal-conc-name description)))
      (mapc-condcons
        #'(lambda (link)
            (let* ((link-name (link.name link))
                   (external-name (form-symbol external-conc-name link-name))
                   (internal-name (form-symbol internal-conc-name link-name))
                   (access-events (link.access-events link))
                   (dimension-list (link.dimensions link))
                   (dimension-subscripts (mapcar #'(lambda (dimension)
                                                     (cons 0 dimension))
                                                 dimension-list)))
              (cond ((and (link.inherited-p link)
                          (link.private link))
                     ;; Don't generate any access forms for private links.
                     nil)
                    (dimension-list
                     `(defun ,external-name (unit &rest dimensions)
                        ,@(event-forms access-events link-name internal-name 'unit)
                        (if dimensions
                            (apply #'aref (,internal-name unit) dimensions)
                            (let ((result nil))
                              (do-array-region (indexes ',dimension-subscripts)
                                (setf result
                                      (append
                                        result
                                        (apply #'aref (,internal-name unit) indexes))))
                              result))))
                    (t
                     `(defun ,external-name (unit)
                        ,@(event-forms access-events link-name internal-name 'unit)
                        (,internal-name unit))))))
        (unit.links description)))))

;;; -----------------------------------------------------------------------

(defun store-link-methods (description &optional
			   (external-conc-name
			     (unit.external-conc-name description)))
  "Builds code to save the link information on the property list
   of the link accessor."

  ;; The links slot of the unit description has all the links for this
  ;; unit, including the links from the included unit, so there is no
  ;; need to recur on the included unit.
  (mapc-condcons
    #'(lambda (link)
        ;; Don't save link descriptions for private links because linkf
        ;; looks for the link description in order to do its stuff.
        (unless (and (link.inherited-p link)
                     (link.private link))
          `(setf (get-link-description
                   ',(form-symbol external-conc-name (link.name link)))
                 ',link)))
    (unit.links description)))


;;; --------------------------------------------------------------------------
;;;                                End of File
;;; --------------------------------------------------------------------------
